// file: spec_win_0.cc
//
// applies a hamming window to the input data
//

// local include files
//
#include "spectrogram_plot.h"
#include "spectrogram_plot_constants.h"

// function: spec_window_hamming_cc
//
// arguments:
//
//  double* data: (input/output) sampled data
//  int_4 num_samples: (input) number of samples in the data stream
//  
// return value: a logical_1 indicating status
//
// this function applies a hamming window to the input data by the function:
//
//   y(n) = x(n) * (b - [(1 - b) * cos(2 * pi * n / (num_samples - 1))])
// 
// the center index of the input data is assumed to be time zero for the window
// the input data is overwritten.
//
logical_1 spec_window_function_cc(double* data_a, int_4 num_samples_a,
				  Spec_plot_item* spec_plot) {


  // check for null data
  //
  if (data_a == (double*)NULL) {
    return ISIP_FALSE;
  }

  // use the hamming window function
  //
  if (strcmp((char*)(spec_plot->window_option_d), (char*)"hamming") == 0) {
    
    // loop over all data
    //
    for (int_4 samp_num = 0; samp_num < num_samples_a; samp_num++) {
      
      // store the sample data
      //
      double x_n = data_a[samp_num];

      // multiply by the window function
      //
      double x_hamm_n = x_n * (double)(SPEC_HAMMING_COEFF -
				       (((double)1.0 - SPEC_HAMMING_COEFF) *
					cos(ISIP_TWOPI * (double)samp_num /
					    (double)(num_samples_a - 1))));
      
      // store the windowed data
      //
      data_a[samp_num] = x_hamm_n;
    }
  }
  
  // use the hanning window function
  //
  else if (strcmp((char*)(spec_plot->window_option_d),(char*)"hanning") == 0) {
    
    // loop over all data
    //
    for (int_4 samp_num = 0; samp_num < num_samples_a; samp_num++) {
      
      // store the sample data
      //
      double x_n = data_a[samp_num];
      
      // multiply by the window function
      //
      double x_hann_n = x_n * (double)(0.5 * (1.0 - cos((ISIP_TWOPI * (double)samp_num) / (double)(num_samples_a - 1))));
      
      // store the windowed data
      //
      data_a[samp_num] = x_hann_n;
    }
  }
    
  // use the blackman window function
  //
  else if(strcmp((char*)(spec_plot->window_option_d),(char*)"blackman") == 0) {
    
    // loop over all data
    //
    for (int_4 samp_num = 0; samp_num < num_samples_a; samp_num++) {
      
      // store the sample data
      //
      double x_n = data_a[samp_num];
      
      // multiply by the window function
      //
      double x_black_n = x_n * ((double)0.42 - (0.50 * cos((ISIP_TWOPI * (double)samp_num) / ((double)num_samples_a - 1))) + (0.08 * cos((2 * ISIP_TWOPI * (double)samp_num) / ((double)num_samples_a - 1)))); 
      
      // store the windowed data
      //
      data_a[samp_num] = x_black_n;
    }
  }
  
  // use the bartlett window function
  //
  else if(strcmp((char*)(spec_plot->window_option_d),(char*)"bartlett") == 0) {
    
    // loop over all data
    //
    for (int_4 samp_num = 0; samp_num < num_samples_a; samp_num++) {
      
      // store the sample data
      //
      double x_n = data_a[samp_num];
      
      // multiply by the window function
      //
      double x_bart_n = x_n * ((double) (1 - (2 * fabs((double) samp_num - ((double)num_samples_a - 1) /2) / ((double) num_samples_a - 1))));
      
      // store the windowed data
      //
      data_a[samp_num] = x_bart_n;
    }
  }  

  // default to rectangular (i.e. no windowing) if no window function is
  // specified or if an unsupported window is specified
  //
  else {

    // exit gracefully
    //
    return ISIP_TRUE;
  }
  
  // exit gracefully
  //
  return ISIP_TRUE;
}

