// file: spec_draw_0.cc
//

// system include files
//
#include <string.h>

// isip include files
//
#include "spectrogram_plot.h"
#include <signal.h>

#include "spectrogram_plot_constants.h"
#include <signal_constants.h>

// function spec_plot_plot_points_cc
//
// parameters:
//     Spec_plot_item* spec_plot: a spec plot widget
//     Display* display: the display to draw to
//     Drawable drawable: the drawable to plot on
//     int width: width of the plot
//     int height: height of the plot
//     int x_offset: virtual upper-left x coordinate
//     int y_offset: virtual upper-left y coordinate
//     int xcoord: starting x coord
//     int plot_width: width or region to be plotted
//
//
// returns: int indicating status
//
// this function produces a spectrogram image and plots it in spec_plot->Gimage
//
int spec_plot_draw_spectrogram_cc(Spec_plot_item* spec_plot, Display* display,
				  Drawable drawable, int_4 width, int_4 height,
				  int_4 x_offset, int_4 y_offset, int_4 xcoord,
				  int_4 plot_width) {

  // declare local variables
  //
  int_4 flen = spec_plot->fft_len_d / 2;

  // check fft limits
  //
  if (spec_plot->fft_max_d == spec_plot->fft_min_d) {
    
    // increment the max value by 1
    //
    spec_plot->fft_max_d = spec_plot->fft_max_d + (double)1.0;
  }
  
  // check number of points for outrageous conditions
  //
  if ((width <= 1) || (height <= 1)) {
    return TCL_ERROR;
  }

  // declare an array to hold data
  //
  double* v = new double[flen + 1];
  
  // loop over all of the fft data
  //
  if (spec_plot->num_fft_d > 0) {

    double k = (double) (spec_plot->contrast_d *
			 (spec_plot->num_colors_d - 1) /
			 (spec_plot->fft_max_d - spec_plot->fft_min_d));
    
    // define scale factors for the plotting
    //
    int_4 top = (int_4) ((1.0 - (spec_plot->max_freq_d /
				 (spec_plot->sf_d / 2))) * flen);
    double yscale = (double) (flen - top) / height;

    // loop over each horizontal pixel value
    //
    for (int_4 i = 0; i < spec_plot->num_fft_d; i++) {

      // determine position in the fft data buffer
      //
      int left_index = i * flen;
      int right_index = left_index + flen;
      
      // copy the fft data with brightness scaling
      //
      int_4 q = 0;
      for (int_4 j = left_index; j < right_index; j++) {
	v[q] = (double) (k * (spec_plot->fft_data_d[j] -
			      spec_plot->fft_min_d -
			      spec_plot->brightness_d));
	q++;
      }

      // loop over pixels in the vertical direction
      //
      for (int_4 j = 0; j < height; j++) {
	
	// define the color value for this pixel
	//
	int_4 c;
	
	// declare the scale factors in the vertical direction
	//
	double fy = yscale * j;
	int_4 iy  = (int_4) fy;
	double deltay = fy - (double)iy;
	
	// subsample in the y direction if necessary
	//
	if (height <= flen) {
	  c = (int_4)v[iy];
	}
	
	// interpolate in the y direction if necessary
	//
	else {
	  c = (int_4) (v[iy] + deltay * (v[iy + 1] - v[iy]));
	}
	
	if (c >= spec_plot->num_colors_d) {
	  c = spec_plot->num_colors_d - 1;
	}
	if (c < 0) {
	  c = 0;
	}
	
	XPutPixel(spec_plot->Gimage, i, height - j - 1,
		  spec_plot->colormap_d[c]->pixel);
      }
    }
  }  
  else {
    for (int_4 i = 0; i < width; i++) {
      for (int_4 j = 0; j < height; j++) {
	XPutPixel(spec_plot->Gimage, i, j, spec_plot->colormap_d[0]->pixel);
      }
    }
  }

  // exit gracefully
  //
  return(TCL_OK);
}


