// file: spec_comp_1.cc
//

// system include files
//
#include <string.h>

// isip include files
//
#include "spectrogram_plot.h"
#include <signal.h>

#include "spectrogram_plot_constants.h"
#include <signal_constants.h>

// function spec_plot_compute_spectrogram_cc
//
// parameters:
//     Spec_plot_item* spec_plot: a spec plot widget
//
// returns: int indicating status
//
// this function produces an array of points based on the speech file in
// spec_plot->file_name_d
//
int spec_plot_compute_spectrogram_cc(Spec_plot_item* spec_plot, int_4 width) {

  // define local variables
  //
  float_4 start_time = spec_plot->start_time_d;
  float_4 end_time = spec_plot->end_time_d;

  // check time boundaries
  //
  if (start_time > end_time) {
    start_time = end_time;
  }

  // define the number of pixels per second
  //
  float_4 pixels_per_sec = (float_4)width / (end_time - start_time);
  
  // check that we have a valid file name to work with
  //
  if (spec_plot->file_name_d == (char*)NULL) {
    return TCL_ERROR;
  }
  
  // declare a signal object to retrieve the signal data
  //
  Signal signal;

  // set the parameters of the signal object
  //
  if (spec_plot->swap_byte_flag_d == SPEC_TRUE) {
    
    signal.set_cc((unichar*)spec_plot->file_name_d, spec_plot->num_chans_d,
		  spec_plot->num_bytes_d, spec_plot->sf_d, ISIP_TRUE);
  }
  else {

    signal.set_cc((unichar*)spec_plot->file_name_d, spec_plot->num_chans_d,
		  spec_plot->num_bytes_d, spec_plot->sf_d, ISIP_FALSE);
  }

  // open the file for reading while checking for errors
  //
  if (signal.open_cc((unichar*) spec_plot->file_name_d, ISIP_READ_ONLY) ==
      ISIP_FALSE) {

    // return with error
    //
    return(TCL_ERROR);
  }

  // set the fft max and min values
  //
  spec_plot->fft_max_d = SPEC_INIT_FFT_MAX;
  spec_plot->fft_min_d = SPEC_INIT_FFT_MIN;

  // calculate the fft length
  //
  int_4 win_samples = (int_4)(spec_plot->win_length_d * spec_plot->sf_d);

  // loop until the fft size is larger that the window samples
  //
  while (spec_plot->fft_len_d < win_samples) {

    spec_plot->fft_len_d <<= 1;
  }
  
  // allocate memory to hold each of the fft outputs
  //
  double* fft_data_in = (double*)Tcl_Alloc(spec_plot->fft_len_d *
					   sizeof(double));
  double* fft_data_out = (double*)Tcl_Alloc(spec_plot->fft_len_d *
					    sizeof(double));
  
  // set the number of fft's computed
  //
  spec_plot->num_fft_d = width;

  // allocate memory to hold the spectrogram data. we need
  // num_fft * fft_len / 2 doubles
  //
  if (spec_plot->fft_data_d != (double*)NULL) {
    Tcl_Free((char*)(spec_plot->fft_data_d));
    spec_plot->fft_data_d = (double*)NULL;
  }
  spec_plot->fft_data_d = (double*)Tcl_Alloc(spec_plot->num_fft_d *
					     spec_plot->fft_len_d / 2 *
					     sizeof(double));
  
  // declare the number of samples read for each frame
  //
  int_4 num_samples = (int_4)0;

  // loop over each pixel in the horizontal direction
  //
  for (int_4 i = 0; i < width; i++) {

    // compute the time in seconds that correspond to the pixel location
    //
    float_4 pixel_time = i / pixels_per_sec + start_time;

    // determine the bounds of the window around the pixel location
    //
    float_4 win_start = pixel_time - (spec_plot->win_length_d / 2);
    float_4 win_end = pixel_time + (spec_plot->win_length_d / 2);

    // zero out the fft input data
    //
    memset((void*)fft_data_in, (int)0, spec_plot->fft_len_d * sizeof(double));
    
    // declare a pointer to some data
    //
    double* data = (double*)NULL;
    
    // read the frame in
    //
    if (signal.read_cc(data, win_start, win_end,
		       (long)spec_plot->display_channel_d, num_samples) ==
	ISIP_FALSE) {

      if (data != (double*)NULL) {
	delete [] data;
      }
      return TCL_ERROR;
    }

    // pre-emphasize the data if desired
    //
    if (spec_plot->pre_emph_flag_d ==  SPEC_TRUE) {

      // preemphasize the data
      //
      spec_pre_emphasize_data_cc(data, spec_plot->pre_emph_coeff_d,
				 num_samples);
    }

    // use the desired window function for the data
    //
    spec_window_function_cc(data, num_samples, spec_plot);
    
    // stuff the data into power-of-2 array for the fft computation
    //
    memcpy((void*)fft_data_in, (void*)data, num_samples * sizeof(double));
     
    // compute the fft (routine returns N magnitude points - not real and imag)
    //
    compute_fft_cc(fft_data_out, fft_data_in,
		   spec_plot->fft_len_d, spec_plot->amp_scale_d);
    
    // stuff the first N/2 points of the fft into spec_plot->fft_data_d
    //
    int_4 num_cpy = spec_plot->fft_len_d / 2;
    
    for (int_4 q = 0; q < num_cpy; q++) {
      spec_plot->fft_data_d[q + num_cpy * i] = fft_data_out[q];

      // find the max and min fft values across all frames
      //
      if (fft_data_out[q] < spec_plot->fft_min_d) {
        spec_plot->fft_min_d = fft_data_out[q];
      }
      if (fft_data_out[q] > spec_plot->fft_max_d) {
        spec_plot->fft_max_d = fft_data_out[q];
      }      
    }

    // clean up data memory
    //
    if (data != (double*)NULL) {
      delete [] data;
      data = (double*)NULL;
    }
  }
  
  // close the input file
  //
  signal.close_cc();

  // deallocate memory for fft data
  //
  if (fft_data_in != (double*)NULL) {
    Tcl_Free((char*)(fft_data_in));
    fft_data_in = (double*)NULL;
  }
  if (fft_data_out != (double*)NULL) {
    Tcl_Free((char*)(fft_data_out));
    fft_data_out = (double*)NULL;
  }

  // exit gracefully
  //
  return(TCL_OK);
}


