// file: sigpl_plot_0.cc
//

// system include files
//
#include <string.h>

// isip include files
//
#ifndef __ISIP_SIGNAL_PLOT
#include "signal_plot.h"
#endif

#ifndef __ISIP_SIGNAL_PLOT_CONSTANTS
#include "signal_plot_constants.h"
#endif

// declare a static variable for the maximum amplitude
//
// see if the global maximum amplitude needs to be found. if so, do it
//
static float_4 max_ampl;  

// function signal_plot_plot_points_cc
//
// parameters:
//     Signal_plot_item* sig_plot: a signal plot widget
//     int_4& num_pts: (output) number of points returned
//     XPoint*& points: (output) pointer to an array of points
//     int width: width of the plot
//     int height: height of the plot
//     int x_offset: virtual upper-left x coordinate
//     int y_offset: virtual upper-left y coordinate
//     Display* display: the display to draw to
//     Drawable drawable: the drawable to plot on
//
// returns: int indicating status
//
// this function produces an array of points based on the speech file in
// sig_plot->file_name_d
//
int signal_plot_plot_points_cc(Signal_plot_item* sig_plot, int_4& num_pts,
			       XPoint** points, int width, int height,
			       int x_offset, int y_offset, Display* display,
			       Drawable drawable) {

  // define local variables
  //
  int_4 npix_x = width;
  int_4 npix_y = height;
  double sample_freq = sig_plot->sf_d;
  int_4 chan = sig_plot->display_channel_d;
  int_4 num_bytes = sig_plot->num_bytes_d;
  int_4 num_chan = sig_plot->num_chans_d;
  float_4 start_time = sig_plot->start_time_d;
  float_4 end_time = sig_plot->end_time_d;
  int_4 buf_size = SIGPLOT_DEFAULT_BUFFER_SIZE;
  int_4 resolution = npix_x * SIGPLOT_DEFAULT_RES_SCALE;

  // check number of points for outrageous conditions
  //
  if ((npix_x <= 1) || (npix_y <= 1)) {
    return TCL_ERROR;
  }

  // reset the number of points
  //
  num_pts = (int_4)0;

  // open the signal file
  //
  if (sig_plot->file_name_d == (char*)NULL) {

    // return with error
    //
    return(TCL_ERROR);
  }

  // open the data file
  //
  FILE *fp_in = fopen((char*)(sig_plot->file_name_d), "r");
  
  // error if the input file does not exist
  //
  if (fp_in == NULL) {

    // return with error
    //
    return(TCL_ERROR);
  }
  
  // compute an i/o buffer size
  //
  int_4 num_samples = (int_4)((end_time - start_time) * sample_freq);
  if (num_samples*num_chan < SIGPLOT_DEFAULT_BUFFER_SIZE) {
    buf_size = num_samples;
  }
  else {
    buf_size = SIGPLOT_DEFAULT_BUFFER_SIZE;
  }

  // allocate an i/o buffer
  //
  char_1* buffer = (char_1*)Tcl_Alloc(buf_size*num_chan*num_bytes *
				      sizeof(char_1*));
    
  // go to the beginning of the input file
  //
  rewind(fp_in);

  // compute the decimation factor and adjust the buf_size
  //
  buf_size /= (int_4)4;
  int_4 decimation_factor = num_samples / resolution;
  if (decimation_factor < (int_4)1)
    {decimation_factor = (int_4)1;}
  if (buf_size % decimation_factor != 0) {
    buf_size += decimation_factor;
    buf_size /= decimation_factor;
    buf_size *= decimation_factor;
  }
  buf_size *= (int_4)4;

  // determine the maximum number of output points
  //
  int_4 out_pts = (int_4)(num_samples / decimation_factor + (int_4)1) *
    (int_4)2;
  
  // create an output buffer
  //
  float_4 out_pts_x[out_pts];
  float_4 out_pts_y[out_pts];

  // initialize counters
  //
  int_4 i_cur_time = (int_4)(start_time * sample_freq);
  int_4 i_end_time = (int_4)(end_time * sample_freq);
  int_4 i_output_time = i_cur_time + decimation_factor - (int_4)1;
  
  // seek to the starting point in the file
  //
  fseek(fp_in, i_cur_time*num_bytes*num_chan, SEEK_SET);
  
  // initialize parameters to find min/max
  //
  int_4 sample_val;
  int_4 min_val = 32767;
  int_4 min_loc = (int_4)0;
  int_4 max_val = -min_val;
  int_4 max_loc = (int_4)0;
  float_4 time_scale = 1.0/sample_freq;
  
  // main loop
  //
  while (i_cur_time < i_end_time) {
    
    // read data
    //
    int_4 num_samples = fread(buffer, num_chan*num_bytes, buf_size, fp_in);

    // determine the number of bytes read
    //
    int_4 num_bytes_read = num_samples * num_chan * num_bytes;
    
    // byte-swap the signal if necessary
    //
    if (sig_plot->swap_byte_flag_d == SIGPLOT_TRUE) {

      // set a pointer to the beginning of the buffer
      //
      uint_1* temp = (uint_1*)buffer;

      for (int_4 i = 0; i < num_bytes_read; i += 2) {
	uint_1 samp = temp[0];
	temp[0] = temp[1];
	temp[1] = samp;
	temp += 2;
      }
    }
    
    // decimate the signal
    //
    for (int_4 i=0; i<buf_size; i++) {
      
      // compute the sample value
      //
      if (num_bytes == (int_4)2) {
	sample_val = *((int_2*)(buffer + i*(num_chan*num_bytes) +
				chan*num_bytes));
      }
      else if (num_bytes == (int_4)4) {
	sample_val = *((int_4*)(buffer + i*(num_chan*num_bytes) +
				chan*num_bytes));
      }
      
      else {

	// free memory
	//
	Tcl_Free((char*)buffer);

	// return with error
	//
	return(TCL_ERROR);
	
      }

      // check for min/max
      //
      if (sample_val < min_val) {
	min_val = sample_val;
	min_loc = i_cur_time;
      }
      if (sample_val > max_val) {
	max_val = sample_val;
	max_loc = i_cur_time;
      }
      
      // output the critical points if necessary
      //
      if ((i_cur_time == i_output_time) && (i_cur_time < i_end_time)) {
	if (min_loc < i_end_time) {
	  out_pts_x[(int)num_pts] = i_output_time * time_scale;
	  out_pts_y[(int)num_pts] = (float_4)min_val;
	  num_pts++;
	}
	if (max_loc < i_end_time) {
	  out_pts_x[(int)num_pts] = i_output_time * time_scale;
	  out_pts_y[(int)num_pts] = (float_4)max_val;
	  num_pts++;
	}
	i_output_time += decimation_factor;
	min_val = 32767;
	max_val = -min_val;
      }
      
      // update counters
      //
      i_cur_time++;
    }
    
  }

  // find the maximum magnitude
  //
  float_4 val = 0;
  for (int_4 i=0; i<num_pts; i++) {
    val = out_pts_y[i];
    if (out_pts_y[i] < 0)
      {val = -out_pts_y[i];}
    if (val > max_ampl)
      {max_ampl = val;}
  }

  
  // normalize the data and output
  //
  float_4 scale_x = 1.0 / (end_time - start_time) * (float_4)(npix_x-1);
  float_4 scale_y_1 = -1.0 / max_ampl * (float_4)(npix_y-1) / 2.0;
  float_4 scale_y_2 = npix_y / 2;

  // release memory if it is already allocated
  //
  if (*points != (XPoint*)NULL) {
    Tcl_Free((char*)*points);
    *points = (XPoint*)NULL;
  }
  *points = (XPoint*) Tcl_Alloc(num_pts*sizeof(XPoint));
  
  // add each x,y pair to the points array
  //
  for (int_4 i=0; i<num_pts; i++) {
    
    ((*points)[i]).x = ((int_2)rint((out_pts_x[i] - start_time) * scale_x) +
			x_offset);
    ((*points)[i]).y = ((int_2)rint((sig_plot->gain_factor_d * out_pts_y[i] *
				     scale_y_1) + scale_y_2) + y_offset);
  }

  // draw the signal plot lines to the drawable
  //
  XDrawLines(display, drawable, sig_plot->text_gc, *points, num_pts,
	     CoordModeOrigin);

  // free the buffer memory
  //
  Tcl_Free((char*)buffer);
  
  // close the input file
  //
  fclose(fp_in);
  
  // exit gracefully
  //
  return(TCL_OK);
  
}


