// file: energy_plot_0.cc
//

// system include files
//
#include <string.h>

// isip include files
//
#include "energy_plot.h"
#include <signal.h>

#include "energy_plot_constants.h"
#include <signal_constants.h>

// function energy_plot_plot_points_cc
//
// parameters:
//     Energy_plot_item* energy_plot: a energy plot widget
//     int_4& num_pts: (output) number of points returned
//     XPoint*& points: (output) pointer to an array of points
//     int width: width of the plot
//     int height: height of the plot
//     int x_offset: virtual upper-left x coordinate
//     int y_offset: virtual upper-left y coordinate
//     Display* display: the display to draw to
//     Drawable drawable: the drawable to plot on
//
// returns: int indicating status
//
// this function produces an array of points based on the speech file in
// energy_plot->file_name_d
//
int energy_plot_plot_points_cc(Energy_plot_item* energy_plot, int_4& num_pts,
			       XPoint** points, int width, int height,
			       int x_offset, int y_offset, Display* display,
			       Drawable drawable) {

  // define local variables
  //
  float_4 start_time = energy_plot->start_time_d;
  float_4 end_time = energy_plot->end_time_d;
  float_4 max_val = (float_4)0;
  float_4 min_val = (float_4)0;

  // check number of points for outrageous conditions
  //
  if ((width <= 1) || (height <= 1)) {
    return TCL_ERROR;
  }

  // reset the number of points
  //
  num_pts = (int_4)0;

  // declare a signal object to retrieve the signal data
  //
  Signal signal;

  // set the parameters of the signal object
  //
  if (energy_plot->swap_byte_flag_d == ENERGY_TRUE) {
    
    signal.set_cc((unichar*)energy_plot->file_name_d, energy_plot->num_chans_d,
		  energy_plot->num_bytes_d, energy_plot->sf_d, ISIP_TRUE);
  }
  else {

    signal.set_cc((unichar*)energy_plot->file_name_d, energy_plot->num_chans_d,
		  energy_plot->num_bytes_d, energy_plot->sf_d, ISIP_FALSE);
  }

  // open the file for reading while checking for errors
  //
  if (signal.open_cc((unichar*) energy_plot->file_name_d, ISIP_READ_ONLY) ==
      ISIP_FALSE) {

    // return with error
    //
    return(TCL_ERROR);
  }

  // check time boundaries
  //
  if (start_time > end_time) {
    start_time = end_time;
  }
    
  // create the energy array
  //
  double* energy_array = new double[width];

  // set the number of pixels per second
  //
  float_4 pixels_per_sec = (float_4)width / (end_time - start_time);
  
  // declare the number of samples read for each frame
  //
  int_4 num_samples = (int_4)0;
  
  // loop over each pixel
  //
  for (int_4 i = 0; i < width; i++) {

    // declare a pointer to some data
    //
    double* data = (double*)NULL;

    // compute the time in seconds that correspond to the pixel location
    //
    float_4 pixel_time = i / pixels_per_sec + start_time;

    // determine the bounds of the window around the pixel location
    //
    float_4 win_start = pixel_time - (energy_plot->win_length_d / 2);
    float_4 win_end = pixel_time + (energy_plot->win_length_d / 2);
    
    // read the window of data in
    //
    if (signal.read_cc(data, win_start, win_end,
		       (long)energy_plot->display_channel_d, num_samples) ==
	ISIP_FALSE) {

      if (data != (double*)NULL) {
	delete [] data;
      }
      return TCL_ERROR;
    }
    
    // pre-emphasize the data if desired
    //
    if (energy_plot->pre_emph_flag_d == ENERGY_TRUE) {

      // preemphasize the data
      //
      energy_pre_emphasize_data_cc(data, energy_plot->pre_emph_coeff_d,
				   num_samples);
    }

    // use the desired window function for the data
    //
    energy_window_function_cc(data, num_samples, energy_plot); 

    // compute the energy of the data
    //
    double frame_energy = (double)0.0;
    compute_energy_cc(data, num_samples, frame_energy,
		      energy_plot->rms_scale_d, energy_plot->amp_scale_d);

    // store the energy value in the energy array
    //
    energy_array[i] = frame_energy;

    // clean up data memory
    //
    if (data != (double*)NULL) {
      delete [] data;
      data = (double*)NULL;
    }
  }

  // set the energy maximum and minimum values
  //
  max_val = (float_4)ENERGY_MAX_AMPLITUDE;
  min_val = (float_4)ENERGY_MIN_AMPLITUDE;
    
  // find the normalizing values for the data
  //
  float_4 scale_y = (float_4)-1.0 / (max_val-min_val) * (float_4)(height);

  // release memory if it is already allocated
  //
  if (*points != (XPoint*)NULL) {
    Tcl_Free((char*)*points);
    *points = (XPoint*)NULL;
  }

  // allocate memory for the points
  //
  *points = (XPoint*) Tcl_Alloc(width * sizeof(XPoint));

  // declare a variable for the energy value in decibles
  //
  double energy_log = (double)0.0;

  // loop over all energy points and compute the pixel location of that value
  //
  for (int_4 i = 0; i < width; i++) {
    
    ((*points)[i]).x = (int_2)(i + x_offset);

    // check the bounds of the energy value
    //
    if (energy_array[i] < (double)ENERGY_MIN_VALUE) {
      energy_log = log(ENERGY_MIN_VALUE);
    }    
    else {

      // compute the energy in decibles
      //
      energy_log = log(energy_array[i]);
    }

    ((*points)[i]).y = (int_2)rint(((energy_log - min_val) * scale_y) +
			height) + y_offset;
  }

  // draw the energy plot lines to the drawable
  //
  XDrawLines(display, drawable, energy_plot->text_gc, *points, width,
	     CoordModeOrigin);

  // close the input file
  //
  signal.close_cc();

  // clean up memory
  //
  if (energy_array != (double*)NULL) {
    delete [] energy_array;
  }
  
  // exit gracefully
  //
  return(TCL_OK);
}


