/* $Id: execute.c,v 1.1 1998/04/16 18:21:31 hamaker Exp hamaker $
 *
 * Copyright 1991 Brent Townshend (bst%tt@cam.org)
 * Townshend Computer Tools
 * Montreal, Quebec
 *
 * Mon Aug 5 02:31:14 EDT 1991
 *
 * Revision History: $Log: execute.c,v $
 * Revision History: Revision 1.1  1998/04/16 18:21:31  hamaker
 * Revision History: Initial revision
 * Revision History:
 * Revision 2.62  1996/10/11  16:42:03  bst
 * Increased minimum samplesBeforeStart to 100,000 instead of 50,000.
 *
 * Revision 2.61  1996/03/21  16:43:27  bst
 * Made fn static
 *
 * Revision 2.60  1994/08/30  19:00:32  bst
 * Fixed coercion of cmd->clock
 *
 * Revision 2.59  1994/05/12  03:04:18  bst
 * change file->channels if isect overrides it.
 *
 * Revision 2.58  1994/04/07  01:46:39  bst
 * Corrected bug introduced in handling of empty formats.
 *
 * Revision 2.57  1994/03/15  16:53:46  bst
 * Ignore errors for cmd->format == "none"
 *
 * Revision 2.56  1994/03/07  18:39:07  bst
 * Use labs() instead of abs() for longs.
 *
 * Revision 2.55  1993/11/19  03:23:00  bst
 * Added #pragma unused.
 *
 * Revision 2.54  1993/11/10  00:33:20  bst
 * Don't reset precision to 8 for NA_ENCODING_CONTROL.
 *
 * Revision 2.53  1993/10/04  02:00:39  bst
 * Removed extern decl.
 *
 * Revision 2.52  1993/08/31  17:26:36  bst
 * Avoid sampling rate conversion in Pro-port mode.
 * Don't set CS and user bits when in Pro-port mode.
 *
 * Revision 2.51  1993/08/10  02:50:47  bst
 * Coerce %l args to long.
 * Moved abs() macro to support.h
 *
 * Revision 2.50  1993/07/05  15:07:57  bst
 * Added arg to interrupt_hit()
 *
 * Revision 2.49  1993/06/23  20:38:22  bst
 * Changed 'long' to 'int32'
 *
 * Revision 2.48  1993/06/16  17:08:15  bst
 * Print correct count of samples transferred if error occurs during drain.
 *
 * Revision 2.47  1993/05/26  13:31:03  bst
 * Check return code from NADrain().
 *
 * Revision 2.46  1993/04/22  18:42:30  bst
 * Added include of dso.h
 *
 * Revision 2.45  1993/04/22  03:28:45  bst
 * Made use of '=' in conditional less ambiguous.
 * Check for abs before redefining.
 *
 * Revision 2.44  1993/04/21  02:13:24  bst
 * Changed function interrupt() to interrupt_hit() to avoid conflict
 *
 * Revision 2.43  1993/04/12  02:28:39  bst
 * Use long when 32-bit integers required.
 *
 * Revision 2.42  1993/03/01  04:08:27  bst
 * Fixed use of sys vs. local includes.
 *
 * Revision 2.41  1993/02/23  16:47:57  bst
 * Pass 'playrec' arg to NASetFilter() calls.
 *
 * Revision 2.40  1993/02/04  23:21:10  bst
 * Only use SIGUSR1 if it has been defined.
 *
 * Revision 2.39  1992/12/31  05:02:45  bst
 * Made Execute() return the exit status instead of exitting.
 *
 * Revision 2.38  1992/11/22  04:59:09  bst
 * Renamed SIOMODE_AES*
 *
 * Revision 2.37  1992/11/09  04:50:49  bst
 * Make sure nsamples, seek do not specify partial byte positions.
 *
 * Revision 2.36  1992/09/29  03:57:54  bst
 * Don't check lock unless we're using AES/EBU input.
 *
 * Revision 2.35  1992/09/08  02:28:22  bst
 * Make sure samplesBeforeStart is large enough to fill PIO buffer.
 * Display true count of samples if playback encounters an error.
 *
 * Revision 2.34  1992/08/18  01:13:10  bst
 * Use cleanup() instead of exit() to guarantee data connection is closed.
 *
 * Revision 2.33  1992/07/13  05:24:19  bst
 * Call NAClose() after an interrupt.
 * Don't drain output if tx was == 0.
 *
 * Revision 2.32  1992/07/05  03:14:18  bst
 * Use 'snap.pb' instead of 'snap'.
 *
 * Revision 2.31  1992/06/27  22:19:35  bst
 * Added coercion.
 *
 * Revision 2.30  1992/05/15  22:59:41  bst
 * Sed file->channels and file->srate if they weren't set by rdheader()
 *
 * Revision 2.29  1992/05/15  15:10:46  bst
 * Always execute seek before transfers (to set sampleSize, at least)
 * Fixed tests for mismatches between settings and file settings.
 * Set isect->precision from file[0].precision only if file has it set.
 * Always check if file->name is nonzero before using it.
 * Fixed bug that prevented use of stdin.
 *
 * Revision 2.28  1992/05/12  18:59:32  bst
 * Use _NAencodingNames[] to display names of encodings in error msgs.
 *
 * Revision 2.27  1992/05/10  19:05:49  bst
 * Use samples instead of words for counts of transfers.
 * Support multiple precisions, encodings.
 *
 * Revision 2.26  1992/04/09  23:50:03  bst
 * Added support for playback of multiple files.
 *
 * Revision 2.25  1992/04/09  17:59:21  bst
 * Coerc arg to sleep() to unsigned int
 *
 * Revision 2.24  1992/04/07  22:57:06  bst
 * Changed coercions for comparison of a 'NAclock' with 'int'
 * ,
 *
 * Revision 2.23  1992/04/06  21:57:58  bst
 * Added coercion to match prototype.
 *
 * Revision 2.22  1992/03/06  03:14:56  bst
 * Computer samplesBeforeStart using externalRate if sampleRate==0
 *
 * Revision 2.21  1992/03/06  03:09:04  bst
 * Compute nsamples using externalRate if sampleRate==0
 *
 * Revision 2.20  1992/03/06  00:33:04  bst
 * Don't set sampleRate to externalRate before processing command line.
 *
 * Revision 2.19  1992/02/17  04:50:32  bst
 * Removed 'nsamples' arg for wrtrailer()
 * Call VerifyCUBits if CS is read from a file.
 * Use/set file channel status bits.
 * Only store CU bits if recording (was a bug)
 *  Added sampio.c, identify.c, aifc.c, ieee_externd.c to library.
 * Added checks for file->name == 0
 *
 * Revision 2.18  1992/02/15  19:52:28  bst
 * Exit if NAOpen() returns NULL
 *
 * Revision 2.17  1992/02/12  04:33:12  bst
 * Added coercions for cmd->clock.
 *
 * Revision 2.16  1992/02/02  22:00:16  bst
 * If -a option is given, stop tape if command is interrupted.
 *
 * Revision 2.15  1992/01/27  06:21:29  bst
 * Use cmd->unit for NetAudio server to connect to.
 *
 * Revision 2.14  1992/01/27  03:57:54  bst
 * Changed handling in interrupt() routine
 *     - open a new connection to server
 *     - use (*file->aborted)() for waves repositioning.
 * Added extFirstSample to keep track of first output sample # for intr routine.
 *
 * Revision 2.13  1992/01/24  05:07:20  bst
 * Added SIGUSR handling.
 * Close the signal file with the correct number of samples after an intr.
 *
 * Revision 2.12  1992/01/22  21:14:17  bst
 * Rationalized setup of options (server,defaults,file,command line)
 * Use 'channels' instead of 'left','right'
 * If the sampleRate for playback is close to 48k or 44.1k set external thus.
 *
 * Revision 2.11  1992/01/20  03:56:42  bst
 * Modified a comment dlplay->naplay
 *
 * Revision 2.10  1992/01/19  05:27:52  bst
 * Abort repeats for a transfer() return value of 0 as well as negative.
 *
 * Revision 2.9  1992/01/19  04:30:10  bst
 * Check for error return from wrheader()
 *
 * Revision 2.8  1992/01/11  19:21:04  bst
 * For record, set file->srate from isect->sampleRate
 *
 * Revision 2.7  1992/01/10  04:08:19  bst
 * Removed old printf().
 *
 * Revision 2.6  1992/01/08  18:31:24  bst
 * Leave sampleRate at 0 for recording.
 * Don't setup filtering if sampleRate==0
 *
 * Revision 2.5  1991/12/17  04:47:41  bst
 * Added setup of filtering.
 *
 * Revision 2.4  1991/12/16  05:24:36  bst
 * Removed old code to write 0 padding.
 * Stop repeating a segment if any transfer failes.
 * Exit with code 9 if a transfer error occurred.
 *
 * Revision 2.3  1991/12/16  00:43:59  bst
 * Removed 2nd arg to NAOpen()
 * Added call to NAOpenData()
 *
 * Revision 2.2  1991/12/10  22:45:24  bst
 * Merged in changes from 1.27 -> 1.30
 * Check for error returns from file->open,file->create, and file->seek
 * Do autostart before checking for lock.
 * IFDEF-ed out 0 padding at end
 *
 * Revision 2.1  1991/12/08  21:08:19  bst
 * Network version.
 *
 * Revision 1.30  1991/12/06  02:17:59  bst
 * Keep transfer counts in terms of words instead of samples.
 *
 * Revision 1.29  1991/12/05  18:48:26  bst
 * Make sure zero padding is a multiple of BLOCK_LENGTH
 *
 * Revision 1.28  1991/11/22  05:02:25  bst
 * Use dat_dspProgram() instead of dat_mode() for converter.
 * Print progress messages to stderr instead of stdout.
 *
 * Revision 1.27  1991/11/14  05:35:27  bst
 * Use new symbolic parameter retrieval from dsp programs.
 *
 * Revision 1.26  1991/11/14  01:06:29  bst
 * Correct transferred count to account for stereo.
 * Print sample counts returned from dspGetParams() while waiting for flush.
 *
 * Revision 1.25  1991/11/11  04:13:15  bst
 * Use new Debug()
 * Call wrtrailer() explicitly - file->close() no longer calls it.
 *
 * Revision 1.24  1991/11/09  03:44:58  bst
 * Handle interrupt signal - stop transfers immediately.
 *
 * Revision 1.23  1991/11/08  21:25:35  bst
 * Pass number of samples to wrheader() routine
 *
 * Revision 1.22  1991/11/08  06:00:47  bst
 * Broke into multiple sources files:
 *     main.c -> execute.c cmdline.c main.c
 *
 * Revision 1.21  1991/11/07  14:51:21  bst
 * If left==0 and right==0 use mono output
 * Replace call to cmd.seek() when cmd.repeats==1
 * Added -m option for mono playback
 * Added conditional support for SPW
 *
 * Revision 1.20  1991/11/04  14:14:06  bst
 * Coerce file.srate into an int before calling dat_setMode().
 *
 * Revision 1.19  1991/11/04  01:44:13  bst
 * Support more file cmd.formats.
 * Figure out incoming sampling rate from channel status info.
 * Many rearrangements of code.
 *
 * Revision 1.17  1991/10/30  04:16:06  bst
 * Renamed 'c' to 'cb' to avoid aliasing
 *
 * Revision 1.16  1991/10/28  23:23:33  bst
 * Check for input lock when playing back locked to input source.
 * Default input source is RCA connectors.
 * Removed -b option.
 * Added -g option to generate clock.
 * Added more input selection arguments to -S option.
 *
 * Revision 1.15  1991/10/20  18:20:04  bst
 * Define bzero() as memset() on MAC
 * Write 2 seconds of zeroes after each write to let filters settle.
 * Fixed handling of error returns from transfer().
 *
 * Revision 1.14  1991/10/17  21:13:58  bst
 * Fixed bug: used to continue repeating after an interrupt.
 * Define bzero as memset on Mac.
 *
 * Revision 1.13  1991/10/03  14:19:15  bst
 * Added delay handling in remote sequences.
 * Moved order of operations - Remote later
 * Removed pause before starting transfers.
 *
 * Revision 1.12  1991/09/29  06:47:25  bst
 * Allow file sampling rates > 48000
 *
 * Revision 1.11  1991/09/27  14:58:09  bst
 * Set sampling rate conversion using dat_mode().
 * Added 'press return' before beginning transfers.
 *
 * Revision 1.10  1991/09/16  05:16:48  bst
 * Changed some function names to make them more consistent.
 * Added input source selection.
 * Fixed bug if t=0 and stereo output.
 * Changed filename for CU bits when using stdio to "stdio.cu"
 *
 * Revision 1.9  1991/09/14  03:22:51  bst
 * Multiple cmd.nsamples by 2 if running in stereo.
 *
 * Revision 1.8  1991/09/12  03:00:10  bst
 * Added coercion of first argument to bcopy().
 *
 * Revision 1.7  1991/09/09  04:56:34  bst
 * Fixed setting of channel status bits for pro mode.
 *
 * Revision 1.6  1991/08/23  23:51:41  bst
 * Modified function declarations to work with non-ANSI compilers
 *
 * Revision 1.5  1991/08/23  16:05:33  bst
 * Reversed bit order of channel status bits.
 *
 * Revision 1.4  1991/08/21  09:32:47  bst
 * Removed unimplemented cmd.formats.
 * Revised help messages.
 * Modified for new parameter lists for getU,setU,getC,setC
 *
 * Revision 1.3  1991/08/12  13:43:43  bst
 * Added automatic generation of correct C bits.
 * Fixed bug in setting C bits.
 * Added -A option.
 *
 * Revision 1.2  1991/08/05  19:20:38  bst
 * Changed environment variable names.
 * Added include of dsp_mode.h.
 *
 * Revision 1.1  1991/08/05  06:32:45  bst
 * Initial revision
 */

/*
 * Record/Play to DAT-Link from a file
 *
 * Set RECORD macro for narecord, otherwise naplay
 */

/* Exit Codes:
 *    0 - normal exit
 *    1 - command line error 
 *    2 - Unable to open connection to DAT-Link
 *    3 - No signal from DAT machine
 *    4 - Unable to seek tape or file
 *    5 - Unable to open file
 *    6 - Unable to transfer C or U information
 *    7 - DAT-Link communications error
 *    8 - Internal error
 *    9 - Underrun error
 */

#include <assert.h>
#include <signal.h>
#include "playrec.h"
#include "netaudio.h"
#include "siomode.h"
#include "support.h"

#ifdef MSDOS
#include <dos.h>	/* For declaration of sleep() */
#endif

static SampFile *extFile = 0;
static const char *extUnit = 0;
#ifndef RECORD
static int32 extFirstSample = 0;	/* First sample number of playback */
#endif
static int stopIfInterrupted = 0;
static char *stopCommand = "stop";	   /* Command to stop DAT */
NAport *extPort = 0;

void cleanup(status)
int status;
{
    if (extPort)
	NAClose(extPort);
    exit(status);
}

static void interrupt_hit(sig)
int sig;
{
#ifdef PRAGMA_UNUSED
#pragma unused(sig)
#endif /* PRAGMA_UNUSED */
    NAport *port;

    fprintf(stderr,"Interrupt\n");
    /* Abort transfers */
    /* We need a new connection to server, since the state of the current
	connection is unknown */
    port = NAOpen(extUnit);
    if (port == 0)
	fprintf(stderr,"Interrupt routine unable to open connection to server\n");
    else {
#ifdef RECORD
	NAFlush(port,NA_RECORD);
	(*extFile->wrtrailer)(extFile);
#else
	/* For playback, we need to adjust the count due to internal buffering */
	NAsnapshots snap;
	NAPause(port,NA_PLAY,1);
	NAGetSnapshot(port,&snap);
	NAFlush(port,NA_PLAY);
	if (extFile) {
	    fprintf(stderr,"Closing %s at sample %ld\n",
		    extFile->name?extFile->name:"stdin",(long)snap.pb.sampleNumber);
	    (*extFile->aborted)(extFile,(unsigned int32)
				(extFirstSample+snap.pb.sampleNumber));
	    fprintf(stderr,"Done\n");
	}
#endif /* Record */
	/* Stop DAT machine if needed */
	if (stopIfInterrupted) {
	    if (NARemote(port,getenv("DL_RC"),stopCommand) < 0)
		fprintf(stderr,"Unable to execute '%s' on DAT machine via remote control: ",stopCommand);
	}
    }
    NAClose(port);
    cleanup(0);
}


int Execute(cmd,file)
Params *cmd;
SampFile file[];
{
    int32 samplesBeforeStart;	 /* Number of samples to send before starting PB */
    int32 nxfr=0;		 /* Number of samples transfered */
#ifdef RECORD
    static char *startCommands[] = {"stop","play",0};  /* Command to start DAT */
#else
    static char *startCommands[] = {"stop", "record","%6","play",0};
#endif
    char *device = 0;			/* Device for remote commands */
    int32 tx;
    NAport *port;
    NAinfo info;
#ifdef RECORD
    NAinfosect *isect = &info.record;
#else
    NAinfosect *isect = &info.playback;
#endif
    int i;
    
    if (cmd->numfiles == 0) {
	cmd->names[0] = 0;
	cmd->numfiles = 1;
    }
#ifdef RECORD
    else if (cmd->numfiles > 1) {
	fprintf(stderr,"Sorry, cannot handle multiple files for recording.\n");
	cleanup(1);
    }
#endif
    for (i=0;i<cmd->numfiles;i++) {
	file[i].name = cmd->names[i];
	if (cmd->format) {
	    if (strcmp(cmd->format,"none") != 0) {
		if (SampFileSetFormat(&file[i],cmd->format) < 0)
		    cleanup(1);
	    }
	} else {
#ifdef RECORD
	    fprintf(stderr,
		    "%s: No file type specified - saving as raw data.\n",
		    file[i].name?file[i].name:"stdout");
#else /* RECORD */
	    const char *fmt;
	    if (file[i].name&& ((fmt = SampFileIdentify(file[i].name)) != 0)) {
		if (SampFileSetFormat(&file[i],fmt) < 0)
		    cleanup(1);
	    }
	    else
		fprintf(stderr,
			"%s: Unknown file type - assuming headerless data.\n",
			file[i].name?file[i].name:"stdin");
#endif /* RECORD */
	}

#ifndef RECORD
	if ((*file[i].open)(&file[i]) < 0)
#else /* RECORD */
        if ((*file[i].create)(&file[i]) < 0)
#endif /* RECORD */
	    {
		fprintf(stderr,"Unable to open %s\n",
			file[i].name?file[i].name:"stdio");
		cleanup(5);
	    }
    }
	    
#ifdef PTIME
    ptime("after open");
#endif

    /* Open DAT-Link */
    if ((port = NAOpen(cmd->unit)) == 0)
	cleanup(1);
    extUnit = cmd->unit;

    if (port == 0) {
	perror("NAOpen");
	cleanup(1);
    }
    extPort = port;

    /*****************************/
    /* Get current info settings */
    /*****************************/
    NAGetInfo(port,&info);

    /*****************************/
    /* Default overrides         */
    /*****************************/
    isect->sampleRate = 0;
    isect->channels = NA_CHANNELS_STEREO;
    isect->precision = 16;
    isect->encoding = NA_ENCODING_LINEAR;
    info.clock = NA_CLOCK_INTERNAL;
    info.master = 0;
    info.skipSilence = 0;
    info.source = DL_ISRC_ALL;
    info.loopback = 0;
#ifndef RECORD

    /*****************************/
    /* Settings from file        */
    /*****************************/
    if (file[0].channels != -1)
	isect->channels = file[0].channels;
    if (file[0].srate != -1)
	isect->sampleRate = file[0].srate;
    if (file[0].precision != -1)
	isect->precision = file[0].precision;
#endif

    /*****************************/
    /* Command line settings     */
    /*****************************/
    if (cmd->source != -1)
	info.source = cmd->source;
    if (cmd->sampleRate != -1)
	isect->sampleRate = cmd->sampleRate;
    if (cmd->precision != -1)
	isect->precision = cmd->precision;
    if (cmd->encoding != -1) {
	isect->encoding = cmd->encoding;
	if (isect->encoding == NA_ENCODING_IEEE)
	    isect->precision = 32;
	else if ((isect->encoding == NA_ENCODING_ULAW) ||
		 (isect->encoding == NA_ENCODING_ALAW))
	    isect->precision = 8;
    }
#ifdef RECORD
    if (cmd->skipSilence != -1)
	info.skipSilence = cmd->skipSilence;
    if (SIOMODE_DP_ISPRO(info.dspSIOMode)) {
	/* Pro-port - try not to use sampling rate conversion */
	info.record.externalRate = info.playback.externalRate = isect->sampleRate;
    }
#else
    if (cmd->externalRate != -1)
	isect->externalRate = cmd->externalRate;
    else if (SIOMODE_DP_ISPRO(info.dspSIOMode)) {
	/* Pro-port - try not to use sampling rate conversion */
	info.record.externalRate = info.playback.externalRate = isect->sampleRate;
    }
    else {
	/* If not set on cmd line, use the matching external rate if possible */
	if ((int32)(isect->sampleRate+0.5) == 44100)
	    isect->externalRate = 44100;
	if ((int32)(isect->sampleRate+0.5) == 48000)
	    isect->externalRate = 48000;
	/* else leave it the way it was */
	/* Don't use 32k, since it may not be supported */
    }

    if (cmd->master != -1)
	info.master = cmd->master;
    if (cmd->clock != ((NAclock)-1))
	info.clock = cmd->clock;
    
    if (!SIOMODE_DP_ISPRO(info.dspSIOMode)) {
	/*
	 * Set C.S. and User bits
	 */
	if (cmd->cbits == 1) {
	    assert(cmd->cfilename != 0);
	    Debug("execute",1,"Reading channel status from %s.\n",cmd->cfilename);
	    ReadCUBits(isect,cmd->cfilename);
	    VerifyCUBits(isect,cmd->consumer,isect->externalRate);
	}
	else if (file[0].cstatValid) {
	    Debug("execute",1,"Using channel status from data file '%s'.\n",
		  file[0].name?file[0].name:"stdin");
	    memcpy(isect->leftCS,file[0].leftCS,24);
	    memcpy(isect->rightCS,file[0].rightCS,24);
	    VerifyCUBits(isect,cmd->consumer,isect->externalRate);
	}
	else
	    /* cmd->cbits == 0 */
	    SetDefaultCUBits(isect,cmd->consumer,isect->externalRate);
    }
#endif /* Record */

    if (cmd->channels != -1)
	isect->channels = cmd->channels;

#ifndef RECORD
	/* Print warnings for mismatches between files and/or command line */
	/* Also, set values of file fields that are still unknown */
    for (i=0;i<cmd->numfiles;i++) {
	if (file[i].channels == -1)
	    file[i].channels = isect->channels;
	else if ((isect->channels == NA_CHANNELS_STEREO) !=
		 (file[i].channels == NA_CHANNELS_STEREO)) {
	    fprintf(stderr,"Warning: File %s contains %s data, but playback is %s.\n",
		    file[i].name?file[i].name:"stdin",
		    ((file[i].channels == NA_CHANNELS_STEREO)?"stereo":"mono"),
		    ((isect->channels == NA_CHANNELS_STEREO)?"stereo":"mono"));
	    file[i].channels = isect->channels;
	}
	if (file[i].srate == -1)
	    file[i].srate = isect->sampleRate;
	else if (isect->sampleRate != file[i].srate)
	    fprintf(stderr,"Warning: File %s sample rate is %.0f, but playback is at %.0f.\n",
		    file[i].name?file[i].name:"stdin",file[i].srate,
		    isect->sampleRate?isect->sampleRate:isect->externalRate);

	if (file[i].precision == -1)
	    file[i].precision = isect->precision;
	else if (file[i].precision != isect->precision)
	    fprintf(stderr,"Warning: File %s has %d bits/sample, but playback is at %d bits/sample.\n", file[i].name?file[i].name:"stdin",
		    file[i].precision, isect->precision);

	if (file[i].encoding == -1)
	    file[i].encoding = isect->encoding;
	else if (isect->encoding != file[i].encoding )
	    fprintf(stderr,"Warning: File %s encoding is %s, but data is being interpretted as %s.\n",
		    file[i].name?file[i].name:"stdin",
		    _NAencodingNames[file[i].encoding],
		    _NAencodingNames[(int)isect->encoding]);
    }
#endif

    /********************************/
    /* Miscellaneous setup          */
    /********************************/

    /* Setup filtering */
    if (isect->sampleRate && (isect->externalRate != isect->sampleRate)) {
	NAfilter filter;
	if (isect->encoding != NA_ENCODING_LINEAR) {
	    fprintf(stderr,"Sorry, only linear encoding supported with sample-rate conversion\n");
	    cleanup(1);
	}
	if (isect->precision != 16) {
	    fprintf(stderr,"Sorry, only 16-bit precision supported with sample-rate conversion\n");
	    cleanup(1);
	}

	GetFilterDefaults(&filter.atten,&filter.frequency.frequency_len,
			  &filter.frequency.frequency_val,&filter.level.level_val,
#ifdef RECORD
			  isect->externalRate, isect->sampleRate
#else
			  isect->sampleRate, isect->externalRate
#endif
			  );
	filter.level.level_len = filter.frequency.frequency_len;
	NASetFilter(port,
#ifdef RECORD
		    NA_RECORD,
#else
		    NA_PLAY,
#endif
		    &filter);
    }

#ifndef RECORD
    if (cmd->duration != -1)
	cmd->nsamples = cmd->duration * (isect->sampleRate?isect->sampleRate:
					 isect->externalRate);
    
    samplesBeforeStart = (int32)(cmd->bufferTime*
			(isect->sampleRate?isect->sampleRate:isect->externalRate));
    /* Make sure at least enough to fill up the DAT-Link's PIO buffer is sent */
    if (samplesBeforeStart < 100000)
	samplesBeforeStart = 100000;
    Debug("execute",2,"seek=%ld\n",(long)cmd->seek);
#endif

    Debug("execute",2,"channels=%d, nsamples=%ld\n",
	  isect->channels, (long)cmd->nsamples);
    Debug("execute",2,"file[0] rate=%f, DAT rate=%f, transfer rate=%f\n",
	  file[0].srate,isect->externalRate,isect->sampleRate);

    /* Set info */
    NASetInfo(port,&info);

    /*************************************/
    /* Finished setting up configuration */
    /*************************************/

    /*
     * Start tape if needed
     */
    if (cmd->autoStart) {
	char **s;
	device = getenv("DL_RC");
	if (device == 0) {
	    fprintf(stderr,"DL_RC must be set to the remote device name to use the -a option\n");

	    cleanup(2);
	}
	/* Make sure DAT machine is stopped if we get interrupted */
	stopIfInterrupted = 1;
	for (s=startCommands;*s;s++) {
	    Debug("",1,"Starting tape with command: %s\n",*s);
	    if (**s == '%') {
		int pt = atoi((*s)+1);
		sleep((unsigned int)pt);
	    }
	    else {
		if (NARemote(port,device,*s) < 0) {
		    fprintf(stderr,"Unable to execute '%s' on '%s' via remote control: ",*s,device);
		    cleanup(4);
		}
		sleep(1);
	    }
	}
    }

    /*
     * Read back current info if needed and make sure we have lock
     */
#ifdef RECORD
    if ((info.dspSIOMode == SIOMODE_AES) ||
	(info.dspSIOMode == SIOMODE_AES_OVERRIDE) ||
	(info.dspSIOMode == SIOMODE_AESRX))
#else
    if (info.clock == NA_CLOCK_INPUT)
#endif /* RECORD */
    /* Check if data is coming in */
    {
	int retries;
	for (retries=0;retries<3;retries++) {
	    NAGetInfo(port,&info);
	    if (info.locked)
		break;
	    sleep(1);
	}
	if (!info.locked) {
	    fprintf(stderr,"No signal on digital audio input.\n");
	    cleanup(3);
	}
    }

#ifdef RECORD
    if (isect->sampleRate == 0)
	/* Set sample rate for record to true value */
	isect->sampleRate = isect->externalRate;
    file[0].srate = isect->sampleRate;
    if (cmd->duration != -1)
	cmd->nsamples = cmd->duration * isect->sampleRate;
    file[0].channels = isect->channels;
    file[0].precision = isect->precision;
    file[0].encoding = isect->encoding;
    /*
     * Store CU bits if requested 
     */
    if (cmd->cbits == 1) {
	assert(cmd->cfilename != 0);
	StoreCUBits(isect,cmd->cfilename);
    }
    /* Copy CU bits to file header */
    memcpy(file[0].leftCS, isect->leftCS,24);
    memcpy(file[0].rightCS, isect->rightCS,24);
    file[0].cstatValid = 1;
#endif

    if ((cmd->nsamples != -1) && ((cmd->nsamples*isect->precision) % 8)) {
	int32 ns = (cmd->nsamples*isect->precision/8)*8/isect->precision;
	fprintf(stderr,"Transferring %ld samples instead of %ld to transfer integer bytes.\n",
		(long)ns,(long)cmd->nsamples);
	cmd->nsamples = ns;
    }
#ifndef RECORD
    /* Make sure we do not seek to a partial sample */
    if ((cmd->seek*isect->precision) % 8) {
	int32 ns = (cmd->seek*isect->precision/8)*8/isect->precision;
	fprintf(stderr,"Seeking to sample %ld instead of %ld to align to byte boundary.\n",
		(long)ns,(long)cmd->seek);
	cmd->seek = ns;
    }
#endif

#ifdef RECORD
    /*
     *   Write header to output file
     */
    if ((*file[0].wrheader)(&file[0],(unsigned int32)cmd->nsamples) < 0)
	cleanup(5);
#endif

    /*
     * Set up signal so interrupt or SIGUSR1 halts I/O before exitting
     */
    extFile = &file[0];
    (void)signal(SIGINT, interrupt_hit);
#ifdef SIGUSR1
    (void)signal(SIGUSR1, interrupt_hit);
#endif

    /*
     * Do data transfers
     */

#ifdef RECORD
    if (NAOpenData(port,NA_RECORD) < 0) {
	fprintf(stderr,"Unable to open data connection\n");
	cleanup(1);
    }
#else
    if (NAOpenData(port,NA_PLAY) < 0) {
	fprintf(stderr,"Unable to open data connection\n");
	cleanup(1);
    }
#endif

    tx = -1;
#ifndef RECORD
    if (cmd->repeats > 1)
	while (cmd->repeats--) {
	    for (i=0;i<cmd->numfiles;i++) {
		if ((*file[i].seek)(&file[i],(unsigned int32)cmd->seek) < 0) {
		    fprintf(stderr,"%s: Unable to seek to sample %ld.\n",
			    file[i].name?file[i].name:"stdin",(long)cmd->seek);
		    cleanup(4);
		}
		extFirstSample = cmd->seek;
		extFile = &file[i];
		tx = transfer(&file[i],port,cmd->nsamples,
			      (samplesBeforeStart > 0)?&samplesBeforeStart:0);
	    }
	    nxfr += labs(tx);
	    if (tx <= 0)
		break;
	}
    else
#endif
	{
	    for (i=0;i<cmd->numfiles;i++) {
#ifndef RECORD
		if ((*file[i].seek)(&file[i],(unsigned int32)cmd->seek) < 0) {
		    fprintf(stderr,"%s: Unable to seek to sample %ld.\n",
			    file[i].name?file[i].name:"stdin",(long)cmd->seek);
		    cleanup(4);
		}
		extFirstSample = cmd->seek;
#endif
		extFile = &file[i];
		tx = transfer(&file[i],port,cmd->nsamples,&samplesBeforeStart);
	    }
	    nxfr = labs(tx);
	}

#ifndef RECORD
    if (tx > 0) {
	int dresult;
	printf("Waiting for output to finish...");
	fflush(stdout);
	dresult = NADrain(port);
	if (dresult < 0) {
	    fprintf(stderr,"Failed drain of playback buffer\n");
	    tx = -1;
	}
	else if (dresult != 0) {
	    fprintf(stderr,"Transfer error: %s\n",_NAxfrErrorNames[dresult]);
	    tx = -1;
	}
	else
	    printf("Done\n");
    }
#endif
    /* Stop tape if needed */
    if (cmd->autoStart) {
	if (NARemote(port,device,stopCommand) < 0) {
	    fprintf(stderr,"Unable to execute '%s' on DAT machine via remote control: ",stopCommand);
	    cleanup(7);
	}
    }

    /* All done, clean up signals, close DAT-Link and file */
    (void)signal(SIGINT, SIG_DFL);
#ifdef SIGUSR1
    (void)signal(SIGUSR1, SIG_DFL);
#endif

#ifndef RECORD
    if (tx < 0) {
	/* For playback, we need to adjust the count due to internal buffering */
	NAsnapshots snap;
	if (NAGetSnapshot(port,&snap) >= 0)
	    nxfr = snap.pb.sampleNumber;
    }
#endif

    fprintf(stderr,"Transferred %ld samples\n",(long)nxfr);
    for (i=0;i<cmd->numfiles;i++) {
#ifdef RECORD
	(*file[i].wrtrailer)(&file[i]);
#endif
        (*file[i].close)(&file[i]);
    }
    NAClose(port);

    if (tx < 0)
	return 9;
    else
	return 0;
}
