/* $Id: sampfile.c,v 1.36 1996/03/21 16:41:35 bst Exp $
 *
 * Copyright 1991 Brent Townshend (bst%tt@cam.org)
 * Townshend Computer Tools
 * Montreal, Quebec
 *
 * Sun Nov 3 20:42:31 EST 1991
 *
 * Revision History: $Log: sampfile.c,v $
 * Revision 1.36  1996/03/21  16:41:35  bst
 * Made some fns static.
 *
 * Revision 1.35  1994/04/25  20:31:06  bst
 * Fixed arg types for *_wrheader(), *_write*()
 *
 * Revision 1.34  1993/11/19  03:22:19  bst
 * Removed fcntl.h
 *
 * Revision 1.33  1993/08/10  02:43:47  bst
 * Use O_BINARY
 * Coerce lseek arg2 to long.
 * Coerce %l args to long.
 *
 * Revision 1.32  1993/07/08  18:36:30  bst
 * Use SamplesToBytes(), BytesToSamples() macros.
 *
 * Revision 1.31  1993/07/05  15:10:15  bst
 * Changed nsamples arg to int32.
 *
 * Revision 1.30  1993/06/23  20:10:15  bst
 * Changed 'long' to 'int32'
 *
 * Revision 1.29  1993/05/18  19:45:44  bst
 * Use NALog() to log/print errors and warnings.
 *
 * Revision 1.28  1993/04/22  03:25:00  bst
 * Moved *_construct() declarations to formats.h
 *
 * Revision 1.27  1993/04/12  02:14:08  bst
 * Use long when 32-bit integers required.
 *
 * Revision 1.26  1993/02/07  19:37:32  bst
 * Added unused() #pragma.
 *
 * Revision 1.25  1992/12/28  22:23:30  bst
 * Added spw3.0 support.
 *
 * Revision 1.24  1992/10/16  01:06:55  bst
 * Compute sample size before computing seek position.
 *
 * Revision 1.23  1992/08/19  00:22:55  bst
 * Removed include of config.h (sysdefs.h gets it).
 *
 * Revision 1.22  1992/05/15  15:10:03  bst
 * Set sampleSize in seek() if it's not already set.
 * Don't set precsion,encoding, and sampleSize in rdheader() for raw files.
 *
 * Revision 1.21  1992/05/10  19:02:32  bst
 * Added encoding, precision.
 *
 * Revision 1.20  1992/04/09  17:58:51  bst
 * Changed strings.h to string.h
 *
 * Revision 1.19  1992/02/17  04:49:23  bst
 * Moved sample I/O routines to sampio.c
 * Moved format enum to formats.h
 * Added AIFC/AIFF handling
 * Added format field to this
 *
 * Revision 1.18  1992/02/13  17:44:48  bst
 * Accept minimal length format names
 *
 * Revision 1.17  1992/02/08  03:10:59  bst
 * Don't use SEEK_SET macro.
 *
 * Revision 1.16  1992/01/30  04:07:37  bst
 * Use '#ifdef' instead of '#if' for ESPSFILES
 *
 * Revision 1.15  1992/01/28  05:50:49  bst
 * Use P() macro for prototypes.
 * Removed old spwa_contruct()
 * Added include of ../config.h
 *
 * Revision 1.14  1992/01/27  03:57:45  bst
 * Added aborted() function.
 *
 * Revision 1.13  1992/01/24  05:05:01  bst
 * Added function pointer coercions.
 * close() now take a sample count argument.
 *
 * Revision 1.12  1992/01/22  21:05:30  bst
 * Use 'channels' instead of 'left','right'
 * Removed 'COMDISCO' conditional, always compile in SPW support.
 * Changed sampleSize from a byte count to a bit count.
 *
 * Revision 1.11  1992/01/19  04:28:53  bst
 * Added headerLength,sampleSize,totalSamples.
 * Created rd/wrheader() functions.
 *
 * Revision 1.10  1991/12/10  22:12:58  bst
 * Use currentSample to keep track of current sample number position.
 * Added ESPS file support.
 *
 * Revision 1.9  1991/12/06  18:41:42  bst
 * Removed 'const' qualifier from buf arg to SampFile_write()
 *
 * Revision 1.8  1991/12/06  06:02:07  bst
 * Include sysdefs.h, but no longer libdefs.h
 * Added char * coercions.
 *
 * Revision 1.7  1991/11/22  05:08:43  bst
 * Added 'swap' file fromat.
 *
 * Revision 1.6  1991/11/11  04:27:42  bst
 * Remove debug message.
 * SampFile_close() no longer calls wrtrailer() since sampfile may be
 *     only open for reading.
 *
 * Revision 1.5  1991/11/08  21:29:58  bst
 * Removed params argument form SampFileSetDefaults
 * Made parts conditional on COMDISCO==1
 *
 * Revision 1.4  1991/11/08  06:02:48  bst
 * Added SampFileSetFormat()
 * Renamed SampFile_construct -> SampFileSetDefaults
 * Pass params structure into SampFileSetDefaults()
 * Open & create now use stdio if name is null.
 *
 * Revision 1.3  1991/11/07  14:52:49  bst
 * Use SampFile_null() for do-nothing functions.
 *
 * Revision 1.2  1991/11/06  16:42:01  bst
 * Use 'unsigned int' instead of 'u_int'
 *
 * Revision 1.1  1991/11/04  01:47:10  bst
 * Initial revision
 *
 */
#include <stdio.h>
#include <assert.h>
#include <string.h>
#include "sysdefs.h"
#include "debug.h"
#include "sampfile.h"
#include "formats.h"
#include "support.h"

char *fmtnames[] = {"raw","native","swap", "AIFF", "AIFC", "sppack","spw2",
#ifdef ESPSFILES
			   "esps",
#endif
			       "spw3"};
static char *minfmtnames[] = {"r","n","sw","aiff", "aifc", "spp","spw2",
#ifdef ESPSFILES
			   "e",
#endif
			       "spw3"};


static int SampFile_open(this)
SampFile *this;
{
    if (this->name)
	this->fd = open(this->name,O_RDONLY|O_BINARY);
    else
	this->fd = 0;
    if (this->fd < 0) {
	NALog(LOG_ERR,"Unable to open %s.\n",this->name);
	exit(5);
    }
    return (*this->rdheader)(this);
}

static int SampFile_create(this)
SampFile *this;
{
    if (this->name) {
#ifdef macintosh
	this->fd = open(this->name,O_WRONLY|O_CREAT|O_TRUNC|O_BINARY);
#else
	this->fd = open(this->name,O_WRONLY|O_CREAT|O_TRUNC|O_BINARY,0644);
#endif
    }
    else
	this->fd = 1;
    if (this->fd < 0) {
	NALog(LOG_ERR,"Unable to open %s.\n",this->name);
	exit(5);
    }
    this->currentSample = 0;
    return 0;
}

static int SampFile_rdheader(this)
SampFile *this;
{
    this->headerLength = 0;
    return 0;
}

static int SampFile_wrheader(this,nsamples)
SampFile *this;
unsigned int32 nsamples;
{
    if (this->encoding == -1)
	this->encoding = 0;
    if (this->precision == -1)
	this->precision = 16;
    this->sampleSize = this->precision*((this->channels==3)?2:1);
    Debug("SampFile.wrheader",1,"sampleSize=%d, precision=%d, encoding=%d\n",
	  this->sampleSize,this->precision,this->encoding);
    this->totalSamples = nsamples;
    this->headerLength = 0;
    return 0;
}

static int SampFile_close(this)
SampFile *this;
{
    return close(this->fd);
}

int SampFile_null(this)
SampFile *this;
{
#ifdef PRAGMA_UNUSED
#pragma unused(this)
#endif /* PRAGMA_UNUSED */
    return 0;
}

/* Seek so given sampleNumber is next to be read from this->fd.
   If sampleSize is not a multiple of 8, then the fd will be positioned
   as (sampleNumber-1) had just been read (i.e. may be past first bit of
   sampleNumber */
static int SampFile_seek(this,sampleNumber)
SampFile *this;
unsigned int32 sampleNumber;
{
    if (this->sampleSize == 0) {
	assert(this->precision > 0);
	this->sampleSize = this->precision*((this->channels==3)?2:1);
    }
    if (this->currentSample != sampleNumber) {
	int32 bnum = this->headerLength+
	    SamplesToBytes(sampleNumber,this->sampleSize)+
	    (((sampleNumber*this->sampleSize)%8)+7)/8;
	this->currentSample = sampleNumber;
	Debug("SampFile.seek",1,"Seeking to sample %ld = byte %ld\n",
	      (long)sampleNumber,(long)bnum);
	if (lseek(this->fd,(long)bnum,0) < 0) {
	    NALog(LOG_ERR,"lseek failed: %s\n",NAsyserr());
	    return -1;
	}
    }
    return 0;
}

void SampFileSetDefaults(this,argc,argv)
SampFile *this;
int argc;
char *argv[];
{
    this->fd = -1;
    this->io = NULL;
    this->name = 0;
    this->srate = -1;
    this->channels = -1;
    this->encoding = -1;
    this->precision = -1;
    this->scale = 1;
    this->currentSample = 0;
    this->headerLength = 0;
    this->sampleSize = 0;	/* Unknown */
    this->totalSamples = 0;	/* Unknown */
    this->cstatValid = 0;
    this->open = SampFile_open;
    this->create = SampFile_create;
    this->wrheader = SampFile_wrheader;
    this->rdheader = SampFile_rdheader;
    this->write = samp_write_16_B;
    this->read = samp_read_16_B;
    this->wrtrailer = SampFile_null;
    this->seek = SampFile_seek;
    this->close = SampFile_close;
    this->aborted = (int (*)P((SampFile*,unsigned int32)))SampFile_null;

    /* Copy the command line into cmd */
    *this->cmd = 0;
    while (argc--) {
	if (strlen(this->cmd)+strlen(*argv)> (sizeof(this->cmd)-3)) {
	    strcat(this->cmd,"...");
	    break;
	}
	strcat(this->cmd,*argv);
	strcat(this->cmd," ");
	argv++;
    }
}

int SampFileSetFormat(this,fmtName)
SampFile *this;
const char *fmtName;
{
    int i;
    this->format = (int)RAW;

    for (i=0;i<(int)LASTFORMAT;i++)
	if (strncasecmp(fmtName,minfmtnames[i],strlen(minfmtnames[i])) == 0) {
	    this->format = i;
	    i = -1;
	    break;
	}
    if (i != -1) {
	if (strcasecmp(fmtName,"help") != 0)
	    fprintf(stderr,"Unsupported file format: %s\n", fmtName);
	fprintf(stderr,"Valid formats are:");
	for (i=0;i<(int)LASTFORMAT;i++)
	    fprintf(stderr," %s",fmtnames[i]);
	fprintf(stderr,"\n");
	return -1;
    }
    
    /* File in the structure with the appropriate method calls */
    switch ((FileFormats)this->format) {
      case RAW:
	/* Default initialization was for raw */
	break;
      case NATIVE:
	native_construct(this);
	break;
      case SWAP:
	swap_construct(this);
	break;
      case AIFC:
      case AIFF:
	AIFC_construct(this);
	break;
      case SPPACK:
	sppack_construct(this);
	break;
      case SPW2:
	SPW_construct(this);
	break;
      case SPW3:
	SPW3_construct(this);
	break;
#ifdef ESPSFILES
      case ESPS:
	ESPS_construct(this);
	break;
#endif
      default:
	NALog(LOG_ERR,"Bad file format: %d\n",this->format);
	exit(1);
    }
    return 0;
}
