/* $Id: defaults.c,v 1.41 1996/03/21 16:40:44 bst Exp $
 *
 * Copyright 1991 Brent Townshend (bst%tt@cam.org)
 * Townshend Computer Tools
 * Montreal, Quebec
 *
 * Sun Nov 10 23:03:54 EST 1991
 *
 * Revision History: $Log: defaults.c,v $
 * Revision 1.41  1996/03/21  16:40:44  bst
 * Made UNIXToMac() static.
 * Set default INSTALLROOT to :: on mac
 *
 * Revision 1.40  1995/02/21  07:47:04  bst
 * Check in /usr/local/lib/datlink if all fails.
 *
 * Revision 1.39  1994/05/26  15:14:54  bst
 * Make sure float/int args are just numbers.
 *
 * Revision 1.38  1994/04/20  13:50:43  bst
 * Added int coercion to avoid Solaris 2.3 warning.
 *
 * Revision 1.37  1993/11/16  01:52:53  bst
 * Changed pragma unused arg.
 *
 * Revision 1.36  1993/08/27  00:35:03  bst
 * Coerce size_t args to long and use %l formats.
 *
 * Revision 1.35  1993/08/10  02:27:41  bst
 * Moved abs() macro to support.h
 * Convert filenames to UNIX style.
 * Add implicit '.' to PATH under DOS.
 *
 * Revision 1.34  1993/06/01  02:45:31  bst
 * Added coercions for GCC 2.4.1 unsigned 'size_t'.
 *
 * Revision 1.33  1993/05/18  19:44:30  bst
 * Use NALog() to log/print errors and warnings.
 *
 * Revision 1.32  1993/04/22  18:39:46  bst
 * Made 'kend' const char * to avoid warning messages.
 *
 * Revision 1.31  1993/04/22  03:23:00  bst
 * Added static function declarations..
 *
 * Revision 1.30  1993/04/21  02:11:36  bst
 * Define R_OK if it's not defined under DOS. (Don't include sys/file.h)
 * Check if 'abs' is predefined before defining it.
 *
 * Revision 1.29  1993/02/23  18:45:16  bst
 * Removed some debugging printf's()
 *
 * Revision 1.28  1993/02/18  21:37:59  bst
 * Added MSDOS support.
 *
 * Revision 1.27  1993/02/07  19:36:23  bst
 * Added unused() #pragma.
 *
 * Revision 1.26  1992/09/16  23:37:03  bst
 * Append '/' to end of value in DL_DIR.
 * Added FindFile() debug message.
 *
 * Revision 1.25  1992/08/24  16:35:34  bst
 * Don't UNIXToMac on UNIX host.
 *
 * Revision 1.24  1992/08/20  00:19:27  bst
 * Added better support for mac filenames.
 *
 * Revision 1.23  1992/05/12  18:53:35  bst
 * Fixed bug - was running past end of path.
 *
 * Revision 1.22  1992/04/09  23:49:29  bst
 * Check if InitDefaults() is called multiple times.
 *
 * Revision 1.21  1992/04/04  23:30:56  bst
 * New algorithm for locating installation root.
 *
 * Revision 1.20  1992/02/15  19:51:29  bst
 * Look for DATLink.Defaults in source directory if not installed.
 *
 * Revision 1.19  1992/02/11  18:18:02  bst
 * FindFile() always returns value of DATLink on Mac.
 * Use strchr() instead of index()
 * Don't include sys/file.h on Mac.
 *
 * Revision 1.18  1992/02/08  03:10:07  bst
 * Fixed bugs in FindFile()
 * Added include of sys/file.h for X_OK macro
 *
 * Revision 1.17  1992/02/02  21:59:47  bst
 * Added checks to make sure array is not overflowed in FindFile()
 *
 * Revision 1.16  1992/01/30  06:11:04  bst
 * Made fullPath static so it can be used by calling routine.
 *
 * Revision 1.15  1992/01/30  04:06:18  bst
 * Added FindFile() for position independent installation
 *
 * Revision 1.14  1992/01/27  06:19:56  bst
 * Move debug stuff to debug.c
 * Use InitDefaults without options processing.
 * Removed unit prepend to option names.
 * Made AddDefaults() externally visible.
 *
 * Revision 1.13  1992/01/22  20:55:35  bst
 * Added check for valid int/float options
 *
 * Revision 1.12  1992/01/17  04:18:42  bst
 * Use varargs or stdargs for Debug()
 *
 * Revision 1.11  1992/01/10  04:06:15  bst
 * Sense of return value of strcasecmp() was reversed in GetBoolDefault()
 *
 * Revision 1.10  1991/12/12  04:37:35  bst
 * Remove define of REAL_DEBUG_PROTOTYPE since debug.h is not included.
 *
 * Revision 1.9  1991/12/10  22:11:52  bst
 * Log time with each debug message.
 *
 * Revision 1.8  1991/12/06  18:36:33  bst
 * Define 'const' as null if not ANSI -C
 *
 * Revision 1.7  1991/12/06  06:01:31  bst
 * New defaults (No longer dat_*)
 *
 * Revision 1.6  1991/11/22  05:06:13  bst
 * Increased maximum number of args to Debug()
 *
 * Revision 1.5  1991/11/14  01:01:38  bst
 * Ignore blank lines in .datlinkrc
 *
 * Revision 1.4  1991/11/12  05:49:38  bst
 * Replace calls to getOptDefaults() with getDefaults()
 *
 * Revision 1.3  1991/11/12  05:42:17  bst
 * Made all get*Default() routines return a default value if default not found.
 * Added boolean,int,float defaults get routines.
 * set Debug_defaults from libdatlink.defaults.Trace
 *
 * Revision 1.2  1991/11/11  05:20:20  bst
 * Don't use varargs for Debug() (Its not very portable)
 *
 * Revision 1.1  1991/11/11  04:04:03  bst
 * Initial revision
 *
 */
#include <stdio.h>
#include <ctype.h>
#include <string.h>
#if !defined(macintosh) && !defined(MSDOS)
#include <sys/file.h>
#endif /* !macintosh && !MSDOS */
#include "sysdefs.h"
#include "defaults.h"
#include "debug.h"

#ifdef MSDOS
#ifndef R_OK
#define R_OK 4
#endif /*!R_OK */
#endif /* MSDOS */

int Debug_Defaults = 0;

#define MAXDEFNAME 1000
static void LocateInstallRoot P((const char *));

typedef struct InitLines {
    char *name;
    char *value;
    struct InitLines *next;
} InitLines;

static int DefaultsInitialized = 0;

static struct Defaults {
    char sysfile[MAXDEFNAME];
    const char *cmdName;
    InitLines *il,*lastline;
} defaults;

void AddDefaults(name,value)
const char *name,*value;
{
    InitLines *newLine;

    if (DefaultsInitialized == 0) {
	InitDefaults("?");
	Debug("defaults",1,"Warning: Main program should call InitDefaults()\n");
    }

    /* Create a new entry in structure and link it into defaults */
    newLine = (InitLines *)malloc(sizeof(*newLine));
    newLine->next = 0;
    if (defaults.lastline) {
	defaults.lastline->next = newLine;
	defaults.lastline = newLine;
    }
    else
	defaults.il = defaults.lastline = newLine;

    newLine->name = malloc(strlen(name)+1);
    strcpy(newLine->name,name);
    newLine->value = malloc(strlen(value)+1);
    strcpy(newLine->value,value);
    if (Debug_Defaults)
	printf("%s:%s\n",newLine->name,newLine->value);
}

static void AddDefaultsLine P((const char *fname, char *buf));
static void AddDefaultsLine(fname, buf)
const char *fname;
char *buf;
{
    char *name,*name_end,*value,*value_end;

    /* Break into parts */
    name = buf;
    while (isspace(*name))
	name++;
    /* Ignore a blank line or comment */
    if ((*name == 0) || (*name == '!'))
	return;
    name_end = name;
    while (*name_end && !isspace(*name_end) && (*name_end != ':'))
	name_end++;
    value = name_end;
    if (*value == ':')
	value++;
    while (*value && isspace(*value))
	value++;
    value_end = value;
    while (*value_end && (*value_end != '\n'))
	value_end++;
    if (*value_end == 0) {
	NALog(LOG_ERR,"Bad line in %s: %s",fname,buf);
	return;
    }
    *name_end = 0;
    *value_end = 0;
    AddDefaults(name,value);
}

/* Read in file placing new entries at beginning of list */
static void ReadDefaultsFile P((const char *fname));
static void ReadDefaultsFile(fname)
const char *fname;
{
    FILE *initfd;
    if ((initfd = fopen(fname,"r")) == NULL)
	return;
    while (1) {
	char buf[MAXDEFNAME];

	if (fgets(buf,sizeof(buf),initfd) == NULL)
	    break;
	AddDefaultsLine(fname,buf);
    }
}

void InitDefaults(pgmName)
const char *pgmName;
{
    char initfile[MAXDEFNAME];
    const char *defFile;
    char *home;

    if (DefaultsInitialized) {
	NALog(LOG_ERR,
	      "Programs attempted to call InitDefaults() multiple times.\n");
	exit(1);
    }

    DefaultsInitialized = 1;
    /* Locate install root of DAT-Link software */
    LocateInstallRoot(pgmName);

    if (strrchr(pgmName,'/'))
	/* Only keep basename of command */
	defaults.cmdName = strrchr(pgmName,'/')+1;
    else
	defaults.cmdName = pgmName;

    defaults.il = defaults.lastline = 0;

    /* Read system init file */
    defFile = FindFile("etc/DATLink.Defaults");
    if (defFile == 0)
	/* For running out of source directory */
	defFile = FindFile("scripts/DATLink.Defaults");

    if (defFile) {
	strcpy(defaults.sysfile,defFile);
	ReadDefaultsFile(defaults.sysfile);
    }
    else
	NALog(LOG_ERR,
	      "Warning: System defaults file not found. (etc/DATLink.Defaults)\n");

    /* Now HOME init file */
    if ((home = getenv("HOME")) != NULL) {
	sprintf(initfile,"%s/.datlinkrc",home);
	ReadDefaultsFile(initfile);
    }

    DefaultsInitialized = 2;
    Debug("defaults",1,"%s - Installroot = %s\n",pgmName,FindFile(""));
    Debug_Defaults = GetIntDefault("libdatlink.defaults.Trace",0);
}

/* Check if components key and name match - advance them if they do */
static int MatchComponent P((const char **key,const char **name));
static int MatchComponent(key,name)
const char **key,**name;
{
    if (**key == '.')
	(*key)++;
    if (**name == '*') {
	(*name)++;
	while (1) {
	    const char *kend = strchr(*key,'.');
	    size_t len;

	    if (kend == 0)
		len = strlen(*key);
	    else
		len = kend-*key;
	    
	    if (Debug_Defaults > 1)
		printf("Comparing %s to %s for %ld\n",*key,*name-1,(long)len);
	    if ((strncasecmp(*key,*name,len) == 0) &&
		(((*key)[len]==(*name)[len]) ||
		 (((*key)[len]=='.') && ((*name)[len]=='*')))) {
		*name += len;
		*key += len;
		return 1;
	    }
	    else {
		*key += len;
		if (**key == 0)
		    return  (**name == 0);
		(*key)++;  /* Skip . */
	    }
	}
    }
    else {
	const char *kend = strchr(*key,'.');
	size_t len;
	if (kend == 0)
	    len = strlen(*key);
	else
	    len = kend-*key;
	
	if (**name  == '.')
	    (*name)++;

	if (Debug_Defaults > 1)
	    printf("Comparing %s to %s for %ld\n",*key,*name,(long)len);
	if ((strncasecmp(*key,*name,len) == 0) &&
	    (((*key)[len]==(*name)[len]) ||
	     (((*key)[len]=='.') && ((*name)[len]=='*')))) {
	    *name += len;
	    *key += len;
	    return 1;
	}
	else
	    return 0;
    }
}

/* Check if a request matches a given name */
static int DefaultMatch P((const char *key,const char *name));
static int DefaultMatch(key,name)
const char *key,*name;
{
    while (*key && *name) {
	if (MatchComponent(&key,&name) == 0)
	    return 0;
    }
    return (*key == 0) && (*name == 0);
}

/* Get an optional default - return null if not found */
const char *GetDefault(p,defv)
const char *p;
const char *defv;
{
    const char *result = 0;
    char key[MAXDEFNAME];
    InitLines *l;

    if (DefaultsInitialized == 0) {
	InitDefaults("?");
	Debug("defaults",1,"Warning: Main program should call InitDefaults()\n");
    }

    sprintf(key,"%s.%s",defaults.cmdName,p);

    for (l=defaults.il;l;l=l->next) {
	if (DefaultMatch(key,l->name))
	    result = l->value;
    }
    if (Debug_Defaults)
	printf("GetDefault(%s) -> %s\n",p,result);
    return result?result:defv;
}

double GetFloatDefault(p,fv)
const char *p;
double fv;
{
    const char *s = GetDefault(p,0);
    const char *sp;
    if (s == 0)
	return fv;
    while (isspace(*s))
	s++;
    sp = s;
    while (*s++) {
	if (!isdigit(*s) && (*s != '+') && (*s != '.') && (*s != '-'))
	    break;
    }
    if (*s == 0)
	return atof(sp);
    NALog(LOG_ERR,"Bad value for default: %s.\n",p);
    NALog(LOG_ERR,"    Expected a float number, but found '%s'\n",s);
    return fv;
}

int GetIntDefault(p,iv)
const char *p;
int iv;
{
    const char *s = GetDefault(p,0);
    const char *sp;
    if (s == 0)
	return iv;
    while (isspace(*s))
	s++;
    sp = s;
    while (*s++) {
	if (!isdigit(*s) && (*s != '+') && (*s != '-'))
	    break;
    }
    if (*s == 0)
	return atoi(sp);
    NALog(LOG_ERR,"Bad value for default: %s.\n",p);
    NALog(LOG_ERR,"    Expected an integer, but found '%s'\n",s);
    return iv;
}

int GetBoolDefault(p,defv)
const char *p;
int defv;
{
    const char *s = GetDefault(p,0);
    if (s == 0)
	return defv;
    if ((strcasecmp(s,"on") == 0) || (strcasecmp(s,"true") == 0))
	return True;
    if ((strcasecmp(s,"off") == 0) || (strcasecmp(s,"false") == 0))
	return False;
    NALog(LOG_ERR,"Boolean default, %s, should be 'on','off','true', or 'false', not '%s'\n",
	    p,s);
    return defv;
}

#define MAXPATHLENGTH 200
static char InstallRoot[MAXPATHLENGTH];

#ifdef MSDOS
/* Convert a path from UNIX format to DOS format */
const char *DOSToUNIX(const char *path)
{
    static char unixpath[MAXPATHLENGTH];
    const char *p = path;
    char *u = unixpath;
    for (;*p;p++,u++) {
	if (*p == '\\')
	    *u = '/';
	else
	    *u = *p;
    }
    *u = 0;
    return unixpath;
}
#endif /* MSDOS */

#ifdef macintosh
/* Convert a path from UNIX format to MAC format */
static const char *UNIXToMac(path)
const char *path;
{
    static char macpath[MAXPATHLENGTH];
    const char *p = path;
    char *q = macpath;
    int firsttime = 1;
    while (*p) {
	if (strncmp(p,"./",2) == 0)
	    p++;
	else if (strncmp(p,"../",3)==0) {
	    p+=3;
	    if (firsttime)
		*q++ = ':';
	    *q++ = ':';
	}
	else if (strcmp(p,"..")==0) {
	    p+=2;
	    if (firsttime)
		*q++ = ':';
	    *q++ = ':';
	}
	else {
	    const char *elem = strchr(p,'/');
	    if (elem == 0) {
		strcpy(q,p);
		break;
	    }
	    else {
		int len = elem-p;
		if (len) {
		    strncpy(q,p,len);
		    q+=len;
		    *q++ = ':';
		}
	    }
	    p = elem+1;
	}
	firsttime = 0;
    }
    return macpath;
}
#endif

/* Locate a file - return the full path name */
const char* FindFile(filename)
const char *filename;		/* Name of file or directory to locate */
{
    static char fullPath[MAXPATHLENGTH];
    if ((int)(strlen(InstallRoot)+strlen(filename)+1) >= MAXPATHLENGTH) {
	NALog(LOG_ERR,"Internal error - FindFile(%s) - path too long\n",
		filename);
	return 0;
    }
#ifdef macintosh
    sprintf(fullPath,"%s%s",InstallRoot,UNIXToMac(filename));
#else
    sprintf(fullPath,"%s%s",InstallRoot,filename);
    if (access(fullPath,R_OK) != 0)
	return 0;
#endif
    if (DefaultsInitialized == 2) 
	Debug("defaults",2,"FindFile(%s) -> %s\n",filename,fullPath);
    return fullPath;
}

/* Locate the root directory of the DAT-Link installation */
static void LocateInstallRoot(ocmdName)
const char *ocmdName;
{
#ifdef macintosh
#ifdef PRAGMA_UNUSED
#pragma unused(ocmdName)
#endif /* PRAGMA_UNUSED */
    if (getenv("DATLink"))
	strcpy(InstallRoot,getenv("DATLink"));
    else
	strcpy(InstallRoot,"::");
#else /* macintosh */
    size_t dirlen;
    const char *path,*colon;
    const char *cmdName;

    if (getenv("DL_DIR")) {
	strcpy(InstallRoot,getenv("DL_DIR"));
	strcat(InstallRoot,"/");
	return;
    }

#ifdef MSDOS
    cmdName = DOSToUNIX(ocmdName);
#else /* MSDOS */
    cmdName = ocmdName;
#endif /* !MSDOS */
    /* Check if command included a path (./cmd or /a/b/cmd ) */
    if (strchr(cmdName,'/')) {
	/* Yes, strip off executable name */
	const char *base = strrchr(cmdName,'/');
	dirlen = (int)(base-cmdName+1);
	if (dirlen+3 >= MAXPATHLENGTH) {
	    NALog(LOG_ERR,"Internal error - LocateInstallRoot(%s) - path too long\n",
		    cmdName);
	    return;
	}
	strncpy(InstallRoot, cmdName, dirlen);
	InstallRoot[dirlen] = 0;
	strcat(InstallRoot,"../");
	if (FindFile("etc/dl"))
	    return;
    }

    /* Locate '../etc/dl' on the user's path */
#ifdef MSDOS
    /* MSDOS always checks . first */
    strcpy(InstallRoot,"../");
    if (FindFile("etc/dl"))
	return;
#endif /* MSDOS */
    path= getenv("PATH");
    colon = path;
    while (path && *path) {
#ifdef MSDOS
	colon = strchr(path,';');
#else
	colon = strchr(path,':');
#endif
	if (colon == 0)
	    dirlen = strlen(path);
	else
	    dirlen = (int)(colon-path);
	if (dirlen+1+strlen("../etc/dl") >= MAXPATHLENGTH) {
	    NALog(LOG_ERR,"Internal error - LocateInstallRoot(%s) - path too long\n",
		    cmdName);
	    return;
	}
	if (dirlen == 0)
	    InstallRoot[0] = 0;
	else {
	    strncpy(InstallRoot,path,dirlen);
	    strcpy(&InstallRoot[dirlen],"/../");
	}
	if (FindFile("etc/dl"))
	    return;
	if (colon)
	  path = colon+1;
	else
	  path = 0;
    }
    
    /* Try a common path name - /usr/local/lib/datlink/etc/dl (for G. Erdmann) */
    strcpy(InstallRoot,"/usr/local/lib/datlink/");
    if (FindFile("etc/dl"))
	return;

    NALog(LOG_ERR,"Unable to locate ../etc/dl on PATH\n");
    InstallRoot[0] = 0;
#endif /* macintosh */
}

