/* $Id: debug.c,v 1.32 1996/03/19 20:53:16 eryk Exp $
 *
 * Copyright 1992 Brent Townshend (bst%tt@cam.org)
 * Townshend Computer Tools
 * Montreal, Quebec
 *
 * Mon Jan 27 01:40:16 EST 1992
 *
 * Revision History: $Log: debug.c,v $
 * Revision 1.32  1996/03/19  20:53:16  eryk
 * Added __BORLANDC__ conditions with __STDC__
 *
 * Revision 1.31  1994/08/30  18:59:52  bst
 * Modified debug time display so it works on mac also.
 *
 * Revision 1.30  1994/04/25  20:30:21  bst
 * Use st for macro arg in vfprintf() macro to avoid conflict with %s
 *
 * Revision 1.29  1994/04/14  19:22:24  bst
 * Use HAS_stdarg instead of HAS_stdarg_h
 *
 * Revision 1.28  1993/11/16  01:52:30  bst
 * Added PRAGMA_UNUSED.
 * Check for HAS_sys_errlist
 *
 * Revision 1.27  1993/08/27  00:34:13  bst
 * Use HAS_* defines.
 * Use NEEDS_VA
 * Moved declarations of system functions to sysdefs.h
 *
 * Revision 1.26  1993/08/10  02:27:22  bst
 * Added include of support.h
 *
 * Revision 1.25  1993/06/30  01:29:30  bst
 * Use vsyslog() instead of vsprintf()+syslog()
 * Prepend NALog() messages with error level if not using syslog().
 *
 * Revision 1.24  1993/06/16  17:23:41  bst
 * Use static buffer for syslog messages..
 *
 * Revision 1.23  1993/06/14  18:39:00  bst
 * Changed return type of vsprintf() to int.
 *
 * Revision 1.22  1993/06/14  16:19:29  bst
 * Use vsprintf+syslog instead of vsyslog.
 *
 * Revision 1.21  1993/06/01  02:45:26  bst
 * Added coercions for GCC 2.4.1 unsigned 'size_t'.
 *
 * Revision 1.20  1993/05/18  21:01:57  bst
 * Set fullfmt[0] to null if negative level.
 *
 * Revision 1.19  1993/05/18  19:43:37  bst
 * Added DebugCont(), NALog(), NAOpenlog().
 * Log errors, warnings, debug to syslog() if NAOpenlog() called.
 * Added NAsyserr()
 *
 * Revision 1.18  1993/05/07  15:28:59  bst
 * Fixed bug in SetDebug() - unterminated string.
 *
 * Revision 1.17  1993/04/22  03:22:28  bst
 * Don't use gettimeofday() under MSDOS.
 *
 * Revision 1.16  1993/04/16  01:29:38  bst
 * Avoid writing into const char * in SetDebug().
 *
 * Revision 1.15  1993/02/18  21:37:48  bst
 * Added MSDOS support.
 *
 * Revision 1.14  1993/02/05  15:40:24  bst
 * Define NO_DEBUG_PROTOTYPE.
 *
 * Revision 1.13  1992/11/04  04:39:40  bst
 * Don't change argument to SetDebug().
 *
 * Revision 1.12  1992/11/01  19:33:52  bst
 * Move last accessed entry in Debug cache to top.
 *   -> speeds up accesses.
 *
 * Revision 1.11  1992/10/06  21:53:41  bst
 * Removed bad call to free().
 *
 * Revision 1.10  1992/09/16  23:36:48  bst
 * Clear cached debug entries when level is changed.
 *
 * Revision 1.9  1992/08/19  00:22:33  bst
 * Removed include of config.h (sysdefs.h gets it).
 *
 * Revision 1.8  1992/05/27  14:45:49  bst
 * Use non-ANSI declaration of Debug()
 *
 * Revision 1.7  1992/04/07  22:56:25  bst
 * Added DebugCheck() - Use it in Debug()
 *
 * Revision 1.6  1992/03/02  04:00:10  bst
 * Removed %0 from format.
 *
 * Revision 1.5  1992/02/15  19:51:07  bst
 * Use config.h to decide vfprintf(), _doprnt() usage.
 *
 * Revision 1.4  1992/02/12  04:32:29  bst
 * Define vfprintf() as _doprnt() on Alliant.
 *
 * Revision 1.3  1992/02/11  18:17:27  bst
 * Use strchr() instead of index()
 * Use time() for debug messages on Mac.
 *
 * Revision 1.2  1992/01/30  06:10:55  bst
 * Added VARARGS comments.
 *
 * Revision 1.1  1992/01/27  06:40:19  bst
 * Initial revision
 */

#define NEEDS_VA
#include <stdio.h>
#include <ctype.h>
#include <string.h>
#include "sysdefs.h"
#include "defaults.h"
/* If Debug() uses varargs then the prototype won't match the definition
   -> disable prototype */
#if !defined(HAS_stdarg) || (!defined(__STDC__) && !defined(__BORLANDC__))
#define NO_DEBUG_PROTOTYPE
#endif
#include "debug.h"
#include "support.h"
#ifdef HAS_syslog_h
#include <syslog.h>
#endif /* HAS_syslog_h */

#ifdef MISSING_vfprintf
/* No vfprintf(), try _doprnt() */
#ifdef HAS__doprnt
#define vfprintf(s,f,a) _doprnt(f,a,s)
#else /* HAS__doprnt */
/* That's missing too, punt on the args */
#define vfprintf(st,f,a) fprintf(st,"%s",f)
#endif /* !HAS__doprnt */
#endif /* MISSING_vfprintf */

struct dtable {
    struct dtable *next;
    const char *request;
    int level;
} *DTable = 0;

int DebugCheck(dstr,level)
const char *dstr;
int level;
{
    register struct dtable *p=DTable,*lastp=0;
    int dblev;

    while (p) {
	if (p->request == dstr)
	    break;
	lastp = p;
	p = p->next;
    }

    if (p) {
	dblev = p->level;
	if (p != DTable) {
	    /* Move located entry to front of list */
	    if (lastp)
		lastp->next = p->next;
	    p->next = DTable;
	    DTable = p;
	}
    }
    else {
	char buf[1000];
	sprintf(buf,"%s.Debug",dstr);
	dblev = GetIntDefault(buf,0);
	p = (struct dtable *)malloc(sizeof(struct dtable));
	p->request = dstr;
	p->level = dblev;
	p->next = DTable;
	DTable = p;
    }
    return dblev >= abs(level);
}

static int use_syslog = 0;

void NAOpenlog(ident, logopt, facility)
const char *ident;
int logopt, facility;
{
#ifdef HAS_openlog
    openlog(ident,logopt,facility);
#else /* HAS_openlog */
#ifdef PRAGMA_UNUSED
#pragma unused(ident,logopt,facility)
#endif /* PRAGMA_UNUSED */
#endif /* !HAS_openlog */
#ifdef HAS_syslog
    use_syslog = 1;
#endif /* HAS_syslog */
}

#ifdef HAS_stdarg
#if defined(__STDC__) || defined(__BORLANDC__)
void NALog(int level,const char *fmt,...)
#else /* __STDC__ */
void NALog(level,fmt)
int level;
const char *fmt;
#endif /* __STDC__ */
#else /* HAS_stdarg */
void NALog(level,fmt,va_alist)
const char *fmt;
int level;
va_dcl
#endif /* HAS_stdarg */
{
    va_list args;
#ifdef HAS_stdarg
    va_start(args,fmt);
#else /* HAS_stdarg */
    va_start(args);
#endif /* !HAS_stdarg */
#ifdef HAS_syslog
    if (use_syslog)
      vsyslog(level,fmt,args);
    else
#endif /* HAS_syslog */
    {
	switch (level) {
	  case LOG_ERR:
	    fprintf(stderr,"ERROR:   ");
	    break;
	  case LOG_WARNING:
	    fprintf(stderr,"WARNING: ");
	    break;
	  case LOG_INFO:
	    fprintf(stderr,"INFO:    ");
	    break;
	  case LOG_DEBUG:
	    fprintf(stderr,"DEBUG:   ");
	    break;
	  default:
	    fprintf(stderr,"LEV%d:    ",level);
	    break;
	}
	vfprintf(stderr,fmt,args);
    }
    va_end(args);
}

void SetDebug(str)
const char *str;
{
    register struct dtable *p,*np;
    if (isdigit(*str))
	AddDefaults("*Debug",str);
    else {
	const char *level = strchr(str,':');
	if (level == (char *)0)
	    NALog(LOG_WARNING,"Illegal debug option: %s\n",str);
	else {
	    char buf[100];
	    buf[0]='*';
	    strncpy(&buf[1],str,(size_t)(level-str));
	    strcpy(&buf[level-str+1],"*Debug");
	    AddDefaults(buf,level+1);
	}
    }
    /* Clear cached debug entries */
    for (p=DTable;p;p=np) {
	np = p->next;
	free(p);
    }
    DTable = 0;
}

#ifdef HAS_stdarg
#if defined(__STDC__) || defined(__BORLANDC__)
int Debug(const char *dstr,int level,const char *fmt,...)
#else /* __STDC__ */
int Debug(dstr,level,fmt)
const char *dstr;
int level;
const char *fmt;
#endif /* __STDC__ */
#else /* HAS_stdarg */
int Debug(dstr,level,fmt,va_alist)
const char *dstr,*fmt;
int level;
va_dcl
#endif /* HAS_stdarg */
{
    if (DebugCheck(dstr,level)) {
	va_list args;
	int slen;
	char fullfmt[200];
	static char blanks[] = "                  ";
	if (level >= 0) {
	    double t = mtime();
	    t -= ((long)(t/100.0))*100.0;
	    sprintf(fullfmt,"%6.3f: ",t);
	    slen = strlen(dstr);
	    if (slen < sizeof(blanks))
		sprintf(fullfmt,"%s%s:%s",fullfmt, dstr,&blanks[slen]);
	    else
		sprintf(fullfmt,"%s>%s:",fullfmt, dstr-sizeof(blanks)+slen+2);
	    while (level--)
		strcat(fullfmt,".");
	}
	else
	    fullfmt[0] = 0;
#ifdef HAS_stdarg
	va_start(args,fmt);
#else /* HAS_stdarg */
	va_start(args);
#endif /* !HAS_stdarg */
	if (strlen(fullfmt) + strlen(fmt) + 1 > sizeof(fullfmt)) {
	    NALog(LOG_WARNING,"Debug format overflow: %s\n", fmt);
	    return 1;
	}

	strcat(fullfmt,fmt);
#ifdef HAS_syslog
	if (use_syslog)
	  vsyslog(LOG_DEBUG,fullfmt,args);
	else
#endif /* HAS_syslog */
	    vfprintf(stderr,fullfmt,args);
	va_end(args);
	return 1;
    }
    else
	return 0;
}


#ifdef HAS_stdarg
#if defined(__STDC__) || defined(__BORLANDC__)
void DebugCont(const char *fmt,...)
#else /* __STDC__ */
void DebugCont(fmt)
const char *fmt;
#endif /* __STDC__ */
#else /* HAS_stdarg */
void DebugCont(fmt,va_alist)
const char *fmt;
va_dcl
#endif /* HAS_stdarg */
{
    va_list args;
#ifdef HAS_stdarg
    va_start(args,fmt);
#else /* HAS_stdarg */
    va_start(args);
#endif /* !HAS_stdarg */
#ifdef HAS_syslog
    if (use_syslog)
	vsyslog(LOG_DEBUG,fmt,args);
    else
#endif
	vfprintf(stderr,fmt,args);
    va_end(args);
}


char *NAsyserr()
{
#ifdef HAS_sys_errlist
    extern char *sys_errlist[];
    return sys_errlist[errno];
#else /* HAS_sys_errlist */
    return "???";
#endif /* !HAS_sys_errlist */
}
