/* $Id: comdisco3.c,v 1.17 1996/03/21 16:40:37 bst Exp $
 *
 * Copyright 1992 Brent Townshend (bst%tt@cam.org)
 * Townshend Computer Tools
 * Montreal, Quebec
 *
 * Mon Dec 28 18:02:55 EST 1992
 *
 * Revision History: $Log: comdisco3.c,v $
 * Revision 1.17  1996/03/21  16:40:37  bst
 * Made some fns static
 *
 * Revision 1.16  1994/05/12  03:03:22  bst
 * Fixed handling of complex files.
 * Fixed handling of double files.
 * Added support for playback of float files.
 *
 * Revision 1.15  1993/10/26  16:16:04  bst
 * Added coercions for GCC 2.5.0
 *
 * Revision 1.14  1993/08/31  17:29:31  bst
 * Change 'equal' to 'equalpos' to avoid conflict with macros.h
 *
 * Revision 1.13  1993/08/27  00:42:12  bst
 * Define NEEDS_TIME
 * Removed read() declaration.
 * Added DatIO() args coercion.
 *
 * Revision 1.12  1993/08/10  02:26:30  bst
 * Use atol() instead of atoi().
 * Coerce %l args to long.
 *
 * Revision 1.11  1993/07/08  18:35:22  bst
 * Use SamplesToBytes(), BytesToSamples() macros.
 *
 * Revision 1.10  1993/07/05  23:50:30  bst
 * Change time type to time_t
 *
 * Revision 1.9  1993/06/23  20:10:15  bst
 * Changed 'long' to 'int32'
 *
 * Revision 1.8  1993/06/01  02:45:21  bst
 * Added coercions for GCC 2.4.1 unsigned 'size_t'.
 *
 * Revision 1.7  1993/05/18  19:43:33  bst
 * Use NALog() to log/print errors and warnings.
 *
 * Revision 1.6  1993/04/22  03:22:18  bst
 * Added static function declarations.
 *
 * Revision 1.5  1993/04/12  02:12:47  bst
 * Moved DataAlloc() and DataIO() declarations to sampfile.h
 * Use long when 32-bit integers required.
 *
 * Revision 1.4  1993/02/04  23:20:58  bst
 * Added read() prototype for Mac.
 *
 * Revision 1.3  1993/01/29  21:59:51  bst
 * Fixed support for writing SPW 3.0 files.
 *
 * Revision 1.2  1992/12/31  05:02:27  bst
 * Fixed some initial bugs.
 *
 * Revision 1.1  1992/12/28  23:02:58  bst
 * Initial revision
 *
 */
#define NEEDS_TIME
#include <stdio.h>
#include <assert.h>
#include <ctype.h>
#include <string.h>
#include "sysdefs.h"
#include "sampfile.h"
#include "debug.h"
#include "support.h"
#include "formats.h"

/* Read samples */
static int32 SPW3_read_dbl(this,buf,nbytes)
SampFile *this;
byte *buf;
int32 nbytes;
{
    int stereo = (this->channels==3);
    int32 nsamples = nbytes/(sizeof(short)*(stereo?2:1));

    if (nsamples > this->totalSamples-this->currentSample) {
	nsamples = this->totalSamples-this->currentSample;
	nbytes = nsamples*sizeof(short)*(stereo?2:1);
    }

    if (nbytes == 0)
	return 0;

    assert(this->sampleSize);
    DataAlloc((size_t)SamplesToBytes(nsamples,this->sampleSize),0);
    if (DataIO(this,(int32 (*) P((int,char*,int32)))longread,
	       (int32)SamplesToBytes(nsamples,this->sampleSize),
	       nsamples,0) < 0)
	return -1;

    {
	register int32 i;
	register double *dlp = dleft;
	register short *sbuf = (short *)buf;
	register double sc = 32767*this->scale;

	for (i=0;i<nsamples*(stereo?2:1);i++)
	    *sbuf++ = sc * *dlp++;
    }

    return nbytes;
}

/* Read samples */
static int32 SPW3_read_float(this,buf,nbytes)
SampFile *this;
byte *buf;
int32 nbytes;
{
    int stereo = (this->channels==3);
    int32 nsamples = nbytes/(sizeof(short)*(stereo?2:1));

    if (nsamples > this->totalSamples-this->currentSample) {
	nsamples = this->totalSamples-this->currentSample;
	nbytes = nsamples*sizeof(short)*(stereo?2:1);
    }

    if (nbytes == 0)
	return 0;

    assert(this->sampleSize);
    DataAlloc((size_t)SamplesToBytes(nsamples,this->sampleSize),0);
    if (DataIO(this,(int32 (*) P((int,char*,int32)))longread,
	       (int32)SamplesToBytes(nsamples,this->sampleSize),
	       nsamples,0) < 0)
	return -1;

    {
	register int32 i;
	register float *dlp = (float *)dleft;
	register short *sbuf = (short *)buf;
	register double sc = 32767*this->scale;

	for (i=0;i<nsamples*(stereo?2:1);i++)
	    *sbuf++ = sc * *dlp++;
    }

    return nbytes;
}

/*
 * Input operations
 */

/* Get a line from an fd and store in given buffer - return length */
static int getline P((int fd, char *buf, int buflen));
static int getline(fd,buf,buflen)
int fd;
char *buf;
int buflen;
{
    int i;
    for (i=0;i<buflen-1;i++) {
	if (read(fd,&buf[i],1) != 1) {
	    NALog(LOG_WARNING,"read failed: %s\n",NAsyserr());
	    return 0;
	}
	if (buf[i] == '\n') {
	    buf[i] = 0;
	    return i;
	}
    }
    buf[buflen-1] = 0;
    /* Discard rest of line */
    Debug("SPW",1,"Discarding end of long line in header: '%s...'\n", buf);
    for (;;i++) {
	char c;
	if (read(fd,&c,1) != 1) {
	    NALog(LOG_ERR,"getline: read failed: %s\n",NAsyserr());
	    return 0;
	}
	if (c == '\n')
	    return i;
    }
}

/* Locate the first '=' in line and return a pointer to the next non-white
   character following.  Also, insert a null after the last non-white char
   in 'line' prior to the '=' */
static char *breakline P((char *line));
static char *breakline(line)
char *line;
{
    char *equalpos = strchr(line,'=');
    char *end;
    if (equalpos == 0) {
	NALog(LOG_ERR,"Bad line in SPW 3.0 file: '%s'\n", line);
	return 0;
    }
    for (end=equalpos-1;(end>line)&&isspace(*end);end--)
	;
    end[1] = 0;
    for (equalpos++;isspace(*equalpos);equalpos++)
	;
    return equalpos;
}

#define MAXLINE 200

/* Read the header from the file */
static int SPW3_rdheader P((SampFile *this));
static int SPW3_rdheader(this)
SampFile *this;
{
    char line[MAXLINE];
    enum sections { PREFACE, USER_COMMENT, COMMON_INFO, DATA_INFO, DATA } section = PREFACE;
    static char *section_names[] = { "","$USER_COMMENT","$COMMON_INFO","$DATA_INFO",
				     "$DATA" };

    this->encoding = 0;
    this->precision = 16;
    this->sampleSize = 0;
    this->channels = 0;
    this->cmd[0] = 0;
    if (getline(this->fd,line,sizeof(line)) < 12) {
	NALog(LOG_ERR,"Unable to read ident from SPW 3.0 file, %s: %s\n",
	      this->name, NAsyserr());
	exit(5);
    }
    if (strncmp(line,"$SIGNAL_FILE",12) != 0) {
	NALog(LOG_ERR,"File %s is not a SPW version 3.0 file.\n", this->name);
	exit(5);
    }
    while (section != DATA) {
	if (getline(this->fd,line,sizeof(line)) < 0) {
	    NALog(LOG_ERR,"SPW 3.0 header on %s ended prematurely\n", this->name);
	    exit(5);
	}
	if (line[0] == '$') {
	    /* Section */
	    int i;
	    for (i=0;i<sizeof(section_names)/sizeof(section_names[0]);i++) {
		if (strcmp(line,section_names[i]) == 0)
		    section = (enum sections)i;
	    }
	}
	else {
	    char *value;
	    switch (section) {
	      case PREFACE:
		Debug("SPW",1,"Unexpected preface line in SPW 3.0 file: '%s'\n", line);
		break;
	      case USER_COMMENT:
		strncat(this->cmd,line,sizeof(this->cmd)-strlen(this->cmd)-1);
		break;
	      case COMMON_INFO:
		value = breakline(line);
		if (strcmp(line,"Sampling Frequency") == 0)
		    this->srate = atof(value);
		break;
	      case DATA_INFO:
		value = breakline(line);
		if (strcmp(line,"Signal Type") == 0) {
		    if (strcmp(value,"Double") == 0) {
			this->sampleSize = 8*sizeof(double);
			this->read = SPW3_read_dbl;
		    }
		    else if (strcmp(value,"Float") == 0) {
			this->sampleSize = 8*sizeof(float);
			this->read = SPW3_read_float;
		    }
		    else if (strcmp(value,"Fixedpoint") == 0) {
			/* Assume it's 16-bits/sample */
			this->sampleSize = 8*sizeof(short);
			this->read = samp_read_16_B;
		    }
		    else {
			NALog(LOG_ERR,"%s: SPW 3.0 Signal Type '%s' unknown\n",
			      this->name, value);
		    }
		}
		else if (strcmp(line,"Number of points") == 0)
		    this->totalSamples = atol(value);
		else if (strcmp(line,"Complex Format") == 0)
		    this->channels = 3;
		break;
	      case DATA:
		break;
	    }
	}
    }
    if (this->sampleSize == 0) {
	NALog(LOG_ERR,"%s: Signal Type not specified in file header\n",
	      this->name);
	exit(5);
    }
    if (this->channels == 3)
	this->sampleSize *= 2;
    this->headerLength = tell(this->fd);
    Debug("comdisco3",1,"sampleSize=%d, channels=%d, samples=%ld\n",
	  this->sampleSize, this->channels, (long)this->totalSamples);
    return 0;
}

/*
 * Output operations
 */

static void putline P((int fd, char *s));
static void putline(fd,s)
int fd;
char *s;
{
    write(fd,s,(unsigned int)strlen(s));
    write(fd,"\n",1);
}

/* Write the header */
static int SPW3_wrheader P((SampFile *this, unsigned int32 nsamples));
static int SPW3_wrheader(this,nsamples)
SampFile *this;
unsigned int32 nsamples;
{
    time_t t;
    char buf[100];
    int numchannels = (this->channels == 3)?2:1;

    if (this->encoding == -1)
	this->encoding = 0;
    if (this->precision == -1)
	this->precision = 16;
    this->sampleSize = this->precision * numchannels;
    putline(this->fd, "$SIGNAL_FILE 9");
    putline(this->fd, "$USER_COMMENT");
    time(&t);
    sprintf(buf,"Recorded via DAT-Link on %s",ctime(&t));
    putline(this->fd, buf);
    putline(this->fd, "$COMMON_INFO");
    putline(this->fd, "SPW Version        = 3.00");
#ifdef sparc
    putline(this->fd, "System Type        = sun4");
#endif
    sprintf(buf,      "Sampling Frequency = %g", this->srate);
    putline(this->fd, buf);
    putline(this->fd, "Starting Time      = 0");
    putline(this->fd, "$DATA_INFO");
    this->totalSamples = nsamples;
    sprintf(buf, "Number of points   = %010ld",(long)nsamples);
    putline(this->fd,buf);
    putline(this->fd, "Signal Type        = Fixedpoint");
    putline(this->fd, "Fixed Point Format = <16,0,t>");
    if (numchannels == 2)
        putline(this->fd, "Complex Format     = Real_Imag");
    putline(this->fd,"$DATA");
    this->headerLength = tell(this->fd);
    return 0;
}

/* 
 * Write trailer - actually rewrites header.
 */
static int SPW3_wrtrailer P((SampFile *this));
static int SPW3_wrtrailer(this)
SampFile *this;
{
    unsigned int32 nsamples = (tell(this->fd)-this->headerLength)/
	(2*((this->channels==3)?2:1));
    if (nsamples != this->currentSample)
	NALog(LOG_ERR,
		"File position indicates %ld samples, but struct says %ld\n",
		(long)nsamples,(long)this->currentSample);
    if (lseek(this->fd,0L,0) < 0)
	return -1;
    /* Rewrite header over original */
    if (this->totalSamples != nsamples) {
	Debug("comdisco3",1,"Updating header for %ld samples instead of %ld\n",
	      (long)nsamples,(long)this->totalSamples);
	return SPW3_wrheader(this,nsamples);
    }
    return 0;
}

/* 
 * Set up for SPW3
 */
void SPW3_construct(this)
SampFile *this;
{
    this->wrheader = SPW3_wrheader;
    this->wrtrailer = SPW3_wrtrailer;
    this->rdheader = SPW3_rdheader;
    this->write = samp_write_16_B;
}
