/* $Id: comdisco.c,v 1.39 1996/03/21 16:40:28 bst Exp $
 *
 * Copyright 1991 Brent Townshend (bst%tt@cam.org)
 * Townshend Computer Tools
 * Montreal, Quebec
 *
 * Sun Nov 3 20:42:31 EST 1991
 *
 * Interface to Comdisco's SPW system
 */
#define NEEDS_TIME
#include <stdio.h>
#include <assert.h>
#include <math.h>
#include <string.h>
#include "sysdefs.h"
#include "sampfile.h"
#include "debug.h"
#include "defaults.h"
#include "spw_signal.h"
#include "support.h"
#include "formats.h"

#define HPP ((struct spw_signal *)this->private)

double *dleft=0,*dright=0;
int leftSize=0, rightSize=0;

void DataAlloc(nbytes,stereo)
size_t nbytes;
int stereo;
{
    if (nbytes > leftSize) {
	if (leftSize == 0)
	    dleft = (double *)malloc(nbytes);
	else
	    dleft = (double *)realloc(dleft,nbytes);
	if (dleft == 0) {
	    NALog(LOG_ERR,"Out of memory\n");
	    exit(1);
	}
	leftSize = nbytes;
    }
    if (stereo && (nbytes > rightSize)) {
	if (rightSize == 0)
	    dright = (double *)malloc(nbytes);
	else
	    dright = (double *)realloc(dright,nbytes);
	if (dright == 0) {
	    NALog(LOG_ERR,"Out of memory\n");
	    exit(1);
	}
	rightSize = nbytes;
    }
}

int32 DataIO(this,fn,nbytes,nsamples,stereo)
SampFile *this;
int32 (*fn)P((int,char*,int32));
int32 nbytes,nsamples;
int stereo;
{
    int nr;
    Debug("DataIO",2,"stereo=%d, nbytes=%ld, nsamples=%ld\n", stereo,
	  (long)nbytes, (long)nsamples);
    if ((nr=(*fn)(this->fd,(char *)dleft,nbytes)) != nbytes) {
	NALog(LOG_ERR,"I/O Error on left channel (%ld/%ld)\n",(long)nr,
	      (long)nbytes);
	exit(1);
    }
    
    if (stereo) {
	/* Advance to imag array */
	if ((*this->seek)(this,this->currentSample+this->totalSamples) < 0)
	    return -1;
	
	/* Read/Write imaginary (right) data */
	if ((*fn)(this->fd,(char *)dright,nbytes) != nbytes) {
	    NALog(LOG_ERR,"I/O Error on right channel\n");
	    exit(1);
	}
	
	/* Seek to next sample in real array */
	if ((*this->seek)(this,this->currentSample-this->totalSamples+nsamples) < 0)
	    return -1;
    }
    else
	this->currentSample += nsamples;

    return 0;
}

static int32 SPW_read_dbl(this,buf,nbytes)
SampFile *this;
byte *buf;
int32 nbytes;
{
    int stereo = TYPE_IS_COMPLEX(HPP->type);
    int32 nsamples = nbytes/(sizeof(short)*(stereo?2:1));

    if (nsamples > this->totalSamples-this->currentSample) {
	nsamples = this->totalSamples-this->currentSample;
	nbytes = nsamples*sizeof(short)*(stereo?2:1);
    }

    if (nbytes == 0)
	return 0;

    assert(this->sampleSize);
    DataAlloc((size_t)SamplesToBytes(nsamples,this->sampleSize),stereo);
    if (DataIO(this,(int32 (*) P((int,char*,int32)))longread,
	       SamplesToBytes(nsamples,this->sampleSize),nsamples,stereo) < 0)
	return -1;

    if (stereo) {
	register double sc = 32767*this->scale;
	register short *sbuf = (short *)buf;
	register double *dlp = dleft;
	register double *drp = dright;
	register int32 i;

	for (i=0;i<nsamples;i++) {
	    *sbuf++ = sc * *dlp++;
	    *sbuf++ = sc * *drp++;
	}
    }
    else {
	register int32 i;
	register double *dlp = dleft;
	register short *sbuf = (short *)buf;
	register double sc = 32767*this->scale;

	for (i=0;i<nsamples;i++)
	    *sbuf++ = sc * *dlp++;
    }

    return nbytes;
}

static int32 SPW_read_fxp_cplx(this,buf,nbytes)
SampFile *this;
byte *buf;
int32 nbytes;
{
    int32 nsamples = nbytes/(2*sizeof(short));

    if (nsamples > this->totalSamples-this->currentSample) {
	nsamples = this->totalSamples-this->currentSample;
	nbytes = nsamples*sizeof(short)*2;
    }
    if (nbytes == 0)
	return 0;

    assert(this->sampleSize);
    DataAlloc((size_t)SamplesToBytes(nsamples,this->sampleSize),1);
    if (DataIO(this,(int32 (*) P((int,char*,int32)))longread,
	       SamplesToBytes(nsamples,this->sampleSize),
	       nsamples,1) < 0)
	return -1;

    if (this->scale == 1) {
	register short *slp = (short *)dleft;
	register short *srp = (short *)dright;
	register short *sbuf = (short *)buf;
	register int32 i;

	for (i=0;i<nsamples;i++) {
	    *sbuf++ = *slp++;
	    *sbuf++ = *srp++;
	}
    }
    else {
	register double sc = this->scale;
	register short *slp = (short *)dleft;
	register short *srp = (short *)dright;
	register short *sbuf = (short *)buf;
	register int32 i;

	for (i=0;i<nsamples;i++) {
	    *sbuf++ = (short)(sc * *slp++);
	    *sbuf++ = (short)(sc * *srp++);
	}
    }
    return nbytes;
}

static int32 SPW_read_fxp_scalar(this,buf,nbytes)
SampFile *this;
byte *buf;
int32 nbytes;
{
    int32 nsamples = nbytes/sizeof(short);

    if (nsamples > this->totalSamples-this->currentSample) {
	nsamples = this->totalSamples-this->currentSample;
	nbytes = nsamples*sizeof(short);
    }
    if (nbytes == 0)
	return 0;

    if (HPP->flag.packed && (HPP->sdata.fixed.numbits == 16)) {
	/* Special case for short read/write */
	if (longread(this->fd,(char *)buf,(unsigned int32)nbytes) != nbytes)
	    return -1;
	if (this->scale != 1) {
	    register int32 i;
	    register short *sbuf = (short *)buf;
	    register double sc = this->scale;
	    
	    for (i=0;i<nsamples;i++,sbuf++)
		*sbuf = (short)(*sbuf * sc);
	}
	this->currentSample += nsamples;
    }
    else if (HPP->flag.packed) {
	/* General case, any number of packed bits/sample <= 32 */
	int bitsRemaining;	/* Bits remaining in current word */
	int numbits = HPP->sdata.fixed.numbits;
	unsigned int *curword;
	int32 nread = (nsamples*numbits+7)/8+1;
	       /* Number of bytes to read from file including 1 prior byte */
	short *sbuf = (short *)buf;
	unsigned short mask;

	/* Setup a mask to zero low bits if numbits<16 */
	if (numbits < 16)
	    mask = (0xffff >> (16-numbits))<<(16-numbits);
	else
	    mask = 0xffff;
	/* Read one extra byte at beginning to continue where we last left off */
	if (lseek(this->fd,-1L,1) < 0) {
	    NALog(LOG_ERR,"lseek failed: %s\n",NAsyserr());
	    return -1;
	}
	/* Leave extra space for addressing past end */
	DataAlloc((size_t)(nread+4),0);
	if (DataIO(this,(int32 (*) P((int,char*,int32)))longread,
		   nread,nsamples,0) < 0)
	    return -1;
	curword = (unsigned int *)dleft;
	/* Compute how many bits remain to be used in curword[0] (24..31) */
	bitsRemaining = 24+(8-(this->currentSample*numbits)%8)%8;
	while (nsamples--) {
	    if (bitsRemaining <= 0) {
		bitsRemaining += 32;
		curword++;
	    }
	    if (bitsRemaining < 16)
		/* Be careful not to use -ve shifts, or shifts >= 32, 
		   they're implementation dep. */
		*sbuf++ = ((curword[0]<<(16-bitsRemaining))|
			   (curword[1]>>(16+bitsRemaining))) & mask;
	    else
		*sbuf++ = (curword[0]>>(bitsRemaining-16)) & mask;

	    bitsRemaining -= numbits;
	}
    }
    else {
	/* Not packed */
	assert(0);
    }

    return nbytes;
}

static int32 SPW_write_fxp_cplx(this,buf,nbytes)
SampFile *this;
const byte *buf;
int32 nbytes;
{
    int32 nsamples = nbytes/(sizeof(short)*2);
    register short *slp,*srp;
    register short *sbuf = (short *)buf;
    register int32 i;

    assert(this->sampleSize);
    DataAlloc((size_t)SamplesToBytes(nsamples,this->sampleSize),1);

    slp = (short *)dleft;
    srp = (short *)dright;
    for (i=0;i<nsamples;i++) {
	*slp++ = *sbuf++;
	*srp++ = *sbuf++;
    }
    
    if (DataIO(this,(int32 (*) P((int,char*,int32)))longwrite,
	       SamplesToBytes(nsamples,this->sampleSize),nsamples,1) < 0)
	return -1;

    return nbytes;
}

static int32 SPW_write_dbl(this,buf,nbytes)
SampFile *this;
const byte *buf;
int32 nbytes;
{
    int stereo = TYPE_IS_COMPLEX(HPP->type);
    int32 nsamples = nbytes/(sizeof(short)*(stereo?2:1));
    register double *dlp,*drp;
    register short *sbuf = (short *)buf;
    register int32 i;

    assert(this->sampleSize);
    DataAlloc((size_t)SamplesToBytes(nsamples,this->sampleSize),stereo);

    if (stereo) {
	dlp = dleft;
	drp = dright;
	for (i=0;i<nsamples;i++) {
	    *dlp++ = *sbuf++/32767.0;
	    *drp++ = *sbuf++/32767.0;
	}
    }
    else {
	dlp = dleft;
	for (i=0;i<nsamples;i++)
	    *dlp++ = *sbuf++/32767.0;
    }
    
    if (DataIO(this,(int32 (*)P((int,char*,int32)))longwrite,
	       SamplesToBytes(nsamples,this->sampleSize),nsamples,stereo) < 0)
	return -1;

    return nbytes;
}

static int SPW_open(this)
SampFile *this;
{
    Debug("SPW",1,"Read SPW signal: %s\n",this->name);
    if (this->name)
	this->fd = open(this->name,O_RDONLY|O_BINARY);
    else
	this->fd = 0;
    if (this->fd < 0) {
	NALog(LOG_ERR,"Unable to open %s.\n",this->name);
	exit(5);
    }
    return (*this->rdheader)(this);
}

static int sig_ReadRev4HeaderNA P((int fd, struct spw_signal *sig));
static int sig_ReadRev4HeaderNA(fd, sig)
int fd;
struct spw_signal *sig;
{
  struct spw_signal_rev4 sig4;

  if (lseek(fd, 0L, 0) < 0) { /* reset to beginning of file */
      NALog(LOG_ERR,"lseek failed: %s\n", NAsyserr());
      return -1;
  }

  if (read(fd,(char *)&sig4, sizeof(sig4)) != sizeof(sig4)) {
      NALog(LOG_ERR,"Error reading rev 4 header\n");
      return -1;
  }

  if (sig4.magic != MAGIC)
    return -1;

  sig->magic = MAGIC;
  sig->samp_freq = sig4.samp_freq;
  sig->num_of_pts = sig4.num_of_pts;
  sig->sdata.fixed.exponent = sig4.sdata.fixed.exp;
  sig->sdata.fixed.numbits = sig4.sdata.fixed.numbits;

  if (TYPE_IS_COMPLEX(sig4.type))
      sig->type = TYPE_COMPLEX;
  else
      sig->type = TYPE_REAL;

  switch(sig4.size) {
    case 1 : /* INTEGER */
      sig->type |= TYPE_INTEGER;
      break;
    case 2 : /* LOGICAL */
      sig->type |= TYPE_LOGICAL;
      break;
    case 3 : /* FIXED */
      sig->type |= TYPE_FXDPT;
      break;
    case 4 : /* FLOAT */
      sig->type |= TYPE_FLOAT;
      break;
    default :
    case 0 : /* DOUBLE */
      sig->type |= TYPE_DOUBLE;
      break;
  }
  return 0;
}

static int SPW_rdheader(this)
SampFile *this;
{
    int revision;
    int commentLength;

    Debug("SPW",1,"Read SPW header from %s\n",this->name);
    this->precision = 16;  /* Use 16 bit transfers */
    this->encoding = 0;

    this->private = (char *)malloc(sizeof(*HPP));
    if (read(this->fd,(char *)HPP,sizeof(struct spw_signal)) !=	sizeof(*HPP))
	return -1;
    if (HPP->magic != MAGIC) {
	NALog(LOG_ERR,"%s: Not an SPW file\n",this->name);
	return -1;
    }
    revision = atoi(HPP->revision);
    if (revision == 4) {
	if (sig_ReadRev4HeaderNA(this->fd,HPP) < 0)
	    return -1;
    }
    else if (revision != 7) {
	NALog(LOG_ERR,"Sorry, can only handle SPW revision 4 or 7 .sig files, not revision %d\n",revision);
	return -1;
    }
    Debug("SPW",1,"Read revision %d SPW file from %s\n",revision,this->name);
    
    if (TYPE_IS_FXDPT(HPP->type)) {
	Debug("SPW",1,"Fixed point signal (%d,%d)\n",
	      HPP->sdata.fixed.numbits,HPP->sdata.fixed.exponent);
	if (HPP->flag.packed) {
	    this->sampleSize = HPP->sdata.fixed.numbits;

	    this->scale *= pow(2.0,HPP->sdata.fixed.exponent-1.0);
	    if (TYPE_IS_COMPLEX(HPP->type) && (HPP->sdata.fixed.numbits != 16)) {
		NALog(LOG_ERR,"Unable to read packed, fixed-point SPW files with %d bits/sample.\n",HPP->sdata.fixed.numbits);
		return -1;
	    }
		
	    Debug("SPW",1,"Using FXP read with scale=%f\n",this->scale);
	}
	else {
	    NALog(LOG_ERR,"Unable to read unpacked, fixed-point SPW files\n");
	    return -1;
	}
    }
    else if (TYPE_IS_DOUBLE(HPP->type))
	this->sampleSize = sizeof(double)*8;
    else {
	NALog(LOG_ERR,"Unrecognized SPW file data type: 0x%x\n",HPP->type);
	return -1;
    }

    if (TYPE_IS_COMPLEX(HPP->type)) {
	this->channels = 3;
	Debug("SPW",1,"Complex signal\n");
	if (TYPE_IS_FXDPT(HPP->type))
	    this->read = SPW_read_fxp_cplx;
	else
	    this->read = SPW_read_dbl;
    }
    else {
	this->channels = 0;
	Debug("SPW",1,"Scalar signal\n");
	if (TYPE_IS_FXDPT(HPP->type))
	    this->read = SPW_read_fxp_scalar;
	else
	    this->read = SPW_read_dbl;
    }

    if (this->srate < 0)
	this->srate = HPP->samp_freq;

    /* Get comment length */
    read(this->fd,(char *)&commentLength,sizeof(commentLength));
    /* Skip remainder of comment */
    if (lseek(this->fd,(long)commentLength,1) < 0)
	return -1;

    /* Get header length */
    this->headerLength = tell(this->fd);
    Debug("SPW",1,"Header length = %ld bytes\n",(long)this->headerLength);
    
    this->totalSamples = HPP->num_of_pts;
    this->currentSample = 0;
    return 0;
}

static int SPW_create(this)
SampFile *this;
{
    char buf[100];
    time_t t;

    if (this->name)
	this->fd = open(this->name,O_WRONLY|O_CREAT|O_TRUNC|O_BINARY
#ifndef macintosh
			,0644
#endif /* macintosh */
			);
    else
	this->fd = 0;
    if (this->fd < 0) {
	NALog(LOG_ERR,"Open of %s failed: %s\n",this->name,NAsyserr());
	return -1;
    }

    this->private = (char *)malloc(sizeof(*HPP));
    strcpy(HPP->revision,"7");
    HPP->magic = MAGIC;
    HPP->starttime = 0;
    HPP->filename = this->name;
    if (GetBoolDefault("SPW.WriteFixed",True)) {
	HPP->type = TYPE_FXDPT;
	HPP->sdata.fixed.numbits = 16;
	HPP->sdata.fixed.exponent = 1;
	HPP->sdata.fixed.format = 1;  /* Twos-complement */
	HPP->flag.packed = 1;
	Debug("SPW",1,"Create fixed-point SPW signal: %s\n",this->name);
    }
    else {
	HPP->type = TYPE_DOUBLE;
	Debug("SPW",1,"Create double SPW signal: %s\n",this->name);
    }
    time(&t);
    sprintf(buf,"Recorded via DAT-Link on %s",ctime(&t));
    HPP->comment = malloc(strlen(buf)+1);
    strcpy(HPP->comment,buf);
    return 0;
}

static int SPW_wrheader(this,nsamples)
SampFile *this;
unsigned int32 nsamples;
{
    int commentLength;
    if (this->precision == -1)
	this->precision = 16;
    if (this->encoding == -1)
	this->encoding = 0;
    if ((this->encoding != 0) || (this->precision != 16)) {
	NALog(LOG_ERR,"Sorry, only 16-bit, linear supported for SPW files\n");
	return -1;
    }

    if (((int32)nsamples) == -1) {
	NALog(LOG_ERR,"Number of samples to transfer must be specified in advance for SPW format\n");
	exit(1);
    }
    this->totalSamples = HPP->num_of_pts = HPP->alloc_size = nsamples;
    this->currentSample = 0;
    HPP->samp_freq = this->srate;
    if (this->channels == 3) {
	HPP->type |= TYPE_COMPLEX;
	if (TYPE_IS_DOUBLE(HPP->type))
	    this->write = SPW_write_dbl;
	else
	    this->write = SPW_write_fxp_cplx;
    }
    else if (TYPE_IS_DOUBLE(HPP->type))
	this->write = SPW_write_dbl;
    /* else use SampFile_write */

    if (write(this->fd,(char *)HPP,sizeof(*HPP)) != sizeof(*HPP)) {
	NALog(LOG_ERR,"Unable to write SPW header to %s\n",this->name);
	return -1;
    }
    /* Write comment */
    commentLength = strlen(HPP->comment);
    if (write(this->fd,(char *)&commentLength,sizeof(commentLength)) !=
	sizeof(commentLength))
	return -1;
    if (write(this->fd,HPP->comment,(unsigned int32)commentLength) !=
	commentLength)
	return -1;
    this->totalSamples = nsamples;
    if (TYPE_IS_DOUBLE(HPP->type))
	this->sampleSize = sizeof(double)*8;
    else
	this->sampleSize = sizeof(short)*8;
    /* Get header length */
    this->headerLength = tell(this->fd);
    Debug("SPW",1,"Header length=%ld\n",(long)this->headerLength);
    return 0;
}

void SPW_construct(this)
SampFile *this;
{
    this->scale = 1.0;
    this->open = SPW_open;
    this->create = SPW_create;
    this->rdheader = SPW_rdheader;
    this->wrheader = SPW_wrheader;
}

#if 0
/* Override SPW csigReadSignal() so we can read the file locally even if the
   SPW libraries are linked in */
int csigReadSignal(cp_filename, hpp)
char *cp_filename;
struct spw_signal **hpp;
{
    Debug("SPW",1,"csigReadSignal(%s) ignored.\n",cp_filename);   
    *hpp = (struct spw_signal *)malloc(sizeof(*hpp));
    (*hpp)->filename = malloc(strlen(cp_filename)+1);
    strcpy((*hpp)->filename,cp_filename);
    return 0;
}
#endif
