/* $Id: aifc.c,v 1.17 1993/10/26 16:15:53 bst Exp $
 *
 * Copyright 1992 Brent Townshend (bst%tt@cam.org)
 * Townshend Computer Tools
 * Montreal, Quebec
 *
 * Sun Feb 16 23:44:07 EST 1992
 *
 * Revision History: $Log: aifc.c,v $
 * Revision 1.17  1993/10/26  16:15:53  bst
 * Added coercions for GCC 2.5.0
 *
 * Revision 1.16  1993/08/10  02:25:08  bst
 * Coerce PutInt() arg2 to in32.
 * Coerce %l args to long.
 * Coerce lseek() arg2 to long
 * int32 vs int changes.
 *
 * Revision 1.15  1993/06/23  20:10:15  bst
 * Changed 'long' to 'int32'
 *
 * Revision 1.14  1993/06/09  14:14:18  bst
 * Changed some types, added some coercions.
 *
 * Revision 1.13  1993/06/01  02:45:11  bst
 * Added coercions for GCC 2.4.1 unsigned 'size_t'.
 *
 * Revision 1.12  1993/05/18  19:43:25  bst
 * Use NALog() to log/print errors and warnings.
 *
 * Revision 1.11  1993/04/22  18:39:00  bst
 * Added some coercions.
 * Added static function prototypes.
 *
 * Revision 1.10  1993/04/22  03:21:05  bst
 * Moved IEEE declarations to ieee_extend.h
 *
 * Revision 1.9  1993/04/12  02:11:35  bst
 * Use long when 32-bit integers required.
 *
 * Revision 1.8  1992/08/28  03:14:02  bst
 * Coerce chars to unsigned int before shifting more than 8 bits.
 *
 * Revision 1.7  1992/08/19  00:22:19  bst
 * Removed include of config.h (sysdefs.h gets it).
 *
 * Revision 1.6  1992/07/05  03:13:32  bst
 * Use lseek() to move file pointer to beginning of data instead of this->seek
 *     (this->seek)() though file was already positiioned.
 *
 * Revision 1.5  1992/05/10  19:02:58  bst
 * Added encoding, precision.
 *
 * Revision 1.4  1992/04/09  17:56:09  bst
 * Coerce first arg to strncmp() to const char *
 * Remove goto
 * Added missing arg to strncmp() call for check if compression == NONE
 *
 * Revision 1.3  1992/04/07  22:54:27  bst
 * Added int coercions of AIFF and AIFC enum elements.
 * Removed extraneous %0 from Debug() formats.
 *
 * Revision 1.2  1992/03/02  03:59:59  bst
 * Added missing args to error printf.
 *
 * Revision 1.1  1992/02/17  04:45:04  bst
 * Initial revision
 *
 */
#include <stdio.h>
#include <string.h>
#include "sysdefs.h"
#include "sampfile.h"
#include "debug.h"
#include "formats.h"
#include "support.h"
#include "ieee_extend.h"

#define AIFC_VERSION 0xa2805140

typedef struct Chunk_COMM {
    short numChannels;
    unsigned int32 numSampleFrames;
    short sampleSize;
    double sampleRate;
    char compressionType[4];
    unsigned char compressionNameLength;
    char compressionName[255];
} Chunk_COMM;


static unsigned int32 GetInt P((const unsigned char *s));
static unsigned int32 GetInt(s)
const unsigned char *s;
{
    return (((unsigned int32)s[0])<<24)|(((unsigned int32)s[1])<<16)|
	(((unsigned int32)s[2])<<8)|((unsigned int32)s[3]);
}

static unsigned short GetShort P((const unsigned char *s));
static unsigned short GetShort(s)
const unsigned char *s;
{
    return (((unsigned int)s[0])<<8)|s[1];
}

/*
 * Input operations
 */
/* Read the header from the file */
static int AIFC_rdheader P((SampFile *this));
static int AIFC_rdheader(this)
SampFile *this;
{
    char buf[12];
    int found_comm = 0;
    int found_ssnd = 0;
    int32 aifc_version = -1;
    int32 formLength;
    int32 nread;
    int nr;

    this->precision = 16;	/* Use 16 bit transfers */
    this->encoding = 0;

    if ((nr=read(this->fd,buf,12)) != 12) {
	if (nr < 0)
	    NALog(LOG_ERR,"read failed: %s\n",NAsyserr());
	return -1;
    }
    if (strncmp((const char *)buf,"FORM",4) != 0) {
	NALog(LOG_ERR,"%s is not an AIFF or AIFF-C file\n",this->name);
	return -1;
    }
    formLength = GetInt((const unsigned char *)&buf[4]);
    if (strncmp((const char *)&buf[8],"AIFC",4) == 0) {
	this->format = (int)AIFC;
    }
    else if (strncmp((const char *)&buf[8],"AIFF",4) == 0) {
	this->format = (int)AIFF;
    }
    else {
	NALog(LOG_ERR,"%s is a FORM file, but not an AIFF or AIFF-C file\n",
		this->name);
	return -1;
    }
    nread = 4;
    /* Locate all the important chunks */
    while (nread < formLength) {
	int32 ckDataSize;

	if ((nr=read(this->fd,buf,8)) != 8) {
	    if (nr < 0)
		NALog(LOG_ERR,"read failed: %s\n",NAsyserr());
	    break;
	}

	ckDataSize = GetInt((const unsigned char *)&buf[4]);

	Debug("aifc",2,"Reading %s chunk at %ld/%ld: %.4s with %ld data bytes\n",
	      fmtnames[this->format],(long)nread,(long)formLength,buf,
	      (long)ckDataSize);
	if (strncmp((const char *)buf,"COMM",4) == 0) {
	    unsigned char common_buf[512];
	    Chunk_COMM common;

	    if (found_comm == 1) {
		NALog(LOG_ERR,"%s: Malformed %s file, multiple COMM chunks.\n",
			this->name,fmtnames[this->format]);
		return -1;
	    }
	    found_comm = 1;
	    if (ckDataSize > sizeof(common)) {
		NALog(LOG_ERR,"%s: Bad COMM chunk size: %ld\n",
			this->name,(long)ckDataSize);
		return -1;
	    }
	    if (longread(this->fd,(char *)common_buf,
			 (unsigned int32)ckDataSize) != ckDataSize) {
		NALog(LOG_ERR,"%s: Error reading COMM chunk\n",this->name);
		return -1;
	    }
	    common.numChannels = GetShort(&common_buf[0]);
	    switch (common.numChannels) {
	      case 1:
		this->channels = 0;
		break;
	      case 2:
		this->channels = 3;
		break;
	      default:
		NALog(LOG_ERR,"%s: %s file with %d channels not supported\n",
			this->name,fmtnames[this->format],common.numChannels);
	    }

	    common.numSampleFrames = GetInt(&common_buf[2]);
	    this->totalSamples = common.numSampleFrames;

	    common.sampleSize = GetShort(&common_buf[6]);
	    this->sampleSize = common.sampleSize;
	    if ((this->sampleSize < 1) || (this->sampleSize > 32)) {
		NALog(LOG_ERR,"%s: Illegal sample size: %d bits\n",
			this->name,this->sampleSize);
		return -1;
	    }
	    if (this->sampleSize != 16)
		this->read = samp_read_N_B;

	    common.sampleRate =
		ConvertFromIeeeExtended((unsigned char *)&common_buf[8]);
	    this->srate = common.sampleRate;
	    if (this->format == (int)AIFC) {
		memcpy(common.compressionType,&common_buf[18],4);
		memcpy(common.compressionName,&common_buf[23],
		       (size_t)common_buf[22]);
		common.compressionName[common_buf[22]] = 0;
		if (strncmp((const char *)common.compressionType,"NONE",4) != 0) {
		    NALog(LOG_ERR,
			    "%s: Unable to read AIFC files compressed with %s\n",
			    this->name,common.compressionName);
		    return -1;
		}
	    }
	}
	else if ((this->format == (int)AIFC) && (strncmp((const char *)buf,"FVER",4) == 0)) {
	    char version_buf[4];
	    if ((nr=read(this->fd,version_buf,4)) != 4) {
		if (nr < 0)
		    NALog(LOG_ERR,"read failed: %s\n",NAsyserr());
		return -1;
	    }
	    aifc_version = GetInt((const unsigned char *)version_buf);
	}
	else if (strncmp((const char *)buf,"SSND",4) == 0) {
	    char ssnd_buf[8];
	    unsigned int32 offset,blockSize;
	    if (found_ssnd == 1) {
		NALog(LOG_ERR,"%s: Malformed %s file, multiple SSND chunks.\n",
			this->name,fmtnames[this->format]);
		return -1;
	    }
	    found_ssnd = 1;
	    if ((nr=read(this->fd,ssnd_buf,8)) != 8) {
		NALog(LOG_ERR,"%s: Error while reading SSND chunk\n",
			this->name);
		if (nr < 0)
		    NALog(LOG_ERR,"read failed: %s\n",NAsyserr());
		return -1;
	    }
	    offset = GetInt((const unsigned char *)ssnd_buf);
	    blockSize = GetInt((const unsigned char *)&ssnd_buf[4]);
	    if (blockSize != 0) {
		NALog(LOG_ERR,"%s: Sorry, block-aligned %s files not supported\n",
			this->name,fmtnames[this->format]);
		return -1;
	    }
	    /* FD is positioned at first byte of soundData */
	    this->headerLength = tell(this->fd)+offset;
	    /* Advance to next chunk */
	    if (lseek(this->fd,(long)((ckDataSize+1)&~1L)-8,1) < 0) {
		NALog(LOG_ERR,"lseek failed: %s\n", NAsyserr());
		return -1;
	    }
	}
	else if (strncmp((const char *)buf,"AESD",4) == 0) {
	    if (ckDataSize != 24) {
		NALog(LOG_ERR,"%s: AESD chunk has %ld data bytes instead of 24\n",
			this->name,(long)ckDataSize);
		/* Advance to next chunk */
		if (lseek(this->fd,(long)(ckDataSize+1)&~1L,1) < 0) {
		    NALog(LOG_ERR,"lseek failed: %s\n", NAsyserr());
		    break;
		}
	    }
	    if ((nr=read(this->fd,(char *)this->leftCS,24)) != 24) {
		if (nr < 0)
		    NALog(LOG_ERR,"read failed: %s\n",NAsyserr());
		break;
	    }
	    memcpy(this->rightCS,this->leftCS,24);
	    this->cstatValid = 1;
	}
	else {
	    Debug("aifc",1,"Skipping unknown %s chunk: %.4s\n",fmtnames[this->format],buf);
	    
	    /* Advance to next chunk */
	    if (lseek(this->fd,(long)(ckDataSize+1)&~1L,1) < 0) {
		NALog(LOG_ERR,"lseek failed: %s\n",NAsyserr());
		break;
	    }
	}
	nread += 8+((ckDataSize+1)&~1);
    }
    if ((nread != formLength) && (nread != formLength-8)) {
	/* Note that someone's been creating AIFF files with the FORM size
	    8 bytes too large! */
	NALog(LOG_ERR,"%s: Malformed %s file: incorrect size.\n",
		this->name,fmtnames[this->format]);
    }

    if (!found_comm || !found_ssnd) {
	/* Missing some important chunks */
	if ((this->format == (int)AIFC) && aifc_version &&
	    (aifc_version != AIFC_VERSION)) {
	    NALog(LOG_ERR,
		    "%s: Unrecognized AIFC version: 0x%lx instead of 0x%lx\n",
		    this->name,(long)aifc_version,(long)AIFC_VERSION);
	    return -1;
	}
	if (!found_comm) {
	    NALog(LOG_ERR,"%s: Malformed %s file, missing COMM chunk.\n",
		    this->name,fmtnames[this->format]);
	    return -1;
	}
	if (!found_ssnd) {
	    NALog(LOG_ERR,"%s: Malformed %s file, missing SSND chunk.\n",
		    this->name,fmtnames[this->format]);
	    return -1;
	}
    }
    /* Position file-pointer to beginning of data */
    if (lseek(this->fd,(long)this->headerLength,0) < 0) {
	NALog(LOG_ERR,"lseek failed: %s\n", NAsyserr());
	return -1;
    }
    return 0;
}

/*
 * Output operations
 */

static void PutStr P((int fd, char *s, int n));
static void PutStr(fd,s,n)
int fd;
char *s;
int n;
{
    int nw;
    if ((nw=write(fd,s,(unsigned int)n)) != n) {
	if (nw < 0)
	    NALog(LOG_ERR,"PutStr: write failed: %s\n", NAsyserr());
	exit(1);
    }
}


static void PutInt P((int fd, unsigned int32 x));
static void PutInt(fd,x)
int fd;
unsigned int32 x;
{
    char buf[4];
    buf[0] = (x>>24);
    buf[1] = (x>>16)&0xff;
    buf[2] = (x>>8)&0xff;
    buf[3] = x&0xff;
    PutStr(fd,buf,4);
}

static void PutShort P((int fd, int x));
static void PutShort(fd,x)
int fd;
int x;
{
    char buf[4];
    buf[0] = (x>>8);
    buf[1] = x&0xff;
    PutStr(fd,buf,2);
}

static void PutByte P((int fd, int x));
static void PutByte(fd,x)
int fd;
int x;
{
    PutStr(fd,(char *)&x,1);
}

/* Write the header */
static int AIFC_wrheader P((SampFile *this, unsigned int32 nsamples));
static int AIFC_wrheader(this,nsamples)
SampFile *this;
unsigned int32 nsamples;
{
    int nchannels = (this->channels==3)?2:1;
    int32 ssnd_length = nsamples*nchannels*2 + 16;
    int vers_length = (this->format==(int)AIFC)?12:0;
    int comm_length = (this->format==(int)AIFC)?46:26;
    int aesd_length = this->cstatValid?32:0;
    unsigned int32 length = ssnd_length+vers_length+comm_length+aesd_length+4;
    char srate_buf[10];

    if (this->precision == -1)
	this->precision = 16;
    if (this->encoding == -1)
	this->encoding = 0;
    if ((this->encoding != 0) || (this->precision != 16)) {
	NALog(LOG_ERR,"Sorry, only 16-bit, linear supported for AIFF files\n");
	return -1;
    }

    /* FORM AIFC File */
    PutStr(this->fd,"FORM",4);
    PutInt(this->fd,length);
    if (this->format == (int)AIFC)
	PutStr(this->fd,"AIFC",4);
    else
	PutStr(this->fd,"AIFF",4);

    /* FVER chunk */
    if (this->format == (int)AIFC) {
	PutStr(this->fd,"FVER",4);
	PutInt(this->fd,(unsigned int32)4);
	PutInt(this->fd,AIFC_VERSION);
    }

    /* COMM Chunk */
    PutStr(this->fd,"COMM",4);
    PutInt(this->fd,(unsigned int32)((this->format==(int)AIFC)?38:18));
    PutShort(this->fd,nchannels);
    PutInt(this->fd,nsamples);
    PutShort(this->fd,16);
    ConvertToIeeeExtended(this->srate,srate_buf);
    PutStr(this->fd,srate_buf,10);
    if (this->format == (int)AIFC) {
	PutStr(this->fd,"NONE",4);
	PutByte(this->fd,14);
	PutStr(this->fd,"not compressed",14);
	PutByte(this->fd,0);
    }
    
    /* AESD Chunk */
    if (this->cstatValid) {
	PutStr(this->fd,"AESD",4);
	PutInt(this->fd,(unsigned int32)24);
	PutStr(this->fd,(char *)this->leftCS,24);
    }
    
    /* SSND Chunk */
    PutStr(this->fd,"SSND",4);
    PutInt(this->fd,(unsigned int32)(ssnd_length-8));
    PutInt(this->fd,(unsigned int32)0);
    PutInt(this->fd,(unsigned int32)0);
    this->headerLength = tell(this->fd);
    this->sampleSize = this->precision*nchannels;
    this->totalSamples = nsamples;
    return 0;
}

/* Write a trailer */
static int AIFC_wrtrailer P((SampFile *this));
static int AIFC_wrtrailer(this)
SampFile *this;
{
    unsigned int32 nsamples = (tell(this->fd)-this->headerLength)/
	(2*((this->channels==3)?2:1));
    if (nsamples != this->currentSample)
	NALog(LOG_ERR,
		"File position indicates %ld samples, but struct says %ld\n",
		(long)nsamples,(long)this->currentSample);
    if (lseek(this->fd,0L,0) < 0)
	return -1;
    /* Rewrite header over original */
    if (this->totalSamples != nsamples) {
	Debug("aifc",1,"Updating header for %ld samples instead of %ld\n",
	      (long)nsamples,(long)this->totalSamples);
	return AIFC_wrheader(this,nsamples);
    }
    return 0;
}

/* 
 * Set up for AIFC
 */
void AIFC_construct(this)
SampFile *this;
{
    this->wrheader = AIFC_wrheader;
    this->rdheader = AIFC_rdheader;
    this->wrtrailer = AIFC_wrtrailer;
}

