/* $Id: misc.c,v 2.28 1994/03/07 18:34:51 bst Exp $
 *
 * Copyright 1991 Brent Townshend (bst%tt@cam.org)
 * Townshend Computer Tools
 * Montreal, Quebec
 *
 * Sun Dec 8 15:30:46 EST 1991
 *
 * Revision History: $Log: misc.c,v $
 * Revision 2.28  1994/03/07  18:34:51  bst
 * Added prototype for RcvWait()
 *
 * Revision 2.27  1993/11/16  01:50:30  bst
 * Added PRAGMA_UNUSED
 *
 * Revision 2.26  1993/08/30  18:01:34  bst
 * Fixed uninitialized snapshot variable.
 *
 * Revision 2.25  1993/08/30  14:39:31  bst
 * Use RcvWait() in NADrain() and NABegin()
 * Increased timeout waiting for server to receive data, reset timeout
 *   whenever amount of received data increases.
 *
 * Revision 2.24  1993/08/10  02:14:19  bst
 * Moved min() to support.h
 * Coerce %l args to long.
 * Don't verify transmission count if NADIRECT.
 *
 * Revision 2.23  1993/06/16  17:22:00  bst
 * Removed dead code.
 * Check for errors while waiting for server to receive data.
 * Wait longer (100 vs 10 iterations) for server to receive data.
 *
 * Revision 2.22  1993/06/15  18:23:50  bst
 * Check if data has been sent/received when NABegin is called.
 *
 * Revision 2.21  1993/06/07  14:20:22  bst
 * Moved padding of last block of data back to server side.
 * Wait until all bytes sent to server have been received before sending
 * 	NA_DRAIN driective.
 *
 * Revision 2.20  1993/06/05  02:45:17  bst
 * Added size_t coercion to match memcpy() arg types.
 * Pad output on client side during NADrain() call instead of in server.
 *
 * Revision 2.19  1993/05/26  13:15:44  bst
 * Wait for drain while state is anything but INACTIVE.
 * Return transfer error code as return result.
 *
 * Revision 2.18  1993/05/18  19:41:44  bst
 * Use NALog() to log/print errors and warnings.
 *
 * Revision 2.17  1993/03/01  16:45:48  bst
 * Removed special THINK-C declaration (use enum==int option under Think-C)
 *
 * Revision 2.16  1993/03/01  04:03:46  bst
 * Workaround bug in THINK-C
 *
 * Revision 2.15  1993/02/23  16:45:59  bst
 * Added 'playrec' arg to NASetFilter().
 *
 * Revision 2.14  1993/02/04  23:12:07  bst
 * NADIRECT support.
 *
 * Revision 2.13  1992/10/25  20:58:17  bst
 * No info returned by CONFIGURE_NOTIFY events - use NAGetInfo().
 *
 * Revision 2.12  1992/10/25  18:30:41  bst
 * Use _NAFreeReply() to free reply.
 *
 * Revision 2.11  1992/10/04  17:20:13  bst
 * Added NAGetServerVersion().
 *
 * Revision 2.10  1992/09/08  02:27:42  bst
 * While draining, continue waiting if state == NA_PAUSED.
 *
 * Revision 2.9  1992/07/05  03:09:20  bst
 * DRAIN now waits until state is not ACTIVE, it used to wait until state was INACTIVE.
 *
 * Revision 2.8  1992/04/06  21:56:17  bst
 * Added coercion to match prototype.
 *
 * Revision 2.7  1992/02/15  19:48:56  bst
 * End DRAIN if playback becomes INACTIVE
 * Added NAServer()
 *
 * Revision 2.6  1992/01/28  05:48:51  bst
 * Added NAGetFilterImpulse()
 *
 * Revision 2.5  1991/12/17  04:47:07  bst
 * Copy filter into request rather than passing a pointer.
 *
 * Revision 2.4  1991/12/16  05:23:07  bst
 * NADrain() - send DRAIN request to server.
 * Verify that client is allowed to be calling NADrain()
 *
 * Revision 2.3  1991/12/16  00:42:17  bst
 * NAPause now uses a new request rather than SetInfo.
 * NADrain() always calls NABegin(), regardless of state.
 *
 * Revision 2.2  1991/12/10  22:35:37  bst
 * In NADrain() - Call NABegin() and be careful about flushing old CONFIGURE events
 *
 * Revision 2.1  1991/12/08  21:31:39  bst
 * Network version
 *
 * Revision 1.1  1991/12/08  20:32:14  bst
 * Initial revision
 *
 */
#include <assert.h>
#include "NAlibint.h"

#ifdef NADIRECT
#include "dlio.h"
#endif
#include "support.h"

#ifndef BLOCK_LENGTH
#define BLOCK_LENGTH 512
#endif /* !BLOCK_LENGTH */

/* Pause control */
int NAPause(port,playrec,pause)
NAport *port;
int playrec;
int pause;
{
    NArequest request;

    request.request = NA_PAUSE;
    request.NArequest_u.ps.playrec = playrec;
    request.NArequest_u.ps.pause = pause;
    if (_NASendRequest(port,&request) < 0)
	exit(1);
    return 0;
}

/* Wait until all data sent to server has been received or an error
   has occurred.  Return playback error code (positive) if one occured,
   -1 for other types of errs */
static int RcvWait P((NAport *port));
static int RcvWait(port)
NAport *port;
{
#ifdef NADIRECT
#ifdef PRAGMA_UNUSED
#pragma unused(port)
#endif /* PRAGMA_UNUSED */
#else /* NADIRECT */
    NAsnapshots snap;
    int i,timeout;

    /* Wait until all transmitted bytes have been received by server */
    for (i=0,timeout=0;timeout<100;i++,timeout++) {
	NAinfo info;
	static int oldtx = 0;
	NAGetInfo(port,&info);
	if (info.playback.error != NA_XFRERR_NOERROR)
	    return (int)info.playback.error;

	NAGetSnapshot(port,&snap);
	if (snap.pb.bytesTransferred >= port->bytesSent)
	    break;
	if (snap.pb.bytesTransferred != oldtx)
	    timeout = 0;	/* Restart timeout counter */
    }
    if (snap.pb.bytesTransferred != port->bytesSent) {
	NALog(LOG_ERR,
	      "%ld bytes of data sent to server, but %ld bytes received.\n",
	      (long)port->bytesSent, (long)snap.pb.bytesTransferred);
	return -1;
    }
    if (i>0)
	Debug("NABegin",1,"Took %d iterations to send data\n", i);
#endif /* !NADIRECT */
    return 0;
}

/* Wait for data to drain */
int NADrain(port)
NAport *port;
{
    NArequest request;
#ifndef NADIRECT
    NAevent event;
#endif
    NAinfo info;
    int rwait;

    if ((port->playrec & NA_PLAY) == 0) {
	NALog(LOG_WARNING,"Attempt to drain data on a port not open for playback\n");
	return -1;
    }

    /* Wait until all transmitted bytes have been received by server */
    rwait = RcvWait(port);
    if (rwait != 0)
	return rwait;

    /* Let server know we're waiting for buffer to drain */
    request.request = NA_DRAIN;
    if (_NASendRequest(port,&request) < 0)
	exit(1);

#ifndef NADIRECT
    /* Flush any pending CONFIGURENOTIFY events */
    while (NACheckMaskEvent(port,1<<NA_CONFIGURENOTIFY,&event))
	;
#endif

    NAGetInfo(port,&info);
    /* Wait until an error occurs (usually an underrun) or state changes */
    if (info.playback.error == NA_XFRERR_NOERROR) {
	/* Wait for playback to stop */
#ifndef NADIRECT
	if ((port->eventmask & (1<<NA_CONFIGURENOTIFY)) == 0)
	    /* Enable CONFIGURENOTIFY events */
	    if (NASelectInput(port,port->eventmask | (1<<NA_CONFIGURENOTIFY)) < 0)
		return -1;
#endif /* !NADIRECT */
	do {
#ifndef NADIRECT
	    if (NAMaskEvent(port,1<<NA_CONFIGURENOTIFY,&event) < 0)
		return -1;
#endif /* !NADIRECT */
	    NAGetInfo(port,&info);
	} while ((info.playback.error == NA_XFRERR_NOERROR) &&
		 (info.playback.state != NA_INACTIVE));
    }

    return (int)info.playback.error;
}

/* Immediately flush all buffered data */
int NAFlush(port,playrec)
NAport *port;
int playrec;
{
    NArequest request;

    request.request = NA_FLUSH;
    request.NArequest_u.fl_playrec = playrec;  /* Storage location for info */
    if (_NASendRequest(port,&request) < 0)
	exit(1);
    return 0;
}

/* Immediately begin transfers */
int NABegin(port,playrec)
NAport *port;
int playrec;
{
    NArequest request;

    if (playrec & NA_PLAY) {
	/* Make sure data has been sent to server */
	int rwait;

	if (port->bytesSent == 0) {
	    NALog(LOG_ERR,
		  "NABegin called before any data has been written to DAT-Link\n");
	    return -1;
	}
	if ((rwait = RcvWait(port)) != 0)
	    return rwait;
    }

    request.request = NA_BEGIN;
    request.NArequest_u.beg_playrec = playrec;  /* Storage location for info */
    if (_NASendRequest(port,&request) < 0)
	exit(1);

    return 0;
}

/* Set sample-rate conversion filter */
int NASetFilter(port,playrec,filter)
NAport *port;
int playrec;
const NAfilter *filter;
{
    NArequest request;

    request.request = NA_SETFILTER;
    request.NArequest_u.filter.playrec = playrec;
    memcpy(&request.NArequest_u.filter.filter,filter,sizeof(*filter));
    if (_NASendRequest(port,&request) < 0)
	exit(1);

    return 0;
}

/* Set sample-rate conversion filter */
int NAGetFilterImpulse(port,playrec,impulseRate,response,length)
NAport *port;
int playrec;
float *impulseRate;
float *response;
int *length;
{
    NArequest request;
    NAreply *reply;

    request.request = NA_GETFILTER;
    request.NArequest_u.gf_playrec = playrec;
    if (_NASendRequest(port,&request) < 0)
	exit(1);

    /* Wait for reply */
    if ((reply = _NAReply(port)) == 0)
	exit(1);

    memcpy(response,reply->NAreply_u.response.response.response_val,
	   (size_t)(min(*length,
			reply->NAreply_u.response.response.response_len)*
		    sizeof(*response)));
    *length = reply->NAreply_u.response.response.response_len;
    *impulseRate = reply->NAreply_u.response.rate;
    _NAFreeReply(reply);
    return 0;
}

int NAServer(port,op)
NAport *port;
NAserverOps op;
{
    NArequest request;

    request.request = NA_SERVER;
    request.NArequest_u.serverOp = op;
    if (_NASendRequest(port,&request) < 0)
	exit(1);
    return 0;
}

int NAGetServerVersion(port)
NAport *port;
{
    NArequest request;
    NAreply *reply;
    int version;

    request.request = NA_GETVERSION;
    if (_NASendRequest(port,&request) < 0)
	exit(1);

    /* Wait for reply */
    if ((reply = _NAReply(port)) == 0) {
	NALog(LOG_WARNING,"Server may be running an incompatible version of NetAudio.\n");
	exit(1);
    }
    version = reply->NAreply_u.sversion;
    _NAFreeReply(reply);
    return version;
}
