/* $Id: remote.c,v 1.41 1996/03/21 16:36:03 bst Exp $
 *
 * Copyright 1991 Brent Townshend (bst%tt@cam.org)
 * Townshend Computer Tools
 * Montreal, Quebec
 *
 * Mon Aug 5 02:03:19 EDT 1991
 *
 * Revision History: $Log: remote.c,v $
 * Revision 1.41  1996/03/21  16:36:03  bst
 * Fixed path separator on mac
 * Check for out of memory from realloc
 * Print message for unsupported device listing on mac
 *
 * Revision 1.40  1996/03/20  01:18:49  eryk
 * Undef WIN32 before include dirent.h
 *
 * Revision 1.39  1995/07/06  17:27:13  bst
 * Made check on device name case insensitive.
 *
 * Revision 1.38  1995/04/21  22:29:38  bst
 * Only read remote device once, unless device changes.
 * Added some debug messages.
 *
 * Revision 1.37  1993/08/27  00:31:49  bst
 * Use HAS_, MISSING_ defines.
 *
 * Revision 1.36  1993/08/10  02:08:50  bst
 * Change some int to int32
 *
 * Revision 1.35  1993/06/01  02:44:26  bst
 * Added coercions for GCC 2.4.1 unsigned 'size_t'.
 *
 * Revision 1.34  1993/05/18  19:41:14  bst
 * Use NALog() to log/print errors and warnings.
 *
 * Revision 1.33  1993/04/22  03:14:47  bst
 * Added static function prototypes.
 *
 * Revision 1.32  1993/03/09  02:12:33  bst
 * Fixed error in memory allocation for remote sequences.
 *
 * Revision 1.31  1993/03/01  04:01:09  bst
 * Use for remote device descriptions.
 *
 * Revision 1.30  1993/02/18  21:35:23  bst
 * Added coercion for opendir() arg.
 *
 * Revision 1.29  1993/02/07  19:34:52  bst
 * Added additional unused #pragma.
 *
 * Revision 1.28  1993/02/05  22:01:34  bst
 * Use PRAGMA_UNUSED instead of UNUSED macro.
 *
 * Revision 1.27  1993/02/05  20:37:53  bst
 * Added UNUSED() macro call for ununused parameters to functions.
 *
 * Revision 1.26  1992/08/19  00:21:54  bst
 * Removed include of config.h (sysdefs.h gets it).
 *
 * Revision 1.25  1992/04/09  17:55:51  bst
 * Changed strings.h to string.h
 *
 * Revision 1.24  1992/04/06  21:55:47  bst
 * Added coercion to match prototype.
 *
 * Revision 1.23  1992/03/02  03:59:24  bst
 * Added missing argument to error message printf.
 *
 * Revision 1.22  1992/02/12  04:29:53  bst
 * Better handling of old machines that don't have dirent.h
 *
 * Revision 1.21  1992/02/11  18:16:37  bst
 * Don't mess with directories on Mac.
 *
 * Revision 1.20  1992/02/08  20:13:58  bst
 * Include dirent.h if USE_DIRENT is set in ../config.h.
 *
 * Revision 1.19  1992/02/08  03:09:01  bst
 * Removed RCFPATH, always use INSTALLROOT/etc/rcf
 *
 * Revision 1.18  1992/02/02  21:58:35  bst
 * Added checks for array overflow in pathSearch, listDevices.
 *
 * Revision 1.17  1992/01/30  04:04:46  bst
 * Position-independent installation (no more DL_DIR or DL_DIR_DEFAULT)
 *
 * Revision 1.16  1992/01/27  03:55:59  bst
 * Addd implementation of dat_listDevices()
 * Use RCSUFFIX macro instead of ".rcf"
 *
 * Revision 1.15  1992/01/17  04:17:36  bst
 * Corrected type of 'mode' (was int)
 *
 * Revision 1.14  1991/12/06  05:59:11  bst
 * New dat_listCommands()
 * Added dat_listDevices() - but it doesn't work yet.
 * New defaults.
 *
 * Revision 1.13  1991/11/12  05:44:43  bst
 * Added '$' at beginning of path names to specify installation directoy.
 * Added hard-wired default path for RCF files.
 *
 * Revision 1.12  1991/11/11  04:26:59  bst
 * Use new Debug procedure.
 * Move dat_strcasecmp() to strcasecmp.c
 * Search for rcf files in path defined in defaults files.
 *
 * Revision 1.11  1991/10/25  04:21:42  bst
 * Made dat_strcasecmd() non-ANSI.
 * Expanded 'u_int' (MAC complained)
 * Removed (void) coercion from free()
 *
 * Revision 1.10  1991/10/25  03:40:40  bst
 * Made indep. of host byte-order.
 * Use ':' for path separator on Macintosh.
 *
 * Revision 1.9  1991/10/20  18:25:50  bst
 * Use dynamically allocated RMTCommands
 *
 * Revision 1.8  1991/10/09  17:19:00  bst
 * Always use dat_strcasecmp included here rather than strcasecmp()
 * Deleted declarations of tolower().
 *
 * Revision 1.7  1991/10/04  02:28:19  bst
 * Sun386 needs strcasecmp().
 *
 * Revision 1.6  1991/09/16  05:21:22  bst
 * Changed some function names to make them more consistent.
 *
 * Revision 1.5  1991/09/12  03:02:37  bst
 * Fixed handling of stdlib.h and unistd.h
 * Don't use prototype for tolower() if it's a macro.
 * Removed declaration of getenv() since its in libdefs.h now.
 *
 * Revision 1.4  1991/08/23  23:50:52  bst
 * Modified function declarations to work with non-ANSI compilers
 *
 * Revision 1.3  1991/08/23  16:00:16  bst
 * Changed dat_sendmsg() interface.
 *
 * Revision 1.2  1991/08/05  19:12:46  bst
 * Changed environment variable names.
 *
 * Revision 1.1  1991/08/05  06:06:53  bst
 * Initial Revision
 *
 *
 */
#include <assert.h>
#include "libdefs.h"
#include <string.h>
#include "dat_remote.h"
#include "defaults.h"

#ifndef MISSING_dirent_h
/* on Borland, dirent.h includes windows.h which has different
	definitions of hton* and ntoh* */
#ifdef __WIN32__
#define TEMPWIN32
#undef __WIN32__
#endif /* __WIN32__ */

#include <dirent.h>

#ifdef TEMPWIN32
#define __WIN32__
#endif /* TEMPWIN32 */

#else /* !MISSING_dirent_h */
#ifdef HAS_sys_dir_h 
#include <sys/dir.h>
#define dirent direct
#endif /* HAS_sys_dir_h */
#endif /* MISSING_dirent_h */

#define RCSUFFIX ".rcf"
#define RCFDIR "etc/rcf"

typedef struct {
    char cmdName[80];
    int seqLen;   /* Number of transitions in sequence (should be even) */
    word data[1024];   /* Relative timing of transitions in 8 uS steps */
} Sequence;

/* Remote control access */
typedef struct {
    char deviceName[50];    /* Model number of device */
    int carrierOn, carrierOff;
    int repeats,polarity;
    int allocops, numops;
    Sequence *seqs;
} RDesc;

static RDesc desc = { "",0,0,0,0,0,0,(Sequence *)0 };

/* Low level remote access */
void dat_sendIR(dt,ircmd)
dat_fd dt;
const RMTCommand *ircmd;
{
    unsigned int dlen = sizeof(*ircmd)+(ircmd->ndata-1)*sizeof(ircmd->data[0]);
    byte *buffer = (byte *)malloc((size_t)dlen);
    int i;
    int x;

    /* Make sure the byte ordering is little endian
	(assumes all fields of RMTCommand are shorts) */
    assert((x=sizeof(*ircmd)) == 8);	/* In case someone changes RMTCommand */
    for (i=0;i<dlen;i+=2) {
	buffer[i] = ((word *)ircmd)[i/2] >> 8;
	buffer[i+1] = ((word *)ircmd)[i/2] & 0xff;
    }
    (void)dat_sendmsg(dt,DL_PUTRMTSEQ,dlen,buffer,0,0);
    free(buffer);
}

/* Low level remote access */
void dat_stopIR(dt)
dat_fd dt;
{
    (void)dat_sendmsg(dt,DL_ENDRMTSEQ,0,0,0,0);
}

#define MAXPATHLENGTH 200

#if 0
char *pathSearch(path,file,mode)
const char *path;
const char *file;
const char *mode;
{
    const char *sysdir = FindFile("");
    static char fname[MAXPATHLENGTH];
    const char *comp = path;

    if (sysdir == 0) {
	NALog(LOG_ERR,"Unable to locate system directory\n");
	exit(1);
    }
    Debug("libdatlink.pathSearch",2,"Searching for %s in %s with mode '%s'\n",
	  file,path,mode);
    while (1) {
	char *colon = index(comp,':');
	FILE *fd;
	int compLength;

	if (colon)
	    compLength = (int)(colon-comp);
	else
	    compLength = strlen(comp);

	if (*comp == '$') {
	    comp++; compLength--;
	    if (*comp == '/') {
		comp++; compLength--;
	    }
	    if (compLength+strlen(sysdir)+strlen(file)+2 >= MAXPATHLENGTH) {
		NALog(LOG_ERR,
			"Internal error - pathSearch(%s,%s,%s): path too long\n",
			path,file,mode);
		return 0;
	    }
	    sprintf(fname,"%s%c%s",sysdir,DSEP,comp);
	    compLength += strlen(sysdir)+1;
	}
	else {
	    if (compLength+strlen(file)+1 >= MAXPATHLENGTH) {
		NALog(LOG_ERR,
			"Internal error - pathSearch(%s,%s,%s): path too long\n",
			path,file,mode);
		return 0;
	    }
	    strncpy(fname,comp,compLength);
	}
	fname[compLength++] = DSEP;
	strcpy(&fname[compLength],file);

	Debug("libdatlink.pathSearch",3,"Checking %s\n",fname);
	if ((fd=fopen(fname,mode)) != NULL) {
	    fclose(fd);
	    return fname;
	}
	if (colon == 0)
	    return 0;
	comp = colon+1;
    }
}
#endif

static FILE *rcfile P((char *devname, char *mode));

static FILE *rcfile(devname,mode)
char *devname;
char *mode;
{
    FILE *result;
    char *devfile;

    /* Build name of RCF file */
    const char *rcfdir = FindFile(RCFDIR);
    if (rcfdir == 0)
	/* So we can find rcf files when running from source directory */
	rcfdir = FindFile("rcf");
    if (rcfdir == 0) {
	NALog(LOG_ERR,"Unable to locate remote control directory (%s)\n",
		RCFDIR);
	return NULL;
    }
    devfile = malloc(strlen(rcfdir)+1+strlen(devname)+strlen(RCSUFFIX)+1);
#ifdef macintosh
    sprintf(devfile,"%s:%s%s",rcfdir,devname,RCSUFFIX);
#else
    sprintf(devfile,"%s/%s%s",rcfdir,devname,RCSUFFIX);
#endif
    Debug("libdatlink.rcfile",1,"RCF file=%s\n",devfile);
    result = fopen(devfile,mode);
    if ((result == NULL) && (*mode == 'w'))
	/* Unable to open file for device */
	NALog(LOG_ERR,"Unable to open %s for writing\n",devfile);
    free(devfile);
    return result;
}

/* Set remote descriptor for DAT unit */
int dat_setDevice(dt,devname)
dat_fd dt;
char *devname;
{
#ifdef PRAGMA_UNUSED
#pragma unused(dt)
#endif /* PRAGMA_UNUSED */
    FILE *fd;
    int op;
    char fileDevName[500];

    if (devname == 0) {
	devname = getenv("DL_RC");
	if (devname == NULL) {
	    NALog(LOG_WARNING,
		  "No remote device specified and DL_RC environment variable not set.\n");
	    return -2;
	}
    }

    if (strncmp(devname, desc.deviceName, sizeof(desc.deviceName)-1) == 0) {
	Debug("libdatlink.setDevice",1,"Reusing RCF descriptor for %s\n", devname);
	return 0;
    }
    else 
	Debug("libdatlink.setDevice",1,"Load new RCF file for %s\n", devname);

    fd = rcfile(devname,"r");
    desc.numops = 0;
    if (fd == NULL)  {
	/* No such device, but build a new descriptor in case it was rclearn
	    that called us */
	strncpy(desc.deviceName,devname,sizeof(desc.deviceName)-1);
	/* Default to 40kHz carrier, 50% duty cycle */
	desc.carrierOn = 100;
	desc.carrierOff = 100;
	desc.repeats = 5;   /* 5 repetitions */
	desc.polarity = 1;  /* Positive polarity */
	NALog(LOG_WARNING,"Unknown remote device: %s\n",desc.deviceName);
	return -1;
    }
    strncpy(desc.deviceName, devname, sizeof(desc.deviceName)-1);
    /* Potential problem if fileDevName is overflowed! */
    fscanf(fd,"%s %d %d %d %d",fileDevName,&desc.carrierOn,&desc.carrierOff,
	   &desc.repeats,&desc.polarity);
    if (strcasecmp(fileDevName, devname)) {
	NALog(LOG_WARNING,"RCF file: %s.rcf contains descriptor for %s\n",
	      devname, fileDevName);
    }
    op = 0;
    while (1) {
	int count = 0;
	int interval;
	
	if (desc.allocops <= op) {
	    if (op == 0) {
		desc.allocops = 20;
		desc.seqs = (Sequence *)malloc(desc.allocops*sizeof(*desc.seqs));
	    }
	    else {
		desc.allocops = op+10;
		if ((desc.seqs = (Sequence *)realloc(desc.seqs,
						     desc.allocops*sizeof(*desc.seqs))) == 0) {
		    NALog(LOG_ERR,"Out of memory\n");
		    exit(1);
		}
	    }
	}
	if (fscanf(fd,"%s",desc.seqs[op].cmdName) != 1)
	    break;
	fscanf(fd,"%d",&interval);
	while (interval != -1) {
	    desc.seqs[op].data[count] = interval;
	    count++;
	    fscanf(fd,"%d",&interval);
	}
	desc.seqs[op].seqLen = count;
	op++;
    }
    desc.numops = op;
    fclose(fd);
    return 0;
}

/* Save remote descriptor for DAT unit */
int dat_saveDevice(dt,devname)
dat_fd dt;
char *devname;
{
#ifdef PRAGMA_UNUSED
#pragma unused(dt)
#endif /* PRAGMA_UNUSED */
    FILE *fd = rcfile(devname,"w+");
    int i,op;

    if (fd == NULL)
	return -1;
    fprintf(fd,"%s %d %d %d %d\n",devname,desc.carrierOn,desc.carrierOff,
	   desc.repeats,desc.polarity);

    for (op=0;op<desc.numops;op++) {
	fprintf(fd,"%s\n",desc.seqs[op].cmdName);
	for (i=0;i<desc.seqs[op].seqLen;i++)
	    fprintf(fd,"%d ",desc.seqs[op].data[i]);
	fprintf(fd,"-1\n");
    }

    fclose(fd);
    return 0;
}

/* Update a sequence for a descriptor */
void dat_remoteLearn(dt,cmd,seqLen,data)
dat_fd dt;
char *cmd;
int seqLen;
word data[];
{
#ifdef PRAGMA_UNUSED
#pragma unused(dt)
#endif /* PRAGMA_UNUSED */
    int i;
    int op = dat_lookupCommand(cmd);
    
    /* Add command if its not already present */
    if (op == -1) {
	if (desc.numops == desc.allocops) {
	    if (desc.allocops == 0) {
		desc.allocops = 20;
		desc.seqs = (Sequence *)malloc(desc.allocops*sizeof(*desc.seqs));
	    }
	    else {
		desc.allocops = desc.numops+10;
		if ((desc.seqs = (Sequence *)realloc(desc.seqs,
					     desc.allocops*sizeof(*desc.seqs))) == 0) {
		    NALog(LOG_ERR,"Out of memory\n");
		    exit(1);
		}
	    }
	}
	op = desc.numops++;
	strncpy(desc.seqs[op].cmdName,cmd,49);
    }

    desc.seqs[op].seqLen = seqLen;
    for (i=0;i<seqLen;i++)
	desc.seqs[op].data[i] = data[i];
}

/* Execute an operation given by op
 * Return -1 if failure (not found in descriptor)
 */
int dat_remote(dt,command)
dat_fd dt;
char *command;
{
    int i;
    Sequence *seq;
    RMTCommand *cmd;
    word *dptr;
    int op;
    int cmdlen;

    if (desc.deviceName[0] == 0) {
	NALog(LOG_WARNING,"dat_execute: No device set\n");
	return -1;
    }

    op = dat_lookupCommand(command);
    if (op < 0) {
	NALog(LOG_WARNING,"dat_remote: Unknown operation: %s\n",command);
	return -1;
    }

    seq = &desc.seqs[op];

    if (seq->seqLen == 0)
	/* Not applicable on this device */
	return -1;

    cmdlen = seq->seqLen * desc.repeats+2; 
    if (desc.polarity != 1)
	cmdlen+=2;

    cmd = (RMTCommand *)malloc(sizeof(*cmd)+(cmdlen-1)*sizeof(cmd->data[0]));
    cmd->ndata = cmdlen;
    cmd->carrierOn = desc.carrierOn;
    cmd->carrierOff = desc.carrierOff;
    dptr = cmd->data;

    if (desc.polarity != 1) {
	/* Reverse polarity at beginning */
	*dptr++ = 10000;  /* 1.25 ms leader */
    }

    for (i=0;i<desc.repeats;i++) {
	int j;
	word *seqPtr = seq->data;
	for (j=0;j<seq->seqLen;j++)
	    *dptr++ = *seqPtr++;
    }

    /* Gap so back-to-back commands are not confused with a held-down button */
    *dptr++ = 0;
    *dptr++ = 32767;

    if (desc.polarity != 1) {
	/* Reverse polarity back to normal at end */
	*dptr++ = 10000; /* 1.25 ms trailer */
    }

    dat_sendIR(dt,cmd);
    free((char *)cmd);
    return 0;
}

/* Lookup a remote command (-1 if failure) */
int dat_lookupCommand(cmdName)
char *cmdName;
{
    int i;
    for (i=0;i<desc.numops;i++)
	if (strcasecmp(cmdName,desc.seqs[i].cmdName) == 0)
	    return i;
    return -1;
}

#define MAXRCFFILES 1000

char **dat_listDevices(len)
unsigned int32 *len;
{
#ifdef macintosh
    static char *unsupported = "(others)";
    fprintf(stderr,
	    "Sorry, device listing not supported on macintosh; see etc:rcf folder.\n");
    *len = 1;
    return &unsupported;
#else /* macintosh */
    DIR *dirp;
    const char *dirName = FindFile(RCFDIR);
    char **result = (char **)malloc(sizeof(*result)*MAXRCFFILES);

    *len = 0;

    if (dirName == 0)
	/* So we can find rcf files when running from source directory */
	dirName = FindFile("rcf");

    if (dirName == 0) {
	NALog(LOG_ERR,"Unable to locate remote control directory, %s\n",RCFDIR);
	return 0;
    }
    
    Debug("libdatlink.listDevices",3,"Checking %s\n",dirName);
    dirp = opendir((char *)dirName);
    if (dirp == NULL) {
	NALog(LOG_ERR,"Unable to open remote control directory: %s\n", dirName);
	return 0;
    }
    else {
	struct dirent *dp;
	for (dp=readdir(dirp);dp!=NULL;dp=readdir(dirp)) {
	    size_t dlen = strlen(dp->d_name);
	    size_t baselen = dlen-strlen(RCSUFFIX);
	    if (strcasecmp(&dp->d_name[baselen],RCSUFFIX) == 0) {
		if (*len == MAXRCFFILES) {
		    NALog(LOG_ERR,"Too many RCF files ... truncating list to %d\n",
			  MAXRCFFILES);
		    return result;
		}
		result[*len] = malloc(baselen+1);
		strncpy(result[*len],dp->d_name,baselen);
		result[*len][baselen] = 0;
		*len = *len + 1;
	    }
	}
	closedir(dirp);
    }
    return result;
#endif /* macintosh */
}

char **dat_listCommands(len)
unsigned int32 *len;
{
    int cmd;
    char **result = (char **)malloc(desc.numops*sizeof(char *));

    *len = desc.numops;
    for (cmd=0;cmd<desc.numops;cmd++) {
	result[cmd] = malloc(strlen(desc.seqs[cmd].cmdName)+1);
	strcpy(result[cmd],desc.seqs[cmd].cmdName);
    }
    return result;
}

#ifdef NOTDEF
/* Set for automatic execution 
 *  Read,Write, etc. automatically start/stop DAT machine.
 */
void dat_setAuto(dt)
dat_fd dt;
{
}

/* Set for manual execution (default) */
void dat_setManual()
dat_fd dt;
{
}

/* Reposition tape to give sample address number and start reading into
 * internal buffer.
 * Returns: -1 if not possible to position (insufficient data recorded on tape)
 *           0 if positioned, subsequent reads will begin with this sample #
 */
int dat_seekRead(dt,sampleAddress)
dat_fd dt;
int sampleAddress;
{
    return -1;
}

#endif

void dat_trainBegin(dt)
dat_fd dt;
{
    (void)dat_sendmsg(dt,DL_GETRMTBEGIN,0,0,0,0);
}

void dat_trainEnd(dt)
dat_fd dt;
{
    (void)dat_sendmsg(dt,DL_GETRMTEND,0,0,0,0);
}
