/* $Id: reload.c,v 1.34 1996/03/21 16:35:45 bst Exp $
 *
 * Copyright 1991 Brent Townshend (bst%tt@cam.org)
 * Townshend Computer Tools
 * Montreal, Quebec
 *
 * Sun Oct 20 16:11:22 EDT 1991
 *
 * Revision History: $Log: reload.c,v $
 * Revision 1.34  1996/03/21  16:35:45  bst
 * Use binary mode for fopen on mac.
 *
 * Revision 1.33  1994/10/26  23:03:51  bst
 * Added error message about writing update firmware unsuccessful
 *
 * Revision 1.32  1994/05/26  15:55:55  bst
 * Added extern dat_FirmwareRootName to set firmware name.
 * Reload firmware if currently running is not an exact match to desired.
 *
 * Revision 1.31  1994/03/07  18:32:56  bst
 * Fixed some coercions.
 *
 * Revision 1.30  1993/08/31  17:28:23  bst
 * Print installroot when unable to find firmware directory.
 *
 * Revision 1.29  1993/08/10  02:07:04  bst
 * Open firmware file as binary.
 * Fixed rounding up to block multiples.
 * Retry incomplete freads().
 *
 * Revision 1.28  1993/06/23  20:08:03  bst
 * Changed 'long' to 'int32'
 *
 * Revision 1.27  1993/06/12  20:47:11  bst
 * Clear cached parameters even if unable to close datlink fd.
 *
 * Revision 1.26  1993/05/18  19:41:10  bst
 * Use NALog() to log/print errors and warnings.
 *
 * Revision 1.25  1993/04/22  03:14:19  bst
 * Made use of '=' in conditional more clear.
 *
 * Revision 1.24  1993/04/12  02:04:47  bst
 * Use long when 32-bit integers required.
 *
 * Revision 1.23  1993/03/01  04:00:34  bst
 * Use malloc() instead of static storage for buffers.
 *
 * Revision 1.22  1992/11/08  22:54:42  bst
 * Fix warning from gcc 2.3.1
 *
 * Revision 1.21  1992/11/01  19:33:08  bst
 * Made firmware reload capability mandatory.
 *
 * Revision 1.20  1992/08/24  16:23:24  bst
 * Fixed error in use of DSEP.
 *
 * Revision 1.19  1992/08/20  00:18:44  bst
 * Handle MAC filenames.
 *
 * Revision 1.18  1992/06/27  22:16:56  bst
 * Use dat_clearParameters()
 *
 * Revision 1.17  1992/05/20  01:46:18  bst
 * Close DAT-Link after reloading software.
 * Increased sleep after reloading software.
 *
 * Revision 1.16  1992/05/20  00:21:13  bst
 * Execute a cold boot if the DAT-Link is already running out of RAM
 *
 * Revision 1.15  1992/04/23  22:04:08  bst
 * Changed parsing of revision file to accept symbolic HW revision.
 *
 * Revision 1.14  1992/04/09  17:55:21  bst
 * Changed strings.h to string.h
 *
 * Revision 1.13  1992/04/06  21:55:36  bst
 * Added coercion to match prototype.
 *
 * Revision 1.12  1992/04/04  23:28:56  bst
 * Unable to locate firmware is no longer fatal.
 *
 * Revision 1.11  1992/02/11  18:16:26  bst
 * Use ':' for path separators on Mac.
 *
 * Revision 1.10  1992/02/08  03:32:16  bst
 * Added dat_clearTransfers() call before sending new executable.
 *
 * Revision 1.9  1992/01/30  04:04:11  bst
 * Use position independent location of etc/dl/Version
 *
 * Revision 1.8  1992/01/27  03:55:40  bst
 * Use dat_writeany() instead of dat_write() for reloading program.
 *
 * Revision 1.7  1991/12/05  18:52:10  bst
 * Round up length of executable to a multiple of BLOCK_LENGTH
 *
 * Revision 1.6  1991/12/05  02:16:26  bst
 * Pass 0 instead of -1 as nrecv in dat_sendmsg() call.
 *
 * Revision 1.5  1991/11/14  01:03:28  bst
 * Sleep for 2 seconds after reloading software to give it time to get going.
 * Clear cached version number after reloading.
 *
 * Revision 1.4  1991/11/11  04:26:40  bst
 * Use new Debug() procedure.
 *
 * Revision 1.3  1991/10/30  04:17:20  bst
 * Removed unused declaration of 'i'.
 *
 * Revision 1.2  1991/10/24  22:30:42  bst
 * Use ';' for pathname separator on MAC.
 * Made buffer static so it doesn't overflow stack on MAC.
 *
 * Revision 1.1  1991/10/20  20:11:25  bst
 * Initial revision
 *
 */
/* Reload firmware if needed.
	return -1 for failure
	       0 if no firmware is available to load, or firmware is up-to-date
	       1 if software was reloaded
	       2 if DAT-Link was cold-booted 
	For return values 1 or 2, the DAT-Link will be closed as a side effect
*/
#include <string.h>
#include "libdefs.h"
#include "defaults.h"

#define BUFSIZE ((unsigned int)OPTIMUM_TRANSFER_LENGTH)

/* May be changed in main program */
const char *dat_FirmwareRootName = "";

int dat_reload(dt)
dat_fd dt;
{
    const char *dirname;
    char filename[100];
    FILE *vfile,*ofile;
    int FWRev;
    char HWRevision[100];
    char ofname[100];
    char fwname[100];
    int version;
    byte *buf = 0;
    unsigned int nread;

    if ((version = dat_version(dt)) < 0)
	return -1;
    
    if (version%1000 < 14) {
	NALog(LOG_ERR,"Fatal Error - The ROMs in this DAT-Link do not support firmware reload.\n");
	NALog(LOG_ERR,"              Please contact your DAT-Link supplier for a free ROM upgrade.\n");
	exit(1);
    }

    dirname = FindFile("etc/dl");
    if (dirname == 0) {
	dirname = FindFile("../system");
	if (dirname == 0) {
	    NALog(LOG_ERR,"Unable to locate firmware directory. (Installroot=%s)\n",
		  FindFile("."));
	    NALog(LOG_ERR,"DAT-Link software was not properly installed.\n");
	    return 0;
	}
    }
	    
    sprintf(fwname,"%s%d",dat_FirmwareRootName,version/1000);
    sprintf(filename,"%s%cVersion",dirname,DSEP);
    vfile = fopen(filename,"r");
    if (vfile == NULL) {
	NALog(LOG_ERR,"Firmware revision file (%s) not found.\n",filename);
	NALog(LOG_ERR,"DAT-Link software was not properly installed.\n");
	Debug("libdatlink.reload",2,
	      "Firmware version file, %s, not found.\n",filename);
	return 0;
    }
    do {
	if (fscanf(vfile,"%s %d %s",HWRevision,&FWRev,ofname) != 3) {
	    Debug("libdatlink.reload",2,
		  "No match found for '%s'\n",fwname);
	    fclose(vfile);
	    return 0;
	}
    } while (strcmp(HWRevision,fwname));
    fclose(vfile);
    Debug("libdatlink.reload",2,"New firmware: FWRev=%d, HWRev=%s, fname=%s\n",
	       FWRev,HWRevision,ofname);
    if (version%1000 == FWRev) {
	Debug("libdatlink.reload",2,"Firmware up-to-date\n");
	return 0;
    }
    sprintf(filename,"%s%c%s",dirname,DSEP,ofname);
#if defined(MSDOS) || defined(macintosh)
    ofile = fopen(filename,"rb");
#else
    ofile = fopen(filename,"r");
#endif
    if (ofile == NULL) {
	NALog(LOG_ERR,"Unable to open %s: Check %s%cVersion.\n",
	      filename,dirname,DSEP);
	return -1;
    }
    if (dat_RAMExecution(dt) == 1) {
	    /* Already running out of RAM */
	Debug("libdatlink.reload",1,"Rebooting DAT-Link\n");
	if (dat_sendmsg(dt,DL_COLDBOOT,0,0,0,0) < 0) {
	    Debug("libdatlink.reload",1,"Reboot of DAT-Link failed\n");
	    return -1;
	}
	    /* Clear cached value of version number */
	dat_clearParameters(dt);
	sleep(2);
	if (dat_closefast(dt) < 0)
	    return -1;
	return 2;
    }
    /* Make sure we start with a clean buffer */
    dat_clearTransfers(dt);
    buf = (byte *)malloc(BUFSIZE);
    while ((nread = fread((char *)buf,1,BUFSIZE,ofile)) != 0) {
	int rounded = 0;
	unsigned int nw;

	/* Try to complete a partial read */
	while (nread < BUFSIZE) {
	    unsigned int n2 = fread((char *)&buf[nread],1,BUFSIZE-nread,ofile);
	    if (n2 == 0)
		break;
	    nread += n2;
	}
	/* Round up to a block */
	if (nread != BUFSIZE) {
	    nread = ((nread+BLOCK_LENGTH-1)/BLOCK_LENGTH)*BLOCK_LENGTH;
	    rounded = 1;
	}
	if ((nw=dat_writeany(dt,buf,(int32)nread)) != (int32)nread)  {
	    NALog(LOG_ERR,"Error writing updated firmware to DAT-Link (%ld/%ld)\n",
		  (long)nw,(long)nread);
	    free(buf);
	    return -1;
	}
	if (rounded)
	    break;
    }
    free(buf);
    Debug("libdatlink.reload",2,"About to execute new firmware.\n");

    if (dat_sendmsg(dt,DL_EXECUTE,0,0,0,0) < 0)
	return -1;
    /* Give the new firmware time to get running before accessing it again */
    sleep(4);

    /* Clear cached value of version number */
    dat_clearParameters(dt);

    if (dat_closefast(dt) < 0)
	return -1;

    Debug("libdatlink.reload",1,"New firmware started.\n");

    return 1;
}
