/* $Id: params.c,v 1.35 1994/03/07 18:30:22 bst Exp $
 *
 * Copyright 1991 Brent Townshend (bst%tt@cam.org)
 * Townshend Computer Tools
 * Montreal, Quebec
 *
 * Mon Aug 5 02:03:02 EDT 1991
 *
 * Revision History: $Log: params.c,v $
 * Revision 1.35  1994/03/07  18:30:22  bst
 * Fixed some long coercions.
 *
 * Revision 1.34  1993/11/19  14:18:50  bst
 * Added missing ';'
 *
 * Revision 1.33  1993/11/19  03:16:26  bst
 * Increased size of params result to 256 bytes.
 *
 * Revision 1.32  1993/10/26  16:15:11  bst
 * Added coercions for GCC 2.5.0
 *
 * Revision 1.31  1993/10/04  01:40:42  bst
 * Added dat_getSN().
 *
 * Revision 1.30  1993/10/02  05:01:15  bst
 * Handle ROM parameters
 * Fixed algorithm for determining size of param block.
 * Added SN to parameters
 *
 * Revision 1.29  1993/09/29  03:00:49  bst
 * Added dat_get[TR]XModel() and underlying support.
 * Compressed debug printout.
 *
 * Revision 1.28  1993/08/27  00:32:53  bst
 * Use %ld for size_t args and coerce to long.
 *
 * Revision 1.27  1993/08/10  02:05:43  bst
 * Coerce %l args to long.
 *
 * Revision 1.26  1993/06/28  20:04:45  bst
 * Properly handle short returns from DL_PARAMETERS.
 * Handle firmware revs prior to 44 (No max*Size).
 *
 * Revision 1.25  1993/06/23  20:08:03  bst
 * Changed 'long' to 'int32'
 *
 * Revision 1.24  1993/05/24  20:27:24  bst
 * Added dat_getMaxReadSize(), dat_getMaxWriteSize()
 *
 * Revision 1.23  1993/05/18  19:40:31  bst
 * Use DebugCont() for debug message continuations.
 *
 * Revision 1.22  1993/04/22  03:14:09  bst
 * Added static function prototypes.
 *
 * Revision 1.21  1993/04/12  02:04:14  bst
 * Use long when 32-bit integers required.
 *
 * Revision 1.20  1993/03/15  20:33:24  bst
 * Set value of 'dt' variable in params.
 *
 * Revision 1.19  1993/03/01  04:00:18  bst
 * Use dynamic storage for Parameters.
 *
 * Revision 1.18  1992/11/06  01:45:25  bst
 * Cleanup warnings from GNU C 2.3.1
 *
 * Revision 1.17  1992/09/16  23:36:32  bst
 * Made error message more understandable.
 *
 * Revision 1.16  1992/08/28  03:13:44  bst
 * Coerce chars to unsigned int before shifting more than 8 bits.
 *
 * Revision 1.15  1992/08/18  01:11:29  bst
 * Fixed name dat_Checksum -> dat_checksum.
 *
 * Revision 1.14  1992/07/09  21:34:40  bst
 * Stop displaying partitions at end of table.
 *
 * Revision 1.13  1992/07/01  17:06:32  bst
 * Fixed bug in dat_getDSP*() routines.
 * Read DSP partition table for ROM rev >=32
 * Added debug printout of DSP partition table.
 *
 * Revision 1.12  1992/06/27  22:16:30  bst
 * Made each parameter a function call instead of a global.
 * Allow different parameters for different DAT-Link units.
 *
 * Revision 1.11  1992/05/20  01:44:23  bst
 * Set extra variables to -1 if the firmware doesn't support them.
 *
 * Revision 1.10  1992/05/20  00:20:34  bst
 * Added additional data fields to CHECKSUM reply.
 *
 * Revision 1.9  1991/11/22  06:33:20  bst
 * Removed uneeded initializers.
 *
 * Revision 1.8  1991/11/14  01:01:08  bst
 * Made cached version number a global so dat_reload() can clear it.
 *
 * Revision 1.7  1991/11/08  05:56:14  bst
 * Cache version number to speed up transfers.
 *
 * Revision 1.6  1991/10/25  03:36:07  bst
 * Build checksum from byte stream indep. of host byte-order.
 *
 * Revision 1.5  1991/10/20  18:23:34  bst
 * Read the correct number of bytes in dat_checksum() even though they're not used.
 *
 * Revision 1.4  1991/09/09  04:52:25  bst
 * Added dat_version()
 *
 * Revision 1.3  1991/08/23  23:50:52  bst
 * Modified function declarations to work with non-ANSI compilers
 *
 * Revision 1.2  1991/08/23  16:00:16  bst
 * Changed dat_sendmsg() interface.
 *
 * Revision 1.1  1991/08/05  06:06:37  bst
 * Initial Revision
 */
#include "libdefs.h"

struct Parameters {
    struct Parameters *next;
    int dt;
    int Loaded;
    int VersionNumber;
    int RAMExecution;
    int Emulation;
    int SCSISwitchSetting;
    int DIPSwitchSetting;
    int Checksum;
    int32 DSPStartAddrs[MAXDSPPARTITIONS];
    int32 DSPMemSizes[MAXDSPPARTITIONS];
    int DSPMemTypes[MAXDSPPARTITIONS];
    int32 MaxReadSize;	/* Maximum read transaction seen by DAT-Link */
    int32 MaxWriteSize;  /* Maximum write transaction seen by DAT-Link */
    byte TXModel, RXModel;	/* Model of AES/EBU transceivers */
    int32 SN;
};

static struct Parameters *Params = 0;

/* Get Version numbers */
static struct Parameters *getParameters P((dat_fd dt));
static struct Parameters *getParameters(dt)
dat_fd dt;
{
    struct Parameters *p, *prev;
    for (p=Params,prev=0;p;p=p->next) {
	if (p->dt == dt)
	    break;
	prev = p;
    }

    if (p == 0) {
	p = (struct Parameters *)malloc(sizeof(*p));
	if (prev == 0)
	    Params = p;
	else
	    prev->next = p;
	p->next = 0;
	p->Loaded = 0;
	p->dt = dt;
    }

    if (!p->Loaded) {
	int part;
	byte result[256];
	int nrecv;

	if ((nrecv=dat_sendmsg(dt,DL_PARAMETERS,0,0,sizeof(result),result)) < 0)
	    return 0;
	if (nrecv < 4) {
	    NALog(LOG_ERR,"getParameters only retrieved %d/%ld bytes.\n",
		  nrecv,(long)sizeof(result));
	    return 0;
	}
	p->Checksum = result[0]<<8|result[1];
	p->VersionNumber = result[3]*1000+result[2];

	/* Default values for DSP partitions */
	for (part=0;part<MAXDSPPARTITIONS;part++) {
	    p->DSPStartAddrs[part] = 0;
	    p->DSPMemSizes[part] = 0;
	}
	p->DSPMemSizes[0] = 0x2000;
	/* Other defaults */
	p->RAMExecution = -1;
	p->Emulation = -1;
	p->SCSISwitchSetting = -1;
	p->DIPSwitchSetting = -1;
	p->MaxReadSize = -1;
	p->MaxWriteSize = -1;
	p->TXModel = p->RXModel = 0;
	p->SN = -1;

	/* Firmware revisions >= 31 also send some extra info */
	if (result[2] >= 31) {
	    if (nrecv < 8) {
		NALog(LOG_ERR,"getParameters only retrieved %d/%ld bytes.\n",
		      nrecv,(long)sizeof(result));
		return 0;
	    }
	    p->RAMExecution = result[4];
	    p->Emulation = result[5];
	    p->SCSISwitchSetting = result[6];
	    p->DIPSwitchSetting = result[7];
	    if (result[2] >= 32) {
		int i=8;

		for (part=0;part<MAXDSPPARTITIONS;part++) {
		    p->DSPStartAddrs[part] =
			(((unsigned int32)result[i])<<24)|
			(((unsigned int32)result[i+1])<<16)|
			(((unsigned int32)result[i+2])<<8)|result[i+3];
		    i+=4;
		    p->DSPMemSizes[part] = (((unsigned int32)result[i])<<24)|
			(((unsigned int32)result[i+1])<<16)|
			(((unsigned int32)result[i+2])<<8)|result[i+3];
		    i+=4;
		    p->DSPMemTypes[part] = result[i++];
		    if (p->DSPMemSizes[part] == 0)
			break;
		}
		if (part == 0) {
		    NALog(LOG_ERR,"GetParameters: Received %d bytes-no dsp info\n",
			  nrecv);
		}
		if (result[2] >= 44) {
		    p->MaxReadSize = (((unsigned int32)result[i])<<24)|
			(((unsigned int32)result[i+1])<<16)|
			(((unsigned int32)result[i+2])<<8)|result[i+3];
		    i+=4;
		    p->MaxWriteSize = (((unsigned int32)result[i])<<24)|
			(((unsigned int32)result[i+1])<<16)|
			(((unsigned int32)result[i+2])<<8)|result[i+3];
		    i+=4;
		}
		if (result[2] >= 47)  {
		    int j;
		    int nentries;
		    word entries[20];

		    p->TXModel = result[i++];
		    p->RXModel = result[i++];
		    nentries = (result[i]<<8)|(result[i+1]);
		    i+=2;
		    if (nentries > sizeof(entries)/2) {
			NALog(LOG_WARNING,"Too many parameter entries: %d > %ld\n",
			      nentries,(long)sizeof(entries)/2);
			nentries = sizeof(entries)/2;
		    }
		    for (j=0;j<nentries;j++) {
			entries[j] = (result[i]<<8)|(result[i+1]);
			i+=2;
		    }
		    if (nentries >= 2)
			p->SN = entries[1];
		}
	    }
	}
	p->Loaded = 1;
	if (DebugCheck("libdatlink.params",1)) {
	    DebugCont("Parameters for dt at fd %d, ",dt);
	    DebugCont(" SN = %d\n",p->SN);
	    DebugCont(" Loaded = %d,",p->Loaded);
	    DebugCont(" VersionNumber = %d,",p->VersionNumber);
	    DebugCont(" RAMExecution = %d,",p->RAMExecution);
	    DebugCont(" Emulation = %d\n",p->Emulation);
	    DebugCont(" SCSISwitchSetting = %d,",p->SCSISwitchSetting);
	    DebugCont(" DIPSwitchSetting = 0x%x,",p->DIPSwitchSetting);
	    DebugCont(" Checksum = 0x%x\n",p->Checksum);
	    DebugCont(" Max Rd/Wr = %ld/%ld, ",(long)p->MaxReadSize,
		      (long)p->MaxWriteSize);
	    DebugCont(" AES/EBU Tx/Rx = %s/%s\n",
		      TXModelNames[p->TXModel],RXModelNames[p->RXModel]);
	    for (part=0;p->DSPMemSizes[part]&&(part<MAXDSPPARTITIONS);part++) {
		DebugCont(" DSP Addr %6lx, Length %6lx, Type %d\n",
		       (long)p->DSPStartAddrs[part],(long)p->DSPMemSizes[part],
		       p->DSPMemTypes[part]);
	    }
	}
    }
    return p;
}

int dat_version(dt)
dat_fd dt;
{
    struct Parameters *p;
    if ((p=getParameters(dt)) == 0)
	return -1;
    return p->VersionNumber;
}

int dat_RAMExecution(dt)
dat_fd dt;
{
    struct Parameters *p;
    if ((p=getParameters(dt)) == 0)
	return -1;
    return p->RAMExecution;
}

int dat_Emulation(dt)
dat_fd dt;
{
    struct Parameters *p;
    if ((p=getParameters(dt)) == 0)
	return -1;
    return p->Emulation;
}

int dat_SCSISwitchSetting(dt)
dat_fd dt;
{
    struct Parameters *p;
    if ((p=getParameters(dt)) == 0)
	return -1;
    return p->SCSISwitchSetting;
}

int dat_DIPSwitchSetting(dt)
dat_fd dt;
{
    struct Parameters *p;
    if ((p=getParameters(dt)) == 0)
	return -1;
    return p->DIPSwitchSetting;
}

/* Get ROM p->Checksum */
int dat_checksum(dt)
dat_fd dt;
{
    struct Parameters *p;
    if ((p=getParameters(dt)) == 0)
	return -1;
    return p->Checksum;
}

void dat_clearParameters(dt)
dat_fd dt;
{
    struct Parameters *p;
    for (p=Params;p;p=p->next) {
	if (p->dt == dt)
	    p->Loaded = 0;
    }
}

int32 dat_getDSPMemSize(dt,block)
dat_fd dt;
int block;
{
    struct Parameters *p;
    if ((p=getParameters(dt)) == 0)
	return -1;
    return p->DSPMemSizes[block];
}

int32 dat_getDSPStartAddr(dt,block)
dat_fd dt;
int block;
{
    struct Parameters *p;
    if ((p=getParameters(dt)) == 0)
	return -1;
    return p->DSPStartAddrs[block];
}

int dat_getDSPMemType(dt,block)
dat_fd dt;
int block;
{
    struct Parameters *p;
    if ((p=getParameters(dt)) == 0)
	return -1;
    return p->DSPMemTypes[block];
}

int32 dat_getMaxReadSize(dt)
dat_fd dt;
{
    struct Parameters *p;
    if ((p=getParameters(dt)) == 0)
	return -1;
    return p->MaxReadSize;
}

int32 dat_getMaxWriteSize(dt)
dat_fd dt;
{
    struct Parameters *p;
    if ((p=getParameters(dt)) == 0)
	return -1;
    return p->MaxWriteSize;
}

int dat_getTXModel(dt)
dat_fd dt;
{
    struct Parameters *p;
    if ((p=getParameters(dt)) == 0)
	return -1;
    return p->TXModel;
}

int dat_getRXModel(dt)
dat_fd dt;
{
    struct Parameters *p;
    if ((p=getParameters(dt)) == 0)
	return -1;
    return p->RXModel;
}

int dat_getSN(dt)
dat_fd dt;
{
    struct Parameters *p;
    if ((p=getParameters(dt)) == 0)
	return -1;
    return p->SN;
}
