/* $Id: dsp.c,v 1.54 1996/03/21 01:22:17 bst Exp $
 *
 * Copyright 1991 Brent Townshend (bst%tt@cam.org)
 * Townshend Computer Tools
 * Montreal, Quebec
 *
 * Mon Aug 5 02:03:10 EDT 1991
 *
 * Revision History: $Log: dsp.c,v $
 * Revision 1.54  1996/03/21  01:22:17  bst
 * Changed message to debug.
 *
 * Revision 1.53  1994/05/12  03:02:14  bst
 * Coerce bzero 1st arg to char*
 *
 * Revision 1.52  1993/11/19  03:15:15  bst
 * Check for out  of memory when allocating memory for DSP programs.
 *
 * Revision 1.51  1993/08/10  02:04:00  bst
 * Coerce %l args to long
 *
 * Revision 1.50  1993/06/23  20:08:03  bst
 * Changed 'long' to 'int32'
 *
 * Revision 1.49  1993/06/23  03:33:21  bst
 * Initialize end of dsp program buffer to avoid UMR.
 *
 * Revision 1.48  1993/06/01  02:43:07  bst
 * Added coercions for GCC 2.4.1 unsigned 'size_t'.
 *
 * Revision 1.47  1993/05/24  18:04:44  bst
 * Use playback buffer to download DSP program.
 *
 * Revision 1.46  1993/05/18  19:40:01  bst
 * Use NALog() to log/print errors and warnings.
 *
 * Revision 1.45  1993/04/22  03:12:32  bst
 * Added ANSI prototypes for static functions.
 *
 * Revision 1.44  1993/04/12  02:03:16  bst
 * Use long when 32-bit integers required.
 *
 * Revision 1.43  1993/03/17  17:39:50  bst
 * Load play-then-record programs if bidirectional I/O is requested.
 *
 * Revision 1.42  1993/03/01  03:58:33  bst
 * Use malloc() instead of static storage for buffers.
 *
 * Revision 1.41  1993/02/28  22:46:50  bst
 * Use PGMSpecs* for args to SetupDSPProgram()
 * Many changes.
 *
 * Revision 1.40  1993/02/04  23:08:07  bst
 * Always compile - even on Mac.
 *
 * Revision 1.39  1992/12/31  05:00:49  bst
 * Made DL_DSPMESSAGE send a reply to acknowledge.
 *
 * Revision 1.38  1992/11/21  03:10:07  bst
 * Added needed parentheses in error message.
 *
 * Revision 1.37  1992/11/08  22:54:22  bst
 * Added dat_dspMessage().
 *
 * Revision 1.36  1992/11/04  22:03:03  bst
 * Added include of support.h
 *
 * Revision 1.35  1992/11/04  04:32:58  bst
 * Added 'pgm' arg to dat_getDSPProgram().
 * Choose DSP program here.
 * Check capabilities of DSP program.
 *
 * Revision 1.34  1992/11/01  20:55:41  bst
 * Added 'suppressNoLoad' arg to EncodeProgram().
 *
 * Revision 1.33  1992/11/01  19:31:50  bst
 * Removed check for firmware revision.
 *
 * Revision 1.32  1992/10/04  17:16:35  bst
 * Use host byte-order for DSP floating points stored in ints.
 *
 * Revision 1.31  1992/09/09  04:05:27  bst
 * Moved dat_dspSetSIOMode() to mode.c
 *
 * Revision 1.30  1992/08/14  03:01:59  bst
 * Check for DSP program too large and print error message.
 *
 * Revision 1.29  1992/08/11  20:58:16  bst
 * Don't use 'u' qualifier for integers (not all C compilers support it).
 *
 * Revision 1.28  1992/07/13  05:20:41  bst
 * Fixed bug in setting word parameters (was using wrong half of int).
 *
 * Revision 1.27  1992/07/11  20:24:26  bst
 * Read DSP program from system directory.
 *
 * Revision 1.26  1992/07/10  04:23:40  bst
 * Added 'isword' arg to dst_dsp[GS]etParameter().
 *
 * Revision 1.25  1992/07/05  04:28:44  bst
 * Use pgm_FIXnn() instead of pgm_FIX[cy]()
 *
 * Revision 1.24  1992/07/05  03:04:31  bst
 * Added 'todat' arg to dat_getDSPProgram().
 * Added dat_dspSetSIOMode() function.
 *
 * Revision 1.23  1992/06/27  22:14:57  bst
 * Allow variable size DSP memory space.
 *
 * Revision 1.22  1992/04/06  21:54:41  bst
 * Use fabs() not abs()  (Hit this bug on HP)
 * Changed type of plen to match dat_sendmsg() prototype.
 *
 * Revision 1.21  1992/03/02  03:59:03  bst
 * Added missing argument to error message printf.
 *
 * Revision 1.20  1992/02/12  04:29:28  bst
 * Made a local array static to prevent compiler stack overrun on FX/80
 *
 * Revision 1.19  1992/02/11  18:14:05  bst
 * Moved dspRead(), dspWrite(), DSPFlaot() and NativeFloat() to dspwr.c
 * Check if impulse pointers are non-zero before using them.
 *
 * Revision 1.18  1992/01/30  06:08:55  bst
 * Set impulse response to 1,0,0,... if no conversion is done.
 * Fixed up a possible pointer alignment problem.
 *
 * Revision 1.17  1992/01/28  05:48:34  bst
 * Added impulse response parameters as parameters.
 *
 * Revision 1.16  1992/01/27  03:54:37  bst
 * Moved dat_getDSPProgram() here from mode.c
 *
 * Revision 1.15  1992/01/11  16:18:13  bst
 * Fixed DSPFloat debug message to print correct floating point value
 *
 * Revision 1.14  1991/12/17  02:52:07  bst
 * Removed 'dat_dspProgram' global variable.
 * Moved NativeFloat() and DSPFloat() here from status.c
 * Added dat_dspGetFloat() and dat_dspSetFloat()
 * Added 'pgm' arg to dat_dsp{Get,Set}Parameter()
 *
 * Revision 1.13  1991/12/16  00:33:46  bst
 * Added defaultProgram variable to fix problem with extra delete of programs.
 *
 * Revision 1.12  1991/12/12  04:35:51  bst
 * Added dat_dspLoadProgram() (Extracted from mode.c)
 *
 * Revision 1.11  1991/12/08  20:27:11  bst
 * Removed include of memory.h
 *
 * Revision 1.10  1991/12/06  05:58:39  bst
 * More coercions before shifting.
 *
 * Revision 1.9  1991/12/05  18:49:39  bst
 * Coerce bytes to unsigned int before shifting to form a word
 *
 * Revision 1.8  1991/11/14  05:37:59  bst
 * Added dat_dspGetParameter() and dat_dspSetParameter().
 * Added global variable 'dat_dspProgram'
 *
 * Revision 1.7  1991/11/14  01:01:52  bst
 * Renamed 'dsp_write'->dat_dspWrite and 'dsp_read'->'dat_dspRead'
 * Added dat_dspGetParams()
 * Removed dsp_getpcr and dsp_setpcr
 *
 * Revision 1.6  1991/11/04  02:10:05  bst
 * Fixed bug which was overrunning memory while byte-swapping.
 *
 * Revision 1.5  1991/10/25  03:36:42  bst
 * Made indep. of host byte-order.
 * Went back to byte transfers for dsp_write() and dsp_read() (but only even cnts)
 * dsp_write() and dsp_read() now word with data in DSP32c byte-order, indep.
 *     of host byte-order.
 *
 * Revision 1.4  1991/09/12  04:48:56  bst
 * Fixed problem with too many local variables on Alliant.
 *
 * Revision 1.3  1991/09/09  04:53:06  bst
 * Changed lowlevel interface to dsp_write().
 *
 * Revision 1.2  1991/08/23  23:50:52  bst
 * Modified function declarations to work with non-ANSI compilers
 *
 * Revision 1.1  1991/08/05  06:06:53  bst
 * Initial Revision
 */
#include <math.h>
#include "libdefs.h"
#include "host_if.h"
#include "pgm.h"
#include "support.h"
#include "pgmspecs.h"

#define DSP_MAXPROGRAM DL_MAXMSGLENGTH-DL_HEADER_LEN-6

static int SetupDSPProgram P((dat_fd dt,Program *result,PGMSpecs *specs,
			      int cap,int todat));

static int SetupDSPProgram(dt,result,specs,cap,todat)
dat_fd dt;
Program *result;
PGMSpecs *specs;
int cap;
int todat;
{
    double irate, orate;

    if ((cap & (((specs->mode&(MODE_LEFT|MODE_RIGHT))==(MODE_LEFT|MODE_RIGHT))?
	      DSP_CAP_STEREO:DSP_CAP_MONO)) == 0) {
	NALog(LOG_WARNING,"DSP program '%s' does not support %s transfers.\n",
		result->name,
		((specs->mode&(MODE_LEFT|MODE_RIGHT))==(MODE_LEFT|MODE_RIGHT))?
		"STEREO":"MONO");
	dat_errno = EDLBADPGM;
	return -1;
    }

    if (todat) {
	irate = specs->hostrate;
	orate = specs->datrate;
    }
    else {
	irate = specs->datrate;
	orate = specs->hostrate;
    }
	
    if (cap & DSP_CAP_CONVERSION) {
	int maxImpulse = specs->impulseAlloc;
	if (SetupConverter(irate,orate,todat?1:0,result,specs->mode,
			   dat_getDSPMemSize(dt,0),
			   specs->atten,specs->nf,specs->freqs,
			   specs->levels,&specs->impulseRate,
			   specs->impulse,&maxImpulse) < 0) {
	    dat_errno = EDLSCONV;
	    return -1;
	}
	specs->impulseLength = maxImpulse;
    }
    else if (fabs(specs->datrate-specs->hostrate) > 1) {
	NALog(LOG_WARNING,"Sample rate conversion (%.0f->%.0f) requested, but DSP program '%s' does not support it.\n",
		irate,orate,result->name);
	dat_errno = EDLBADPGM;
	return -1;
    }
    return 0;
}

/* Build a program to do sample rate conversion */
Program *dat_getDSPProgram(dt,pgm,pspecs,rspecs)
dat_fd dt;
const Program *pgm;
PGMSpecs *pspecs, *rspecs;
{
    int version = dat_version(dt);
    Program *result;
    int cap;


    Debug("libdatlink.dat_getDSPProgram",2,"Building program for version %d\n",
	  version/1000);
	
    /* Find the basic template of the DSP program */
    if (pgm)
	/* Make a copy of the given program */
	result = DuplicateProgram(pgm);
    else {
	char pgmName[20];
	
	if (pspecs && rspecs) {
	    /* Bidirectional */
	    if ((fabs(pspecs->datrate-pspecs->hostrate) < 1) &&
		(fabs(rspecs->datrate-rspecs->hostrate) < 1)) {
		/* Use natural rate */
		if (pspecs->impulseAlloc >= 1) {
		    pspecs->impulse[0] = 1;
		    pspecs->impulseLength = 1;
		}
		if (rspecs->impulseAlloc >= 1) {
		    rspecs->impulse[0] = 1;
		    rspecs->impulseLength = 1;
		}
		sprintf(pgmName,"FIX%d",version/1000);
	    }
	    else if ((pspecs->mode & (MODE_LEFT|MODE_RIGHT)) ==
		     (MODE_LEFT|MODE_RIGHT)) {
		/* Stereo */
		if ((rspecs->mode & (MODE_LEFT|MODE_RIGHT)) !=
		    (MODE_LEFT|MODE_RIGHT)) {
		    NALog(LOG_WARNING,"Attempt to do bidirectional transfers with different input/output modes.\n");
		    dat_errno = EDLBADPGM;
		    return 0;
		}
		sprintf(pgmName,"FTS%d",version/1000);
	    }
	    else
		/* Mono */
		sprintf(pgmName,"FTM%d",version/1000);
	}
	else if ((pspecs == 0) && (rspecs == 0)) {
	    NALog(LOG_WARNING,"dat_getDSPProgram() called with both pspecs and rspecs null\n");
	    dat_errno = EDLBADPGM;
	    return 0;
	}
	else {
	    /* Uni-directional */
	    PGMSpecs *specs = pspecs?pspecs:rspecs;

	    if (fabs(specs->datrate-specs->hostrate) < 1) {
		/* Use natural rate */
		if (specs->impulseAlloc >= 1) {
		    specs->impulse[0] = 1;
		    specs->impulseLength = 1;
		}
		sprintf(pgmName,"FIX%d",version/1000);
	    }
	    else if ((specs->mode & (MODE_LEFT|MODE_RIGHT)) ==
		     (MODE_LEFT|MODE_RIGHT))
		/* Stereo */
		sprintf(pgmName,"%s%d",pspecs?"TS":"FS",version/1000);
	    else
		/* Mono */
		sprintf(pgmName,"%s%d",pspecs?"TM":"FM",version/1000);
	}
	result = ReadDSPProgram(pgmName);
	
	if (result == 0) {
	    NALog(LOG_ERR,"Unable to read DSP program: %s\n",pgmName);
	    dat_errno = EDLBADPGM;
	    return 0;
	}
    }
    
    cap = GetCapabilities(result);
    
    Debug("libdatlink.dat_getDSPProgram",2,
	  "Using program '%s' with capabilities: 0x%x\n",result->name,cap);
    
    if (pspecs) {
	if ((cap & DSP_CAP_TODAT) == 0) {
	    NALog(LOG_WARNING,
		    "DSP program '%s' does not support transfers to DAT.\n",
		    result->name);
	    DeleteProgram(result);
	    dat_errno = EDLBADPGM;
	    return 0;
	}
	    
	if (SetupDSPProgram(dt,result,pspecs,cap,1) < 0) {
	    DeleteProgram(result);
	    return 0;
	}
    }

    if (rspecs) {
	if ((cap & DSP_CAP_FROMDAT) == 0) {
	    NALog(LOG_WARNING,
		    "DSP program '%s' does not support transfers to DAT.\n",
		    result->name);
	    DeleteProgram(result);
	    dat_errno = EDLBADPGM;
	    return 0;
	}
	    
	if (SetupDSPProgram(dt,result,rspecs,cap,0) < 0) {
	    DeleteProgram(result);
	    return 0;
	}
    }

	
    return result;
}

/* Load a program into the DSP32 */
int dat_dspLoadProgram(dt,pgm)
dat_fd dt;
const Program *pgm;
{
    static Program *defaultProgram = 0;
    int version = dat_version(dt);
    unsigned char *buf = 0;
    int32 plen,wrlen;
    DL_Status stat;
    size_t bufSize;
    byte ackcode;

    if (pgm == 0) {
	if (defaultProgram == 0) {
	    char pgmName[20];
	    sprintf(pgmName,"FIX%d",version/1000);
	    defaultProgram =  ReadDSPProgram(pgmName);
	    if (defaultProgram == 0) {
		NALog(LOG_ERR,"No DSP program found for HW Revision %d\n",
			version/1000);
		return -1;
	    }
	}
	pgm = defaultProgram;
    }

    Debug("libdatlink.loadProgram",1,"Loading program: %s\n",pgm->name);

    /* Send program, complete with symbol table, to DAT-Link but with ".noload" segments
       omitted */

    dat_getStatus(dt, &stat);
    bufSize = stat.pb.bufferSize;

    buf = (byte *)realmalloc(bufSize);
    while (buf == 0) {
	Debug("libdatlink.loadProgram",1,"Out of memory allocating %ld bytes for DSP program.\n",
	       (long)bufSize);
	bufSize /= 2;
	buf = (byte *)realmalloc(bufSize);
    }
    plen = EncodeProgram(pgm,buf,(int32)bufSize,1);
    wrlen = ((plen+BLOCK_LENGTH-1)/BLOCK_LENGTH)*BLOCK_LENGTH;
    if ((plen < 0) || (wrlen > bufSize)) {
	NALog(LOG_ERR,"Internal Error: DSP Program (%s) too large to download.\n",
	    pgm->name);
	NALog(LOG_ERR,
	      "Max program size (including symtab) is %ld bytes\n", (long)bufSize);
	free(buf);
	return -1;
    }

    if (wrlen != plen)
	/* Avoid a Purify error (UMR) */
	bzero((char *)&buf[plen],(size_t)(wrlen-plen));

    /* Make sure we start with a clean buffer */
    dat_clearTransfers(dt);

    if (dat_writeany(dt,buf,wrlen) != wrlen)  {
	free(buf);
	return -1;
    }
    free(buf);

    if (dat_sendmsg(dt,DL_SETDSPPGM,0,0,1,&ackcode) < 0)
	return -1;
    if (ackcode != 1) {
	NALog(LOG_WARNING,"Firmware rejected DSP program download (code %d)\n",
	      ackcode);
	return -1;
    }
    return 0;
}

int dat_dspSetParameter(dt,pgm,parameter,value,isword)
dat_fd dt;
const Program *pgm;
const char *parameter;
int32 value;
int isword;
{
    byte swapped[4];
    int nr;
    Symbol *sym;

    if (pgm == 0) {
	NALog(LOG_WARNING,"No DSP program loaded, unable to locate parameter: %s\n",
		parameter);
	return -1;
    }
    if ((sym = LookupDSPSymbol(pgm,parameter)) == 0) {
	NALog(LOG_WARNING,"Unable to locate DSP parameter: %s\n",parameter);
	return -1;
    }
    
    /* Covert parameter to LSB-first byte order */
    swapped[0] = value&0xff;
    swapped[1] = (value>>8)&0xff;
    swapped[2] = (value>>16)&0xff;
    swapped[3] = (value>>24)&0xff;

    if (isword) {
	/* Swap bytes assuming params is a word */
	if ((short)(value &0xffff) != value) {
	    NALog(LOG_ERR,"Attempt to set word parameter (%s) to 0x%lx\n",
		    parameter,(long)value);
	}
	nr = dat_dspWrite(dt,sym->addr,swapped,2);
    }
    else {
	nr = dat_dspWrite(dt,sym->addr,swapped,4);
    }
    if (nr < 0)
	return nr;
    Debug("libdatlink.dspSetParameter",1,"%s <- %ld (0x%lx)\n",
	  parameter,(long)value,(long)value);
    return 0;
}

int dat_dspGetParameter(dt,pgm,parameter,value,isword)
dat_fd dt;
const Program *pgm;
const char *parameter;
int32 *value;
int isword;
{
    int swapped;
    byte *bptr = (byte *)&swapped;
    int nr;
    Symbol *sym;

    if (pgm == 0) {
	NALog(LOG_WARNING,"No DSP program loaded, unable to locate parameter: %s\n",
		parameter);
	return -1;
    }
    if ((sym = LookupDSPSymbol(pgm,parameter)) == 0) {
	NALog(LOG_WARNING,"Unable to locate DSP parameter: %s\n",parameter);
	return -1;
    }
    
    nr = dat_dspRead(dt,sym->addr,(byte *)&swapped,(unsigned int)(isword?2:4));
    if (nr < 0)
	return nr;
    if (isword) 
	/* Swap bytes assuming parameter is a word */
	*value = bptr[0]|(((unsigned int)bptr[1])<<8);
    else
	/* Swap bytes assuming parameter is an integer */
	*value = bptr[0]|(((unsigned int32)bptr[1])<<8)|
	    (((unsigned int32)bptr[2])<<16)|(((unsigned int32)bptr[3])<<24);
    Debug("libdatlink.dspGetParameter",1,"%s = %ld\n",parameter,(long)*value);
    return 0;
}

int dat_dspGetFloat(dt,pgm,parameter,value)
dat_fd dt;
const Program *pgm;
const char *parameter;
float *value;
{
    dsp32float x;
    if (dat_dspGetParameter(dt,pgm,parameter,&x,0) < 0)
	return -1;
    *value = NativeFloat(x);
    return 0;
}

int dat_dspSetFloat(dt,pgm,parameter,value)
dat_fd dt;
const Program *pgm;
const char *parameter;
double value;
{
    dsp32float x = DSPFloat(value);
    return dat_dspSetParameter(dt,pgm,parameter,x,0);
}

/* Send a message to the DSP */
int dat_dspMessage(dt,nwords,msg)
dat_fd dt;
unsigned int nwords;
const word *msg;
{
    byte reply;
    unsigned int nbytes = nwords*2;

    if (dat_sendmsg(dt,DL_DSPMESSAGE,nbytes,(const byte *)msg,1,&reply) < 0)
	return -1;
    if (reply != 0)
	return -1;
    return 0;
}
