// file: excise_signal.cc
//
// this is a simple program that excises a signal with 1 or 2 channels
//

// system include files
//
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

// isip include files
//
#include <integral.h>
#include <integral_constants.h>

// local include files
//
#include "excise_signal.h"

// main program
//
main (int argc, char** argv) {
  
  // variables for command line arguments
  //
  float_4 sample_freq = DEFAULT_SAMPLE_FREQ;
  float_4 start_time = DEFAULT_START_TIME;
  float_4 end_time = DEFAULT_END_TIME;
  float_4 pad_time = DEFAULT_PAD_TIME;
  float_4 scale_value = DEFAULT_SCALE;    
  int_4 num_bytes = DEFAULT_NUMBER_BYTES_PER_SAMPLE;
  int_4 num_chan = DEFAULT_NUMBER_OF_CHANNELS;
  int_4 out_mode = DEFAULT_OUTPUT_MODE;
  char_1 file_in[999];
  char_1 file_out[999];  
  FILE* fp_in;
  FILE* fp_out;
  logical_1 feather_flag = DEFAULT_FEATHER_FLAG;
  logical_1 swap_bytes = DEFAULT_SWAP_BYTES;
  
  // set the filenames to defaults
  //
  strcpy((char*)file_in, (char*)DEFAULT_INPUT_FILE);
  strcpy((char*)file_out, (char*)DEFAULT_OUTPUT_FILE);
  
  // get the configuration parameters
  //
  get_parameters_C(argc, argv, sample_freq, start_time, end_time,
		   num_bytes, num_chan, out_mode, pad_time, feather_flag,
		   swap_bytes, scale_value, file_in, file_out);
  
  // check that the feather flag is only set for files with 2 byte samples
  //
  if ((feather_flag == ISIP_TRUE) || (swap_bytes == ISIP_TRUE)) {
    if (num_bytes != (int_4)2) {
      fprintf(stderr,
	      "%s: feathering only supported on 16-bit linear files\n",
	      ISIP_PROGRAM_NAME);
      return(ISIP_ERROR);
    }
  }
  
  // open the file to be read
  //
  if (strcmp((char*)file_in, (char*)DEFAULT_INPUT_FILE) == 0) {
    fp_in = stdin;
  }
  else {
    fp_in = fopen((char*)file_in, "r");
  }
  
  // open the file to write to
  //
  if (strcmp((char*)file_out, (char*)DEFAULT_OUTPUT_FILE) == 0){
    fp_out = stdout;
  }
  else {
    fp_out = fopen((char*)file_out, "w");
  }
  
  if (fp_in == NULL) {
    fprintf(stderr,
	    "%s: error opening file [%s]\n",
	    ISIP_PROGRAM_NAME, file_in);
    return(ISIP_ERROR);
  }
  
  if (fp_out == NULL) {
    fprintf(stderr,
	    "%s: error opening file [%s]\n",
	    ISIP_PROGRAM_NAME, file_out);
    return(ISIP_ERROR);
  }

  // compute the end of file if necessary
  //
  fseek(fp_in, 0, SEEK_END);
  int_4 file_size = ftell(fp_in);
  float_4 end_file_time = (float_4)file_size / 
    (sample_freq * (float_4)num_bytes * (float_4)num_chan);
  rewind(fp_in);
  
  // check for default end time
  //
  if (end_time == DEFAULT_END_TIME) {
    end_time = end_file_time;
  }
  
  // add the pad buffers if necessary
  //
  int_4 pad_samples = (int_4)(pad_time * sample_freq);
  
  // do boundary checking on the provided end-time
  //
  if (end_time < start_time) { 
    fprintf(stderr, "%s: end_time (%f) invalid (%f %f)\n",
	    ISIP_PROGRAM_NAME, end_time, start_time, end_file_time);
    fclose(fp_in);
    return(ISIP_ERROR);
  }
  
  if  (end_time > end_file_time) {
    end_time = end_file_time;
  }
  
  // compute the number of samples to read including the pad samples if
  // the feather mode is set
  //
  int_4 num_samples = (int_4)((end_time - start_time) * sample_freq);
  
  // compute an i/o buffer size
  //
  int_4 buf_size = DEFAULT_BUFFER_SIZE;
  if (num_samples < buf_size) {
    buf_size = num_samples;
  }
  
  // allocate an i/o buffer for the input signal and a single
  // channel of output
  //
  char_1* buffer = (char_1*)malloc(buf_size*num_chan*num_bytes);
  char_1* single_buffer = (char_1*)malloc(buf_size*num_chan*num_bytes/2);
  
  // initialize counters
  //
  int_4 i_cur_time = (int_4)(start_time * sample_freq);
  int_4 i_end_time = (int_4)(i_cur_time + num_samples);
  
  // seek to the starting point in the file
  //
  fseek(fp_in, i_cur_time*num_bytes*num_chan, SEEK_SET);
  
  // create a buffer to hold the pad samples
  //
  char_1* pad_buffer = (char_1*)malloc(pad_samples*num_chan*num_bytes);
  char_1* single_pad_buffer = (char_1*)malloc(pad_samples*num_bytes);
  
  // main loop
  //
  if (num_chan == 2) {
    
    
    // reset the padding buffer
    //
    if (pad_samples > (int_4)0) {
      
      memset((void*)pad_buffer, (int)0, (pad_samples * sizeof(char_1) *
					 num_chan * num_bytes));
    }
    
    // output both channels
    //
    if (out_mode == 2) {
      
      // see if we need to feather, if not, then dump a buffer of zeros
      //
      if (feather_flag == ISIP_FALSE) {
	
	// dump the padding buffer
	//
	fwrite(pad_buffer, num_chan*num_bytes, pad_samples, fp_out);
	
      }
      else {
	
	// make sure that the padding can be achieved without zero-stuffing
	//
	if (i_cur_time < pad_samples) {
	  
	  // seek to the beginning of the file
	  //
	  rewind(fp_in);
	  
	  // read in the buffer
	  //
	  fread(pad_buffer + ((pad_samples - i_cur_time) * num_chan *
			      num_bytes), num_chan*num_bytes,
		pad_samples - i_cur_time, fp_in);
	  
	}
	
	else {
	  
	  // seek to the starting point in the file
	  //
	  fseek(fp_in, (i_cur_time-pad_samples)*num_bytes*num_chan, SEEK_SET);
	  
	  // read in the buffer
	  //
	  fread(pad_buffer, num_chan*num_bytes, pad_samples, fp_in);
	}
	
	// swap bytes if necessary
	//
	if (swap_bytes == ISIP_TRUE) {
	  buffer_swap_bytes_cc(pad_buffer, pad_samples * num_chan);
	}
	
	// loop over all samples and feather for each channel
	//
	buffer_feather_cc(pad_buffer, pad_samples, num_chan,
			  FEATHER_BEGINNING);
	
	// scale data if necessary
	//
      	buffer_scale_cc(pad_buffer, pad_samples, num_chan, scale_value);
	
	// dump the padding buffer
	//
	fwrite(pad_buffer, num_chan * num_bytes, pad_samples, fp_out);
      }
      
      while (i_cur_time < i_end_time) {
	
	// read data
	//
	int_4 samples_read = fread(buffer, num_chan*num_bytes,
				   buf_size, fp_in);
	
	if((i_cur_time + samples_read) > i_end_time) {
	  samples_read = i_end_time - i_cur_time;
	}
	
	// swap bytes if necessary
	//
	
	if (swap_bytes == ISIP_TRUE) {
	  buffer_swap_bytes_cc(buffer, samples_read * num_chan);
	}
	
	// scale data
	//
	buffer_scale_cc(buffer, samples_read, num_chan, scale_value);
	
	// output the data
	//
	int_4 samples_writ = fwrite(buffer, num_chan*num_bytes,
				    samples_read, fp_out);
	if (samples_writ != samples_read) {
	  fprintf(stderr, "%s: samples_writ (%ld) != samples_read (%ld)\n",
		  ISIP_PROGRAM_NAME, samples_writ, samples_read);
	  fclose(fp_in);
	  return(ISIP_ERROR);
	}
	
	// update counters
	//
	i_cur_time += samples_read;
      }
      
      // reset the padding buffer
      //
      if (pad_samples > (int_4)0) {
	
	memset((void*)pad_buffer, (int)0, (pad_samples * sizeof(char_1) *
					   num_chan * num_bytes));
      }
      
      // see if we need to feather, if not, then dump a buffer of zeros
      //
      if (feather_flag == ISIP_FALSE) {
	
	// dump the padding buffer
	//
	fwrite(pad_buffer, num_chan * num_bytes, pad_samples, fp_out);
      }
      else {
	
	fread(pad_buffer, num_chan*num_bytes, pad_samples, fp_in);
	
	// swap bytes if necessary
	//
	if (swap_bytes == ISIP_TRUE) {
	  buffer_swap_bytes_cc(pad_buffer, pad_samples * num_chan);
	}
	
	// loop over all samples and feather for each channel
	//
	buffer_feather_cc(pad_buffer, pad_samples, num_chan,
			  FEATHER_ENDING);
	
	// scale data
	//
      	buffer_scale_cc(pad_buffer, pad_samples, num_chan, scale_value);
	
	
	// dump the padding buffer
	//
	fwrite(pad_buffer, num_chan * num_bytes, pad_samples, fp_out);
      }
      
    }
    
    // output channel 0 only
    //
    else if (out_mode == 0) {
      
      // see if we need to feather, if not, then dump a buffer of zeros
      //
      if (feather_flag == ISIP_FALSE) {
	
	// dump the padding buffer
	//
	fwrite(pad_buffer, num_bytes, pad_samples, fp_out);
      }
      else {
	
	// make sure that the padding can be achieved without zero-stuffing
	//
	if (i_cur_time < pad_samples) {
	  rewind(fp_in);
	  fread(pad_buffer + ((pad_samples - i_cur_time) * num_chan *
			      num_bytes), num_chan*num_bytes,
		pad_samples - i_cur_time, fp_in);
	  
	}
	else {
	  
	  // seek to the starting point in the file
	  //
	  fseek(fp_in, (i_cur_time-pad_samples)*num_bytes*num_chan, SEEK_SET);
	  fread(pad_buffer, num_chan*num_bytes, pad_samples, fp_in);
	}
	
	// swap bytes if necessary
	//
	if (swap_bytes == ISIP_TRUE) {
	  buffer_swap_bytes_cc(pad_buffer, pad_samples * num_chan);
	}
	
	// loop over all samples and feather for each channel
	//
	buffer_feather_cc(pad_buffer, pad_samples, num_chan,
			  FEATHER_BEGINNING);
	
	// scale data
	//
	buffer_scale_cc(pad_buffer, pad_samples, num_chan, scale_value);
	
	// copy channel zero of the pad buffer to the single pad buffer
	//
	int_2* sing_ptr = (int_2*)single_pad_buffer;
	int_2* doub_ptr = (int_2*)pad_buffer;	  
	for (int_4 i = 0; i < pad_samples; i++) {
	  
	  // copy a sample and increment the buffer pointers
	  //
	  sing_ptr[0] = doub_ptr[0];
	  sing_ptr++;
	  doub_ptr = doub_ptr + 2;
	}
	
	// dump the padding buffer
	//
	fwrite(single_pad_buffer, num_bytes, pad_samples, fp_out);
      }
      
      while (i_cur_time < i_end_time) {
	
	// read data
	//
	int_4 samples_read = fread(buffer, num_chan*num_bytes,
				   buf_size, fp_in);
	// truncate at the end time
	//
	if((i_cur_time + samples_read) > i_end_time) {
	  samples_read = i_end_time - i_cur_time;
	}
	
       	// swap bytes if necessary
	//
	if (swap_bytes == ISIP_TRUE) {
	  buffer_swap_bytes_cc(buffer, samples_read * num_chan);
	}
	
	// scale data
	//
	buffer_scale_cc(buffer, samples_read, num_chan, scale_value);
	
	for (int_4 i = 0; i < samples_read; i++) {
	  for (int_4 j = 0; j < num_bytes; j++) {
	    single_buffer[num_bytes*i+j] = buffer[i*num_chan*num_bytes+j];
	  }
	}
	
	int_4 samples_writ = fwrite(single_buffer, num_bytes,
				    samples_read, fp_out);
	
	if (samples_writ != samples_read) {
	  fprintf(stderr, "%s: samples_writ (%ld) != samples_read (%ld)\n",
		  ISIP_PROGRAM_NAME, samples_writ, samples_read);
	  fclose(fp_in);
	  return(ISIP_ERROR);
	}
	
	// update counters
	//
	i_cur_time += samples_read;
      }
      
      // reset the padding buffer
      //
      if (pad_samples > (int_4)0) {
	
	memset((void*)pad_buffer, (int)0, (pad_samples * sizeof(char_1) *
					   num_chan * num_bytes));
      }
      
      // see if we need to feather, if not, then dump a buffer of zeros
      //
      if (feather_flag == ISIP_FALSE) {
	
	// dump the padding buffer
	//
	fwrite(pad_buffer, num_bytes, pad_samples, fp_out);
      }
      else {
	
	fread(pad_buffer, num_chan*num_bytes, pad_samples, fp_in);
	
	// swap bytes if necessary
	//
	if (swap_bytes == ISIP_TRUE) {
	  buffer_swap_bytes_cc(pad_buffer, pad_samples * num_chan);
	}
	
	// loop over all samples and feather for each channel
	//
	buffer_feather_cc(pad_buffer, pad_samples, num_chan,
			  FEATHER_ENDING);
	
	// scale data
	//
	buffer_scale_cc(pad_buffer, pad_samples, num_chan, scale_value);
	
	// copy channel zero of the pad buffer to the single pad buffer
	//
	int_2* sing_ptr = (int_2*)single_pad_buffer;
	int_2* doub_ptr = (int_2*)pad_buffer;	  
	for (int_4 i = 0; i < pad_samples; i++) {
	  
	  // copy a sample and increment the buffer pointers
	  //
	  sing_ptr[0] = doub_ptr[0];
	  sing_ptr++;
	  doub_ptr = doub_ptr + 2;
	}
	
	// dump the padding buffer
	//
	fwrite(single_pad_buffer, num_bytes, pad_samples, fp_out);
      }
      
    }
    
    // output channel 1 only
    //
    else {
      
      // reset the padding buffer
      //
      if (pad_samples > (int_4)0) {
	
	memset((void*)pad_buffer, (int)0, (pad_samples * sizeof(char_1) *
					   num_chan * num_bytes));
      }
      
      // see if we need to feather, if not, then dump a buffer of zeros
      //
      if (feather_flag == ISIP_FALSE) {
	
	// dump the padding buffer
	//
	fwrite(pad_buffer, num_bytes, pad_samples, fp_out);
      }
      else {
	
	// make sure that the padding can be achieved without zero-stuffing
	//
	if (i_cur_time < pad_samples) {
	  
	  // go to the beginning of the file
	  //
	  rewind(fp_in);
	  fread(pad_buffer + ((pad_samples - i_cur_time) * num_chan *
			      num_bytes), num_chan*num_bytes,
		pad_samples - i_cur_time, fp_in);
	  
	}
	else {
	  
	  // seek to the starting point in the file
	  //
	  fseek(fp_in, (i_cur_time-pad_samples)*num_bytes*num_chan, SEEK_SET);
	  fread(pad_buffer, num_chan*num_bytes, pad_samples, fp_in);
	}
	
	// swap bytes if necessary
	//
	if (swap_bytes == ISIP_TRUE) {
	  buffer_swap_bytes_cc(pad_buffer, pad_samples * num_chan);
	}
	
	// loop over all samples and feather for each channel
	//
	buffer_feather_cc(pad_buffer, pad_samples, num_chan,
			  FEATHER_BEGINNING);
	
	// scale data
	//
	buffer_scale_cc(pad_buffer, pad_samples, num_chan, scale_value);
	
	// copy channel zero of the pad buffer to the single pad buffer
	//
	int_2* sing_ptr = (int_2*)single_pad_buffer;
	int_2* doub_ptr = (int_2*)pad_buffer;
	doub_ptr = doub_ptr + 1;
	for (int_4 i = 0; i < pad_samples; i++) {
	  
	  // copy a sample and increment the buffer pointers
	  //
	  sing_ptr[0] = doub_ptr[0];
	  sing_ptr++;
	  doub_ptr = doub_ptr + 2;
	}
	
	// dump the padding buffer
	//
	fwrite(single_pad_buffer, num_bytes, pad_samples, fp_out);
      }
      
      while (i_cur_time < i_end_time) {
	
	// read data
	//
	int_4 samples_read = fread(buffer, num_chan*num_bytes,
				   buf_size, fp_in);
	
	// truncate at the end time
	//
	if((i_cur_time + samples_read) > i_end_time) {
	  samples_read = i_end_time - i_cur_time;
	}
	
	// swap bytes if necessary
	//
	if (swap_bytes == ISIP_TRUE) {
	  buffer_swap_bytes_cc(buffer, samples_read * num_chan);
	}
	
	// scale data
	//
	buffer_scale_cc(buffer, samples_read, num_chan, scale_value);
	
	for (int_4 i = 0; i < samples_read; i++) {
	  for (int_4 j = 0; j < num_bytes; j++) {
	    single_buffer[i*num_bytes+j] =
	      buffer[i*num_chan*num_bytes+j+num_bytes];
	  }
	}
	int_4 samples_writ = fwrite(single_buffer, num_bytes,
				    samples_read, fp_out);
	if (samples_writ != samples_read) {
	  fprintf(stderr, "%s: samples_writ (%ld) != samples_read (%ld)\n",
		  ISIP_PROGRAM_NAME, samples_writ, samples_read);
	  fclose(fp_in);
	  return(ISIP_ERROR);
	}
	
	// update counters
	//
	i_cur_time += samples_read;
      }
      
      // reset the padding buffer
      //
      if (pad_samples > (int_4)0) {
	
	memset((void*)pad_buffer, (int)0, (pad_samples * sizeof(char_1) *
					   num_chan * num_bytes));
      }
      
      // see if we need to feather, if not, then dump a buffer of zeros
      //
      if (feather_flag == ISIP_FALSE) {
	
	// dump the padding buffer
	//
	fwrite(pad_buffer, num_bytes, pad_samples, fp_out);
      }
      else {
	
	fread(pad_buffer, num_chan*num_bytes, pad_samples, fp_in);
	
	// swap bytes if necessary
	//
	if (swap_bytes == ISIP_TRUE) {
	  buffer_swap_bytes_cc(pad_buffer, pad_samples * num_chan);
	}
	
	// loop over all samples and feather for each channel
	//
	buffer_feather_cc(pad_buffer, pad_samples, num_chan,
			  FEATHER_ENDING);
	
	// scale data
	//
	buffer_scale_cc(pad_buffer, pad_samples, num_chan, scale_value);
	
	// copy channel zero of the pad buffer to the single pad buffer
	//
	int_2* sing_ptr = (int_2*)single_pad_buffer;
	int_2* doub_ptr = (int_2*)pad_buffer;
	doub_ptr = doub_ptr + 1;
	
	for (int_4 i = 0; i < pad_samples; i++) {
	  
	  // copy a sample and increment the buffer pointers
	  //
	  sing_ptr[0] = doub_ptr[0];
	  sing_ptr++;
	  doub_ptr = doub_ptr + 2;
	}
	
	// dump the padding buffer
	//
	fwrite(single_pad_buffer, num_bytes, pad_samples, fp_out);
      }
      
    }
    
  }
  
  // number of channels = 1
  //
  else {
    
    // reset the padding buffer
    //
    if (pad_samples > (int_4)0) {
      
      memset((void*)pad_buffer, (int)0, (pad_samples * sizeof(char_1) *
					 num_chan * num_bytes));
    }
    
    // see if we need to feather, if not, then dump a buffer of zeros
    //
    if (feather_flag == ISIP_FALSE) {
      
      // dump the padding buffer
      //
      fwrite(pad_buffer, num_bytes, pad_samples, fp_out);
    }
    else {
      // make sure that the padding can be achieved without zero-stuffing
      //
      if (i_cur_time < pad_samples) {
	
	// go to the beginning of the file
	//
	rewind(fp_in);
	fread(pad_buffer + ((pad_samples - i_cur_time) * num_chan *
			    num_bytes), num_chan*num_bytes,
	      pad_samples - i_cur_time, fp_in);
	
      }
      else {
	
	// seek to the starting point in the file
	//
	fseek(fp_in, (i_cur_time-pad_samples)*num_bytes*num_chan, SEEK_SET);
	fread(pad_buffer, num_chan*num_bytes, pad_samples, fp_in);
      }
      
      // swap bytes if necessary
      //
      if (swap_bytes == ISIP_TRUE) {
	buffer_swap_bytes_cc(pad_buffer, pad_samples * num_chan);
      }
      
      // loop over all samples and feather for each channel
      //
      buffer_feather_cc(pad_buffer, pad_samples, num_chan,
			FEATHER_BEGINNING);
      
      // scale data
      //
      buffer_scale_cc(pad_buffer, pad_samples, num_chan, scale_value);
      
      fwrite(pad_buffer, num_bytes, pad_samples, fp_out);
    }
    
    while (i_cur_time < i_end_time) {
      
      // read data
      //
      int_4 samples_read = fread(buffer, num_bytes, buf_size, fp_in);
      
      // truncate at the end time
      //
      if((i_cur_time + samples_read) > i_end_time) {
	samples_read = i_end_time - i_cur_time;
      }
      
      // swap bytes if necessary
      //
      if (swap_bytes == ISIP_TRUE) {
	buffer_swap_bytes_cc(buffer, samples_read * num_chan);
      }
      
      // scale data
      //
      buffer_scale_cc(buffer, samples_read, num_chan, scale_value);
      
      int_4 samples_writ = fwrite(buffer, num_bytes, samples_read,
				  fp_out);
      if (samples_writ != samples_read) {
	fprintf(stderr, "%s: samples_write (%ld) != samples_read (%ld)\n",
		ISIP_PROGRAM_NAME, samples_writ, samples_read);
	fclose(fp_in);
	return(ISIP_ERROR);
      }
      
      // update counters
      //
      i_cur_time += samples_read;
    }
    
    // reset the padding buffer
    //
    if (pad_samples > (int_4)0) {
      
      memset((void*)pad_buffer, (int)0, (pad_samples * sizeof(char_1) *
					 num_chan * num_bytes));
    }
    
    // see if we need to feather, if not, then dump a buffer of zeros
    //
    if (feather_flag == ISIP_FALSE) {
      
      // dump the padding buffer
      //
      fwrite(pad_buffer, num_bytes, pad_samples, fp_out);
    }
    else {
      
      fread(pad_buffer, num_chan*num_bytes, pad_samples, fp_in);
      
      // swap bytes if necessary
      //
      if (swap_bytes == ISIP_TRUE) {
	buffer_swap_bytes_cc(pad_buffer, pad_samples * num_chan);
      }
      
      // loop over all samples and feather for each channel
      //
      buffer_feather_cc(pad_buffer, pad_samples, num_chan,
			
			FEATHER_ENDING);
      // scale data
      //
      buffer_scale_cc(pad_buffer, pad_samples, num_chan, scale_value);
      
      fwrite(pad_buffer, num_bytes, pad_samples, fp_out);
    }
  }
  
  // close the input file
  //
  fclose(fp_in);
  fclose(fp_out);  
  
  // free memory
  //
  free(buffer);
  free(single_buffer);  
  free(pad_buffer);
  free(single_pad_buffer);
  
  // exit gracefully
  //
  return(ISIP_NO_ERROR);
}
  
