#! /usr/local/bin/wish -f

# file: $ISIP_TRANSCRIBER/gui/src/word_window.tcl
#
# procedures to handle window management for word alignment data
#

# list of procedures:
#
#  word_window_proc {}
#  quit_word_proc {}
#  quit_ww_proc {}
#  save_word_proc {}
#  write_word_proc {filename wordlist}
#  read_words_proc {}
#  load_words_proc {}
#  match_ends_proc {chan}
#

# procedure: word_window_proc
#
# arguments: none
#
# return: none
#
# procedure to open the word alignment popup window
#
proc word_window_proc {} {

    # declare globals
    #
    global num_channels
    global current_channel

    global play_mode
    global word_mode

    global curr_ind

    global transcriber_title
    global bg_col

    global p
    global ww

    global word0_file
    global word_error_flag
    
    global word_zoom_mode
    global word_stop

    global chan0_wordlist
    global chan0_uttlist
    global loadflag

    # return if nothing is loaded
    #
    if {$loadflag == 0} {
	return
    }

    # populate the word list by reading in the word file
    #
    read_words_proc    

    # check if there is a mismatch between the number of utterances in the 
    # transcriptions and the number of utterances in the word alignment file.
    #
    set last_utt [lindex $chan0_uttlist end]
    set utt_uid [lindex $last_utt 1]
    set last_wrd [lindex $chan0_wordlist end]
    set wrd_uid [lindex $last_wrd 1]

    # error out if the uid are not the same in both utter and word lists
    #
    if {$utt_uid != $wrd_uid} {
	warning_proc "There is a mismatch between the number of utterances \
		in the transcriptions and the number of utterances in the \
		word alignment file. Please synchronize the two files and \
		try again."
	return
    }

    # match the endpoints of each word-set to those of the parent
    # utterance
    #
    match_ends_proc 0

    set ww_flag 0
    if {[file exists $word0_file] == 1} {
	set ww_flag 1
    } 
	
    # start word mode only if files exist
    #
    if {$ww_flag == 0} {
	warning_proc "You tried to initiate word alignment mode.\n\n \
		However, the specified word alignment files do not exist."
	return
    }

    # set the word mode on
    #
    set word_mode 1
    set word_zoom_mode 0
    set word_stop 0

    # don't recreate the window, try to reuse existing one
    #
    if {[winfo exists $ww] == 1 } {

        wm deiconify $ww
        raise $ww
	
    } else { 

	# start a new window
	#
	toplevel $ww
	wm title $ww "$transcriber_title Word Align"
	wm geometry $ww "600x300+120+360"
	
	# make a top, body and bottom frame
	#
	frame $ww.top_f -borderwidth 1 -relief ridge -height 40 -width 600 \
		-background $bg_col
	frame $ww.body_f -borderwidth 1 -relief ridge -height 420 -width 600 \
		-background $bg_col
	frame $ww.bot_f -borderwidth 1 -relief ridge -height 120 -width 600 \
		-background $bg_col
	pack $ww.top_f -pady 1 -fill both
	pack $ww.body_f -pady 1 -fill both -expand true
	pack $ww.bot_f -pady 1 -fill both

	# divide the bottom frame for word additions
	#
	frame $ww.bot1_f -borderwidth 0 -relief ridge -height 60 -width 600 \
		-background $bg_col
	frame $ww.bot2_f -borderwidth 0 -relief ridge -height 60 -width 600 \
		-background $bg_col
	pack $ww.bot1_f $ww.bot2_f -fill both -expand true -padx 1 -pady 1 \
		-in $ww.bot_f

	# make a display frame and a control frame
	#
	frame $ww.disp_f -borderwidth 1 -relief ridge -height 420 -width 450 \
		-background $bg_col
	frame $ww.cont_f -borderwidth 1 -relief ridge -height 420 -width 150 \
		-background $bg_col
	pack $ww.disp_f -side left -fill both -expand true -padx 1 -pady 1 \
		-in $ww.body_f
	pack $ww.cont_f -side right -fill both -padx 1 -pady 1 -in $ww.body_f

	# subframes for controls
	#
	frame $ww.cont1_f -borderwidth 0 -relief ridge -height 80 -width 150 \
		-background $bg_col
	frame $ww.cont2_f -borderwidth 0 -relief ridge -height 80 -width 150 \
		-background $bg_col
	frame $ww.cont3_f -borderwidth 0 -relief ridge -height 80 -width 150 \
		-background $bg_col
	frame $ww.cont4_f -borderwidth 0 -relief ridge -height 80 -width 150 \
		-background $bg_col
	frame $ww.cont5_f -borderwidth 0 -relief ridge -height 80 -width 150 \
		-background $bg_col
	pack $ww.cont1_f $ww.cont2_f -fill both -expand true -in $ww.cont_f
	pack $ww.cont3_f $ww.cont4_f $ww.cont5_f -pady 1 -fill both \
		-expand true -in $ww.cont_f
	
	# add a label for speaker id
	#
	label $ww.id_l -justify center -text "Utterance Id:" \
		-background $bg_col
	label $ww.t_l -justify center -background $bg_col -anchor w \
		-text "Start      End          Word"
	pack $ww.id_l $ww.t_l -fill both -expand true -in $ww.top_f

	# add a listbox to display words for this utterance
	#
	listbox $ww.list -selectmode browse -yscrollcommand "$ww.yscroll set" \
		-background $bg_col
	scrollbar $ww.yscroll -command "$ww.list yview" -background $bg_col
	pack $ww.list -side left -padx 1 -pady 1 -fill both -expand true \
		-in $ww.disp_f
	pack $ww.yscroll -side right -padx 1 -pady 1 -fill y -in $ww.disp_f
	bind $ww.list <Double-Button-1> {
	    set select [$ww.list curselection]
	    focus_word_proc [lindex $select 0]
	    show_word_proc
	    play_word_proc 0
	}
	
	# add control buttons for traversing words
	#
	button $ww.prev -text "Prev Word" -width 8 -background $bg_col \
		-command {
	    prev_word_proc
	    play_word_proc 0
	}
	button $ww.next -text "Next Word" -width 8 -background $bg_col \
		-command {
	    next_word_proc
	    play_word_proc 0
	}
	
	# add control buttons for traversing utterances
	#
	button $ww.prev_utt -text "Prev Utt" -width 8 -background $bg_col \
		-command {
	    prev_utt_proc
	}
	button $ww.next_utt -text "Next Utt" -width 8 -background $bg_col \
		-command {
	    next_utt_proc
	}

	# add control buttons for audio play
	#
	button $ww.play_word -text "Play Word" -width 8 \
		-background $bg_col -command {
	    play_word_proc 0
	}
	button $ww.play_utt -text "Play Utt" -width 8 \
		-background $bg_col -command {
	    play_utt_proc 0
	}
	button $ww.play_ahead -text "Play Ahead" -width 8 \
		-background $bg_col -command {
	    $ww config -cursor watch
	    cont_word_proc
	    $ww config -cursor top_left_arrow
	}
	button $ww.play_stop -text "Stop Play" -width 8 \
		-background $bg_col -command {
	    set word_stop 1
	}

	# add control buttons for zooming words
	#
	button $ww.view_word -text "Word View" -width 8 -background $bg_col \
		-command {
	    set word_zoom_mode 1
	    set select [$ww.list curselection]
	    if {$select == ""} {
		warning_proc "No word selected. Please select a word \
			in the listbox."
		return
	    }
	    focus_word_proc [lindex $select 0]
	    show_word_proc
	}
	button $ww.view_normal -text "Full View" -width 8 \
		-background $bg_col -command {
	    set word_zoom_mode 0
	    set select [$ww.list curselection]
	    if {$select == ""} {
		warning_proc "No word selected. Please select a word \
			in the listbox."
		return
	    }
	    focus_word_proc [lindex $select 0]
	    show_word_proc
	}

	# add buttons and entryfields to manipulate words
	#
	button $ww.subst -text "Replace" -width 8 -background $bg_col \
		-command {
	    word_replace_proc
	}
	button $ww.insert -text "Insert" -width 8 -background $bg_col \
		-command {
	    word_insert_proc
	}
	button $ww.delete -text "Delete" -width 8 -background $bg_col \
		-command {
	    word_delete_proc
	}
	entry $ww.entry -textvariable ww_word -width 16 -background $bg_col

	pack $ww.subst $ww.insert $ww.delete -side left -padx 1 -pady 1 \
		-fill both -in $ww.bot1_f
	pack $ww.entry -side right -padx 1 -pady 1 -fill both -expand true \
		-in $ww.bot1_f

	# add control buttons for save and quit
	#
	button $ww.save -text "Save" -width 8 -background $bg_col \
		-command {
	    save_word_proc
	}
	button $ww.quit -text "Quit" -width 8 -background $bg_col -command {
	    quit_word_proc
	}

	# pack the buttons
	#
	pack $ww.play_word -side left -padx 1 -pady 1 -fill both \
		-in $ww.cont1_f
	pack $ww.play_utt -side right -padx 1 -pady 1 -fill both \
		-in $ww.cont1_f
	pack $ww.play_ahead -side left -padx 1 -pady 1 -fill both \
		-in $ww.cont2_f
	pack $ww.play_stop -side right -padx 1 -pady 1 -fill both \
		-in $ww.cont2_f

	pack $ww.prev -side left -padx 1 -pady 1 -fill both -in $ww.cont3_f
	pack $ww.next -side right -padx 1 -pady 1 -fill both -in $ww.cont3_f

	pack $ww.prev_utt -side left -padx 1 -pady 1 -fill both \
		-in $ww.cont4_f
	pack $ww.next_utt -side right -padx 1 -pady 1 -fill both \
		-in $ww.cont4_f

	pack $ww.view_word -side left -padx 1 -pady 1 -fill both \
		-in $ww.cont5_f
	pack $ww.view_normal -side right -padx 1 -pady 1 -fill both \
		-in $ww.cont5_f

	pack $ww.save -side left -padx 1 -pady 1 -fill both -in $ww.bot2_f
	pack $ww.quit -side right -padx 1 -pady 1 -fill both -in $ww.bot2_f
    }

    # populate the listbox with words for the current utterance
    #
    set_current_proc $curr_ind

    # key bindings for the word alignment mode
    #
    bind $ww <Configure> { update_window }
    
    bind $ww <Alt-b> {prev_word_proc}
    bind $ww <Alt-f> {next_word_proc}
    
    bind $ww <Alt-p> {prev_utt_proc}
    bind $ww <Alt-n> {next_utt_proc}
    
    bind $ww <Alt-d> {word_delete_proc}
    bind $ww <Alt-i> {word_insert_proc}
    bind $ww <Alt-r> {word_replace_proc}
    
    bind $ww <Alt-q> {quit_word_proc}
    bind $ww <Alt-s> {save_word_proc}

    bind $ww <Alt-Down> {
	set word_zoom_mode 1
	set select [$ww.list curselection]
	if {$select == ""} {
	    warning_proc "No word selected. Please select a word \
		    in the listbox."
	    return
	}
	focus_word_proc [lindex $select 0]
    }
    bind $ww <Alt-Up> {
	set word_zoom_mode 0
	set select [$ww.list curselection]
	if {$select == ""} {
	    warning_proc "No word selected. Please select a word \
		    in the listbox."
	    return
	}
	focus_word_proc [lindex $select 0]
    }
}

# procedure: quit_word_proc
#
# arguments: none
#
# return: none
#
# procedure to close the word align window
#
proc quit_word_proc {} {

    # create globals
    #
    global ww_quit
    set ww_quit ".quit"
    
    global transcriber_title
    global bg_col

    # don't recreate the window, try to reuse existing one
    #
    if { [winfo exists $ww_quit] == 1 } {
        wm deiconify $ww_quit
        raise $ww_quit
 
    } else {
        # create a new window
        #
        toplevel $ww_quit
        wm title $ww_quit "$transcriber_title Word Exit"
	wm geometry $ww_quit "+500+600"
  
        # make a bottom frame
        #
        frame $ww_quit.foot -background $bg_col
        pack $ww_quit.foot -side bottom -fill x -pady 2
 
        button $ww_quit.save -width 6 -text Save -default active \
		-background $bg_col -command {
	    save_word_proc
	    quit_ww_proc
	}
        button $ww_quit.ok -width 6 -text Quit -background $bg_col \
		-command {quit_ww_proc}
        button $ww_quit.cancel -width 6 -text Cancel -background $bg_col \
		-command "destroy $ww_quit"
        pack $ww_quit.save $ww_quit.ok $ww_quit.cancel -side left  -fill x \
                -padx 1m -pady 1m -expand true -in $ww_quit.foot
 
        # make a message frame
        #
        frame $ww_quit.top -background $bg_col
        pack $ww_quit.top -side top -fill x -padx 1m -pady 1m
        message $ww_quit.msg -justify center -width 2i -background $bg_col \
		-text "Quitting will destroy the autosave files. \
		Do you want to save word alignment data before you quit?"
        pack $ww_quit.msg -side top -fill both -in $ww_quit.top
    }
 
    bind $ww_quit <Return> {quit_ww_proc}
}

# procedure: quit_ww_proc
#
# arguments: none
#
# return: none
#
# procedure to quit cleanly
#
proc quit_ww_proc {} {
    
    # declare globals
    #
    global curr_ind
    global word_mode

    global ww
    global ww_quit

    global auto_w0_file

    global chan0_wordlist

    global p

    # reset word mode
    #
    set word_mode 0

    # remove autosave files
    #
    catch {exec rm -f $auto_w0_file}
    
    # clear the wordlists
    #
    set chan0_wordlist {}

    # shut down all windows
    #
    destroy $ww
    destroy $ww_quit

    # make the original window toplevel again
    #
    wm withdraw .
    set_current_proc $curr_ind
}

# procedure: save_word_proc
#
# arguments: none
#
# return: none
#
# procedure to save data
#
proc save_word_proc {} {
    
    # declare globals
    #
    global word0_file
    global num_channels
    global chan0_wordlist
 
    # write the data as belonging to channel 0
    #
    write_word_proc $word0_file $chan0_wordlist
}

# procedure: write_word_proc
#
# arguments:
#  filename: file to write data to
#  wordlist: word list
#
# return: none
#
# procedure to write data to output file
#
proc write_word_proc {filename wordlist} {
    
    # make sure file is writable
    #
    if {[file writable $filename] != 1} {
	warning_proc "You do not have write permissions for the word \
		alignment file $filename."
	return
    }

    # open the file
    #
    set fileptr [open $filename w]
    
    # for each word output values
    #
    foreach item $wordlist {
	
        # set the output string
        #
        foreach {cid uid spkr sex age start end trans} $item {
            set outstr [format "%s%04d %s %s %s %.6f %.6f %s" \
		    $cid $uid $spkr $sex $age $start $end $trans]
        }
	
        # print this
        #
        puts $fileptr $outstr
    }
    
    # close file
    #
    close $fileptr
}

# procedure: read_words_proc
#
# arguments: none
#
# return: none
#
# procedure to read the word data from file and populate wordlists
#
proc read_words_proc {} {

    # declare globals
    #
    global word0_file
 
    global current_channel
    global num_channels

    global chan0_wordlist
 
    # read the data as belonging to channel 0
    #
    set chan0_wordlist [read_trans_proc $word0_file]

}

# procedure: load_words_proc
#
# arguments: none
#
# return: none
#
# procedure to load the current utterance words in the listbox
#
proc load_words_proc {} {

    # declare globals
    #
    global current_channel
    global num_channels
    global chan0_wordlist

    global word_index_list

    global curr_ind
    global cutt_start
    global cutt_end
    global cutt_trans
    global cutt_id    

    global ww

    # clear the listbox
    #
    $ww.list delete 0 end
    set word_index_list {}
    set word_count 0

    # get all the words for the current utterance and fill the listbox
    #
    set wordlist $chan0_wordlist
 
    # get the string to display
    #
    foreach item $wordlist {
 
        # insert each label heading in the listbox
        #
        foreach {cid uid spkr sex age start end trans} $item {
	    set word_id [format "%s%.4d" $cid $uid]
	    if {$cutt_id == $word_id} {
		set word_entry [format "%.6f     %.6f        %s" \
			$start $end $trans]
		$ww.list insert end $word_entry
		lappend word_index_list $word_count
	    }
        }

	# increment word count
	#
	incr word_count
    }

    # update the label
    #
    $ww.id_l configure -text "Utterance Id: $cutt_id"

    # focus on the first element
    #
    focus_word_proc 0
}

# procedure: match_ends_proc
#
# arguments:
#  chan: input channel
#
# return: none
#
# procedure to match the endpoints of each word-set to those of the
# parent utterance
#
proc match_ends_proc {chan} {

    # declare globals
    #
    global chan0_wordlist
    global chan0_uttlist

    # work on the appropriate channel as indicated
    #
    set newlist {}
    set wordlist {}
    set uttlist {}
    set wordlist $chan0_wordlist
    set uttlist $chan0_uttlist

    # loop over each utterance
    #
    set ind 0
    set curr 0
    set nwui 1
    set wuid 1
    set total [llength $wordlist]

    foreach item $uttlist {

	# for each utterance get the start and end times
	#
	foreach {cid uid spkr sex age start end trans} $item {

	    # as long as the words belong to this utterance
	    #
	    while {$wuid == $uid} {

		# loop over the corresponding words in the word list
		#
		set word [lindex $wordlist $ind]
		incr ind
		incr curr
 		
		# get the params of this word
		#
		foreach {wcid wuid wspkr wsex wage wstart wend wtrans} $word {

		    # if this is the first word fix start time
		    #
		    if {$curr == 0} {
			set wstart $start
		    }  

		    # if this is the end of file change end time
		    #
		    if {$ind == $total} {
			set wend $end
			incr nwui
		    }

		    # if it is end of word change end time
		    #
		    if {$ind < $total} {

			# get the next word
			#
			set next_word [lindex $wordlist $ind]
			foreach {nwci nwui nwsp nwse nwag nwst nwen nwtr}\
				$next_word {

			    if {$nwui != $wuid} {
				set wend $end
				set curr 0
			    }
			}
		    }

		    # make new word for new list
		    #
		    set new_word {}
		    lappend new_word $wcid $wuid $spkr $wsex $age $wstart\
			    $wend $wtrans
		    lappend newlist $new_word
		    
		    # reset variables
		    #
		    set wuid $nwui
		}
	    }
	}
    }
    
    # revert the lists
    #
    set wordlist $newlist
    set chan0_wordlist $newlist
}

#
# end of file
