#! /usr/local/bin/wish -f

# file: $ISIP_TRANSCRIBER/gui/src/word_proc.tcl
#
# procedures to handle word mode functionality
#

# list of procedures:
#
#  plot_words_proc {index}
#  prev_word_proc {}
#  next_word_proc {}
#  play_word_proc {bg_flag}
#  cont_word_proc {}
#  word_boundary_proc {}
#  focus_word_proc {index}
#  show_word_proc {}
#  word_replace_proc {}
#  word_insert_proc {}
#  word_delete_proc {}
#

# procedure: plot_words_proc
#
# arguments:
#  index: wordlist index
#
# return: none
#
# procedure to draw the word boundary segments
#
proc plot_words_proc {index} {

    # declare globals
    #
    global num_channels
    global current_channel

    global plot_width
    global font_1

    global window_start
    global window_end
    global window_duration

    global bracket_start
    global bracket_end

    global word_zoom_mode

    global ww
    global p
    
    # draw the current channel word plot
    #
    set yy 10
        
    # set the coordinates for the text labels
    #
    set y_t [expr $yy-5]
    
    # get the contents of the listbox
    #
    set st $index
    set et $index
    set wordlist {}
    if {$word_zoom_mode == 0} {
	set wordlist [$ww.list get 0 end]
    } else {
	if {$index > 0} {
	    set st [expr $st - 1]
	}
	if {$index < [$ww.list size]} {
	    incr et
	}
	set wordlist [$ww.list get $st $et]
    }

    # get the start and end time for the window
    #
    if {[llength $wordlist] > 0} {
	set st_word [lindex $wordlist 0]
	scan $st_word "%f     %f        %s" start end trans
	set window_start $start
	set st_word [lindex $wordlist end]
	scan $st_word "%f     %f        %s" start end trans
	set window_end $end
	set window_duration [expr $window_end - $window_start]
	
	# get the bracket marks
	#
	if {$word_zoom_mode == 1} {
	    set idd 1
	    if {$st == $index} {
		set idd 0
	    }
	    set st_word [lindex $wordlist $idd]
	    scan $st_word "%f     %f        %s" start end trans
	    set bracket_start $start
	    set bracket_end $end
	} else {
	    set bracket_start $window_start
	    set bracket_end $window_end
	}	    

	# plot the signal with these params
	#
	plot_channel_proc
    }

    # clear previous contents of the canvas
    #
    $p.trn_plot delete all
    
    # get the start and end time for each word in the current utterance
    #
    foreach word_str $wordlist {
	
	# scan the listbox string
	#
	scan $word_str "%f     %f        %s" start end trans
	
	# draw the lines
	#
	set x1 [eval expr ($plot_width / $window_duration) \
		* ($start - $window_start)]
	set x2 [eval expr ($plot_width / $window_duration) \
		* ($end - $window_start)]
	
	set x_t [expr $x1 + 16]
	if {$x_t == 16} {
	    set x_t 24
	}

	# plot the line
	#
	$p.trn_plot create line $x1 $yy $x2 $yy -width 1.0 \
		-arrow both -arrowshape {1 2 4}
	$p.trn_plot create text $x_t $y_t -text $trans -font $font_1
    }

    # get the current display string
    #
    if {$index > -1} {
	set word_str [$ww.list get $index]
	scan $word_str "%f     %f        %s" start end trans
	
	# draw the current channel word plot
	#
	set yy 10
		
	# draw the lines
	#
	set x1 [eval expr ($plot_width / $window_duration) \
		* ($start - $window_start)]
	set x2 [eval expr ($plot_width / $window_duration) \
		* ($end - $window_start)]
	
	# plot the line
	#
	$p.trn_plot create line $x1 $yy $x2 $yy -width 3.0 \
		-arrow both -arrowshape {1 2 4}
    }
}

# procedure: prev_word_proc
#
# arguments: none
#
# return: none
#
# procedure to select previous word
#
proc prev_word_proc {} {

    # declare globals
    #
    global ww

    # set the new word
    #
    set select [$ww.list curselection]
    if {$select == ""} {
	warning_proc "No word selected. Please select a word in the listbox."
	return
    }
    set curr_word [lindex $select 0]
    incr curr_word -1
    if {$curr_word < 0} {
	set curr_word 0
    }
    focus_word_proc $curr_word
}

# procedure: next_word_proc
#
# arguments: none
#
# return: none
#
# procedure to select next word
#
proc next_word_proc {} {

    # declare globals
    #
    global ww

    # set the new word
    #
    set select [$ww.list curselection]
    if {$select == ""} {
	warning_proc "No word selected. Please select a word in the listbox."
	return
    }
    set curr_word [lindex $select 0]
    incr curr_word
    if {$curr_word == [$ww.list size]} {
	incr curr_word -1
    }
    focus_word_proc $curr_word
}

# procedure: play_word_proc
#
# arguments:
#  bg_flag: audio flag
#
# return: none
#
# procedure to play the word
#
proc play_word_proc {bg_flag} {

    # declare globals
    #
    global ww
    global play_mode

    # get the current display string
    #
    set select [$ww.list curselection]
    if {$select == ""} {
	warning_proc "No word selected. Please select a word in the listbox."
	return
    }
    set curr_word [lindex $select 0]
    set word_str [$ww.list get $curr_word]
    scan $word_str "%f     %f        %s" start end trans
    
    # play between endpoints
    #
    audio_play_proc $play_mode $start $end $bg_flag
}

# procedure: cont_word_proc
#
# arguments: none
#
# return: none
#
# procedure for continuous play
#
proc cont_word_proc {} {

    # declare globals
    #
    global ww
    global word_stop

    # start at the current selected word
    #
    set num_word [expr [$ww.list size] - 1]
    set select [$ww.list curselection]
    if {$select == ""} {
	warning_proc "No word selected. Please select a word in the listbox."
	return
    }
    set curr_word [lindex $select 0]
    play_word_proc 1

    # play each word after a slight pause
    #
    while {($curr_word < $num_word) && ($word_stop == 0)} {
	incr curr_word
	focus_word_proc $curr_word
	play_word_proc 1
    }

    # set the new focus
    #
    if {$curr_word == $num_word} {
	focus_word_proc 0
    }

    # reset the stop flag
    #
    set word_stop 0
}

# procedure: word_boundary_proc
#
# arguments: none
#
# return: none
#
# procedure for altering the word boundaries
#
proc word_boundary_proc {} {

    # declare globals
    #
    global current_channel
    global num_channels
    global chan0_wordlist

    global word_index_list

    global bracket_start
    global bracket_end

    global ww
    
    # set channel-specific wordlist
    #
    set wordlist $chan0_wordlist
        
    # index of the current word
    #
    set select [$ww.list curselection]
    if {$select == ""} {
	warning_proc "No word selected. Please select a word in the listbox."
	return
    }
    set curr_word [lindex $select 0]
    set w_this [lindex $word_index_list $curr_word]
    
    # get the new endpoints of the current word
    #
    set word_this [lindex $wordlist $w_this]

    # now change the appropriate values
    #
    foreach {cid uid spkr sex age start end trans} $word_this {

	# change the start time
	#
	if {$curr_word > 0} {
	    if {$start != $bracket_start} {
		set word_this [lreplace $word_this 5 5 $bracket_start]
		set wordlist [lreplace $wordlist $w_this $w_this $word_this]

		# change the previous word as well
		#
		set w_prev [expr $w_this - 1]
		set word_prev [lindex $wordlist $w_prev]
		set word_prev [lreplace $word_prev 6 6 $bracket_start]
		set wordlist [lreplace $wordlist $w_prev $w_prev $word_prev]

		# update the current word listbox entry
		#
		foreach {cid uid spkr sex age start end trans} $word_prev {
		    set word_entry [format "%.6f     %.6f        %s" \
			    $start $end $trans]
		    $ww.list delete [expr $curr_word - 1]
		    $ww.list insert [expr $curr_word - 1] $word_entry
		}		
	    }
	}
	    
	# change the end time
	#
	if {$curr_word < [expr [$ww.list size] - 1]} {
	    if {$end != $bracket_end} {
		set word_this [lreplace $word_this 6 6 $bracket_end]
		set wordlist [lreplace $wordlist $w_this $w_this $word_this]
		
		# change the next word as well
		#
		set w_next [expr $w_this + 1]
		set word_next [lindex $wordlist $w_next]
		set word_next [lreplace $word_next 5 5 $bracket_end]
		set wordlist [lreplace $wordlist $w_next $w_next $word_next]

		# update the current word listbox entry
		#
		foreach {cid uid spkr sex age start end trans} $word_next {
		    set word_entry [format "%.6f     %.6f        %s" \
		    $start $end $trans]
		    $ww.list delete [expr $curr_word + 1]
		    $ww.list insert [expr $curr_word + 1] $word_entry
		}	    
	    }
	}

	# update the current word listbox entry
	#
	foreach {cid uid spkr sex age start end trans} $word_this {
	    set word_entry [format "%.6f     %.6f        %s" \
		    $start $end $trans]
	    $ww.list delete $curr_word
	    $ww.list insert $curr_word $word_entry
	}
    }

    # focus on the new word boundary
    #
    focus_word_proc $curr_word

    # set channel-specific wordlist
    #
    set chan0_wordlist $wordlist
}	

# procedure: focus_word_proc
#
# arguments:
#  index: word list index
#
# return: none
#
# procedure to focus on a particular word
#
proc focus_word_proc {index} {
    
    # declare globals
    #
    global ww
    
    # set the selection focus to stated element
    #
    $ww.list selection clear 0 end
    $ww.list see $index
    $ww.list selection set $index
    $ww.entry delete 0 end
    
    # highlight the word boundary segment
    #
    plot_words_proc $index
}

# procedure: show_word_proc
#
# arguments: none
#
# return: none
#
# procedure to display a particular word in the entry window
#
proc show_word_proc {} {
    
    # declare globals
    #
    global current_channel
    global num_channels
    global chan0_wordlist
    global chan1_wordlist

    global word_index_list

    global ww
    
    # set channel-specific wordlist
    #
    set wordlist $chan0_wordlist
    
    # index of the current word
    #
    set select [$ww.list curselection]
    if {$select == ""} {
	warning_proc "No word selected. Please select a word in the listbox."
	return
    }
    set index [lindex $select 0]
    set w_this [lindex $word_index_list $index]
    set word_this [lindex $wordlist $w_this]
    
    # put the transcription in the entry window
    #
    $ww.entry delete 0 end
    foreach {cid uid spkr sex age start end trans} $word_this {
	$ww.entry insert 0 $trans
    }
}

# procedure: word_replace_proc
#
# arguments: none
#
# return: none
#
# procedure to replace current word
#
proc word_replace_proc {} {

    # declare globals
    #
    global current_channel
    global num_channels
    global chan0_wordlist

    global word_index_list

    global ww

    # the new word
    #
    set new_trans [string trim [string trim [$ww.entry get]]]
    if {[string length $new_trans] == 0} {
	warning_proc "Cannot insert an empty string for word!"
	return
    }

    # set channel-specific wordlist
    #
    set wordlist $chan0_wordlist
    
    # index of the current word
    #
    set select [$ww.list curselection]
    if {$select == ""} {
	warning_proc "No word selected. Please select a word in the listbox."
	return
    }
    set curr_word [lindex $select 0]
    set w_this [lindex $word_index_list $curr_word]
    set word_this [lindex $wordlist $w_this]

    # now change the appropriate values
    #
    foreach {cid uid spkr sex age start end trans} $word_this {
	set word_this [lreplace $word_this 7 7 $new_trans]
    }
    set wordlist [lreplace $wordlist $w_this $w_this $word_this]

    # reset the list
    #
    $ww.entry delete 0 end

    # set channel-specific wordlist
    #
    set chan0_wordlist $wordlist
    
    # update the current word listbox entry
    #
    foreach {cid uid spkr sex age start end trans} $word_this {
	set word_entry [format "%.6f     %.6f        %s" \
		$start $end $trans]
	$ww.list delete $curr_word
	$ww.list insert $curr_word $word_entry
    }
    focus_word_proc $curr_word
}

# procedure: word_insert_proc
#
# arguments: none
#
# return: none
#
# procedure to insert a new word after the current word
#
proc word_insert_proc {} {

    # declare globals
    #
    global current_channel
    global num_channels
    global chan0_wordlist

    global word_index_list

    global ww

    # the new word
    #
    set new_trans [string trim [string trim [$ww.entry get]]]
    if {[string length $new_trans] == 0} {
	warning_proc "Cannot insert an empty string for word!"
	return
    }

    # set channel-specific wordlist
    #
    set wordlist $chan0_wordlist
        
    # final index of words in the current utterance
    #
    set last_word [expr [lindex $word_index_list end] + 1]

    # index of the current word
    #
    set select [$ww.list curselection]
    if {$select == ""} {
	warning_proc "No word selected. Please select a word in the listbox."
	return
    }
    set curr_word [lindex $select 0]
    set w_this [lindex $word_index_list $curr_word]
    set word_this [lindex $wordlist $w_this]

    # now change the appropriate values
    #
    set word {}
    foreach {cid uid spkr sex age start end trans} $word_this {

	# update the current word listbox entry
	#
	set word_this [lreplace $word_this 5 5 [expr $end - 0.001]]
	set word_entry [format "%.6f     %.6f        %s" \
		$start [expr $end - 0.001] $trans]
	$ww.list delete $curr_word
	$ww.list insert $curr_word $word_entry
	
	# create the word to insert
	#
	lappend word $cid $uid $spkr $sex $age [expr $end - 0.001] $end \
		$new_trans
    }
    set wordlist [lreplace $wordlist $w_this $w_this $word_this]
    
    # reset the entry
    #
    $ww.entry delete 0 end
    
    # make this the current word
    #
    incr curr_word
    incr w_this
    set word_index_list [lappend word_index_list $last_word]
    set wordlist [linsert $wordlist $w_this $word]

    # set channel-specific wordlist
    #
    set chan0_wordlist $wordlist
    
    # update the current word listbox entry
    #
    foreach {cid uid spkr sex age start end trans} $word {
	set word_entry [format "%.6f     %.6f        %s" \
		$start $end $trans]
	$ww.list insert $curr_word $word_entry
    } 
    focus_word_proc $curr_word
}

# procedure: word_delete_proc
#
# arguments: none
#
# return: none
#
# procedure to delete the current word and add its time to the 
# previous / next word
#
proc word_delete_proc {} {

    # declare globals
    #
    global current_channel
    global num_channels
    global chan0_wordlist

    global word_index_list

    global ww

    # index of the current word
    #
    set select [$ww.list curselection]
    if {$select == ""} {
	warning_proc "No word selected. Please select a word in the listbox."
	return
    }
    set curr_word [lindex $select 0]

    # set channel-specific wordlist
    #
    set wordlist $chan0_wordlist
        
    # total number of words in the current utterance
    #
    if {[llength $word_index_list] == 1} {
	warning_proc "Cannot delete this word! It is the only \
		word in this utterance."
	return
    }

    # set word flag to keep track of adjacent word
    #
    set wordflag 0

    # get the appropriate adjacent word
    #
    set adj_word [expr $curr_word - 1]
    if {$curr_word == 0} {
	set adj_word [expr $curr_word + 1]
	set wordflag 1
    }
    set w_adj [lindex $word_index_list $adj_word]
    set word_adj [lindex $wordlist $w_adj]

    # get the current word start and end times
    #
    set w_this [lindex $word_index_list $curr_word]
    set word_this [lindex $wordlist $w_this]
    foreach {cid uid spkr sex age start end trans} $word_this {
	set wstart $start
	set wend $end
    }

    # change things in the adjacent word accordingly
    #
    if {$curr_word == 0} {
	set word_adj [lreplace $word_adj 5 5 $wstart]
    } else {
	set word_adj [lreplace $word_adj 6 6 $wend]
    }

    # now adjust the adjacent word
    #
    set wordlist [lreplace $wordlist $w_adj $w_adj $word_adj]
    foreach {cid uid spkr sex age start end trans} $word_adj {
	set word_entry [format "%.6f     %.6f        %s" \
		$start $end $trans]
	$ww.list delete $adj_word
	$ww.list insert $adj_word $word_entry
    }

    # remove the current word
    #
    set word_index_list [lreplace $word_index_list end end]
    set wordlist [lreplace $wordlist $w_this $w_this]
    $ww.list delete $curr_word

    # reset the entry
    #
    $ww.entry delete 0 end
    
    # set channel-specific wordlist
    #
    set chan0_wordlist $wordlist
    
    # get the new current word
    #
    if {$wordflag == 1} {
	set curr_word [expr $adj_word - 1]
    } else {
	set curr_word $adj_word
    }
    focus_word_proc $curr_word
}

#
# end of file
