#! /usr/local/bin/wish -f

# file: $ISIP_TRANSCRIBER/gui/src/utter.tcl
#
# functions to manipulate utterance structures
#

# list of procedures:
#
#  verify_utt_proc {}
#  verify_mean_proc {value}
#  verify_noise_proc {value}
#  setage_proc {value}
#  setsex_proc {value}
#  setspk_proc {value}
#  settype_proc {value}
#  set_current_proc {index}
#  next_utt_proc {}
#  prev_utt_proc {}
#  split_utt_proc {}
#  set_utt_proc {}
#  remove_utt_proc {}
#  insert_utt_proc {}
#  clear_trans_proc {}
#  find_current_proc {time_in}
#  delete_utt_proc {}
#

# procedure: verify_utt_proc
#
# arguments: none
#
# return: 1 or 0 for success
#
# procedure to check if the required fields for an utterance are set
#
proc verify_utt_proc {} {

    # declare globals
    #
    global p

    global mean_flag
    global noise_flag
    global lock

    global arraylist
    global audiolist
    global curr_ind

    # dont do any thing if the current transcription has no audio file
    #
    if {[file exists [lindex $audiolist $curr_ind]] != 1} {
	return 1
    }

    # check if the current utterance has been transcribed
    #
    set ckstr [$p.curr_trans get 1.0 end]
    set ckstr [string trim $ckstr]

    if {$lock == 0} {
	if {$ckstr == ""} { 
	    warning_proc "Please make sure that the current utterance has \
		    been transcribed before proceeding any further."
	    return 0
	}
    }

    # check if the user has set the meaning field
    #
    if {$mean_flag == 0 && $lock == 0} {
	warning_proc "Please set the Meaning field to either correct, \
		incorrect before proceeding any further."
	return 0
    }
    
    # check if the user has set the noise field
    #
    if {$noise_flag == 0 && $lock == 0} {
	warning_proc "Please set the Noise field to either low, high \
		before proceeding any further."
	return 0
    }
    return 1
}

# procedure: verify_noise_proc
#
# arguments: 
#
# return: none
#
# procedure to check and see if the noise field is set
#
proc verify_noise_proc {value} {

    global cspk_noise
    global noise_flag

    # set the noise level and noise flag
    #
    set noise_flag 1
    set cspk_noise $value
}

# procedure: verify_mean_proc
#
# arguments: 
#
# return: none
#
# procedure to check and see if the menaing field is set
#
proc verify_mean_proc {value} {

    global cspk_mean
    global mean_flag

    # set the meaning field and the meaning flag
    #
    set mean_flag 1
    set cspk_mean $value
}

# procedure: setage_proc
#
# arguments: none
#
# return: none
#
# procedure to set the speakers age
#
proc setage_proc {value} {
 
    # declare globals
    #
    global arraylist
    global keycount
    global num_utter

    # set the one per call utterance parameters
    #
    set index 0

    while {$index < $num_utter} {
	catch { set arraylist($index,AGE) $value }
	incr index 1
    }
}  

# procedure: setsex_proc
#
# arguments: none
#
# return: none
#
# procedure to set the speakers gender
#
proc setsex_proc {value} {

    # declare globals
    #
    global arraylist
    global keycount
    global num_utter

    # set the one per call utterance parameters
    #
    set index 0

    while {$index < $num_utter} {
	catch { set arraylist($index,GENDER) $value }
	incr index 1
    }
}  

# procedure: settype_proc
#
# arguments: none
#
# return: none
#
# procedure to set the speaker type
#
proc settype_proc {value} {
    
    # declare globals
    #
    global arraylist
    global keycount
    global num_utter

    # set the one per call utterance parameters
    #
    set index 0

    while {$index < $num_utter} {
	catch { set arraylist($index,TYPE) $value }
	incr index 1
    }
}  

   
# procedure: set_current_proc
#
# arguments:
#  index: input utterance index
#
# return: none
#
# procedure to set the current focus
#
proc set_current_proc {index} {

    # declare globals
    #
    global arraylist
    global keycount
    global keys

    global cutt_start
    global cutt_end
    global maximum_time

    global cutt_trans
    global cutt_id
    global cspk_name
    global cspk_sex
    global cspk_age
    global cspk_name
    global cspk_type
    global cspk_mean
    global cspk_noise
    global cspk_id

    global putt_trans
    global nutt_trans

    global curr_ind
    global prev_ind
    global next_ind
    global arrayindex

    global current_channel

    global audio_file
    global audiolist

    global conv_id

    global bookmark_flag
    global bm_start

    global lock

    global num_channels
    global sample_num_bytes
    global sample_frequency

    global sig_plot_0
    global energy_plot_0
    global spec_plot_0

    global window_start
    global window_end
    global window_duration
    global bracket_start
    global bracket_end

    global word_mode
    global p

    global translist
    global grammar
    global nlresults
    global results
    global wavefile

    global cutt_audio
    global putt_audio
    global nutt_audio

    global mean_flag
    global noise_flag
    global callflag

    global curr_call_index
    global callfile_list
    global cutt_call

    global window_len
    global frame_len
    global preemph_coeff
    global preemph_flag
    global window_option
    global window_size
    global rms_factor
    global amplitude_gain
    global energy_ht
    global signal_ht
  
    global autoplay


    # set the index variables only if necessary
    #
    if {($index >= 0) && ($index <= [llength $audiolist])} {

	# set the current focus
	#
	set curr_ind $index
	
	# get the current audio file
	#
	set audio_file [lindex $audiolist $index]

	# set the current audio file name
	#
	set cutt_str [lindex $audiolist $curr_ind]
	set ind [string last "/" $cutt_str]
	if {$ind != -1} {
	    set cutt_audio [string range $cutt_str [expr $ind + 1] end]
	} else {
	    set cutt_audio $cutt_str
	}

	# set the previous audio file name
	#
	set prev [expr $curr_ind - 1]
	if { $prev < 0 } {
	    set putt_audio ""
	} else {
	    set putt_str [lindex $audiolist $prev]
	    set ind [string last "/" $putt_str]
	    if {$ind != -1} {
		set putt_audio [string range $putt_str [expr $ind + 1] end]
	    } else {
		set putt_audio $putt_str
	    }
	}
	
	# set the next audio file name
	#
	set next [expr $curr_ind + 1]
	if { $next > [llength $audiolist] } {
	    set nutt_audio ""
	} else {
	    set nutt_str [lindex $audiolist $next]
	    set ind [string last "/" $nutt_str]
	    if {$ind != -1} {
		set nutt_audio [string range $nutt_str [expr $ind + 1] end]
	    } else {
		set nutt_audio $nutt_str
	    }
	}

	# set the current call name
	#
	set cutt_call_str [lindex $callfile_list $curr_call_index]
	set ind [string last "/" $cutt_call_str]
	set cutt_call [string range $cutt_call_str [expr $ind + 1] end]    

	# update the labels
	#
	$p.cid_label configure -text "Audio File: $cutt_audio"
	$p.pid_label configure -text "Audio File: $putt_audio"
	$p.nid_label configure -text "Audio File: $nutt_audio"
	$p.csession_label configure -text "Call: $cutt_call"
	
	# retrieve the array index for the current audiofile
	#
	set arrayindex $curr_ind

	# retrieve the grammar and insert it into the text area
	#	
	$p.grammar_entry configure -state normal
	$p.grammar_entry delete 1.0 end

	set key "GRAMMAR"
	set insert_grammar ""
	catch { set insert_grammar $arraylist($arrayindex,$key) }

	$p.grammar_entry insert end $insert_grammar
	$p.grammar_entry configure -state disabled

	# retrieve the results and insert it into the text area
	#
	set results "RESULT\[NaN\]"
	set nlresults "NL_INTERPRETATION\[NaN\]\[NaN\]"

	$p.results_entry delete 0 end
	$p.nlresults_entry delete 0 end

	# get the results in the current log entry if any exists
	#
	set num_results -1
	set key "NUM_RESULTS"
	catch { set num_results $arraylist($arrayindex,$key) }

	# search for results and nlresults in the log file if it exists
	#
	if {$num_results != -1} {

	    # iterate through the number of results for the current log
	    #
	    for {set i 0} {$i < $num_results} {incr i} {
		
		# set up the key for the current result
		#
		regsub NaN $results $i key
		
		# insert the current result in the test area
		#
		set insert_result ""
		catch { set insert_result $arraylist($arrayindex,$key) }
		set insert_result [string trim $insert_result]
		$p.results_entry insert end "$key: $insert_result"
		
		# get the number of interpretations corresponding to the 
		# current entry in the log file, i.e., number of nl results
		#
		set key "NUM_NL_INTERPRETATIONS\[NaN\]"
		regsub NaN $key $i key
		catch { set num_nlresults $arraylist($arrayindex,$key) }

		# iterate throught the interpretations for the current result
		#
		for {set j 0} {$j < $num_nlresults} {incr j} {
		    
		    # set up the key for the current interpretation
		    #
		    regsub NaN $nlresults $i key
		    regsub NaN $key $j key
		    
		    # insert the current interpretation in the text area
		    #
		    set insert_nlresult ""
		    catch { set insert_nlresult $arraylist($arrayindex,$key) }
		    set insert_nlresult [string trim $insert_nlresult]
		    $p.nlresults_entry insert end "$key: $insert_nlresult"
		}
	    }
	}

	# set the current utterance parameters 
	#
	catch { set cspk_mean $arraylist($arrayindex,MEANING) }
	catch { set cspk_noise $arraylist($arrayindex,NOISE) }
	catch { set cutt_trans $arraylist($arrayindex,TRANSCRIPTION) }
	catch { set cspk_id $arraylist($arrayindex,ID) }
	catch { set cspk_type $arraylist($arrayindex,TYPE) }
	catch { set cspk_sex $arraylist($arrayindex,GENDER) }
	catch { set cspk_age $arraylist($arrayindex,AGE) }

	# insert the transcription in the text area
	#
	if {$lock == 1} {
	    $p.curr_trans configure -state normal
	    $p.curr_trans delete 1.0 end
	    $p.curr_trans insert end $cutt_trans
	    $p.curr_trans configure -state disabled
	} else {
	    $p.curr_trans delete 1.0 end
	    $p.curr_trans insert end $cutt_trans
	}

	# insert the transcribers id
	#
	$p.id_entry delete 1.0 end
	$p.id_entry insert end $cspk_id
	
	# set the utterance params for previous
	#
	set prev_ind [expr $curr_ind - 1]

	if {$prev_ind >= 0} {

	    set putt_trans ""
	    
	    catch {
		
		# set the previous transcription 
		#
		set putt_trans $arraylist($prev_ind,TRANSCRIPTION)
	    }
	} else {
	    
	    set putt_trans ""
	}

	$p.prev_trans configure -state normal
	$p.prev_trans delete 1.0 end
	$p.prev_trans insert end $putt_trans
	$p.prev_trans configure -state disabled
	
	# set the utterance params for next
	#
	set next_ind [expr $curr_ind + 1]

	if {$next_ind <= [llength $audiolist]} {

	    set nutt_trans ""

	    catch {

		# get the array index of the previous transcription
		#
		set nutt_trans $arraylist($next_ind,TRANSCRIPTION)
	    }
	} else {
	    set nutt_trans ""
	}

	$p.next_trans configure -state normal
	$p.next_trans delete 1.0 end
	$p.next_trans insert end $nutt_trans
	$p.next_trans configure -state disabled

	# check to see if the specified audio file exists
	#
	if {[file exists $audio_file] != 1} {

	    warning_proc "The specified utterance file does not exist."

	    # make the existing plots disappear
	    #
	    $p.energy0_plot itemconfigure $energy_plot_0 -amp_scale 1
	    $p.chan0_plot itemconfigure $sig_plot_0 -gain_factor 1
	    $p.spec0_plot itemconfigure $spec_plot_0 -framelen 1

	    # activate the listbox
	    #
	    catch { $p.list selection clear 0 end }
	    catch { $p.list see $curr_ind }
	    catch { $p.list selection set $curr_ind }

	    update
	    return
	}
	
	# check to see if the current meaning and noise values are valid
	#
	if {$cspk_mean == ""} {
	    set mean_flag 0
	} else {
	    set mean_flag 1
	}
	
	if {$cspk_noise == ""} {
	    set noise_flag 0
	} else {
	    set noise_flag 1
	}

	# set the maximum time for the audio file
	#
	set maximum_time [format "%.6f" [eval expr [file size $audio_file] \
		/ $sample_num_bytes / $sample_frequency / $num_channels]]

	# set the default start and end times
	#
	set cutt_start 0
	set cutt_end $maximum_time
	set window_start 0
	set window_end $maximum_time
	set window_duration $maximum_time
	set bracket_start 0
	set bracket_end $maximum_time

	# check if call flag is set
	#
	if {$callflag == 0 && $autoplay == 1}  {
	    
	    # play the current utterance
	    #
	    play_utt_proc 0
	}
	
    } elseif {$curr_ind < 0} {
	
	# clear all the elements
	#
	set cutt_trans ""
	$p.curr_trans delete 1.0 end
	$p.curr_trans insert end "$cutt_trans"
    }
    
    # activate the listbox
    #
    catch { $p.list selection clear 0 end }
    catch { $p.list see $curr_ind }
    catch { $p.list selection set $curr_ind }
	
    # if the transcriptions are locked relock the current
    # transcription display
    #
    if {$lock == 1} {
	
	$p.curr_trans configure -state disabled
    }
	
    # set the current window display
    #
    curr_window_proc
}

# procedure: next_utt_proc
#
# arguments: none
#
# return: none
#
# procedure to set the focus on next utterance
#
proc next_utt_proc {} {
    
    # declare globals
    #
    global curr_ind

    global word_mode
    global p

    global lock

    global p
    global audiolist

    global next_flag
    global prev_flag

    global state

    # dont go to the next utterance if busy
    #
    if {$next_flag == 1 || $prev_flag == 1} {
	return
    }

    # set the flag to indicate transition to the next utterance
    #
    set next_flag 1

    # check current utterance first
    #
    if {[verify_utt_proc] == 0} {

	# reset the next flag
	#
	set next_flag 0

	return
    }

    # check if the lock is set 
    #
    if {$lock == 0} {

	# save the current utterance in the uttlist
	#
	bracket_marks_proc
    }

    # sort the utterances
    #
    reorder_proc

    # get the current selected element in the listbox
    #
    set next [expr $curr_ind + 1]

    if {$next >= [llength $audiolist]} {

	# activate the listbox
	#
	catch { $p.list selection clear 0 end }
	catch { $p.list see $curr_ind }
	catch { $p.list selection set $curr_ind }

	# reset the next flag
	#
	set next_flag 0

	return
    }
    
    # reset the combobox state variable
    #
    set state $p.noise

    # set the focus to next
    #
    set_current_proc $next

    # reset the next flag
    #
    set next_flag 0
}

# procedure: prev_utt_proc
#
# arguments: none
#
# return: none
#
# procedure to set the focus on previous utterance
#
proc prev_utt_proc {} {
    
    # declare globals
    #
    global curr_ind
    
    global word_mode
    global p

    global lock

    global p
    global audiolist

    global arraymin
    global arraylist

    global next_flag
    global prev_flag

    global state


    # dont go to the previous utterance if busy
    #
    if {$next_flag == 1 || $prev_flag == 1} {
	return
    }

    # set the flag to indicate transition to the previous utterance
    #
    set prev_flag 1

    # check current utterance first
    #
    if {[verify_utt_proc] == 0} {

	# reset the previous flag
	#
	set prev_flag 0

	return
    }

    # check if the lock is set 
    #
    if {$lock == 0} {

	# save the current utterance in the uttlist
	#
	bracket_marks_proc
    }

    # sort the utterances
    #
    reorder_proc

    # get the current selected element in the listbox
    #
    set prev [expr $curr_ind - 1]

    if {$prev < 0} {

	# activate the listbox
	#
	catch { $p.list selection clear 0 end }
	catch { $p.list see $prev }
	catch { $p.list selection set $prev }

	# reset the previous flag
	#
	set prev_flag 0

	return
    }
    
    # reset the combobox state variable
    #
    set state $p.noise

    # set the focus to next
    #
    set_current_proc $prev

    # reset the previous flag
    #
    set prev_flag 0

}

# procedure: split_utt_proc
#
# arguments: none
#
# return: none
#
# procedure to split the current utterance
#
proc split_utt_proc {} {

    # declare globals
    #
    global cutt_start
    global cutt_end
    global cutt_trans
    global cutt_id
    global cspk_name
    global cspk_sex
    global cspk_age

    global conv_id
    global curr_ind
    global current_channel

    global chlabel

    global window_start
    global window_end

    global zoom_min

    global p
    global lock
    
    # if the transcriptions and / or segments are locked return
    #
    if {$lock == 1} {
	return
    }

    # set the new time limits
    #
    set in_pt [eval expr 0.5 * ($cutt_end + $cutt_start)]

    # create a new node to insert
    #
    set utter {}
    set clabel $chlabel
    
    if {[llength $chan0_uttlist] > 0} {
	set current [lindex $chan0_uttlist $curr_ind]
	foreach {cid uid spkr sex age start end trans} $current {
	    lappend utter $cid $uid $cspk_name $cspk_sex $cspk_age $in_pt \
		    $cutt_end $cutt_trans
	    set current [lreplace $current 6 6 $in_pt]
	    set chan0_uttlist [lreplace $chan0_uttlist $curr_ind \
		    $curr_ind $current]
	}
    } else {
	set cid [format "%s" $conv_id]
	set uid [format "%d" 1]
	lappend utter $cid $uid "" "" "" $window_start \
		$window_end ""
    }
	
    # insert this in the right place
    #
    lappend chan0_uttlist $utter
	
    # reset the utterance ids and update the listbox
    #
    reorder_proc
    
    # set the current utterance to new node
    #
    set_current_proc $curr_ind
}

# procedure: set_utt_proc
#
# arguments: none
#
# return: none
#
# procedure to change the current utterance parameters
#
proc set_utt_proc {} {

    # declare globals
    #
    global cspk_name
    global cspk_sex
    global cspk_age
    global cspk_type
    global cspk_mean
    global cspk_noise
    global cspk_id
    global mean_flag 
    global noise_flag

    global arrayindex
    global arraylist

    global p
    global loadflag
    global lock

    # return if nothing is loaded
    #
    if {$loadflag == 0} {
	return
    }
    
    # return if the lock is set
    #
    if {$lock == 1} {
	return
    }

    # set the current utterance transcription
    #
    catch { set cutt_trans [$p.curr_trans get 1.0 end] }
    set cutt_trans [string trim $cutt_trans]

    # set the transcribers id
    #
    catch { set cspk_id [$p.id_entry get 1.0 end] }
    set cspk_id [string trim $cspk_id]

    # set current utterance parameters to the ones in the window
    #
    catch { set arraylist($arrayindex,TRANSCRIPTION) $cutt_trans }
    catch { set arraylist($arrayindex,GENDER) $cspk_sex }
    catch { set arraylist($arrayindex,AGE) $cspk_age }
    catch { set arraylist($arrayindex,TYPE) $cspk_type }
    catch { set arraylist($arrayindex,MEANING) $cspk_mean }
    catch { set arraylist($arrayindex,NOISE) $cspk_noise }
    catch { set arraylist($arrayindex,ID) $cspk_id }

    # exit gracefully
    #
    return 0
}

# procedure: remove_utt_proc
#
# arguments: none
#
# return: none
#
# procedure to remove current utterance from the list
#
proc remove_utt_proc {} {

    # declare globals
    #
    global curr_ind
    global current_channel

    global chan0_uttlist

    global p
    global lock
    
    # if the transcriptions and / or segments are locked return
    #
    if {$lock == 1} {
	return
    }

    # get the list of utterances selected to be deleted
    #
    catch { set select_list [$p.list curselection] }
    catch { set first [lindex $select_list 0] }
    catch { set last [lindex $select_list end] }
	
    # delete the current utterance
    #
    if {$curr_ind < [llength $chan0_uttlist]} {
	set chan0_uttlist [lreplace $chan0_uttlist $first $last]
    }
	
    # set the current utterance focus
    # 
    if {$curr_ind == [llength $chan0_uttlist]} {
    } else {
	set curr_ind $first
    }
    set num_utt [llength $chan0_uttlist]

    # reset the utterance ids and update the listbox
    #
    reorder_proc

    # change focus to new utterance
    #    
    if {$curr_ind >= $num_utt} {
	set curr_ind [eval expr $num_utt - 1]
    }
    set_current_proc $curr_ind
}

# procedure: insert_utt_proc
#
# arguments: none
#
# return: none
#
# procedure to insert a new utterance in the current position
#
proc insert_utt_proc {} {

    global lock
    global silence_tag

    global loadflag
    global p
    
    # if the transcriptions and / or segments are locked return
    #
    if {($lock == 1) || ($loadflag == 0)} {
	return
    }

    # insert a copy of the selected utterance
    #
    split_utt_proc

    # clear its transcription
    #
    # set the current utterance transcription
    #
    $p.curr_trans delete 1.0 end
    $p.curr_trans insert end $silence_tag
    set_utt_proc
}

# procedure: clear_trans_proc
#
# arguments: none
#
# return: none
#
# procedure to clear current transcription
#
proc clear_trans_proc {} {

    # declare globals
    #
    global cutt_start
    global cutt_end
    global cutt_trans
    global cutt_id

    global p
    global lock

    # if the transcriptions are locked return
    #
    if {$lock == 1} {
	return
    }

    # set the current utterance transcription
    #
    $p.curr_trans delete 1.0 end
    set cutt_trans [$p.curr_trans get 1.0 end]
    bracket_marks_proc
}

# procedure: find_current_proc
#
# arguments:
#  time_in: utterance time
#
# return: none
#
# procedure to find utterance index spanning the specified time value
#
proc find_current_proc {time_in} {

    # declare globals
    #
    global chan0_uttlist

    global current_channel

    global maximum_time

    # set the current utterance list
    #
    set uttlist $chan0_uttlist

    # check if the time entered is valid
    #
    if {$time_in <= 0.0} {
	return 0
    } elseif {$time_in >= $maximum_time} {
	return [llength $uttlist]
    }
    
    # loop through all utterances to find the utterance that spans this
    #
    set i 0
    foreach item $uttlist {
	foreach {cid uid spkr sex age start end trans} $item {

	    # test time span
	    #
	    if {$time_in <= $end} {
		return $i
	    }
	    
	    # increment count
	    #
	    incr i
	}
    }

    # if reached here time is after the last utterance
    #
    return [llength $uttlist]
}

# procedure: delete_utt_proc
#
# arguments: none
#
# return: none
#
# delete utterance method --- change transcription to silence
#
proc delete_utt_proc {} {

    # declare globals
    #
    global chan0_uttlist
    global current_channel
    global curr_ind

    global cutt_start
    global cutt_end
    global cutt_trans
    global cutt_id
    global putt_trans
    global nutt_trans

    global silence_tag
    global loadflag

    global p

    global lock

    # if the transcriptions and / or segments are locked return
    #
    if {($lock == 1) || ($loadflag == 0)} {
	return
    }

    # list of utterances to merge
    #
    set select_list {}

    # the new transcription default
    #
    set new_trans $silence_tag
    
    # set the current utterance transcription
    #
    $p.curr_trans delete 1.0 end
    $p.curr_trans insert 1.0 $silence_tag
    set cutt_trans [$p.curr_trans get 1.0 end]

    # add current utterance
    #
    lappend select_list $curr_ind
    
    # check for following silence, if any
    #
    set other_text [string trim $nutt_trans]
    if {$other_text == $silence_tag} {
	lappend select_list [expr $curr_ind + 1]
    } elseif {$other_text != ""} {
	lappend select_list [expr $curr_ind + 1]
	set new_trans $other_text
    }
    
    # check for preceding silence, if any
    #
    set other_text [string trim $putt_trans]
    if {$other_text == $silence_tag} {
	set select_list [concat [expr $curr_ind - 1] $select_list]
    } elseif {($new_trans == $silence_tag) && ($other_text != "")} {
	set select_list [concat [expr $curr_ind - 1] $select_list]
	set new_trans $other_text
    }
    
    # now merge these utterances
    #
    set num_list [eval expr [llength $select_list] - 1]
    if {$num_list > 0} {
	
	# set the channel-wise params
	#
	set uttlist $chan0_uttlist
	
	set i 0
	foreach num $select_list {
	    set utter [lindex $uttlist $num]
	    
	    # create a new utterance with the corresponding params
	    #
	    foreach {cid uid spkr sex age start end trans} $utter {
		
		# set the end-points
		#
		if {$i == 0} {
		    set new_start $start
		    set new_cid $cid
		    set new_uid $uid
		    set first $num
		}
		if {$i == $num_list} {
		    set new_end $end
		    set last $num
		}
	    }
	    incr i
	}

	# set the speaker related information of the merged utterance
	#
	set fspkr_list [lindex $uttlist $first]
	set lspkr_list [lindex $uttlist $last]
	set fspeaker [lindex $fspkr_list 2]
	set lspeaker [lindex $lspkr_list 2]
	
	if {$fspeaker == $lspeaker} {
	    set new_spkr [lindex $fspkr_list 2]
	    set new_sex [lindex $fspkr_list 3]
	    set new_age [lindex $fspkr_list 4]
	} else {
	    set new_spkr [lindex $lspkr_list 2] 
	    set new_sex [lindex $lspkr_list 3]
	    set new_age [lindex $lspkr_list 4]
	}

	# create a new utterance with these parameters
	#
	set new_utter {}
	lappend new_utter $new_cid $new_uid $new_spkr $new_sex \
		$new_age $new_start $new_end $new_trans
	set uttlist [lreplace $uttlist $first $last]
	lappend uttlist $new_utter
	
	# now reassign the utterance list
	#
	set chan0_uttlist $uttlist
		
	# reorder the list and update the listbox
	#
	reorder_proc
	
	# set the current utterance focus to the new one
	#
	set_current_proc $first
    }
}

#
# end of file
