#! /usr/local/bin/wish -f

# file: $ISIP_TRANSCRIBER/gui/src/save.tcl
#
# procedures to write the contents of the utterance and word alignment
# list to the transcription and word files by manual and auto saving methods
#

# list of procedures:
#
#  save_proc {}
#  save_prompt_proc {}
#  file_switch_proc {}
#  write_proc {filename}
#  autosave_proc {}
#  autosave_filenames_proc {}
#

# procedure: save_proc
#
# arguments: none
#
# return: none
#
# procedure to write data to file from the utterance lists
#
proc save_proc {} {

    # declare globals
    #
    global transfile
    global chan0_uttlist

    global p

    global lock

    global loadflag

    # return if nothing is loaded
    #
    if {$loadflag == 0} {
	return
    }

    # cannot save data if a lock is set
    #
    if {$lock == 1} {
	warning_proc "Save is disabled due to the Transcription lock. \
		Please disable the lock to save data."
	return
    }

    # save the utterance of the current focus 
    #
    set_utt_proc

    # write the data as belonging to channel 0
    #
    write_proc $transfile
}

# procedure: save_prompt_proc
#
# arguments: none
#
# return: none
#
# procedure to prompt the user to save the current data before switching files 
#
proc save_prompt_proc {} {

    # create globals
    #
    global trans0_file
    global chan0_uttlist

    global s
    set s ".save"

    global transcriber_title
    global bg_col

    # don't recreate the window, try to reuse existing one
    #
    if { [winfo exists $s] == 1 } {
        wm deiconify $s
        raise $s
	
    } else {
	
        # create a new window
        #
        toplevel $s
        wm title $s "$transcriber_title Save Prompt"
	wm geometry $s "+300+200"
	wm focusmodel $s active	

	# make a bottom frame
	#
	frame $s.foot -background $bg_col
	pack $s.foot -side bottom -fill x -pady 2m

	# create save/dismiss buttons
	#
        button $s.save -width 6 -default active -text Save \
		-background $bg_col -command {
	    destroy $s
	    save_proc
	    file_switch_proc

	    # update the config window only if it is open
	    #
	    if { $config_flag == 1 } {
		if {[winfo exists $cfg] == 1 } {
		    destroy $cfg 
		}
	    }   
	}
	button $s.diss -width 6 -text Dismiss -background $bg_col \
		-command {
	    destroy $s
	    file_switch_proc

	    # update the config window only if it is open
	    #
	    if { $config_flag == 1 } {
		if {[winfo exists $cfg] == 1 } {
		    destroy $cfg 
		}
	    }
	} 
        pack $s.save $s.diss -side left  -fill x \
		-padx 1m -pady 1m -expand true -in $s.foot

	# make a message frame
	#
	frame $s.top -background $bg_col
	pack $s.top -side top -fill x -padx 1m -pady 1m

	message $s.msg -justify center -width 3i -background $bg_col \
		-text "Do you want to save data before you change files?"
	pack $s.msg -side top -fill both -in $s.top
    }

    # define key bindings
    #
    bind $s <Return> {
	destroy $s
	save_proc
	file_switch_proc

	# update the config window only if it is open
	#
	if { $config_flag == 1 } {
	    if {[winfo exists $cfg] == 1 } {
		destroy $cfg 
	    }
	}
    }
}

# procedure: file_switch_proc
#
# arguments: none
#
# return: none 
#
# procedure to set up and load new transcription, word alignment, log and 
# audio files and also set a stop bookmark if required
#
proc file_switch_proc {} {

    # define globals
    #
    global bm_start
    global ind
    
    global keys
    global keycount
    global arraylist

    global state p

    global bookmark_flag

    # reset the state for the listbox
    #
    set state $p.noise

    # if there is no stop bookmark put one
    #
    if {$bookmark_flag == 1} {
	bookmark_entry_proc
    }
    
    # set up the new transcription, word alignment, log and audio files
    #
    current_list_proc $ind
    
    # load the new transcription, word alignment, log and audio files
    #
    load_proc
}

# procedure: write_proc
#
# arguments: 
#  text file: the transcription file to write to
#  uttlist: the transcrip[tion utterlist
#
# return: none 
#
# procedure to open and write a file and using data in the utterance list
#
proc write_proc {filename} {
    
    # declare globals
    #
    global arraylist
    global keycount
    global num_utter
    global keys
    global file_header
    global file_tail
    global log_separator
    global start_tag
    global end_tag
    
    global saveflag
    global std_error

    # make sure file is writable
    #
    if {[file writable $filename] != 1} {
        warning_proc "Sorry! You do not have write permissions for the file \
		$filename."

        return 1
    }

    # open the file
    #
    set fileptr [open $filename w]

    # set the save flag
    #
    set saveflag 1

    # print the transfile header
    #
    foreach item $file_header {
	puts $fileptr $item
    }

    # print the key value pairs to the transcription file
    #
    set index1 0

    while {$index1 < $num_utter} {

	# iterate through the array containing the value terms
	#
	puts $fileptr $start_tag
	set index2 0

	# check to see if the number of keys are accessible
	#
	set error $std_error
	catch {
	    set error $keycount($index1)
	}

	# display a warning message indicating that an erro occured
	# while writing to the log file and then exit
	#
	if {$error == $std_error} {
	    
	    # display the error message
	    #
	    warning_proc "An error occured while writing to file: \
		    $filename. The data currently being transcribed \
		    will not be saved under the circumstances."

	    # close file
	    #
	    close $fileptr
	    
	    return 1
	}
	
	while {$index2 < $keycount($index1)} {

	    # iterate throught the array containing the key terms
	    #
	    set error $std_error
	    catch { 
		set mykey $keys($index1,$index2) 
		set error 0
	    }

	    # display a warning message indicating that an erro occured
	    # while writing to the log file and then exit
	    #
	    if {$error == $std_error} {

		# display the error message
		#
		warning_proc "An error occured while writing to file: \
			$filename. The data currently being transcribed \
			will not be saved under the circumstances."

		# close file
		#
		close $fileptr

		return 1
	    }

	    set error $std_error
	    catch { 
		set myvalue $arraylist($index1,$mykey) 
		set error 0
	    }

	    if {$error == $std_error} {

		# display the error message
		#
		warning_proc "An error occured while writing to file: \
			$filename. The data currently being transcribed \
			will not be saved under the circumstances."

		# close file
		#
		close $fileptr

		return 1
	    }

	    # display a warning message indicating that an erro occured
	    # while writing to the log file and then exit
	    #

	    set outstr [format "        %-25s" $mykey]
	    append outstr [format "%8s" " = "]
	    append outstr $myvalue
	    puts $fileptr "$outstr"
	    incr index2 1
	}

	puts $fileptr $end_tag
	incr index1 1

	# include the utterance seperator in the transcription file
	#
	if {$index1 != $num_utter} {
	    puts $fileptr $log_separator
	}
    }

    # print the transfile tail
    #
    foreach item $file_tail {
	puts $fileptr $item
    }
    
    # close file
    #
    close $fileptr

    # reset the save flag
    #
    set saveflag 0

    # exit gracefully
    #
    return 0
}

# procedure: autosave_proc
#
# arguments: none
#
# return: none
#
# procedure for autosaving data
#
proc autosave_proc {} {

    # declare globals
    #
    global auto_t0_file
    global auto_w0_file
    global chan0_uttlist
    global chan0_wordlist

    global autosave_time
    global time_updated

    global loadflag

    global lock

    # cannot save data if a lock is set
    #
    if {$lock == 1} {
	return
    }

    # cannot save data if no data is loaded
    #
    if {$loadflag == 0} {
	return
    }

    # if the interval is sufficiently large since the last autosave
    #
    if {[expr [clock seconds] - $time_updated] > $autosave_time} {
	
	# update the last update time
	#
	set time_updated [clock seconds]
	
	# save the transcription and word alignment data
	#
	set fileptr [open $auto_t0_file w]
	close $fileptr     
	set fileptr [open $auto_w0_file w]
	close $fileptr
	    
	# write the data as belonging to channel 0
	#
	write_proc $auto_t0_file $chan0_uttlist
	write_word_proc $auto_w0_file $chan0_wordlist
    }
}

# procedure: autosave_filenames_proc
#
# arguments: none
#
# return: none
#
# procedure to set the autosave filenames
#
proc autosave_filenames_proc {} {

    # declare globals
    #
    global autosave_dir
    global trans0_file

    global auto_t0_file
    global auto_w0_file

    global fname

    global num_channels

    # set the autosave directory
    #
    set autosave_dir [file dirname $trans0_file]

    # update the names of the autosave files
    #
    set auto_prefix "${autosave_dir}/${fname}"
    set auto_t0_file "${auto_prefix}_trans_auto.text"
    set auto_w0_file "${auto_prefix}_words_auto.text"
}

#
# end of file
