#! /usr/local/bin/wish -f

# file: $ISIP_TRANSCRIBER/gui/src/load.tcl
#
# procedures for initially setting up the transcriber by reading in the 
# transcription and word alignment files
#

# list of procedures:
#
#  find_trans_proc {index}
#  load_proc {}
#  load_lists_proc {}
#  current_list_proc {index}
#  read_comment_proc {}
#  parse_curr_proc {}
#  load_globals_proc {}
#  read_trans_proc {}
#  read_file_proc {text_area filename}
#

# procedure: find_tarns_proc
#
# arguments: audio index
#
# return: array index
#
# procedure that returns the corresponding array index for the current 
# audio file to match it with its transcriptions
#
proc find_trans_proc {index} {

    # declare globals
    #
    global p

    global audiolist
    global arraylist

    global num_utter

    # get the current audio file for the audiolist
    #
    set str0 [lindex $audiolist $index]

    set ind0 [string last "/" $str0]
    set audiostr0 [string range $str0 [expr $ind0 + 1] end]
    regsub .raw $audiostr0 .wav audiostr

    # get the corresponding audio file index for the arraylist
    #
    set index1 0

    while {$index1 < $num_utter} {

	# parse the utterance file name
	#
	set arraystr0 ""

	catch { set arraystr0 $arraylist($index1,UTTERANCE_FILENAME) }

	if {$arraystr0 != ""} {
	
	    set ind1 [string last "/" $arraystr0]
	    set arraystr [string range $arraystr0 [expr $ind1 + 1] end]

	    # if the file name exists return it
	    #
	    if {$arraystr == $audiostr} {

		return $index1
	    }
	}
	incr index1 1
    }
}

# procedure: load_proc
#
# arguments: none
#
# return: none
#
# procedure to read data from file and set up utterance lists 
#
proc load_proc {} {

    # declare globals
    #
    global current_channel
    global chan0_uttlist 
    global audiolist
    global audio_list_file

    global p 

    global bookmark_flag

    global arraylist
    global num_utter
    global keys
    global keycount

    # destroy all existing arrays
    #
    catch { unset keys }
    catch { unset arraylist }
    catch { unset keycount }
    catch { unset num_utter }

    # read the new transcription file
    #
    set status [read_trans_proc]

    # exit if there was an error in reading the transcription file
    #
    if {$status == 1} {	return 1 }

    # read in the audio file list
    #
    # make sure the audiofile exists and is readable
    #
    if {[file readable $audio_list_file] != 1} {
	warning_proc "Cannot open $audio_list_file. Please make sure the file \
		exists and has read permissions."

	return 1

    } else {
	
	# open the audio file list
	#
	set audioptr [open $audio_list_file r]
    }

    # inform the user that the hash table is being loaded
    #
    warning_proc "Please wait while the lexicon is being read. \
	    This may take a while depending on the size of the lexicon."

    # clear the audio file list
    #
    set audiolist {}
    
    # clear the previous contents of the audio list
    #
    set tmplist {}

    # create a temporary audio list
    #
    while {![eof $audioptr]} {
	
	# read in the file line by line
	#
	set value [gets $audioptr]
	set audiostr [expand_env_proc $value]

	# ignore all blank lines
	#
	if {[string match "" $audiostr] == 0} {
			
	    # add the parsed line to the audiolist
	    #
	    lappend tmplist $audiostr
	}
    }
    
    # initialize the iterator
    #
    set index 0

    # read each line of the transcription file
    #
    while {$index < $num_utter} {

	# parse the utterance file name
	#
	set arraystr ""
	catch { set arraystr $arraylist($index,UTTERANCE_FILENAME) }

	# search for the corresponding file in the audiolist
	#
	if {$arraystr != ""} {

	    # set up the array file
	    #
	    set ind [string last "/" $arraystr]
	    set arraystr [string range $arraystr [expr $ind + 1] end]

	    foreach item $tmplist {

		# set up a temporary variable
		#
		set tmpstr $item

		# set up the list item 
		#
		set ind [string last "/" $tmpstr]
		set tmpstr [string range $tmpstr [expr $ind + 1] end]
		regsub .raw $tmpstr .wav tmpstr

		# add the file to the audiolist if there is a match
		#
		if {$tmpstr == $arraystr } {

		    # add the parsed line to the audiolist
		    #
		    lappend audiolist $item
		    break
		}
	    }
	} else {

	    # append a no data string to the audiolist
	    #
	    lappend audiolist "no data [expr $index + 1]"
	}

	# increment the array index
	#
	incr index 1
    }

    # close the file
    #
    close $audioptr

    # reorder the lists in order of start times and
    # set up the current channel as default display
    #
    set status 1
    catch { 
	set_channel_proc $current_channel
	set status 0
    }

    # exit if there was an error in executing the above script
    #
    if {$status == 1} {	return 1 }
    
    # update the display
    #
    set status 1
    catch { 
	update_window
	set status 0
    }

    # exit if there was an error in executing the above script
    #
    if {$status == 1} {	return 1 }

    # if there is no start bookmark put one
    #
    if {$bookmark_flag == 0} {

	set status [bookmark_entry_proc]
	
	# exit if there was an error writing the bookmarks to file
	#
	if {$status == 1} { return }
    }

    # build the hash table
    #
    build_hash_proc

    # exit gracefully
    #
    return 0
}

# procedure: load_lists_proc 
#
# arguments: none
#
# return: none
#
# procedure to load list elements
#
proc load_lists_proc {} {

    # declare globals
    #
    global callfile
    global session_id

    global callfile_list

    # clear the previous call file list
    #
    set callfile_list {}

    # make sure the call file exists and is readable
    #
    if {[file readable $callfile] != 1} {
	warning_proc "Cannot open $callfile. Please make sure the file \
		exists and has read permissions."
	
	return 1
	
    } else {
	
	# open the call file
	#
	set callptr [open $callfile r]
    }
    
    while {![eof $callptr]} {
	
	# read file line by line
	#
	set callstr [gets $callptr]
	set callstr [string trim $callstr]
	if {[string match "" $callstr] == 0} {
	    set callstr [expand_env_proc $callstr]
	    lappend callfile_list $callstr
	}
    }
    
    # close the file
    #
    close $callptr

    # exit gracufully
    #
    return 0
}

# procedure: current_list_proc
#
# arguments:
#  index: utterance index
#
# return: none
#
# procedure to set the current focus
#
proc current_list_proc {index} {

    # declare globals
    #
    global f

    global callfile_list
    global uttlist
    global audiolist

    global list_file
    global logfile
    global transfile
    global callfile
    global audio_file

    global config_file
    global audio_list_file

    global curr_call_index
    global std_error

    # set the index variable only if necessary
    #
    if {($index >= 0) && ($index < [llength $callfile_list])} {

	# set the current focus
	#
	set curr_call_index $index

    } else {
	
	# set default focus
	#
	set curr_call_index 0
    }

    # retrieve the current call file path
    #
    set callfile [lindex $callfile_list $curr_call_index]
    set callfile [expand_env_proc $callfile]

    # sets the transcription, audio, log and word boundry files
    #
    set list_file [lindex $callfile_list $curr_call_index]

    # make sure the listfile exists and is readable
    #
    if {[file readable $list_file] != 1} {
	warning_proc "Cannot open $list_file. Please make sure the file \
		exists and has read permissions."

	return 1

    } else {
	
	# open the call file
	#
	set listptr [open $list_file r]
    }

    # extract the logfile, transfile, and audiolist parameters from listfile
    #
    while {[gets $listptr line] >= 0} {
	
	# make sure this is not a comment
	#
	set line [string trim $line]
	if {($line != "") && ([string index $line 0] != "#")} {

	    # split parameters and values
	    #
	    scan $line "%s = %s" parameter value

	    # check to see if the file is in the valid format
	    #
	    set error $std_error
	    catch { set error $parameter }
	    if {$error == $std_error} {

		# display a warning message
		#
		warning_proc "The contents of the file: $list_file \
			is  not in the required format. Please make \
			sure that the file is in the required KEY = \
			VALUE pair format."
		
		# close file and exit
		# 
		close $listptr
		
		return 1
	    }

	    set error $std_error
	    catch { set error $value }
	    if {$error == $std_error} {

		# display a warning message
		#
		warning_proc "The contents of the file: $list_file \
			is  not in the required format. please make \
			sure that the file is in the required key = \
			value pair format for each line of the file."

		# close file and exit
		# 
		close $listptr
		
		return 1
	    }

	    set value [string trim $value \"]

	    # set the parameters specified to the corresponding value
	    #
	    if {$parameter == "audio_list"} {
		set audio_list_file [expand_env_proc $value]
	    } elseif {$parameter == "trans_file"} {
		set transfile [expand_env_proc $value]
	    } elseif {$parameter == "comment_file"} {
                set logfile [expand_env_proc $value]
	    }
	}
    }

    # close the file
    #
    close $listptr

    # exit gracufully
    #
    return 0
}    

# procedure: set_comment_proc
#
# arguments: none
#
# return: none
#
# procedure that loads the comment file parameters for the last stopped utter
#
proc set_comment_proc {} {

    # declare globals
    #
    global last_utter
    global audiolist

    set audiostr $last_utter
    
    # get the corresponding audio file index
    #
    set uttindex 0
    set itr 0

    # get the corresponding audiofile index
    #
    foreach item $audiolist {
	set ind [string last "/" $item]
	set mystr [string range $item [expr $ind + 1] end]
	
	if {[string match $mystr $audiostr] == 1} {
	    set uttindex $itr
	}

	# increment the iterator
	#
	incr itr
    }

    # load the utterance
    #
    set_utt_proc
    set_current_proc $uttindex
}

# procedure: read_comment_proc
#
# arguments: none
#
# return: none
#
# procedure that reads the comment file and finds the last utterance at
# which the transcriber on
#
proc read_comment_proc {} {
    
    # declare globals
    #
    global logfile
    global last_utter

    set final_comment ""
    set parameter ""
    set value ""

    # make sure the comment exists and is readable
    #
    if {[file readable $logfile] != 1} {
	warning_proc "cannot open $logfile. please make sure the file \
		exists and has read permissions."
	return
    } else {
	
	# open the comment file
	#
	set commentptr [open $logfile r]
    }

    # read each line of the comment file
    #
    while {![eof $commentptr]} {

	# read file line by line
	#
	set line [gets $commentptr]

	# parse the line that has been read from the file
	#
	if {($line != "") && ([string index $line 0] != "<")} {
	    regexp {([a-za-z]+:) (.*)} $line dummy parameter value

	    set parameter [string trim $parameter]
	    set value [string trim $value]

	    if {$parameter == "utterance:"} {
		set final_comment $value
	    }
	}
    }

    # close the file
    #
    close $commentptr

    # parse the final comment
    #
    set ind [string last "/" $final_comment]
    set last_utter [string range $final_comment [expr $ind + 1] end]
}

# procedure: load_globals_proc
#
# arguments: none
#
# return: none
#
# procedure to load global values from the configuration file
#
proc load_globals_proc {} {

    # declare globals
    #
    global config_file

    global audio_server
    global audio_play_device
    global autosave_time
    global pause_time
    global transcriber_id
    
    global conv_id

    global audio_file
    global lexicon_file
    global sample_frequency
    global sample_num_bytes
    global num_channels
    global current_channel
    global play_mode

    global auto_t0_file
    global auto_w0_file

    global bm_start
    global std_error

    global callfile
    global session_id

    # make sure the config file exists and is readable
    #
    if {[file readable $config_file] != 1} {
	warning_proc "cannot open the config file $config_file. please \
		make sure the file exists and has read permissions."

	return 1

    } else {

	# open the configuration file
	#
	set cfgptr [open $config_file r]
    }

    # read parameters line by line
    #
    while {[gets $cfgptr line] >= 0} {
	
	# make sure this is not a comment
	#
	set line [string trim $line]
	if {($line != "") && ([string index $line 0] != "#")} {

	    # split parameters and values
	    #
	    scan $line "%s = %s" parameter value

	    # check to see if the file is in the valid format
	    #
	    set error $std_error
	    catch { set error $parameter }
	    if {$error == $std_error} {

		# display a warning message
		#
		warning_proc "The contents of the file: $config_file \
			is  not in the required format. Please make \
			sure that the file is in the required KEY = \
			VALUE pair format."
		
		# close file and exit
		# 
		close $cfgptr
		
		return 1
	    }

	    set error $std_error
	    catch { set error $value }
	    if {$error == $std_error} {

		# display a warning message
		#
		warning_proc "The contents of the file: $config_file \
			is  not in the required format. Please make \
			sure that the file is in the required KEY = \
			VALUE pair format."

		# close file and exit
		# 
		close $cfgptr
		
		return 1
	    }

	    set value [string trim $value \"]

	    # set the parameters specified to the corresponding value
	    #
	    if {$parameter == "speech_data_file"} {
		set audio_file [expand_env_proc $value]
	    } elseif {$parameter == "num_channels"} {
		set num_channels [expand_env_proc $value]
	    } elseif {$parameter == "sample_num_bytes"} {
		set sample_num_bytes [expand_env_proc $value]
	    } elseif {$parameter == "sample_frequency"} {
		set sample_frequency [expand_env_proc $value]
	    } elseif {$parameter == "start_channel"} {
		set current_channel [expand_env_proc $value]
	    } elseif {$parameter == "call_file"} {
		set callfile [expand_env_proc $value]
	    } elseif {$parameter == "session_id"} {
		set session_id [expand_env_proc $value]
	    } elseif {$parameter == "lexicon_file"} {
		set lexicon_file [expand_env_proc $value]
	    } elseif {$parameter == "default_conversation_id"} {
		set conv_id [expand_env_proc $value]
	    } elseif {$parameter == "audio_server"} {
		set audio_server [expand_env_proc $value]
	    } elseif {$parameter == "audio_play_device"} {
		set audio_play_device [expand_env_proc $value]
	    } elseif {$parameter == "transcriber_id"} {
		set transcriber_id [expand_env_proc $value]
	    } elseif {$parameter == "autosave_time"} {
		set autosave_time [expand_env_proc $value]  
	    } elseif {$parameter == "pause_time"} {
		set pause_time [expand_env_proc $value]
	    } else {
                set $parameter [string trim [expand_env_proc $value] \"]
	    }
	}
    }

    # close file
    # 
    close $cfgptr

    # set the play_mode
    #
    set play_mode 0

    # exit gracufully
    #
    return 0
}

# procedure: read_trans_proc
#
# arguments:
#  filename: file to read data from
#
# return: list containing utterances from the input file
#
# procedure to open and read a file and put the data in the utterance list
#
proc read_trans_proc {} {

    # declare globals
    #
    global silence_tag
    global start_tag
    global end_tag

    global transcriber_id

    global transfile
    global arraylist
    global keycount
    global num_utter
    global keys

    global file_header
    global file_tail
    global log_separator

    # declare local lists
    #
    set file_header {}
    set file_tail {}

    # reset the header and the number of utterences
    #
    set num_utter 0
    set header_flag 0

    # make sure the transcription file exists and is readable
    #
    if {[file readable $transfile] != 1} {
	warning_proc "Cannot open the transcription file $transfile. Please \
		make sure the file exists and has read permissions."

	return 1

    } else {
	
	# open the transcription file
	#
	set tranptr [open $transfile]
    }

    # read parameters line by line
    #
    while {[gets $tranptr line] >= 0} {
	
	# make sure this is not a comment
	#
	set line [string trim $line]
	
	if {[string match $start_tag $line] == 1} {

	    # initialize flag to indicate that file header has been read
	    #
	    set header_flag 1
	    
	    # read the next line from the trans file
	    #
	    gets $tranptr line
	    set line [string trim $line]
	    set index 0

	    while {[string match $end_tag $line] == 0} {

		# split parameters and values
		#
		set ind [string first "=" $line]

		# display a warning message if the LOG file is not in the
		# required format, i.e, it cannot be parsed
		#
		if {$ind == -1} {

		    # display a warning message
		    #
		    warning_proc "The contents of the file: $transfile \
			    is  not in the required format. Please make \
			    sure that the file is in the required KEY = \
			    VALUE pair format."
		    
		    # close file and exit
		    # 
		    close $tranptr
		    
		    return 1
		}

		# parse the LOG file in order to retrieve the parameters
		#
		set parameter [string range $line 0 [expr $ind - 1]]
		set value [string range $line [expr $ind + 1] end]

		set parameter [string trim $parameter]
		set value [string trim $value]

		# store key value pair into the array
		#
		catch { set keys($num_utter,$index) $parameter }
		catch { set arraylist($num_utter,$parameter) $value }

		# read the next line from the trans file
		#
		gets $tranptr line
		set line [string trim $line]
		
		# increment the index
		#
		incr index 1
	    }
	    
	    # reset the transcription file footer
	    #
	    set file_tail {}

	    # set the utterance file name if it does not exist
	    #
	    set exist 0
	    
	    catch { set exist $arraylist($num_utter,UTTERANCE_FILENAME) }
	    
	    if {$exist == 0} {
		catch { set arraylist($num_utter,UTTERANCE_FILENAME) "" }
		catch { set keys($num_utter,$index) UTTERANCE_FILENAME }
		incr index 1
	    }
	    
	    # set the transcription if it does not exist
	    #
	    set exist 0
	    
	    catch { set exist $arraylist($num_utter,TRANSCRIPTION) }
	    
	    if {$exist == 0} {
		catch { set arraylist($num_utter,TRANSCRIPTION) "" }
		catch { set keys($num_utter,$index) TRANSCRIPTION }
		incr index 1
	    }
	    
	    # set the meaning if it does not exist
	    #
	    set exist 0
	    
	    catch { set exist $arraylist($num_utter,MEANING) }
	    
	    if {$exist == 0} {
		catch { set arraylist($num_utter,MEANING) "" }
		catch { set keys($num_utter,$index) MEANING }
		incr index 1
	    }
	    
	    # set the speaker type if it does not exist
	    #
	    set exist 0
	    
	    catch { set exist $arraylist($num_utter,TYPE) }
	    
	    if {$exist == 0} {
		
		catch { set arraylist($num_utter,TYPE) "" }
		catch { set keys($num_utter,$index) TYPE }
		incr index 1
	    }
	    
	    # set the noise level if it does not exist
	    #
	    set exist 0
	    
	    catch { set exist $arraylist($num_utter,NOISE) }
	    
	    if {$exist == 0} {
		catch { set arraylist($num_utter,NOISE) "" }
		catch { set keys($num_utter,$index) NOISE }
		incr index 1
	    }
	    
	    # set the gender if it does not exist
	    #
	    set exist 0
	    
	    catch { set exist $arraylist($num_utter,GENDER) }
	    
	    if {$exist == 0} {
		catch { set arraylist($num_utter,GENDER) "" }
		catch { set keys($num_utter,$index) GENDER }
		incr index 1
	    }
	    
	    # set the age if it does not exist
	    #
	    set exist 0
	    
	    catch { set exist $arraylist($num_utter,AGE) }
	    
	    if {$exist == 0} {
		catch { set arraylist($num_utter,AGE) "" }
		catch { set keys($num_utter,$index) AGE }
		incr index 1
	    }
	    
	    # set the transcriber ID if it does not exist
	    #
	    set exist 0
	    
	    catch { set exist $arraylist($num_utter,ID) }
	    
	    if {$exist == 0} {
		catch { set arraylist($num_utter,ID) "" }
		catch { set keys($num_utter,$index) ID }
		incr index 1
	    }
	    
	    # set the utterance count for the current log file
	    #
	    catch { set keycount($num_utter) $index }
	    
	    # increment the log count
	    #
	    incr num_utter 1

	} else {
	    
	    if {$header_flag == 0} {

		# read the transcription file header
		#
		lappend file_header $line
		
	    } else {

		# read the transcription file footer
		#
		lappend file_tail $line
	    }
	}
    }

    # close the file
    #
    close $tranptr

    # exit gracufully
    #
    return 0
}

# procedure: read_file_proc
#
# arguments:
#  text_area:  the text area on the transcriber   to be manipulated  
#  filename: file to read data from
#
# return: none
#
# procedure to read given filename into given text area
#
proc read_file_proc {text_area filename} {

    # enable text
    #
    $text_area configure -state normal
    $text_area delete 1.0 end
    
    # make sure the config file exists and is readable
    #
    if {[file readable $filename] != 1} {
	warning_proc "Cannot open the file $filename. Please \
		make sure the file exists and has read permissions."
	return
    }

    # open file
    #
    set fileptr [open $filename r]
    while {![eof $fileptr]} {
	
	# read the file line by line
	#
	set str [gets $fileptr]
	
	# display it in the help window
	#
	$text_area insert end "$str\n"
    }
    
    # disable text
    #
    $text_area configure -state disabled
    
    # close file
    #
    close $fileptr
}

#
# end of file
