#! /usr/local/bin/wish -f

# file: $ISIP_TRANSCRIBER/gui/src/lexicon.tcl
#
# procedures to view the lexicon and search / add words
#

# list of procedures:
#
#  lexicon_proc {}
#  lex_search_proc {lword direction}
#  lex_add_proc {lword}
#  lex_quit_proc {}
#

# procedure: lexicon_proc
#
# arguments: none
#
# return: none
#
# procedure to create the lexicon window
#
proc lexicon_proc {} {

    # declare globals
    #
    global lx

    global lex_ind
    global lex_word
    global lexicon_file
    global transcriber_title
    global bg_col

    # set the toplevel
    #
    set lx ".lexicon"

    # set the indices
    #
    set lex_ind 1.0

    # if existing window can be reused don't recreate the window
    #
    if {[winfo exists $lx] == 1} {

	# raise window
	#
	wm deiconify $lx
	raise $lx

	# re-read lexicon file
	#
	if {[file exists $lexicon_file] == 1} {
	    read_file_proc $lx.lexicon $lexicon_file
	}

    } else {

	# start a new window
	#
	toplevel $lx
	wm title $lx "$transcriber_title Lexicon"
	wm geometry $lx "+100+100"
	wm focusmodel $lx active

	# make top and bottom frames
	#
	frame $lx.top_f -borderwidth 1 -relief ridge -height 400 -width 300 \
		-background $bg_col
	frame $lx.bot_f -borderwidth 1 -relief ridge -height 120 -width 300 \
		-background $bg_col
	pack $lx.top_f -fill both -expand true -side top
	pack $lx.bot_f -fill both -side bottom

	# create subframes for the bottom half
	#
	frame $lx.word_f -width 300 -height 60 -borderwidth 0 \
		-background $bg_col
	frame $lx.control_f -width 300 -height 40 -borderwidth 0 \
		-background $bg_col
	pack $lx.word_f $lx.control_f -side top -fill both -expand true \
		-padx 1 -pady 1 -in $lx.bot_f

	# create text area to display the lexicon
	#
	text $lx.lexicon -relief sunken -yscrollcommand "$lx.yscroll set" \
                -wrap none -exportselection true -state disabled -width 100 \
		-background $bg_col
	if {[file exists $lexicon_file] == 1} {
	    read_file_proc $lx.lexicon $lexicon_file
	}
	scrollbar $lx.yscroll -command "$lx.lexicon yview" -borderwidth 1 \
		-background $bg_col
	pack $lx.lexicon -padx 1 -pady 1 -side left -fill both -expand true \
		-in $lx.top_f
	pack $lx.yscroll -side right -fill y -in $lx.top_f

	# create entry for word
	#
	label $lx.label -text "Word:" -anchor c -width 10 -background $bg_col
	entry $lx.word -textvariable lex_word -width 60 -background $bg_col
	pack $lx.label -side left -fill both -padx 1m -pady 1m -in $lx.word_f
	pack $lx.word -side right -fill both -expand true -padx 1m -pady 1m \
		-in $lx.word_f
	
	# add control buttons
	#
	button $lx.search1 -text "Search Fwd" -width 12 -background $bg_col \
		-command {
	    lex_search_proc $lex_word 1
	    update
	}
	button $lx.search2 -text "Search Bkw" -width 12 -background $bg_col \
		-command {
	    lex_search_proc $lex_word 0
	    update
	}
	button $lx.add -text "Add Word" -width 12 -background $bg_col \
		-command {
	    lex_add_proc $lex_word
	    update
	}
	button $lx.clear -text "Clear Word" -width 12 -background $bg_col \
		-command {
	    set lex_word ""
	}
	button $lx.quit -text "Dismiss" -width 12 -background $bg_col \
		-command {
	    lex_quit_proc
	}

	pack $lx.search1 $lx.search2 $lx.add -side left -fill both \
		-padx 1 -pady 1 -in $lx.control_f
	pack $lx.quit $lx.clear -side right -fill both -padx 1 -pady 1 \
		-in $lx.control_f
    }

    # define key bindings
    #
    bind $lx <Alt-s> {
	lex_search_proc $lex_word 1
	update
    }
    bind $lx <Alt-r> {
	lex_search_proc $lex_word 0
	update
    }
    bind $lx <Alt-a> {
	lex_add_proc $lex_word
	update
    }
    bind $lx <Alt-c> {
	set lex_word ""
    }
}

# procedure: lex_search_proc
#
# arguments:
#  lword: word to search in the lexicon 
#  direction: lexicon search direction
#
# return: none
#
# procedure to search for a word in the lexicon
#
proc lex_search_proc {lword direction} {

    # declare globals
    #
    global lx
    global lex_ind
    global lexicon_file

    # delete old tags
    #
    $lx.lexicon tag remove sel 1.0 end
    $lx.lexicon tag delete sel

    # make sure a word is specified for searching
    #
    if {$lword == ""} {
	return
    }

    # search for the given word in the lexicon file
    #
    set lex_ind [expr $lex_ind + $direction]
    if {$direction == 1} {
	set lex_ind [$lx.lexicon search -forwards -regexp \
		-nocase -- "^$lword" $lex_ind]
    } elseif {$direction == 0} {
	set lex_ind [$lx.lexicon search -backwards -regexp \
		-nocase -- "^$lword" $lex_ind]
    }	

    # if the word is not found
    #
    if {$lex_ind == ""} {
	
	# reset display to top of file
	#
	set lex_ind 1.0
	$lx.lexicon see $lex_ind
	
	# output a message
	#
	warning_proc "The word [string trim $lword] was not found in the \
		lexicon."

    } else {

	# display the selected line in the window
	#
	$lx.lexicon see $lex_ind
	$lx.lexicon tag add sel $lex_ind \
		[$lx.lexicon search -forwards "\n" $lex_ind]
    }
}

# procedure: lex_add_proc
#
# arguments:
#  lword: word to add to the lexicon
#
# return: none
#
# procedure to add a word to the lexicon
#
proc lex_add_proc {lword} {

    # declare globals
    #
    global lx
    global lexicon_file
    
    # add the word to the lexicon file if writable
    #
    if {[file writable $lexicon_file] == 1} {

	$lx.lexicon configure -state normal
	$lx.lexicon insert end "$lword\n"
	$lx.lexicon configure -state disabled

    } else {
	warning_proc "You do not have permission to modify the lexicon."
    }
}

# procedure: lex_quit_proc
#
# arguments: none
#
# return: none
#
# procedure to quit
#
proc lex_quit_proc {} {

    # declare globals
    #
    global lx
    global lex_word
    global lexicon_file
    
    global p

    # reset the lex word
    #
    set lex_word ""

    # delete old tags
    #
    $lx.lexicon tag remove sel 1.0 end
    $lx.lexicon tag delete sel

    # if the lexicon file is writable save changes
    #
    if {[file writable $lexicon_file] == 1} {
	
	# open the file
	#
	set fileptr [open $lexicon_file w]

	# dump all text
	#
	puts $fileptr [$lx.lexicon get 1.0 end]
	
	# close file
	#
	close $fileptr
    }

    # close window
    #
    destroy $lx
}

#
# end of file
