#! /usr/local/bin/wish -f

# file: $ISIP_TRANSCRIBER/gui/src/expand.tcl
#
# provides event driven (Tab expansion) completion of the words in the 
# transcription area for the user
#

# list of procedures:
#
#  display_options_proc {}
#  buffer_line_proc {}
#  hash_function {str}
#  build_hash_proc {}
#  search_hash_proc {}
#

# procedure: display_options_proc
#
# arguments: list of options
#
# return: none
#
# procedure that displays all of the matching options and allows the user to
# select an option and automatically complete the word 
#
proc display_options_proc {options} {

    # declare globals
    #
    global transcriber_title
    global bg_col

    global p e

    set e ".opt"
    set curr_word_ind 0

    # don't recreate the window, try to reuse existing one
    #
    if {[winfo exists $e] == 1 } {
        wm deiconify $e
        raise $e
    } else {

	# create a new window
	#
	toplevel $e
	wm title $e "$transcriber_title Autofill Options"
	wm geometry $e "+100+100"
	wm focusmodel $e active
	wm minsize $e 375 175
	
	# make the top and bottom frames
	#		
	frame $e.opt_f -borderwidth 0 -relief flat -height 250 -width 300 \
		-background $bg_col
	frame $e.ctrl_f -borderwidth 0 -relief flat -height 250 -width 120 \
		-background $bg_col
	frame $e.lbox_f -borderwidth 0 -relief flat -height 250 -width 180 \
		-background $bg_col
	
	pack $e.opt_f -padx 1 -pady 1 -fill both -expand true
	pack $e.ctrl_f -padx 1 -side left -fill both -expand true -in $e.opt_f
	pack $e.lbox_f -padx 1 -side right -fill both -expand true -in $e.opt_f
    
	# add the listbox stuff
	#
	listbox $e.list -selectmode single -yscrollcommand "$e.yscroll set"\
		-background $bg_col -width 40
	scrollbar $e.yscroll -command "$e.list yview" -background $bg_col 
	pack $e.list -side left -padx 1m -pady 1m -fill both -expand true \
		-in $e.lbox_f
	pack $e.yscroll -side right -padx 1m -pady 1m -fill y -in $e.lbox_f

	# add the control buttons
	#
	button $e.done -text "OK" -background $bg_col -command {
	    
	    # set the initial focus
	    #
	    catch { set ind [lindex [$e.list curselection] 0] }
	    
	    # retrieve the index of the word selected
	    #
	    catch { set ind [lindex [$e.list curselection] 0] }
	    catch { set curr_word_ind $ind }
	    catch { set selection [$e.list get $ind] }
	    catch { set selection [string trim $selection] }
	    
	    # automatically complete the word with a unique match
	    #
	    catch { $p.curr_trans insert end [buffer_line_proc] }
	    catch { $p.curr_trans insert end $selection }
	}
	
	button $e.cancel -text "Dismiss" -background $bg_col -command {
	    destroy $e
	}
	
	pack $e.done -side top -padx 1m -pady 1m -fill both -in $e.ctrl_f
	pack $e.cancel -side top -padx 1m -pady 1m -fill both -in $e.ctrl_f

	# key bindings for the listbox
	#
	bind $e.list <Double-Button-1> {
	    
	    # retrieve the index of the file selected
	    #
	    catch { set ind [lindex [$e.list curselection] 0] }
	    catch { set curr_word_ind $ind }
	    catch { set selection [$e.list get $ind] }
	    catch { set selection [string trim $selection] }
	    
	    # automatically complete the word for the user for a unique match
	    #
	    catch { $p.curr_trans insert end [buffer_line_proc] }
	    catch { $p.curr_trans insert end $selection }
	}
    }

    # fill the list box
    #
    fill_options_proc $options
    
    # activate the list box
    #
    catch { $e.list selection clear 0 end }
    catch { $e.list see 0 }
    catch { $e.list selection set 0 }
    
    # event to view the current list box index
    #
    bind $e.list <Enter> {
	catch { $e.list selection set $curr_word_ind }
    }
}

# procedure: buffer_line_proc
#
# arguments: none
#
# return: string
#
# procedure that reads in the transcription, parses the entire line upto the
# focus word and returns the line leading to the focus word
#
proc buffer_line_proc {} {
   
    # declare globals
    #
    global p

    # set the current utterance transcription
    #
    set prevstr [$p.curr_trans get 1.0 end]
    set prevstr [string trim $prevstr]

    # clear the current transcription box contents
    #
    catch { $p.curr_trans delete 1.0 end }

    set ind [string last " " $prevstr]

    if {$ind == -1} {
	set prevstr ""
    } else {
	set prevstr [string range $prevstr 0 $ind]
    }

    return $prevstr
}

# procedure: hash_function
#
# arguments: input string
#
# return: lower case character
#
# procedure that takes a word as an argument and return the first letter of
# that word in lower case to index the hash table
#
proc hash_function {str} {

    # convert the string to lower case if not already
    #
    set lower [string tolower $str]

    # parse the first character form the string
    #
    set first [string index $lower 0]
    
    # make sure that the hash index is not a special character
    #
    set flag 1
    for {set i 0} {$flag == 1} {incr i} {

	# get the first character
	#
	set first [string index $lower $i]

	# check to see if it is a special character
	#
	if {$first == "\-"} { 
	    set flag 1 
	} elseif {$first == "\*"} { 
	    set flag 1 
	} elseif {$first == "\?"} { 
	    set flag 1 
	} elseif {$first == "\["} { 
	    set flag 1 
	} elseif {$first == "\]"} { 
	    set flag 1 
	} elseif {$first == "\\"} { 
	    set flag 1 
	} else   { 
	    set flag 0 
	}
    }

    # return the hash index
    #
    return $first
}

# procedure: build_hash_proc
#
# arguments: none
#
# return: none
#
# procedure that reads in the dictionary file and creates a hash table 
# that provides for better search time of the data
#
proc build_hash_proc {} {

    # define globals
    #
    global p

    global hash
    global lexicon_file 

    # add a watch on the mouse pointer to indicate we are busy right now
    #
    $p configure -cursor watch

    # check if the dictionary file is readable
    #
    if {[file readable $lexicon_file] != 1} {
	warning_proc "Cannot open $lexicon_file. Please make sure the file \
		exists and has read permissions."
	return
    } else {
	
	# open the dictionary file
	#
	set lexptr [open $lexicon_file r]
    }

    # read the dictionary file 
    #
    while {![eof $lexptr]} {
	
	# read a line from the file and remove all leading white spaces
	#
	set lexstr [gets $lexptr]
	set lexstr [string trim $lexstr]

	# compute the hash index
	#
	set index [hash_function $lexstr]

	# add the word to the hash table using the hash index
	#
	lappend hash($index) $lexstr
    }

    # remove the mouse watch pointer
    #
    $p configure -cursor top_left_arrow

    # close the dictionary file
    #
    close $lexptr
}

# procedure: search_hash_proc
#
# arguments: none
#
# return: none
#
# procedure that takes a string as an input and completes the string if there
# is a unique match else it displays all possible matches 
#
proc search_hash_proc {} {

    # declare globals
    #
    global p e
    global hash
    global lexicon_file

    # declare local variables
    #
    set match {}
    set found 0

    # set the current utterance transcription
    #
    catch { set trans [$p.curr_trans get 1.0 end] }
    catch { set trans [string trim $trans] }

    # exit if the expansion mechanism has been invoked on an empty string
    #
    if {$trans == ""} {

	# exit gracefully
	#
	return
    }

    # find the index of the last white space before the word to be expanded
    #
    set ind [string last " " $trans]

    # get the word to be expanded
    #
    if {$ind == -1} {
	set focus $trans
    } else {
	set focus [string range $trans [expr $ind + 1] end]
    }
    
    # remove all trailing white spaces from the focus word
    #
    set focus [string trim $focus]

    # compute the hash index
    #
    set index [hash_function $focus]

    # make sure that the hash index exists
    #
    set exist 0
    catch { set exist $hash($index) }
    if {$exist == 0} { 
    	bell 
    	return 
    }

    # search the hash table for matches
    #
    foreach item $hash($index) {

	# if the there is a perfect match
	#
	if {$item == $focus} {

	    # clear the listbox
	    #
	    catch { $e.list delete 0 end     }
	    
	    # insert the unique match into the listbox
	    #
	    catch { $e.list insert end $focus }
	    
	    # activate the list box
	    #
	    catch { $e.list selection clear 0 end }
	    catch { $e.list see 0 }
	    catch { $e.list selection set 0 }
	    
	    # exit gracefully
	    #
	    return
	}

	# if no unique match is found then store all the words that 
	# could be a possible completion of the focus word
	#
	if {[string match ${focus}* $item]} {
	    incr found
	    lappend match $item	    
	}
    }
    
    # execute the statements if possible completions for the focus word exist
    #
    if {$found} {
	
	# display all matches if there are more that one
	#
	if {$found > 1} {
	    
	    # dispaly all the options
	    #
	    display_options_proc $match
	    
	} else {
	    
	    # automatically complete the word for the user if only one
	    # possible completion was found
	    #
	    catch { $p.curr_trans insert end [buffer_line_proc] }
	    catch { $p.curr_trans insert end [lindex $match 0] }
	    
	    if {[winfo exists $e] == 1 } {
		
		# clear the listbox
		#
		catch { $e.list delete 0 end     }
		
		# insert the unique match into the listbox
		#
		catch { $e.list insert end [lindex $match 0] }
		
		# activate the list box
		#
		catch { $e.list selection clear 0 end }
		catch { $e.list see 0 }
		catch { $e.list selection set 0 }
	    }
	}
    } else {
	
	# sound a bell indicating that no matches have been found
	#
	bell
    }
}
    	
#
# end of file


