#! /usr/local/bin/wish -f

# file: $ISIP_TRANSCRIBER/gui/src/audio.tcl
#
# procedures to play audio on various systems like for example dat, ncd, x86
# and sparc
#

# list of procedures:
#
#  audio_play_proc {out_mode start end bg_flag}
#  dat_play_proc {out_mode start end bg_flag}
#  ncd_play_proc {out_mode start end bg_flag}
#  x86_play_proc {out_mode start end bg_flag}
#  sparc_play_proc {out_mode start end bg_flag}
#  play_brackets_proc {}
#  proc play_window_proc {}
#  play_utt_proc {index}
#  volume_proc {ch vol}
#

# procedure: audio_play_proc
#
# arguments:
#  out_mode: audio output mode
#  start: audio start time
#  end: audio end time
#  bg_flag: audio flag
# 
# return: none
#
# this procedure cuts a portion of the input signal
#
proc audio_play_proc {out_mode start end bg_flag} {
    
    # set global variables
    #
    global audio_play_device
    global loadflag
    
    # return if nothing is loaded
    #
    if {$loadflag == 0} {
	return
    }

    # play this to a dat or an ncd terminal as specified
    #
    if {$audio_play_device == "dat"} {
	dat_play_proc $out_mode $start $end $bg_flag
    } elseif {$audio_play_device == "ncd"} {
	ncd_play_proc $out_mode $start $end $bg_flag
    } elseif {$audio_play_device == "x86"} {
	x86_play_proc $out_mode $start $end $bg_flag
    } elseif {$audio_play_device == "sparc"} {
 	sparc_play_proc $out_mode $start $end $bg_flag
    } else {
	warning_proc "Audio device $audio_play_device not currently supported."
    }
}

# procedure: dat_play_proc
#
# arguments:
#  out_mode: audio output mode
#  start: audio start time
#  end: audio end time
#  bg_flag: audio flag
# 
# return: none
#
# procedure to play audio data on the dat
#
proc dat_play_proc {out_mode start end bg_flag} {

    # declare globals
    #
    global play_prog_dat
    global excise_signal_prog

    global audio_server
    global audio_file
    
    global volume_ch0

    global sample_frequency
    global sample_num_bytes
    global num_channels

    global byte_swap
    global pause_time

    # set the volume to play
    #
    set volume $volume_ch0
        
    # check for byte swapping
    #
    set au_format " "
    set data_format "raw"
    if {$byte_swap == 1} {
	set au_format "-swap_bytes"
	set data_format "swap"
    }
    
    # need to up-sample data to play well
    #
    set upsample_frequency 44100

    # open the pipe to excise_signal and naplay depending on the
    # channel to be played
    #
    set chan "left"
    set out_mode 2
    
    # play data to the dat link
    #
    if {$bg_flag == 0} {
	exec -- $excise_signal_prog -sf $sample_frequency -start $start \
		-end $end -num_bytes $sample_num_bytes -pad_time $pause_time \
		-num_chan $num_channels -out_mode $out_mode \
		$au_format -infile $audio_file -scale $volume | \
		$play_prog_dat -s $sample_frequency -f $data_format \
		-o stereo -m $chan -u $audio_server -S $upsample_frequency \
		> /dev/null 2> /dev/null &
    } else {
	exec -- $excise_signal_prog -sf $sample_frequency -start $start \
		-end $end -num_bytes $sample_num_bytes -pad_time $pause_time \
		-num_chan $num_channels -out_mode $out_mode \
		$au_format -infile $audio_file -scale $volume | \
		$play_prog_dat -s $sample_frequency -f $data_format \
		-o stereo -m $chan -u $audio_server -S $upsample_frequency \
		> /dev/null 2> /dev/null
    }	    
}

# procedure: ncd_play_proc
#
# arguments:
#  out_mode: audio output mode
#  start: audio start time
#  end: audio end time
#  bg_flag: audio flag
# 
# return: none
#
# procedure to play audio data on ncd
#
proc ncd_play_proc {out_mode start end bg_flag} {
    
    # set global variables
    #
    global play_prog_ncd
    global excise_signal_prog

    global audio_server
    global audio_file

    global volume_ch0

    global sample_frequency
    global sample_num_bytes
    global num_channels

    global byte_swap
    global pause_time

    # set the volume to play
    #
    set num_chans 1
    set ster "mono"
    set volume $volume_ch0
    
    # need to upsample data to play well
    #
    set upsample_frequency 44100

    # check for byte swapping
    #
    set au_format " "
    if {$byte_swap == 1} {
	set au_format "-swap_bytes"
    }

    # play data to the ncd terminal
    #
    if {$bg_flag == 0} {
	exec -- $excise_signal_prog -sf $sample_frequency -start $start \
		-end $end -num_bytes $sample_num_bytes -pad_time $pause_time \
		-num_chan $num_channels -out_mode $out_mode \
		$au_format -infile $audio_file -scale $volume | audioconvert \
		-i linear16,rate=$sample_frequency,$ster \
		-f sun,rate=$upsample_frequency,$ster | \
		$play_prog_ncd -audio $audio_server > /dev/null 2> /dev/null &
    } else {
	exec -- $excise_signal_prog -sf $sample_frequency -start $start \
		-end $end -num_bytes $sample_num_bytes -pad_time $pause_time \
		-num_chan $num_channels -out_mode $out_mode \
		$au_format -infile $audio_file -scale $volume | audioconvert \
		-i linear16,rate=$sample_frequency,$ster \
		-f sun,rate=$upsample_frequency,$ster | \
		$play_prog_ncd -audio $audio_server > /dev/null 2> /dev/null
    }	    
}

# procedure: x86_play_proc
#
# arguments:
#  out_mode: audio output mode
#  start: audio start time
#  end: audio end time
#  bg_flag: audio flag
# 
# return: none
#
# procedure to play audio data on the x86 machines
#
proc x86_play_proc {out_mode start end bg_flag} {
    
    # set global variables
    #
    global play_prog_x86
    global excise_signal_prog

    global audio_server
    global newaudiofile
    global volume_ch0

    global sample_frequency
    global sample_num_bytes
    global num_channels

    global byte_swap
    global pause_time

    # need to upsample data to play on the Soundblasters
    #
    set upsample_frequency 44100

    # set the volume to play
    #
    set num_chans 1
    set ster "mono"
    set volume $volume_ch0
    
    # check for byte swapping
    #
    set au_format " "
    if {$byte_swap == 1} {
	set au_format "-swap_bytes"
    }

    # play data to the ncd terminal
    #
    if {$bg_flag == 0} {
	exec -- $excise_signal_prog -sf $sample_frequency -start $start \
		-end $end -num_bytes $sample_num_bytes -pad_time $pause_time \
		-num_chan $num_channels -out_mode $out_mode \
		$au_format -infile $newaudiofile -scale $volume | \
		audioconvert -i linear16,rate=$sample_frequency,$ster \
		-f sun,rate=$upsample_frequency,$ster | $play_prog_x86 \
		-b 0 -p $audio_server - > /dev/null 2> /dev/null &
    } else {
	exec -- $excise_signal_prog -sf $sample_frequency -start $start \
		-end $end -num_bytes $sample_num_bytes -pad_time $pause_time \
		-num_chan $num_channels -out_mode $out_mode \
		$au_format -infile $newaudiofile -scale $volume | \
		audioconvert -i linear16,rate=$sample_frequency,$ster \
		-f sun,rate=$upsample_frequency,$ster | $play_prog_x86 \
		-b 0 -p $audio_server - > /dev/null 2> /dev/null
    }	    
}

# procedure: sparc_play_proc
#
# arguments:
#  out_mode: audio output mode
#  start: audio start time
#  end: audio end time
#  bg_flag: audio flag
# 
# return: none
#
# procedure to play audio data on the sparc machines
#
proc sparc_play_proc {out_mode start end bg_flag} {
    
    # set global variables
    #
    global play_prog_sparc
    global excise_signal_prog

    global audio_server
    global audio_file

    global volume_ch0

    global sample_frequency
    global sample_num_bytes
    global num_channels

    global byte_swap
    global pause_time

    # need to upsample data to play on the Soundblasters
    #
    set upsample_frequency 44100

    # set the volume to play
    #
    set num_chans 1
    set ster "mono"
    set volume $volume_ch0
    
    # check for byte swapping
    #
    set au_format " "
    if {$byte_swap == 1} {
	set au_format "-swap_bytes"
    }

    # play data to the ncd terminal
    #
    if {$bg_flag == 0} {
	exec -- $excise_signal_prog -sf $sample_frequency -start $start \
		-end $end -num_bytes $sample_num_bytes -pad_time $pause_time \
		-num_chan $num_channels -out_mode $out_mode \
		$au_format -infile $audio_file -scale $volume | audioconvert \
		-i linear16,rate=$sample_frequency,$ster \
		-f sun,rate=$upsample_frequency,$ster | $play_prog_sparc \
		-b 0 -p $audio_server - > /dev/null 2> /dev/null &
    } else {
	exec -- $excise_signal_prog -sf $sample_frequency -start $start \
		-end $end -num_bytes $sample_num_bytes -pad_time $pause_time \
		-num_chan $num_channels -out_mode $out_mode \
		$au_format -infile $audio_file -scale $volume | audioconvert \
		-i linear16,rate=$sample_frequency,$ster \
		-f sun,rate=$upsample_frequency,$ster | $play_prog_sparc \
		-b 0 -p $audio_server - > /dev/null 2> /dev/null
    }	    
}

# procedure: play_brackets_proc
#
# arguments: none
#
# return: none
#
# procedure to play data between bracket markers
#
proc play_brackets_proc {} {
    
    # declare global variables
    #
    global play_mode

    global bracket_start
    global bracket_end

    global verify_flag

    global curr_ind

    global newaudiofile
    global audiolist
    global arraylist

    # find the utterance index to play
    #
    set uttind [eval expr $curr_ind + 0]

    if {($uttind >= 0) && ($uttind <= [llength $audiolist])} {

	# retrieve the current audio file
	#
	set newaudiofile [lindex $audiolist $uttind]
	
	# play the data
	#
	audio_play_proc $play_mode $bracket_start $bracket_end 0
    }
}

# procedure: play_window_proc
#
# arguments: none
#
# return: none
#
# procedure to play data displayed in window
#
proc play_window_proc {} {
    
    # declare global variables
    #
    global window_start
    global window_end
    global play_mode
    
    # play the specified channels
    #
    audio_play_proc $play_mode $window_start $window_end 0
}

# procedure: play_utt_procs 
#
# arguments:
#  index: utterance index
#
# return: none
#
# procedure to play specified utterance
#
proc play_utt_proc {index} {
    
    # declare globals
    #
    global current_channel
    global play_mode
    global curr_ind

    global verify_flag
    global maximum_time
    global newaudiofile
    global audiolist
    global arraylist

    global autoplay
    
    # set the corresponding channel parameters
    #
    set play_mode 0

    # set the start and end times
    #
    set start 0
    set end $maximum_time
    
    # find the utterance index to play
    #
    set uttind [eval expr $curr_ind + $index]

    if {($uttind >= 0) && ($uttind <= [llength $audiolist])} {

	# retrieve the current audio file
	#
	set newaudiofile [lindex $audiolist $uttind]

	# check to see if the specified audio file exists
	#
	if {[file exists $newaudiofile] != 1} {
	    return
	}

	# play the data
	#
	audio_play_proc $play_mode $start $end $verify_flag
    }
}

# procedure: volume_proc
#
# arguments:
#  ch: audio channel
#  vol: input audio volume
#
# return: none
#
# procedure to tie the volume
#
proc volume_proc {ch vol} {

    # declare globals
    #
    global volume_ch0

    global p

    # check if the volume is tied
    #
    $p.volume0 set $vol
}

#
# end of file
