// file: snrc_comp_5.cc
//
// computes the snr of the distribution given the signal+noise and noise
// thresholds
//

// isip include files
//
#include <integral.h>
#include <integral_constants.h>

// local include files
//
#include "snr_calculator.h"

#include "snr_calculator_constants.h"

// function: compute_snr_cc
//
// arguments:
//
//  long** cdf: (input) cumulative distribution of an energy set
//  double signal_thresh : (input) signal + noise threshold
//  double noise_thresh: (input) noise threshold
//  long num_values: (input) number of values in the data array
//  long num_channels: (input) number of channels of data
//  double* max: (input) maximum energy values for the data
//  double* min: (input) minimum energy values for the data
//  double** mean_vals: (input) mean values of each bin
//  double* snr_values: (output) signal-to-noise ratio of each channel
//  
// return value: a logical_1 indicating status
//
// this function finds the signal to noise ratio of each channel by:
//
//    10 * log10[((Es + En) - En) / En] where,
//  
// Es = energy below the signal + noise threshold, signal_thresh
// En = energy below the noise threshold, noise_thresh
//
logical_1 compute_snr_cc(long** cdf_a, double signal_thresh_a,
			 double noise_thresh_a, long num_values_a,
			 long num_channels_a, double* max_a,
			 double* min_a, double** mean_vals_a,
			 double* snr_values_a) {

  // make sure that the data is allocated
  //
  if ((cdf_a == (long**)NULL) || (max_a == (double*) NULL) ||
      (min_a == (double*)NULL) || (snr_values_a == (double*)NULL) ||
      (mean_vals_a == (double**)NULL)) {

    // return with error
    //
    return ISIP_FALSE;
  }

  // determine the cumulative index for the signal + noise and the noise
  // thresholds
  //
  long signal_index = (long)(signal_thresh_a * (double)num_values_a);
  long noise_index = (long)(noise_thresh_a * (double)num_values_a);
  
  // loop over all channels
  //
  for (long chan = 0; chan < num_channels_a; chan++) {

    // find the energy references
    //
    double noise_energy = min_a[chan];
    double signal_energy = min_a[chan];
    
    // loop until the index of the noise threshold is found
    //
    long cumulative_index = cdf_a[0][chan];
    long data_bin = (long)0;
    
    while (cumulative_index < noise_index) {

      cumulative_index = cdf_a[++data_bin][chan];
    }

    // get the noise energy
    //
    noise_energy = mean_vals_a[data_bin][chan];
    
    // loop until the signal threshold is found
    //
    while (cumulative_index < signal_index) {

      cumulative_index = cdf_a[++data_bin][chan];
    }

    // determine the value of the signal + noise energy
    //
    signal_energy = mean_vals_a[data_bin][chan];
    
    // now determine the snr of this data channel and store it
    //
    snr_values_a[chan] = (double)10.0 * log10((signal_energy - noise_energy) /
					       noise_energy);
  }
  
  // exit gracefully
  //
  return ISIP_TRUE;
}

