// file: snrc_comp_1.cc
//
// applies a hamming window to the input data
//

// isip include files
//
#include <integral.h>
#include <integral_constants.h>

// local include files
//
#include "snr_calculator.h"

#include "snr_calculator_constants.h"

// function: window_hamming_cc
//
// arguments:
//
//  double* data: (input/output) sampled data
//  long num_samples: (input) number of samples in the data stream
//  long num_channels: (input) number of channels in the data stream
//  
// return value: a logical_1 indicating status
//
// this function applies a hamming window to the input data by the function:
//
//   y(n) = x(n) * (b - [(1 - b) * cos(2 * pi * n / (num_samples - 1))])
// 
// the center index of the input data is assumed to be time zero for the window
// the input data is overwritten.
//
logical_1 window_hamming_cc(double* data_a, long num_samples_a,
			    long num_channels_a) {

  // loop over all channels
  //
  for (long chan = 0; chan < num_channels_a; chan++) {

    // loop over all data
    //
    for (long samp_num = 0; samp_num < num_samples_a; samp_num++) {

      // store the sample data
      //
      double x_n = data_a[samp_num * num_channels_a + chan];

      // multiply by the window function
      //
      double x_hamm_n = x_n * (double)(SNRC_HAMMING_COEFF -
					 (((double)1.0 - SNRC_HAMMING_COEFF) *
					  cos(ISIP_TWOPI * (double)samp_num /
					      (double)(num_samples_a - 1))));

      // store the windowed data
      //
      data_a[samp_num * num_channels_a + chan] = x_hamm_n;
    }
  }

  // exit gracefully
  //
  return ISIP_TRUE;
}

