// file: $ECE_8993/class/signal/v1.0/sig_write_0.cc
//

// system include files
//
#include <string.h>

// isip include files
//
#include "signal.h"

#include <integral_constants.h>
#include "signal_constants.h"

// method: write_cc
//
// arguments
//   double* data: (input) portion of the signal to output. this memory
//                  is allocated internally. pointer must be null on entry
//   double start_time: (input) beginning point of the desired data
//   long num_samples: (input) number of samples to write
//
// return: logical indicating status
//
// this method writes the desired time window.
//
logical Signal::write_cc(double* data_a, double start_time_a,
			   long num_samples_a) {

  // make sure that the file is open
  //
  if (fp_d == (FILE*)NULL) {

    // return with error
    //
    error_handler_cc((unichar*)"write_cc", (unichar*)"signal file is not open");
    return ISIP_FALSE;
  }

  // make sure that the data pointer is null, else error
  //
  if (data_a == (double*)NULL) {

    // return with error
    //
    error_handler_cc((unichar*)"write_cc", (unichar*)"data is not allocated");
    return ISIP_FALSE;
  }

  // rewind the input file
  //
  rewind(fp_d);

  // test the bounds of the start time
  //
  if (start_time_a > end_file_time_d) {

    // return with error
    //
    error_handler_cc((unichar*)"write_cc", (unichar*)"start time too large");
    return ISIP_FALSE;
  }

  // test the bounds of the start time
  //
  if (start_time_a < (double)0.0) {

    // set start time to 0.0
    //
    start_time_a = 0.0;
  }

  // allocate memory for the output data
  //
  unichar* out_data = new unichar[num_samples_a * num_chans_d * num_bytes_d];

  // loop over the input data and convert it to the proper output type
  //
  long loop_its = num_samples_a * num_chans_d;
  for (long i = 0; i < loop_its; i++) {

    // compute the sample value
    //
    if (num_bytes_d == (long)2) {
      ((short*)(out_data))[i] = (short)data_a[i];
    }
    else if (num_bytes_d == (long)4) {
      ((long*)(out_data))[i] = (long)data_a[i];
    }
    else {

      // free memory
      //
      delete [] out_data;

      // return with error
      //
      error_handler_cc((unichar*)"write_cc",
		       (unichar*)"number of bytes per sample is invalid");
      return ISIP_FALSE;
    }
  }
  
  // swap bytes if necessary
  //
  if (swap_byte_flag_d == ISIP_TRUE) {
    
    // loop over all two-byte sequences and swap bytes
    //
    // determine how many times through the loop we should go
    //
    long loop_its = (num_samples_a * num_bytes_d * num_chans_d) / (long)2;

    // set a pointer to the beginning of the buffer
    //
    unichar* temp = out_data;
    for (long i = 0; i < loop_its; i++) {
      unichar samp = temp[0];
      temp[0] = temp[1];
      temp[1] = samp;
      temp += 2;
    }
  }

  // seek to the starting point in the file
  //
  fseek(fp_d, (long)(start_time_a * sf_d) * num_bytes_d * num_chans_d,
	SEEK_SET);

  // write the data out to file
  //
  long samples_writ = fwrite(out_data, num_chans_d * num_bytes_d,
			     num_samples_a, fp_d);

  // check that the number of samples written is correct
  //
  if (samples_writ != num_samples_a) {

    // return with error
    //
    error_handler_cc((unichar*)"write_cc",
		     (unichar*)"could not read all samples");
    return ISIP_FALSE;
  }

  // determine the new size of the file
  //
  rewind(fp_d);
  fseek(fp_d, 0, SEEK_END);
  file_size_d = ftell(fp_d);
  end_file_time_d = (double)file_size_d / (sf_d * (double)num_bytes_d *
					    (double)num_chans_d);
  
  // delete memory and clean up
  //
  delete [] out_data;
  
  // exit gracefully
  //
  return ISIP_TRUE;
}



