// file: $ECE_8993/class/signal/v1.0/sig_read_1.cc
//

// isip include files
//
#include "signal.h"

#include <integral_constants.h>
#include "signal_constants.h"

// method: read_cc
//
// arguments:
//   double*& data: (output) portion of the signal to output. this memory
//                   is allocated externally
//   long start_samp: (input) beginning point of the desired data
//   long end_samp: : (input) end point of the desired data
//   long& num_samples: : (output) number of samples read
//
// return: logical indicating status
//
// this method gets the desired time window. if the start sample is less than
// zero, then the start of the data is at index 0. if the end time is greater
// than the end of the file, then the end sample is set to the end of the file.
// if the end sample is less than the start sample, then an error is returned.
//
logical Signal::read_cc(double*& data_a, long start_samp_a,long end_samp_a,
			long channel_a, long& num_samples_a) {

  // zero out the number of samples read
  //
  num_samples_a = 0;
  
  // make sure that the file is open
  //
  if (fp_d == (FILE*)NULL) {

    // return with error
    //
    error_handler_cc((unichar*)"read_cc", (unichar*)"signal file is not open");

    return ISIP_FALSE;
  }

  // make sure that the data pointer is null, else error
  //
  if (data_a == (double*)NULL) {

    // return with error
    //
    error_handler_cc((unichar*)"read_cc", (unichar*)"data is not allocated");
    return ISIP_FALSE;
  }

  // rewind the input file
  //
  rewind(fp_d);

  // determine the end time and check against the end of file
  //
  if (end_samp_a > (file_size_d / (num_chans_d * num_bytes_d))) {

    // set the end time to the end of file time
    //
    end_samp_a = (file_size_d / (num_chans_d * num_bytes_d));
  }

  // make sure that the end and start times are within bounds
  //
  if (end_samp_a < start_samp_a) {

    // return with error
    //
    return ISIP_FALSE;
  }

  // test the bounds of the start and end time
  //
  if (start_samp_a < 0) {

    // set start sample to 0
    //
    start_samp_a = (long)0;
  }

  // compute the number of samples to be obtained
  //
  num_samples_a = end_samp_a - start_samp_a;

  // allocate memory for the data and memory for reading in the data
  //
  unichar* in_data = new unichar[num_samples_a * num_chans_d * num_bytes_d];
  
  // seek to the starting point in the file
  //
  fseek(fp_d, start_samp_a * num_bytes_d * num_chans_d, SEEK_SET);

  // read the data into the data buffer
  //
  long samples_read = fread(in_data, num_chans_d * num_bytes_d,
			     num_samples_a, fp_d);

  // check that the number of samples read is correct
  //
  if (samples_read != num_samples_a) {

    // return with error
    //
    error_handler_cc((unichar*)"read_cc",
		     (unichar*)"could not read all samples");
    return ISIP_FALSE;
  }

  // swap bytes if necessary
  //
  if (swap_byte_flag_d == ISIP_TRUE) {

    // loop over all two-byte sequences and swap bytes
    //
    // determine how many times through the loop we should go
    //
    long loop_its = (num_samples_a * num_bytes_d * num_chans_d) / (long)2;

    // set a pointer to the beginning of the buffer
    //
    unichar* temp = in_data;
    for (long i = 0; i < loop_its; i++) {
      unichar samp = temp[0];
      temp[0] = temp[1];
      temp[1] = samp;
      temp += 2;
    }
  }

  // loop over all data and convert it to doubleing point values
  //
  long loop_its = num_samples_a * num_chans_d;
  long sample_val = 0;
  for (long i = 0; i < loop_its; i++) {

    // compute the sample value
    //
    if (num_bytes_d == (long)2) {
      sample_val = *((short*)(in_data +
			      (i * num_chans_d + channel_a) * num_bytes_d));
    }
    else if (num_bytes_d == (long)4) {
      sample_val = *((long*)(in_data +
			     (i * num_chans_d + channel_a) * num_bytes_d));
    }

    else {

      // free memory
      //
      delete [] in_data;

      // return with error
      //
      error_handler_cc((unichar*)"read_cc",
		       (unichar*)"number of bytes per sample is invalid");
      return ISIP_FALSE;
    }
    
    // assign the value of this sample
    //
    data_a[i] = (double)sample_val;
  }

  // delete memory and clean up
  //
  delete [] in_data;
  
  // exit gracefully
  //
  return ISIP_TRUE;
}






