// file: $ISIP_SNR/util/snr_calculator/v1.0/snrc_comp_5.cc
//
// computes the snr of the distribution given the signal+noise and noise
// thresholds
//

// isip include files
//
#include <integral.h>
#include <integral_constants.h>

// local include files
//
#include "snr_calculator.h"

#include "snr_calculator_constants.h"

// function: compute_snr_cc
//
// arguments:
//
//  int_4** cdf: (input) cumulative distribution of an energy set
//  float_4 signal_thresh : (input) signal + noise threshold
//  float_4 noise_thresh: (input) noise threshold
//  int_4 num_values: (input) number of values in the data array
//  int_4 num_channels: (input) number of channels of data
//  float_4* max: (input) maximum energy values for the data
//  float_4* min: (input) minimum energy values for the data
//  float_4** mean_vals: (input) mean values of each bin
//  float_4* snr_values: (output) signal-to-noise ratio of each channel
//  
// return value: a logical_1 indicating status
//
// this function finds the signal to noise ratio of each channel by:
//
//    10 * log10[((Es + En) - En) / En] where,
//  
// Es = energy below the signal + noise threshold, signal_thresh
// En = energy below the noise threshold, noise_thresh
//
logical_1 compute_snr_cc(int_4** cdf_a, float_4 signal_thresh_a,
			 float_4 noise_thresh_a, int_4 num_values_a,
			 int_4 num_channels_a, float_4* max_a,
			 float_4* min_a, float_4** mean_vals_a,
			 float_4* snr_values_a) {

  // make sure that the data is allocated
  //
  if ((cdf_a == (int_4**)NULL) || (max_a == (float_4*) NULL) ||
      (min_a == (float_4*)NULL) || (snr_values_a == (float_4*)NULL) ||
      (mean_vals_a == (float_4**)NULL)) {

    // return with error
    //
    return ISIP_FALSE;
  }

  // determine the cumulative index for the signal + noise and the noise
  // thresholds
  //
  int_4 signal_index = (int_4)(signal_thresh_a * (float_4)num_values_a);
  int_4 noise_index = (int_4)(noise_thresh_a * (float_4)num_values_a);
  
  // loop over all channels
  //
  for (int_4 chan = 0; chan < num_channels_a; chan++) {

    // find the energy references
    //
    float_4 noise_energy = min_a[chan];
    float_4 signal_energy = min_a[chan];
    
    // loop until the index of the noise threshold is found
    //
    int_4 cumulative_index = cdf_a[0][chan];
    int_4 data_bin = (int_4)0;
    
    while (cumulative_index < noise_index) {

      cumulative_index = cdf_a[++data_bin][chan];
    }

    // get the noise energy
    //
    noise_energy = mean_vals_a[data_bin][chan];
    
    // loop until the signal threshold is found
    //
    while (cumulative_index < signal_index) {

      cumulative_index = cdf_a[++data_bin][chan];
    }

    // determine the value of the signal + noise energy
    //
    signal_energy = mean_vals_a[data_bin][chan];
    
    // now determine the snr of this data channel and store it
    //
    snr_values_a[chan] = (float_4)10.0 * log10((signal_energy - noise_energy) /
					       noise_energy);
  }
  
  // exit gracefully
  //
  return ISIP_TRUE;
}

