// file: $ISIP_SNR/util/snr_calculator/v1.0/snr_calculator.cc
//
// this program calculates the snr of a signal using a probabilistic measure
// 

// system include files
//
#include <string.h>

// isip include files
//
#include "snr_calculator.h"
#include <signal.h>

#include "snr_calculator_constants.h"
#include <signal_constants.h>

// main program
//
int main(int argc, char** argv) {

  // declare local variables
  //
  logical_1 use_hamming = SNRC_DEF_USE_HAMMING;	// flag for use of hamming wind
  logical_1 apply_pre_emphasis = SNRC_DEF_PRE_EMPH;
  						// flag for use of pre-emphasis
  float_4 signal_thresh = SNRC_DEF_SIG_THRESH;	// signal threshold for snr
  float_4 noise_thresh = SNRC_DEF_NOISE_THRESH;	// noise threshold for snr
  float_4 frame_dur = SNRC_DEF_FRAME_DUR;	// frame duration in msecs
  float_4 window_dur = SNRC_DEF_WINDOW_DUR;	// window duration in msecs
  float_4 sample_freq = SNRC_DEF_SF;		// sample frequency
  int_4 num_chans = SNRC_DEF_NUM_CHANS;		// number of channels in data
  logical_1 swap_byte = SNRC_DEF_SWAP;		// whether or not to swap bytes
  char_1* input_file = (char_1*)NULL;		// name of input data file
  int_4 debug_mode = ISIP_DEBUG_NONE;		// debugging mode
  
  // allocate memory for the input file name
  //
  input_file = new char_1[ISIP_MAX_FNAME_SIZE];

  // declare an array to hold the energy values for all channels of data
  //
  float_4** energy_array;
  
  // get the command line parameters
  //
  if (get_parameter_cc(argc, argv, use_hamming, apply_pre_emphasis,
		       signal_thresh, noise_thresh, frame_dur, window_dur,
		       sample_freq, num_chans, swap_byte, input_file,
		       debug_mode) == ISIP_FALSE) {

    // check help option here
    //
    if (strcmp((char*)SNRC_OPT_HELP, *argv) == 0) {
      
      // output error message and exit
      //
      fprintf(stderr, "%s : Error reading command line parameters\n",
	      argv[0]);

      exit(ISIP_ERROR);
    }

    else {
      return ISIP_NO_ERROR;
    }
  }

  // check bounds on window and frame durations as well as thresholds and
  // sample frequency
  //
  if ((signal_thresh < (float_4)0.0) || (noise_thresh < (float_4)0.0) ||
      (frame_dur < (float_4)0.0) || (window_dur < (float_4)0.0) ||
      (sample_freq < (float_4)0.0)) {

    // output error message and exit
    //
    fprintf(stderr, "%s : all values must be greater than zero\n",
	    argv[0]);

    exit(ISIP_ERROR);
  }

  // allocate memory for the signal data
  //
  Signal data_sig;

  // set the parameters of the signal object
  //
  data_sig.set_cc(input_file, num_chans, SNRC_NUM_BYTES, sample_freq,
		  swap_byte);

  // open the signal file
  //
  if (data_sig.open_cc(input_file, ISIP_READ_ONLY) == ISIP_FALSE) {

    // output error message and exit
    //
    fprintf(stderr, "%s : Error opening file %s\n",
	    argv[0], input_file);

    exit(ISIP_ERROR);
  }

  // determine the number of samples that will make up a window of data
  // and the number of samples that will make up a frame of data
  //
  int_4 window_num_samples = (int_4)(window_dur * SNRC_MSEC_TO_SEC * sample_freq);
  int_4 frame_num_samples = (int_4)(frame_dur * SNRC_MSEC_TO_SEC * sample_freq);

  // determine the number of frames that will be read in. add some slack factor
  // 2 would be sufficient, but since we are in the middle of debugging, 20
  // is even better.
  //
  int_4 num_frames_to_read = (int_4)(data_sig.get_file_size_cc() /
    (frame_num_samples * num_chans * SNRC_NUM_BYTES)) + (int_4)20;

  // allocate memory for the energy array
  //
  energy_array = new float_4*[num_frames_to_read];

  // allocate memory for each entry in the energy array
  //
  for (int_4 k = 0; k < num_frames_to_read; k++) {
    energy_array[k] = new float_4[num_chans];
  }

  // initialize a counter to keep track of how many energy values have been
  // read
  //
  int_4 num_energy_values = (int_4)0;
  
  // allocate memory for the window buffer
  //
  float_4* window_buffer = new float_4[window_num_samples * num_chans];
  
  // zero out a single window buffer
  //
  memset((void*)window_buffer, (int)0, (window_num_samples * sizeof(float_4) *
	 num_chans));

  // start at sample 0, we will just skip the first bit of data so that we
  // can center the window around the frame of data. thus the window starts
  // at sample 0 while the frame does not
  // 
  int_4 start_sample = (int_4)((frame_num_samples - window_num_samples) /
			       (int_4)2);
  int_4 num_samps_read = (int_4)0;

  // pointer to the window buffer
  //
  float_4* temp_window = window_buffer;
    
  // make sure that we are in positive time.
  //
  if (start_sample < 0) {
    temp_window = window_buffer - start_sample;
  }

  // initialize data loop
  //
  logical_1 continue_loop = data_sig.read_cc(temp_window, start_sample,
					     (start_sample +
					      window_num_samples),
					     num_samps_read);

  // loop until all data has been read
  //
  while (continue_loop == ISIP_TRUE) {

    // preemphasize the data if necessary
    //
    if (apply_pre_emphasis == ISIP_TRUE) {
      pre_emphasize_data_cc(window_buffer, window_num_samples, num_chans);
    }

    // apply the hamming window if necessary
    //
    if (use_hamming == ISIP_TRUE) {
      window_hamming_cc(window_buffer, window_num_samples, num_chans);
    }

    // allocate memory to store the energy values for all channels
    //
    float_4* window_energy_values = new float_4[num_chans];
    
    // get the energy for all channels and store
    //
    compute_energy_cc(window_buffer, window_num_samples, num_chans,
		      window_energy_values);

    // add this to the energy array
    //
    for (int_4 j = 0; j < num_chans; j++) {
      energy_array[num_energy_values][j] = window_energy_values[j];
    }

    // increment the number of energy values present
    //
    num_energy_values++;
    
    // delete memory for the energy values
    //
    delete [] window_energy_values;
    
    // reset the window data to zeros, we do this because if we don't have
    // enough data to fill the window, we are assuming that it is zero outside
    // of the window
    //
    memset((void*)window_buffer, (int)0, (window_num_samples *
					  sizeof(float_4) * num_chans));

    // increment the start of the window
    //
    start_sample += frame_num_samples;

    // pointer to the window buffer
    //
    temp_window = window_buffer;
    
    // make sure that we are in positive time.
    //
    if (start_sample < 0) {
      temp_window = window_buffer - start_sample;
    }

    // data loop
    //
    continue_loop = data_sig.read_cc(temp_window, start_sample,
				     (start_sample + window_num_samples),
				     num_samps_read);
    
  } // end of the while loop

  // debugging output
  //
  if (debug_mode > ISIP_DEBUG_NONE) {
    // loop over the data and output the energy data
    //
    for (int_4 j = 0; j < num_chans; j++) {
      for (int_4 i = 0; i < num_energy_values; i++) {
	fprintf(stdout, "chan %ld :: energy :: %.6f\n", j,
		energy_array[i][j]);
      }
    }
  }
  
  // allocate memory to hold the maximums and minimums of each energy
  // distribution
  //
  float_4* max_energy = new float_4[num_chans];
  float_4* min_energy = new float_4[num_chans];  
  
  // find the minimum and maximum energy levels
  //
  find_max_min_cc(energy_array, max_energy, min_energy, num_energy_values,
		  num_chans);

  // allocate memory for the energy pdf's of each channel and the mean values
  // of each bin
  //
  int_4** pdf = new int_4*[SNRC_NUM_BINS];
  float_4** mean_vals = new float_4*[SNRC_NUM_BINS];  
  for (int_4 jj = 0; jj < SNRC_NUM_BINS; jj++) {
    pdf[jj] = new int_4[num_chans];
    mean_vals[jj] = new float_4[num_chans];
    for (int_4 kk = 0; kk < num_chans; kk++) {
      mean_vals[jj][kk] = (float_4)0.0;
    }
  }

  // create the pdf for each channel
  //
  compute_pdf_cc(energy_array, pdf, mean_vals, max_energy, min_energy,
		 num_energy_values, num_chans);

  // debugging output
  //
  if (debug_mode > ISIP_DEBUG_NONE) {
    // loop over the data and output the pdf data
    //
    for (int_4 j = 0; j < num_chans; j++) {
      for (int_4 i = 0; i < SNRC_NUM_BINS; i++) {
	fprintf(stdout, "chan %ld :: pdf_bins :: %.6f  %ld\n", j,
		mean_vals[i][j], pdf[i][j]);
      }
    }
  }
  
  // allocate memory for the energy cdf's of each channel
  //
  int_4** cdf = new int_4*[SNRC_NUM_BINS];
  for (int_4 jj = 0; jj < SNRC_NUM_BINS; jj++) {
    cdf[jj] = new int_4[num_chans];
  }
  
  // compute the cdf of the data
  //
  // loop over all channels
  //
  for (int_4 chan = 0; chan < num_chans; chan++) {

    // initialize the first level in the cdf
    //
    cdf[0][chan] = pdf[0][chan];

    // loop over all bins
    //
    for (int_4 bin = 1; bin < SNRC_NUM_BINS; bin++) {
      cdf[bin][chan] = pdf[bin][chan] + cdf[bin - (int_4)1][chan];
    }
  }

  // debugging output
  //
  if (debug_mode > ISIP_DEBUG_NONE) {
    // loop over the data and output the cdf data
    //
    for (int_4 j = 0; j < num_chans; j++) {
      for (int_4 i = 0; i < SNRC_NUM_BINS; i++) {
	fprintf(stdout, "chan %ld :: cdf_bins :: %.6f  %ld\n", j,
		mean_vals[i][j], cdf[i][j]);
      }
    }
  }

  // allocate memory for the snr values
  //
  float_4* snr_values = new float_4[num_chans];
  
  // compute the snr for each channel
  //
  compute_snr_cc(cdf, signal_thresh, noise_thresh, num_energy_values,
		 num_chans, max_energy, min_energy, mean_vals, snr_values);

  // output the snr values for each channel
  //
  for (int_4 i = 0; i < num_chans; i++) {
    fprintf(stdout, "File: %s\n", input_file);
    fprintf(stdout, "\t snr for channel %ld: %.6f dB\n", i, snr_values[i]);
  }
  // close the signal file
  //
  data_sig.close_cc();
  
  // clean up
  //
  delete [] input_file;
  delete [] window_buffer;
  delete [] max_energy;
  delete [] min_energy;
  
  // deallocate memory for each entry in the energy array
  //
  for (int_4 k = 0; k < num_frames_to_read; k++) {
    delete [] energy_array[k];
  }

  // deallocate memory for the energy array
  //
  delete [] energy_array;

  // deallocate memory for the energy pdf's of each channel
  //
  for (int_4 jj = 0; jj < SNRC_NUM_BINS; jj++) {
    delete [] pdf[jj];
  }

  delete [] pdf;

  // deallocate memory for the energy means of each channel
  //
  for (int_4 jj = 0; jj < SNRC_NUM_BINS; jj++) {
    delete [] mean_vals[jj];
  }

  delete [] mean_vals;

  // deallocate memory for the energy cdf's of each channel
  //
  for (int_4 jj = 0; jj < SNRC_NUM_BINS; jj++) {
    delete [] cdf[jj];
  }

  delete [] cdf;  
  
  // return without error
  //
  return ISIP_NO_ERROR;
}



