#!/usr/local/bin/perl
#
# file: compute_posterior.pl
#
$ISIP_HELP_FILE = <<__ISIP_HELP_HERE_FILE__;
name: compute_posterior
synopsis: compute_posterior -input_lattice <input_file> -output_lattice <output_file> -lm_scale <value>  -word_penalty <value> 

descr: this program reads the lattice(s) in ISIP and BBN (HTK style) format and computes posterior of the links. the command line value of language model scale and word insertion penalty overrides the values provided in lattices if any

example: compute_posterior -input_lattice input.lat -output_lattice output.lat

options: 
 -input_lattice <input_file> : this option tells the script the location and name of the input lattice file

 -output_lattice <output_file> : this option tells the script the location and name of the output lattice file

 -lm_scale <value> : this option specifies the weight for the langauge model score (default is 1.0) which is used to scale down the acoustic score

 -word_penalty <value> : this option specifies the penalty applied to each word that is hypothesized (default is 0.0)

arguments: 

man page: none
__ISIP_HELP_HERE_FILE__

# system packages
#
use File::Basename;
use File::Path;
use File::Copy;
use FileHandle;
use strict;
autoflush STDOUT;

# CPAN modules
#
use Graph::Directed;

# ISIP packages
#
use command_line;
use lattice_subs;

#----------------------------------------------------------------------------
#
# constant and variable definitions
#
#----------------------------------------------------------------------------

# define the program name
#
my @tmp = split(/\//, $0);
my $ISIP_PROG = pop(@tmp);

# define a separator for output comments
#
my $TI_LINE_SEP = "-------------------  $ISIP_PROG  ----------------";

# define command-line options
#
my $TI_OPT_INPUT_LAT_FILE_TYPE= "-input_lattice";
my $TI_OPT_OUTPUT_LAT_FILE_TYPE= "-output_lattice";
my $TI_OPT_LM_SCALE = "-lm_scale";
my $TI_OPT_WORD_PENALTY = "-word_penalty";

# define vertex attributes
#
my @ATTR_NAME_VERTEX = ("time");
push(@ATTR_NAME_VERTEX, "alpha");
push(@ATTR_NAME_VERTEX, "beta");

# define edge attributes
#
my @ATTR_NAME_EDGE = ("number");
push(@ATTR_NAME_EDGE, "word");
push(@ATTR_NAME_EDGE, "pronunciation_version");
push(@ATTR_NAME_EDGE, "acoustic_score");
push(@ATTR_NAME_EDGE, "language_model_score");
push(@ATTR_NAME_EDGE, "posterior_score");

# define defaults
#
my $TI_LM_SCALE_DEF = "1.0";
my $TI_WORD_PENALTY_DEF = "0.0";

#----------------------------------------------------------------------------
#
# command line processing
#
#----------------------------------------------------------------------------
# print a debugging message
#
print "$TI_LINE_SEP\n";
print "INFO: compute_posterior - checking command-line arguments\n\n";

# parse the command line
#
my $input_lat_file;
my $output_lat_file;
my $lm_scale_cl;
my $wd_penalty_cl;

my ($input_lat_file, $output_lat_file, $lm_scale_cl, $wd_penalty_cl) = 
    command_line(0, $TI_OPT_INPUT_LAT_FILE_TYPE , 1, 
		 $TI_OPT_OUTPUT_LAT_FILE_TYPE, 1, $TI_OPT_LM_SCALE, 1, 
		 $TI_OPT_WORD_PENALTY, 1);

# verify the required options and arguments
#
if ($input_lat_file eq "") {
    isip_die ("ERROR: input lattice file not provided\n ERROR: provide the input lattice file through $TI_OPT_INPUT_LAT_FILE_TYPE option");
}

if ($output_lat_file eq "") {
    isip_die ("ERROR: output lattice file not provided\n ERROR: provide the output lattice file through $TI_OPT_OUTPUT_LAT_FILE_TYPE option");
}

if ($lm_scale_cl eq "") {
    $lm_scale_cl = $TI_LM_SCALE_DEF;
}

if ($wd_penalty_cl eq "") {
    $wd_penalty_cl = $TI_WORD_PENALTY_DEF;
}

print "INFO: compute_posterior - running with parameters: \n";
my $out_line = join (" ", $TI_OPT_INPUT_LAT_FILE_TYPE, $input_lat_file, 
		     $TI_OPT_OUTPUT_LAT_FILE_TYPE, $output_lat_file, 
		     $TI_OPT_LM_SCALE, $lm_scale_cl,
		     $TI_OPT_WORD_PENALTY, $wd_penalty_cl);
print "$out_line\n";

#----------------------------------------------------------------------------
#
# lattice processing
#
#----------------------------------------------------------------------------

# local variables
#
my $g = new Graph::Directed->new;

# delete if the output file already exists
#
(delete_file($output_lat_file) == 1) 
    or isip_die("ERROR: failed delete the file $output_lat_file");

# read the input lattice
#
my $id;
my $lm_scale = $TI_LM_SCALE_DEF;
my $wd_penalty = $TI_WORD_PENALTY_DEF;
my $cur_pos = 0;
my $next_pos = 0;

while (read_lattice($input_lat_file, \$g, \@ATTR_NAME_VERTEX, 
		    \@ATTR_NAME_EDGE, $cur_pos, \$next_pos, \$id, \$lm_scale,
		    \$wd_penalty) == 1) {
    
    # override the lm_scale and wd_penalty values if required
    #
    if ($lm_scale_cl ne "") {
	$lm_scale = $lm_scale_cl;
    }
    if ($wd_penalty_cl ne "") {
	$wd_penalty = $wd_penalty_cl;
    }
    
    # display information
    #
    print "\nINFO: compute_posterior - lm_scale is set to $lm_scale\n";
    print "\nINFO: compute_posterior - wd_penalty is set to $wd_penalty\n";

    # compute forward probability
    #
    (compute_alpha(\$g, \@ATTR_NAME_VERTEX, \@ATTR_NAME_EDGE, \$id, 
		   \$lm_scale, \$wd_penalty) == 1) 
	or isip_die("ERROR: failed to compute foward computations");
    
    # compute backward probability
    #
    (compute_beta(\$g, \@ATTR_NAME_VERTEX, \@ATTR_NAME_EDGE, \$id, 
		  \$lm_scale, \$wd_penalty) == 1) 
	or isip_die("ERROR: failed to compute backward computations");
    
    # compute posterior probability
    #
    (compute_posterior(\$g, \@ATTR_NAME_VERTEX, \@ATTR_NAME_EDGE, \$id, 
		       \$lm_scale, \$wd_penalty) == 1) 
	or isip_die("ERROR: failed to compute posterior probabilities");
    
    # write the output lattice
    #
    (write_lattice($output_lat_file, \$g, \@ATTR_NAME_VERTEX, 
		   \@ATTR_NAME_EDGE, \$id, \$lm_scale, \$wd_penalty) == 1) 
	or isip_die("ERROR: failed to write the output lattice"); 
    
    # update the positions
    #
    $cur_pos = $next_pos;
    $id = "";
    $lm_scale = $TI_LM_SCALE_DEF;
    $wd_penalty = $TI_WORD_PENALTY_DEF;
    
    # clear the graph
    #
    undef $g;
    $g = new Graph::Directed->new;
}

# processing for the last lattice. note that the while loop above
# quits as soon as it finishes reading the last lattice because the
# read_lattice function returns 0 as soon as it sees an EOF
#

# override the lm_scale and wd_penalty values if required
#
if ($lm_scale_cl ne "") {
    $lm_scale = $lm_scale_cl;
}
if ($wd_penalty_cl ne "") {
    $wd_penalty = $wd_penalty_cl;
}

# display information
#
print "\nINFO: compute_posterior - lm_scale is set to $lm_scale\n";
print "\nINFO: compute_posterior - wd_penalty is set to $wd_penalty\n";

# compute forward probability
#
(compute_alpha( \$g, \@ATTR_NAME_VERTEX, \@ATTR_NAME_EDGE, \$id, \$lm_scale, 
		\$wd_penalty) == 1) or
    isip_die("ERROR: failed to compute foward computations");

# compute backward probability
#
(compute_beta( \$g, \@ATTR_NAME_VERTEX, \@ATTR_NAME_EDGE, \$id, \$lm_scale, 
	       \$wd_penalty) == 1) or
    isip_die("ERROR: failed to compute backward computations");

# compute posterior probability
#
(compute_posterior( \$g, \@ATTR_NAME_VERTEX, \@ATTR_NAME_EDGE, \$id, 
		    \$lm_scale, \$wd_penalty) == 1) or
    isip_die("ERROR: failed to compute posterior probabilities");

# write the last lattice
#
(write_lattice($output_lat_file, \$g, \@ATTR_NAME_VERTEX, \@ATTR_NAME_EDGE, 
	       \$id, \$lm_scale, \$wd_penalty) == 1) 
    or isip_die("ERROR: failed to write the output lattice"); 

#----------------------------------------------------------------------------
#
# finished
#
#----------------------------------------------------------------------------
print "\nINFO: compute_posterior - computation complete\n";
print "$TI_LINE_SEP\n";

# return no error
#
exit 0;
