#!/usr/local/bin/perl
#
# file: compute_lattice_error_rate.pl
#
$ISIP_HELP_FILE = <<__ISIP_HELP_HERE_FILE__;
name: compute_lattice_error_rate
synopsis: compute_lattice_error_rate -input_lattice_list <input_file> -input_transcription_file <input_file> -output_error_rate_file <output_file>

descr: this program reads the lattice(s) in ISIP and BBN (HTK style)
format and computes lattice error rate over all the lattices.

example: compute_lattice_error_rate -input_lattice input.list -input_transcription_file trans.text -output_error_rate_file lattice_wer.text

options: 
 -input_lattice_list <input_file> : this option tells the script the location and names of the input lattices list

 -input_transcription_file <input_file> : this option tells the script the location of a file containing all the reference transcriptions

 -output_error_rate_file <output_file> : this option tells the script the location and name of the output lattice error rate file

arguments: 

man page: none
__ISIP_HELP_HERE_FILE__

# system packages
#
use File::Basename;
use File::Path;
use File::Copy;
use FileHandle;
use strict;
autoflush STDOUT;

# CPAN modules
#
use Graph::Directed;

# ISIP packages
#
use command_line;
use lattice_subs;

#----------------------------------------------------------------------------
#
# constant and variable definitions
#
#----------------------------------------------------------------------------

# define the program name
#
my @tmp = split(/\//, $0);
my $ISIP_PROG = pop(@tmp);

# define a separator for output comments
#
my $TI_LINE_SEP = "-------------------  $ISIP_PROG  ----------------";

# define command-line options
#
my $TI_OPT_INPUT_LAT_FILE_LIST_TYPE= "-input_lattice_list";
my $TI_OPT_INPUT_TRANS_FILE_TYPE= "-input_transcription_file";
my $TI_OPT_OUTPUT_ERROR_RATE_FILE_TYPE= "-output_error_rate_file";

# define vertex attributes
#
my @ATTR_NAME_VERTEX = ("time");
push(@ATTR_NAME_VERTEX, "utterance_id");

# define edge attributes
#
my @ATTR_NAME_EDGE = ("number");
push(@ATTR_NAME_EDGE, "word");
push(@ATTR_NAME_EDGE, "pronunciation_version");
push(@ATTR_NAME_EDGE, "acoustic_score");
push(@ATTR_NAME_EDGE, "language_model_score");

#----------------------------------------------------------------------------
#
# command line processing
#
#----------------------------------------------------------------------------

# print a debugging message
#
print "$TI_LINE_SEP\n";
print "INFO: compute_lattice_error_rate - checking command-line arguments\n\n";

# parse the command line
#
my $input_lat_file_list;
my $input_trans_file; 
my $output_error_rate_file;

my ($input_lat_file_list, $input_trans_file, $output_error_rate_file) = 
    command_line(0, $TI_OPT_INPUT_LAT_FILE_LIST_TYPE , 1, $TI_OPT_INPUT_TRANS_FILE_TYPE , 1, $TI_OPT_OUTPUT_ERROR_RATE_FILE_TYPE, 1);

# verify the required options and arguments
#
if ($input_lat_file_list eq "") {
    isip_die ("ERROR: input lattice file list not provided\n ERROR: provide the input lattice file list through $TI_OPT_INPUT_LAT_FILE_LIST_TYPE option");
}

if ($input_trans_file eq "") {
    isip_die ("ERROR: input transcription file not provided\n ERROR: provide the input transcription file through $TI_OPT_INPUT_TRANS_FILE_TYPE option");
}

if ($output_error_rate_file eq "") {
    isip_die ("ERROR: output error rate file not provided\n ERROR: provide the output lattice error rate file through $TI_OPT_OUTPUT_ERROR_RATE_FILE_TYPE option");
}

print "INFO: compute_lattice_error_rate - running with parameters: \n";
my $out_line = join (" ", $TI_OPT_INPUT_LAT_FILE_LIST_TYPE, $input_lat_file_list, $TI_OPT_INPUT_TRANS_FILE_TYPE, $input_trans_file, $TI_OPT_OUTPUT_ERROR_RATE_FILE_TYPE, $output_error_rate_file);
print "$out_line\n";

#----------------------------------------------------------------------------
#
# lattice processing
#
#----------------------------------------------------------------------------

# local variables
#
my @transcriptions;
my @alignments;
my @corrs;
my @subs;
my @dels;
my @ins;
my @file_ids;

# read the transcriptions
#
(read_file_as_lines($input_trans_file, \@transcriptions) == 1) or
    isip_die("ERROR: compute_lattice_error_rate - failed to read the transcriptions");

# compute error rate for the lattices
#
(compute_error_rate($input_lat_file_list, \@transcriptions, \@ATTR_NAME_VERTEX, \@ATTR_NAME_EDGE, \@alignments, \@corrs, \@subs, \@dels, \@ins, \@file_ids) == 1) or
    isip_die("ERROR: compute_lattice_error_rate - failed to compute foward computations");

# write the report to the error rate file
#
(write_error_rate($output_error_rate_file, \@transcriptions, \@alignments, \@corrs, \@subs, \@dels, \@ins, \@file_ids) == 1) or
    isip_die("ERROR: failed to write the output error rate file"); 

#----------------------------------------------------------------------------
#
# finished
#
#----------------------------------------------------------------------------
print "\nINFO: compute_lattice_error_rate - error rate computations complete\n";
print "$TI_LINE_SEP\n";

# return no error
#
exit 0;
