#!/usr/bin/env python

# file: $(NEDC_NFC)/src/python/class/nedc_edf_tools/nedc_edf_tools.py
#                                                                              
# revision history:                                                            
#
# 20230123 (ML): initial version                                              
#
# This file contains a Python implementation of a downsampler meant to work
# with edf file signals
#
# The API is very simple:
#  constructor: creates the class (called at the top of the program)
#  downsample: downsamples the signal from the old sample rate to the new
#              sample rate (should be called after init)
#
# This downsampler rounds the decimation factor (ratio of old sample rate to
# new sample rate) to an integer so the decimation is not always exact
#------------------------------------------------------------------------------

# import required system modules                                               
#
from collections import OrderedDict
import numpy as np
import os
import re
import scipy.signal as signal
import sys

# import NEDC modules
#
import nedc_debug_tools as ndt
import nedc_file_tools as nft

#------------------------------------------------------------------------------
#                                                                              
# global variables are listed here
#                                                                              
#------------------------------------------------------------------------------

# set the filename using basename
#
__FILE__ = os.path.basename(__file__)

# define variables needed for downsampling
#
EDF_DEF_FTYPE = 'fir'
EDF_DEF_AXIS = 1
EDF_DEF_ZPHASE = True
EDF_DEF_DOWSAMPLE_FILTER_ORDER = 16

#------------------------------------------------------------------------------
#
# classes are listed here
#
#------------------------------------------------------------------------------

# class: Edf_Downsample
#
# This class will downsample a signal using an integer rounded decimation
# factor
#
class Edf_Downsample:

    # define static variables for debug and verbosity
    #
    dbgl_d = ndt.Dbgl()
    vrbl_d = ndt.Vrbl()

    # method: Edf_Dowsample::constructor
    #
    # arguments: none
    #
    # returns: none
    #
    def __init__(self):

        # set the class name
        #
        Edf_Downsample.__CLASS_NAME__ = self.__class__.__name__

    #
    # end of method

    # method: Edf_Downsample::downsample
    #
    # arguments:
    #  sig: the input signal matrix
    #  old_samp_rate: old sample rate
    #  new_samp_rate: new sample rate
    #  filter_order: filter order for low pass filter before decimation
    #
    # return: the downsampled signal (rows of the matrix are channels and
    #         columns are samples)
    #
    # this method uses a python library to downsample the signal.
    #
    def downsample(self, sig, old_samp_rate, new_samp_rate,
                   filter_order = EDF_DEF_DOWSAMPLE_FILTER_ORDER):

        # compute the decimation factor 
        #
        dfactor = int(old_samp_rate / new_samp_rate)

        # exit gracefully
        #
        return signal.decimate(x = sig, q = dfactor, n = filter_order,
                               ftype = EDF_DEF_FTYPE, axis = EDF_DEF_AXIS,
                               zero_phase = EDF_DEF_ZPHASE)
              
    #
    # end of method

#
# end of class

#
# end of file
