#!/usr/bin/env python

# file: $(NEDC_NFC)/src/classes/sigplots/demo_sigplots_page_mixed_view.py
#
# This file contains some useful Python functions and classes that are used
# in the nedc scripts.
#
# Revision History
#
# 20230627 (AB): Refactored code to new comment format
#
#------------------------------------------------------------------------------
from pyqtgraph import QtGui, QtCore
import pyqtgraph as pg

from collections import OrderedDict

from .demo_time_axis import DemoTimeAxis
from .demo_sigplot_waveform import DemoWaveform
from .demo_sigplot_spectrogram import DemoSpectrogram
from .demo_sigplot_energy import DemoEnergy

class DemoSigplotsPageMixedView(QtGui.QWidget):
    """
    Class: DemoSigplotsPageMixedView

    arguments:
     NOTE: all arguments passed are used to set the classes
     corresponding internal data
     -montage_names_a: dict with numbered keys and montage name string values
       these name-strings will be used for:
          1) as keys to self.channel_dict
          2) as names for the generated objects
           (ie channel_being_setup.setObjectName.("sometext", <name_sting>))
     -number_signals_a: how many signals are in montage file, maps to
      how many channels should be created
     -cfg_dict_spectrogram

    returns: 
     None
    
    description:
     this method sets up the channel objects which will display the
     various signal plots (ie waveform, spectrogram, energy, etc.)
     
     it also sets up dictionaries that will allow the programmer to
     address all objects of a particular type
     (ie dict_energy_plots, dict_channel_splitters, etc.)
    """

    def __init__(self,
                 montage_names_a=None,
                 number_signals_a=None,
                 time_scale_a=None,
                 cfg_dict_waveform_a=None,
                 cfg_dict_spectrogram_a=None,
                 cfg_dict_energy_a=None,
                 dict_order_a=None):
        QtGui.QWidget.__init__(self)

        # initialize the component layouts and scroll area
        # TODO: is this where are sizing problem is introduced?
        #
        self._init_layout()
        self._init_scroll_area()

        self.layout.addWidget(self.scroll_area)
        
        # add scroll_area to mixed view page and add page to stacked widget
        #
        self.scroll_area.setWidget(self.scroll_area_contents)

        self.montage_names = montage_names_a
        self.number_signals = number_signals_a

        # initialize dictionary for channel objects
        # this also serves to clear the dictionary on later edf loads
        #
        self.dict_channels = OrderedDict()

        # initialize dictionary of form layouts
        # this also serves to clear the dictionary on later edf loads
        #
        self.dict_form_layouts = {}

        # initialize dictionary of channel labels
        # this also serves to clear the dictionary on later edf loads
        #
        self.dict_channel_labels = {}

        # initialize dictionary of channel splitters
        # this also serves to clear the dictionary on later edf loads
        #
        self.dict_channel_splitters = {}

        # initialize one dictionary each for each kind of signal view,
        # as well as one master dictionary of all sigplots
        # this also serves to clear the dictionary on later edf loads
        #
        self.dict_waveform_plots = OrderedDict()
        self.dict_spectrograms = OrderedDict()
        self.dict_energy_plots = OrderedDict()
        self.dict_sigplots_all = {}

        self.set_view_items(dict_order_a)

        # loop to create each channel
        #
        for i in range(self.number_signals):

            # read the channel name from the montage names generated by
            # montage reader
            #
            channel_name = self.montage_names[i]

            # initialize channel dictionary as part of scrollAreaWidgetContents
            #
            channel_being_setup = DemoMixedViewSingleChannel(
                self.scroll_area_contents,
                time_scale_a,
                channel_name,
                cfg_dict_waveform_a,
                cfg_dict_spectrogram_a,
                cfg_dict_energy_a,
                dict_order_a,
                self.view_items)

            # add channel to channel dictionary
            #
            self.dict_channels[channel_name] = channel_being_setup

            self.dict_form_layouts[channel_name] = channel_being_setup.layout

            self.dict_channel_splitters[
                channel_name] = channel_being_setup.splitter

            self.dict_channel_labels[channel_name] = channel_being_setup.label

            # add plot to dict of all waveform plots and dict of all sigplots
            # this should make it easy to address these plots programmatically
            #
            self.dict_waveform_plots[channel_name] =                    \
                self.dict_sigplots_all["waveform_" + channel_name] =    \
                    channel_being_setup.waveform_plot

            self.dict_spectrograms[channel_name] =                      \
                self.dict_sigplots_all["spectrogram_" + channel_name] = \
                    channel_being_setup.spectrogram

            self.dict_energy_plots[channel_name] =                      \
                self.dict_sigplots_all["energy_" + channel_name] =      \
                    channel_being_setup.energy_plot

            # add the channel to the layout for the scroll area of the
            # mixed view page
            #
            self.scroll_area_layout.addWidget(
                channel_being_setup)

            channel_being_setup.label.setText(channel_name)
        #
        # end of for
    #
    # end of function

    def set_sampling_rate(self,
                          sampling_rate_a):
        for spectrogram in self.dict_spectrograms.values():
            spectrogram.set_sampling_rate(sampling_rate_a)
        for plot in self.dict_energy_plots.values():
            plot.set_sampling_rate(sampling_rate_a)

    def set_view_items(self,
                       dict_order_a):
        view_items_string = dict_order_a['view_items']
        self.view_items = view_items_string.replace(" ","").split(",")
    def _init_layout(self):

        # initialize and name the layout
        #
        self.layout = QtGui.QVBoxLayout(self)

        # set some layout parameters to minimize whitespace
        #
        self.layout.setContentsMargins(0, 0, 0, 0)
        self.layout.setSpacing(10) # TODO: TEST, had been 0
    #
    # end of function

    def _init_scroll_area(self):

        # initialize and name the scroll area
        #
        self.scroll_area = QtGui.QScrollArea(self)

        # set some stylistic parameters
        #
        self.scroll_area.setFrameShape(QtGui.QFrame.NoFrame)
        self.scroll_area.setFrameShadow(QtGui.QFrame.Sunken)

        # use the ability of layouts to get automatic resizing functionality
        #
        self.scroll_area.setWidgetResizable(True)

        # initialize and name the scroll area contents
        #
        self.scroll_area_contents = QtGui.QWidget()

        # set the geometry of the scroll area
        # TODO: test some different numbers
        #
        self.scroll_area_contents.setGeometry(
            QtCore.QRect(0, 0, 1056, 1916))

        self.scroll_area_layout = QtGui.QVBoxLayout(
            self.
            scroll_area_contents)
    #
    # end of function  

    def show_or_hide_energy_plots(self, show=True):
        """
        method: show_or_hide_energy
        
        arguments:
         - show: boolean specifying of plot is shown or hidden

        return: 
         none

        description: 
         this method iterates over dictionay of energy plots generated in
         page_mixed_view_channels_setup() method and calls each's show()
         or hide() method depending on the boolean argument <show>
        """

        # iterate over the dictionary of energy plots set up at the
        # creation of the channel widgets
        #
        for plot in self.dict_energy_plots.values():

            if show is True:
                plot.show()
                plot.do_plot()
                plot.repaint()
            else:
                plot.hide()
        #
        # end of for
    #
    # end of function

    def show_or_hide_waveform_plots(self, show=True):
        """
        method: show_or_hide_waveform_plots
        
        arguments:
         - show: boolean specifying of plot is shown or hidden

        return: 
         none

        description: 
         this method iterates over dictionay of waveform plots generated
         in page_mixed_view_channels_setup() method and calls each's
         show() or hide() method depending on the boolean argument <show>
         
         NOTE: this is for the multi-view general case, not the special case of
         waveform-exclusive view
        """

        # iterate over the dictionary of waveform plots set up the
        # creation of the channel widgets
        #
        for plot in self.dict_waveform_plots.values():
            if show is True:
                plot.do_plot()
                plot.show()
                plot.repaint()
            else:
                plot.hide()
        #
        # end of for
    #
    # end of function

    def show_or_hide_spectrogram(self, show=True):
        """
        method: page_mixed_view_sigplots_show_or_hide_spectrogram
        
        arguments:
         - show: boolean specifying of plot is shown or hidden

        return: 
         none

        description: 
         this method iterates over dictionay of spectrogram plots
         generated in page_mixed_view_channels_setup() method and calls
         each's show() or hide() method depending on the boolean argument
         <show>
         
         NOTE: this is for the multi-view general case, not the special case of
         waveform-exclusive view
        """

        # iterate over the dictionary of spectrogram plots set up the
        # creation of the channel widgets
        #
        for plot in self.dict_spectrograms.values():
            if show is True:
                plot.do_plot()
                plot.show()
                plot.update_image_windowing()
                plot.repaint()
            else:
                plot.hide()
        #
        # end of for
    #
    # end of function

    def spec_frequency_changed(self,
                               spec_low_freq_a,
                               spec_high_freq_a):
        """
        method: spec_frequency_changed
        
        arguments:
         none

        return: 
         none

        description: 
         this method gets the frequency range for the spectrogram from the
         pertinent dropdown menues and communicates this data to each of the
         spectrogram plots
        """

        for plot in self.dict_spectrograms.values():
            plot.set_freq_range(spec_low_freq_a, spec_high_freq_a)
            plot.chop_image_array_for_frequency_and_set_image()


class DemoMixedViewSingleChannel(QtGui.QFrame):
    def __init__(self,
                 parent=None,
                 time_scale_a=None,
                 name_a=None,
                 cfg_dict_waveform_a=None,
                 cfg_dict_spectrogram_a=None,
                 cfg_dict_energy_a=None,
                 dict_order_a=None,
                 view_items=None):

        QtGui.QFrame.__init__(self, parent=parent)

        self.time_scale = time_scale_a

        self.channel_name = name_a

        # set the name of the channel object with Qt conventions
        #
        self.setObjectName("channel_{}".format(name_a))

        # intialize form layout, add it dictionary, and set its object name
        # layouts allow the widgets to resize as the window is resized
        #
        self.layout = QtGui.QFormLayout(self)

        # fixes a mac (OSX) problem where plots do not fill horizontally.
        #
        # !!!! please don't mess with unless you have good reason. !!!!
        #
        self.layout.setFieldGrowthPolicy(
            QtGui.QFormLayout.AllNonFixedFieldsGrow)

        # create channel_label_object add it to dictionary, set object name
        #
        self.label = QtGui.QLabel(self)

        self.label.setMinimumSize(QtCore.QSize(50, 0))

        self.layout.setWidget(0,
                              QtGui.QFormLayout.LabelRole,
                              self.label)

        # some stylistic setup functions
        #
        self.setFrameShape(QtGui.QFrame.StyledPanel)
        self.setFrameShadow(QtGui.QFrame.Raised)
        self.setMinimumSize(QtCore.QSize(50, 0))

        # create and orient splitter for channel being set up
        # and then add it to dictionary
        # the splitter is what allows for multiple (split) views
        #
        self.splitter = QtGui.QSplitter(self)
        self.splitter.setOrientation(QtCore.Qt.Vertical)

        # add splitter to layout
        #
        self.layout.setWidget(0,
                              QtGui.QFormLayout.FieldRole,
                              self.splitter)

        # initialize split_view_plots_dict for given channel
        #
        self.view_dict = {}

        self.set_order_of_views(cfg_dict_spectrogram_a,
                                cfg_dict_energy_a,
                                cfg_dict_waveform_a,
                                time_scale_a,
                                view_items,
                                dict_order_a)

    def setup_waveform_plot(self,
                            cfg_dict_energy_a,
                            time_scale_a=None):
        """
        method: setup_waveform
        
        arguments:
         -channel_being_setup_a: the channel widget for the plot is being setup
         -channel_name_a: the name of the channel widget object
          (retrievable from channel widget object via objectName() method)

        return: 
         none

        description: 
         this method sets up a single channel's waveform plot by:
         -creating it using the DemoWaveform class
         -adding it to the channel's splitter
         -adding it to the channel's dictionary of views
         -adding it the  overall dictionary of waveform view plots
         -setting the name of the object
         -does some stylistic formatting
        
         NOTE: this setup is for the general case of the mixed view,
          not the special case of the exclusively waveform view
        """

        axis = DemoTimeAxis(orientation='bottom')

        # create view and add to the channel's splitter
        #
        self.waveform_plot = DemoWaveform(cfg_dict_energy_a,
                                          time_scale_a,
                                          axisItems={'bottom': axis})

        self.splitter.addWidget(self.waveform_plot)

        # add to channels'view dictionary
        #
        self.view_dict['waveform'] = self.waveform_plot

        # show grids on signal plot widget.
        #
        p = pg.mkPen((30, 30, 10), width=1, style=QtCore.Qt.SolidLine)
        self.waveform_plot.getAxis('bottom').setPen(p)
        self.waveform_plot.showGrid(x=True, y=True, alpha=250)
    #
    # end of function

    def setup_spectrogram(self,
                          cfg_dict_spectrogram_a,
                          time_scale_a=None):
        """
        method: setup_spectrogram
        
        arguments:
         none

        return: 
         none

        description: 
         this method sets up a single channel's spectrogram plot by:
         -creating it using the DemoSpectrogram class
         -adding it to the channel's splitter
         -adding it to the channel's dictionary of views
         -adding it the  overall dictionary of spectrogram view plots
         -setting the name of the object
        """

        # create plot and add it to the channels splitter
        #
        self.spectrogram = DemoSpectrogram(cfg_dict_spectrogram_a,
                                           time_scale_a)

        self.splitter.addWidget(self.spectrogram)

        # when the user resizes the various widgets using the splitter handle,
        # the spectrogram should update its image "windowing"
        #
        self.splitter.splitterMoved.connect(
            self.spectrogram.update_image_windowing)

        # add to channel's view dictionary
        #
        self.view_dict['spectrogram'] = self.spectrogram
    #
    # end of function

    def setup_energy_plot(self,
                          cfg_dict_energy_a,
                          time_scale_a=None):
        """
        method: page_mixed_view_sigplots_setup_energy
        
        arguments:
         -channel_being_setup_a: the channel widget for the plot is being setup
         -channel_name_a: the name of the channel widget object
          (retrievable from channel widget object via objectName() method)

        return: 
         none

        description: 
         this method sets up a single channel's energy plot by:
         -creating it using the DemoEnergy class
         -adding it to the channel's splitter
         -adding it to the channel's dictionary of views
         -adding it the  overall dictionary of energy view plots
         -setting the name of the object
        """

        axis = DemoTimeAxis(orientation='bottom')

        # create plot and add it to the channels splitter
        #
        self.energy_plot = DemoEnergy(cfg_dict_energy_a,
                                      axisItems={'bottom': axis})

        self.splitter.addWidget(self.energy_plot)

        # add to the channel's view dictionary
        #
        self.view_dict['energy'] = self.energy_plot

        p = pg.mkPen((30, 30, 10), width=1, style=QtCore.Qt.SolidLine)
        self.energy_plot.getAxis('bottom').setPen(p)
        self.energy_plot.showGrid(x=True, y=True, alpha=250)
    #
    # end of function

    def set_signal_data(self,
                        t_data_a,
                        y_data_a):

        self.waveform_plot.set_signal_data(t_data_a,
                                           y_data_a)
        self.spectrogram.set_signal_data(y_data_a)
        self.energy_plot.set_signal_data(t_data_a,
                                         y_data_a)

    def set_order_of_views(self,
                           cfg_dict_spectrogram_a,
                           cfg_dict_energy_a,
                           cfg_dict_waveform_a,
                           time_scale_a,
                           view_items,
                           dict_order_a=None):

        for view in view_items:
            if view == 'Energy':
                self.setup_energy_plot(cfg_dict_energy_a,
                                       time_scale_a)
            if view == 'Spectrogram':
                self.setup_spectrogram(cfg_dict_spectrogram_a,
                                       time_scale_a)
            if view == 'Waveform':
                self.setup_waveform_plot(cfg_dict_waveform_a,
                                         time_scale_a)
