#!/usr/bin/env python

# file: $NEDC_NFC/class/python/nedc_image_tools/nedc_image_tools.py
#
# revision history:
#
# 20230719 (JP): cosmetic changes
# 20230718 (AB): Created Mil write for svs files
# 20230705 (AB): Refactored code to new comment format
# 20230219 (PM): rewrote the whole class
# 20200604 (JP): refactored the code and created images classes
# 20180617 (RA): separate class and driver to ISIP standard
# 20180620 (RA): initial version
#
# This class contains methods to process images.
#------------------------------------------------------------------------------

# import system modules
#
import os
import sys
import subprocess
import concurrent.futures

# import NEDC modules
#
import nedc_debug_tools as ndt
import nedc_file_tools as nft
import openslide as osl

# import 3rd-party modules
#
import numpy as np
import cv2
from PIL import Image as pil
from PIL import ImageOps, ImageFile
from pillow_heif import register_heif_opener
from torchvision import transforms

ImageFile.LOAD_TRUNCATED_IMAGES = True
register_heif_opener()

# Disable decompression bomb check
#
pil.MAX_IMAGE_PIXELS = None  

try:
    subprocess.check_output("nvidia-smi")

    # set the cuda device for the environment
    #
    # must be placed BEFORE importing cucim
    #
    os.environ["CUDA_VISIBLE_DEVICES"] = "0,1,2,3"

    from cucim import CuImage

    HAS_GPU = True

# CPU Node:
#   use OpenSlide instead
#
except Exception:

    HAS_GPU = False

#------------------------------------------------------------------------------
#
# global variables are listed here
#
#------------------------------------------------------------------------------

# set the filename using basename
#
__FILE__ = os.path.basename(__file__)

# define a flag for the type of image
#
ERR = int(0)
PIL = int(1)
SVS = int(2)
RGB_MODE = "RGB"
PIL_NAME = "PIL"
MIL_NAME = "MIL"
DEF_LEVEL_DIMENSIONS = "level_dimensions"
DEF_RGB_MODE = RGB_MODE

# define important image specific information
#
SVS_MAGIC = b'II*\x00'
SVS_DEF_LEVEL = int(0)
SVS_MODE_CONVERT = DEF_RGB_MODE
DEF_MAX_DIM = int(1600)
DEF_ITYPE = "JPEG"
DEF_OEXT = "jpg"
DEF_SLIDE_LEVEL = int(0)

# define default value for cropping
#
DEF_CENTX = 0
DEF_CENTY = 0
DEF_NPIXX = -1
DEF_NPIXY = -1
DEF_LEVEL = SVS_DEF_LEVEL

# define defaults for multithreading:
#  The number of threads was chosen based on a set of benchmarks conducted.
#  Both OpenSlide and CUCIM had the best performance with 12 threads.
#  Depending on your computing environment, these defaults may need to be
#  adjusted.
#
DEF_NUM_READ_THREADS = int(12)
DEF_NUM_WRITE_THREADS = int(6)

# define number of bytes to read for printing header infromation
#
IMG_VERS_BSIZE_LEN = int(4)

# define printing-related variables for print_header
#
KEY_LEN_PAD = int(5)

# define default values for printing standard properties when printing
# header contents
#
INT_NO_INFO = int(0)
FLOAT_NO_INFO = float(0.0)

#------------------------------------------------------------------------------
#
# + Classes are listed here:
#  There are three classes in this file
#   1. Pil
#   2. Mil
#   3. Nil
#
# + Breakdown of nedc_image_tools' classes:
#
#   Pil: A class that manipulate most images file type
#   Mil: A class that manipulate medical images file
#   Nil: This is a wrapper for all the other classes.
#        You would **ONLY** need to instantiate this class.
#
# + Nil Class:
#
#   There are currently 12 methods:
#       - set_level_svs()
#       - get_level_svs()
#       - get_dimension()
#       - is_image()
#       - resize()
#       - open()
#       - close()
#       - write()
#       - resize_file()
#       - read_data()
#       - read_data_multithread()
#       - read_metadata()
#
# + Usage:
#
#       Image = Nil()
#
#       Image.open("/dir/to/image/image.svs")
#
#       Image.read_data() # returns a numpy matrix of the whole image.
#                         # check the method of the specified class to see
#                         # more options.
#
#------------------------------------------------------------------------------

class Pil:
    """
    Class: Pil

    arguments:
     none

    description:
     This class facilitates manipulation of Python Image Library (PIL) files.
     The image data is contained in image_d.

    """

    #--------------------------------------------------------------------------
    #
    # static data declarations
    #
    #--------------------------------------------------------------------------

    # define static variables for debug and verbosity
    #
    dbgl_d = ndt.Dbgl()
    vrbl_d = ndt.Vrbl()

    #--------------------------------------------------------------------------
    #
    # constructors
    #
    #--------------------------------------------------------------------------

    def __init__(self):
        """
        method: constructor

        arguments:
         none

        returns:
         none

        description:
         Pil Class Constructor
        """

        # create class data
        #
        Pil.__CLASS_NAME__ = self.__class__.__name__
        self.image_d = None

        # exit gracefully
        #
    #
    # end of method

    #--------------------------------------------------------------------------
    #
    # set/get methods
    #
    #--------------------------------------------------------------------------

    def get_dimension(self):
        """
        method: get_dimension

        arguments:
         none

        returns:
         Tuple(width, height)

        description:
         Returns the image dimensions.
        """

        if self.image_d is None:
            print("Error: %s (line: %s) %s::%s: Pil Image empty" %
                  (__FILE__, ndt.__LINE__, Pil.__CLASS_NAME__, ndt.__NAME__))
            return None

        # exit gracefully
        #
        return self.image_d.size
    #
    # end of method

    #--------------------------------------------------------------------------
    #
    # file-based methods
    #
    #--------------------------------------------------------------------------

    def is_pil(self, fname):
        """
        method: is_pil

        arguments:
         fname: input filename

        returns:
         a Boolean value indicating status

        description:
         Checks if the file is an image.
        """

        # display informational message
        #
        if self.dbgl_d > ndt.BRIEF:
            print("%s (line: %s) %s::%s: opening file: %s" %
                  (__FILE__, ndt.__LINE__, Pil.__CLASS_NAME__, ndt.__NAME__,
                   fname))

        # check if it is a PIL image
        #
        try:
            img = pil.open(fname)
            img.verify()
            img.close()
        except pil.UnidentifiedImageError:
            if self.dbgl_d > ndt.BRIEF:
                print("%s (line: %s) %s::%s: Unable to identify file type: %s" %
                    (__FILE__, ndt.__LINE__, Pil.__CLASS_NAME__, ndt.__NAME__,
                    fname))
            return False
        except Exception:
            return False

        # exit gracefully
        #
        return True
    #
    # end of method

    def open(self, fname):
        """
        method: open

        arguments:
         fname: input filename

        returns:
         a Boolean value indicating status

        description:
         Opens and loads an image file.
        """

        # display informational message
        #
        if self.dbgl_d > ndt.BRIEF:
            print("%s (line: %s) %s::%s: opening pil file" %
                  (__FILE__, ndt.__LINE__, Pil.__CLASS_NAME__, ndt.__NAME__))

        # open the file
        #
        self.image_d = pil.open(fname)

        if self.image_d is None:
            return False

        # exit gracefully
        #
        return True
    #
    # end of method

    def close(self):
        """
        method: close

        arguments:
         none

        returns:
         a Boolean value indicating status

        description:
         Closes an image file.
        """

        # display informational message
        #
        if self.dbgl_d > ndt.BRIEF:
            print("%s (line: %s) %s::%s: closing pil file" %
                  (__FILE__, ndt.__LINE__, Pil.__CLASS_NAME__, ndt.__NAME__))

        # close the file
        #
        self.image_d.close()

        # exit gracefully
        #
        return True
    #
    # end of method

    def write(self, fname , ftype):
        """
        method: write

        arguments:
         fname: filename to be written (input)
         ftype: the image type

        returns:
         a Boolean value indicating status

        description:
         Writes an image file.
        """

        # display informational message
        #
        if self.dbgl_d > ndt.BRIEF:
            print("%s (line: %s) %s::%s: writing pil image" %
                  (__FILE__, ndt.__LINE__, Pil.__CLASS_NAME__, ndt.__NAME__))

        # save the image
        #
        try:
            self.image_d.save(fname, ftype)
        except Exception:
            print("Error: %s (line: %s) %s::%s: pil image not written" %
                  (__FILE__, ndt.__LINE__, Pil.__CLASS_NAME__, ndt.__NAME__))
            return False

        # exit gracefully
        #
        return True
    #
    # end of method

    def resize(self, max_dim = DEF_MAX_DIM, replace = False):
        """
        method: resize

        arguments:
         max_dim: the maximum dimension of the new image (1600)
         replace: whether or not we replace the internal image (False)

        returns:
         PIL Image Object

        description:
         Resizes an image to the input dimensions.
         Note that wWhether replace is True or False, this method will
         return a new PIL Image. The difference is whether the internal
         image gets overwritten.
        """

        # display informational message
        #
        if self.dbgl_d > ndt.BRIEF:
            print("%s (line: %s) %s::%s: resizing pil image" %
                  (__FILE__, ndt.__LINE__, Pil.__CLASS_NAME__, ndt.__NAME__))

        # rotate image if detect EXIF is flaged meaning the camera was rotated
        #
        self.image_d = ImageOps.exif_transpose(self.image_d)

        # gets the size of the image in the tuple (width, height)
        #
        width, height = self.get_dimension()

        # check to see if the dimensions exceed the maximum dimension:
        #  do nothing
        #
        if(width < max_dim and height < max_dim):
            return self.image_d

        # gets the resize scale
        #
        resize_scale = max_dim / np.max((width, height))

        # gets the new size of the image
        #
        new_size = np.int64(resize_scale * np.array((width, height)))

        # resize the image object
        #
        new_image = self.image_d.resize(tuple(new_size),
                                        resample = pil.BICUBIC)

        if replace:
            self.image_d = new_image

        # exit gracefully
        #
        return new_image
    #
    # end of method

    def read_data_multithread(self, coordinates,
                              npixx = DEF_NPIXX, npixy = DEF_NPIXY,
                              color_mode = DEF_RGB_MODE,
                              num_threads = DEF_NUM_READ_THREADS):
        """
        method: read_data_multithread

        arguments:
         coordinates: a list of tuple of coordinate (x, y)
             Ex: [(x0, y0), (x1, y1), ..., (xn, yn)]
         npixx : window length (0)
         npixy : window height (0)
         color_mode: the color mode to convert the data (RGB)
            options: RGB, RGBA, L (grayscale), CMYK, etc...
         num_threads: number of thread to use (6)

        returns:
         a list of numpy arrays

        description:
         Returns a numpy array containing the read region of the image.
        """

        # check if image has been loaded
        #
        if self.image_d is None:
            print("Error: %s (line: %s) %s::%s: Image empty" %
                  (__FILE__, ndt.__LINE__, self.__CLASS_NAME__, ndt.__NAME__))
            return None

        # check if all the argument
        #
        if not all(isinstance(arg, int) for arg in [npixx, npixy]):
            print("Error: %s (line: %s) %s::%s: %s" %
                  (__FILE__, ndt.__LINE__, self.__CLASS_NAME__,
                   ndt.__NAME__, "Value needs to be an integer"))
            return None

        # TODO: Please revisit this issue. I am calling load() to put the image
        # into memory but it doesn't close it which means that there's a chance
        # of memory leakage. We need load as the crop() function calls load() internally
        # and that closes the file automatically which means that we can't use crop()
        # for multi-threading.
        #
        # Solution 1: we need to open the image in each of the thread so that
        # it will close the file after it is done automatically. This will require
        # a little bit more information to be save such as the file path.
        #
        self.image_d.load()

        with concurrent.futures.ThreadPoolExecutor(
            max_workers = num_threads
        ) as executor:

            future = executor.map(
                lambda start_loc: self.read_data(coordinate = start_loc,
                                                 npixx = npixx,
                                                 npixy = npixy,
                                                 color_mode = color_mode),
                coordinates,
            )

        # exit gracefully
        #
        return list(future)
    #
    # end of method

    def read_data(self, coordinate = (DEF_CENTX, DEF_CENTY),
                  npixx = DEF_NPIXX, npixy = DEF_NPIXY,
                  color_mode = DEF_RGB_MODE, normalize = False):
        """
        method: read_data

        arguments:
         coordinate: a tuple of coordinate (x, y) (0, 0)
         npixx : window length (0)
         npixy : window height (0)
         color_mode: the color mode to convert the data (RGB)
          color_mode options: RGB, RGBA, L (grayscale), CMYK, etc...
         normalize: flag to determine whether images pixel values
          will be normalized between [0,1], default value is false

        returns:
         a numpy array

        description:
         Return a numpy array containing the read region of the image.
        """

        # check if image has been loaded
        #
        if self.image_d is None:
            print("Error: %s (line: %s) %s::%s: image_d is empty" %
                  (__FILE__, ndt.__LINE__, Pil.__CLASS_NAME__,
                   ndt.__NAME__))
            return None

        # check if all the argument
        #
        if not all(isinstance(arg, int) for arg in [npixx, npixy]):
            print("Error: %s (line: %s) %s::%s: %s" %
                  (__FILE__, ndt.__LINE__, Nil.__CLASS_NAME__,
                   ndt.__NAME__, "value needs to be an integer"))
            return None

        # If the provided patch size equals the default values,
        # use the full image.
        #
        if (npixx, npixy) == (DEF_NPIXX, DEF_NPIXY):

            # fetch full image
            #
            cropped_image = self.image_d.convert(color_mode)

        # else if the patch size is different than the default values,
        # get only patch of intrest
        #
        else:

            # fetch specific patch of intrest
            #
            x, y = coordinate
            crop_box = (x, y, x + npixx, y + npixy)
            cropped_image = self.image_d.crop(crop_box).convert(color_mode)
            
        # If normalization is requested, convert the PIL image to a tensor.
        # this tensor has normalized pixel values between [0,1]
        #
        if normalize:

            # convert the image into a torch tensor 
            #
            image = transforms.ToTensor()(cropped_image)

        # otherwise simply convert the image into a numpy array
        #
        else:

            # Otherwise, return the image as a numpy array.
            #
            image = np.asarray(cropped_image)
            
        # exit gracefully: return the numpy matrix
        #
        return image
    #
    # end of method

    def read_metadata(self):
        """
        method: read_metadata

        arguments:
         none

        returns:
         a dictionary of the image properties

        description:
         Returns the image information and properties.
        """

        # check if image has been loaded
        #
        if self.image_d is None:
            print("Error: %s (line: %s) %s::%s: image_d is empty" %
                  (__FILE__, ndt.__LINE__, Nil.__CLASS_NAME__,
                   ndt.__NAME__))
            return None

        # exit gracefully
        #
        return vars(self.image_d)
    #
    # end of method

#
# end of class

class Mil:
    """
    Class: Mil

    arguments:
     none

    description:
     This class facilitates manipulation of medical images.
     The image data is stored in image_d. The current set level is
     stored in level_d.
    """

    #--------------------------------------------------------------------------
    #
    # static data declarations
    #
    #--------------------------------------------------------------------------

    # define a static variable to hold the value
    #
    dbgl_d = ndt.Dbgl()
    vrbl_d = ndt.Vrbl()

    #--------------------------------------------------------------------------
    #
    # constructors
    #
    #--------------------------------------------------------------------------

    def __init__(self):
        """
        method: constructor

        arguments:
         none

        returns:
         none

        description:
         Default constructor
        """

        # create class data
        #
        Mil.__CLASS_NAME__ = self.__class__.__name__
        self.image_d = None
        self.level_d = SVS_DEF_LEVEL

        # exit gracefully
        #
    #
    # end of method

    #--------------------------------------------------------------------------
    #
    # set/get methods
    #
    #--------------------------------------------------------------------------

    def set_level(self, level = SVS_DEF_LEVEL):
        """
        method: set_level

        arguments:
         level: the level of the hierarchical image to be read (0)

        returns:
         an integer containing the level

        description:
         Sets the level of the image.
        """

        # display informational message
        #
        if self.dbgl_d > ndt.BRIEF:
            print("%s (line: %s) %s::%s: setting level : %d" %
                  (__FILE__, ndt.__LINE__, Mil.__CLASS_NAME__, ndt.__NAME__,
                   level))

        self.level_d = level

        # exit gracefully
        #
        return level
    #
    # end of method

    def get_level(self):
        """
        method: get_level

        arguments:
         none

        returns:
         an integer containing the level

        description:
         Gets the current value of the image level.
        """

        # display informational message
        #
        if self.dbgl_d > ndt.BRIEF:
            print("%s (line: %s) %s::%s: current level : %d" %
                  (__FILE__, ndt.__LINE__, Mil.__CLASS_NAME__, ndt.__NAME__,
                   self.level_d))

        # exit gracefully
        #
        return self.level_d
    #
    # end of method

    def get_dimension(self):
        """
        method: get_dimension

        arguments:
         none

        returns:
         a Tuple(width, height) containing the dimensions

        description:
         Gets the dimensions of the image for the current set level.
        """

        if self.image_d is None:
            print("Error: %s (line: %s) %s::%s: Mil Image empty" %
                  (__FILE__, ndt.__LINE__, Mil.__CLASS_NAME__,
                   ndt.__NAME__))
            return None

        if HAS_GPU:
            (width, height) = \
                self.image_d.resolutions[DEF_LEVEL_DIMENSIONS][self.level_d]
        else:
            (width, height) = self.image_d.level_dimensions[self.level_d]

        # exit gracefully
        #
        return (width, height)
    #
    # end of method

    #--------------------------------------------------------------------------
    #
    # file-based methods
    #
    #--------------------------------------------------------------------------

    def is_mil(self, fname):
        """
        method: is_mil

        arguments:
         fname: input filename

        returns:
         a Boolean value indicating status

        description:
         Returns a boolean value indicating if it is a medical image.
        """

        # display informational message
        #
        if self.dbgl_d > ndt.BRIEF:
            print("%s (line: %s) %s::%s: opening %s file" %
                  (__FILE__, ndt.__LINE__, Mil.__CLASS_NAME__, ndt.__NAME__,
                   fname))

        # check if it is a medical image:
        #
        try:
            if HAS_GPU:
                img = CuImage(fname)
                img.close()
            else:
                img = osl.OpenSlide(fname)
                img.close()
        except Exception:
            if self.dbgl_d > ndt.BRIEF:
                if HAS_GPU:
                    print("%s (line: %s) %s::%s: CUCIM can't open: %s" %
                          (__FILE__, ndt.__LINE__, Mil.__CLASS_NAME__,
                           ndt.__NAME__, fname))
                else:
                    print("%s (line: %s) %s::%s: OpenSlide can't open: %s" %
                          (__FILE__, ndt.__LINE__, Mil.__CLASS_NAME__,
                           ndt.__NAME__, fname))

            # exit ungracefully: it is not a medical image file
            #
            return False

        # exit gracefully: it is a medical image file
        #
        return True
    #
    # end of method

    def open(self, fname):
        """
        method: open

        arguments:
         fname: input filename

        returns:
         a Boolean value indicating status

        description:
         Opens and loads an image from a file.
        """

        # display informational message
        #
        if self.dbgl_d > ndt.BRIEF:
            print("%s (line: %s) %s::%s: opening Mil file (%s)" %
                  (__FILE__, ndt.__LINE__, Mil.__CLASS_NAME__, ndt.__NAME__,
                   fname))
 
        if not self.is_mil(fname):
            print("Error: %s (line: %s) %s::%s: Not an SVS file" %
                  (__FILE__, ndt.__LINE__, Mil.__CLASS_NAME__,
                   ndt.__NAME__))
            return False

        # open the file
        #
        if HAS_GPU:
            self.image_d = CuImage(fname)
        else:
            self.image_d = osl.OpenSlide(fname)

        # exit gracefully
        #
        return True
    #
    # end of method

    def close(self):
        """
        method: close

        arguments:
         none

        returns:
         a Boolean value indicating status

        description:
         Returns a boolean value indicating the status.
        """

        # display informational message
        #
        if self.dbgl_d > ndt.BRIEF:
            print("%s (line: %s) %s::%s: closing Mil file" %
                  (__FILE__, ndt.__LINE__, Mil.__CLASS_NAME__, ndt.__NAME__))

        # close the file
        #
        self.image_d.close()

        # exit gracefully
        #
        return True
    #
    # end of method

    def read_data_multithread(self, coordinates,
                              npixx = DEF_NPIXX, npixy = DEF_NPIXY,
                              color_mode = DEF_RGB_MODE,
                              num_threads = DEF_NUM_READ_THREADS):
        """
        method: read_data_multithread

        arguments:
         npixx : window length (0)
         npixy : window height (0)
         color_mode: the color mode to convert the data (RGB)
          color_mode options: RGB, RGBA, L (grayscale), CMYK, etc...
         num_threads: number of thread to use (12)

        returns:
         a list of numpy arrays

        description:
         Return a numpy array containing the read region of the image.
        """

        # check if image has been loaded
        #
        if self.image_d is None:
            print("Error: %s (line: %s) %s::%s: Mil image empty" %
                  (__FILE__, ndt.__LINE__, self.__CLASS_NAME__,
                   ndt.__NAME__))
            return None

        # check if all the argument
        #
        if not all(isinstance(arg, int) for arg in [npixx, npixy]):
            print("Error: %s (line: %s) %s::%s: %s" %
                  (__FILE__, ndt.__LINE__, Mil.__CLASS_NAME__,
                   ndt.__NAME__, "value needs to be an integer"))
            return None

        with concurrent.futures.ThreadPoolExecutor(max_workers = num_threads) \
             as executor:

            future = executor.map(
                lambda start_loc: self.read_data(coordinate = start_loc,
                                                npixx = npixx,
                                                npixy = npixy,
                                                color_mode = color_mode),
                coordinates,
            )

        # exit gracefully
        #
        return list(future)
    #
    # end of method

    def read_data(self, coordinate = (DEF_CENTX, DEF_CENTY),
                  npixx = DEF_NPIXX, npixy = DEF_NPIXY,
                  color_mode = DEF_RGB_MODE, normalize = False):
        """
        method: read_data

        arguments:
         coordinate: a tuple of coordinate (x, y) (0, 0)
         npixx : window length (-1)
         npixy : window height (-1)
         normalize: flag to determine whether to normalize pixel values
          between [0,1]

        returns:
         a numpy array containing the data

        description:
         Return a numpy array containing the read region of the image.
        """
        # check if image has been loaded
        #
        if self.image_d is None:
            print("Error: %s (line: %s) %s::%s: Mil Image empty" %
                  (__FILE__, ndt.__LINE__, Mil.__CLASS_NAME__,
                   ndt.__NAME__))
            return None

        # check if all the argument
        #
        if not all(isinstance(arg, int) for arg in [npixx, npixy]):
            print("Error: %s (line: %s) %s::%s: %s" %
                  (__FILE__, ndt.__LINE__, Mil.__CLASS_NAME__,
                   ndt.__NAME__, "value needs to be an integer"))
            return None

        if (npixx, npixy) == (DEF_NPIXX, DEF_NPIXY):

            # if no size is specified or the window size is (0,0):
            #   we take the image size of the specified level as the window
            #
            # (width, height)
            #
            (npixx, npixy) = self.get_dimension()

        # get the correct coordinates based on the level we are reading
        #
        if hasattr(self.image_d, "level_downsamples"):
            # OpenSlide path
            downsample = self.image_d.level_downsamples[self.level_d]
        else:
            # CuCIM path: use its get_downsample() API
            downsample = self.image_d.resolutions["level_downsamples"][self.level_d]
        x0 = int(coordinate[0] * downsample)
        y0 = int(coordinate[1] * downsample)
        
        # fetch the area of intrest
        #
        window = self.image_d.read_region(
            location = (x0, y0),
            size = (npixx, npixy),
            level = self.level_d)

        # if using gpu convert to the color mode specified
        #
        if HAS_GPU:
            if color_mode == DEF_RGB_MODE:
                return np.asarray(window)
            else:
                cropped_image = pil.fromarray(np.asarray(window))
                window = np.asarray(cropped_image.convert(color_mode))

        # else convert to color mode without gpu
        #
        else:
            window = np.asarray(window.convert(color_mode))


        # If normalization is requested, convert the PIL image to a tensor.
        # this tensor has normalized pixel values between [0,1]
        #
        if normalize:
            
            # convert the image into a torch tensor
            #
            image = transforms.ToTensor()(window)
            
        # otherwise simply convert the image into a numpy array
        #
        else:
            
            # Otherwise, return the image as a numpy array.
            #
            image = np.asarray(window)

        # exit gracefully
        #  return the extracted image
        #
        return image

    #
    # end of method

    def read_metadata(self):
        """
        method: read_metadata

        arguments:
         none

        returns:
         a dictionary containing the image properties

        description:
         Returns the image information and properties.
        """

        # check if image is loaded
        #
        if self.image_d is None:
            print("Error: %s (line: %s) %s::%s: No Mil Image loaded" %
                  (__FILE__, ndt.__LINE__, Mil.__CLASS_NAME__,
                   ndt.__NAME__))
            return None
        
        # exit gracefully: return a dictionary of the image properties
        #
        return self.image_d.metadata if HAS_GPU else self.image_d.properties
    #
    # end of method

    def write(self, fname, ftype):
        """
        method: write

        arguments:
         fname: file name
         ftype: image type to write as (e.g., bmp, gif, png, jpeg)

        returns:
         a Boolean value indicating status

        description:
         Writes an svs image to disk using opencv2.
        """

        # display informational message
        #
        if self.dbgl_d > ndt.BRIEF:
            print("%s (line: %s) %s::%s: writing svs image" %
                  (__FILE__, ndt.__LINE__, Mil.__CLASS_NAME__, ndt.__NAME__))

        # Save image
        #
        try:
            cv2.imwrite(fname,self.read_data())
        except Exception:
            print("Error: %s (line: %s) %s::%s: svs image not written" %
                  (__FILE__, ndt.__LINE__, Mil.__CLASS_NAME__, ndt.__NAME__))
            return False

        # exit gracefully
        #
        return True
    #
    # end of method

#
# end of class

class Nil:
    """
    Class: Nil

    arguments:
     none

    description:
     This class facilitates manipulation of Python Image Library (PIL) files
     and Mil files. The image data is contained in image_d. The current
     type of image is contained in class_d.
    """

    #--------------------------------------------------------------------------
    #
    # static data declarations
    #
    #--------------------------------------------------------------------------

    # define a static variable to hold the value
    #
    dbgl_d = ndt.Dbgl()
    vrbl_d = ndt.Vrbl()

    #--------------------------------------------------------------------------
    #
    # constructors
    #
    #--------------------------------------------------------------------------

    def __init__(self):
        """
        method: Constructor

        arguments:
         none

        returns:
         none

        description:
         Default constructor
        """

        Nil.__CLASS_NAME__ = self.__class__.__name__

        # create class data
        #
        self.itype_d = None
        self.image_d = None
        self.class_d = None

        # exit gracefully
        #
    #
    # end of method

    #--------------------------------------------------------------------------
    #
    # set/get methods
    #
    #--------------------------------------------------------------------------

    def set_level_svs(self, level = SVS_DEF_LEVEL):
        """
        method: set_level_svs

        arguments:
         level: the level of the hierarchical image to be read (0)

        returns:
         an integer containing the level

        description:
         Sets the level for an svs image.
        """

        # display informational message
        #
        if self.dbgl_d > ndt.BRIEF:
            print("%s (line: %s) %s::%s: setting svs level" %
                  (__FILE__, ndt.__LINE__, Nil.__CLASS_NAME__, ndt.__NAME__))

        if self.class_d is None:
            print("Error: %s (line: %s) %s::%s: Image class not set" %
                  (__FILE__, ndt.__LINE__, Nil.__CLASS_NAME__, ndt.__NAME__))
            return None

        if not isinstance(self.class_d, type(IMG_CLASS[MIL_NAME])):
            print("Error: %s (line: %s) %s::%s: %s" %
                  (__FILE__, ndt.__LINE__, Nil.__CLASS_NAME__, ndt.__NAME__,
                   "can't set level because the image is not svs"))
            return None

        self.class_d.set_level(level)

        # exit gracefully
        #
        return level
    #
    # end of method

    def get_level_svs(self):
        """
        method: get_level_svs

        arguments:
         none

        returns:
         an integer containing the level

        description:
         Get the current value of the image level.
        """

        # display informational message
        #
        if self.dbgl_d > ndt.BRIEF:
            print("%s (line: %s) %s::%s: setting svs level" %
                  (__FILE__, ndt.__LINE__, Nil.__CLASS_NAME__, ndt.__NAME__))

        if self.class_d is None:
            print("Error: %s (line: %s) %s::%s: Image class not set" %
                  (__FILE__, ndt.__LINE__, Nil.__CLASS_NAME__, ndt.__NAME__))
            return None

        if not isinstance(self.class_d, type(IMG_CLASS[MIL_NAME])):
            print("Error: %s (line: %s) %s::%s: %s" %
                  (__FILE__, ndt.__LINE__, Nil.__CLASS_NAME__, ndt.__NAME__,
                   "can't get level because the image is not an svs file"))
            return None

        # exit gracefully
        #
        return self.class_d.get_level()
    #
    # end of method

    def get_dimension(self):
        """
        method: get_dimension

        arguments:
         none

        returns:
         a Tuple(width, height) containg the dimensions

        description:
         Returns the image dimensions.
        """

        # display informational message
        #
        if self.dbgl_d > ndt.BRIEF:
            print("%s (line: %s) %s::%s: reading the image metadate" %
                  (__FILE__, ndt.__LINE__, Nil.__CLASS_NAME__, ndt.__NAME__))

        if self.class_d is None:
            print("Error: %s (line: %s) %s::%s: %s" %
                  (__FILE__, ndt.__LINE__, Nil.__CLASS_NAME__, ndt.__NAME__,
                   "can't close file because the Image class is not set"))
            return None

        # exit gracefully
        #
        return self.class_d.get_dimension()
    #
    # end of method

    #--------------------------------------------------------------------------
    #
    # file-based methods
    #
    #--------------------------------------------------------------------------

    def is_image(self, fname):
        """
        method: is_image

        arguments:
         fname: input filename

        returns:
         an Integer code containing the type of image

        description:
         Checks the file type.
        """

        # display informational message
        #
        if self.dbgl_d > ndt.BRIEF:
            print("%s (line: %s) %s::%s: opening %s file" %
                  (__FILE__, ndt.__LINE__, Pil.__CLASS_NAME__, ndt.__NAME__,
                   fname))

        # check for an svs image:
        #  note we do this first because it is faster than is_pil
        #
        if IMG_CLASS[MIL_NAME].is_mil(fname):
            return SVS

        if IMG_CLASS[PIL_NAME].is_pil(fname):
            return PIL

        # exit gracefully: it is not a supported image type
        #
        return ERR
    #
    # end of method

    def resize(self, max_dim = DEF_MAX_DIM, replace = False):
        """
        method: resize

        arguments:
         max_dim: the maximum dimension of the new image (1600)
         replace: whether or not we replace the internal image (False)

        returns:
         a PIL Image Object

        description:
         Resizes an image to the input dimensions.
         Note that wWhether replace is True or False, this method will
         return a new PIL Image. The difference is whether the internal
         image gets overwritten.
        """

        # display informational message
        #
        if self.dbgl_d > ndt.BRIEF:
            print("%s (line: %s) %s::%s: resizing image" %
                  (__FILE__, ndt.__LINE__, Nil.__CLASS_NAME__, ndt.__NAME__))

        if self.class_d is None:
            print("Error: %s (line: %s) %s::%s: image class not set" %
                  (__FILE__, ndt.__LINE__, Nil.__CLASS_NAME__,
                   ndt.__NAME__))
            return None

        # check if the current class is a Mil class
        #
        if isinstance(self.class_d, type(IMG_CLASS[MIL_NAME])):
            print("Error: %s (line: %s) %s::%s: %s" %
                  (__FILE__, ndt.__LINE__, Nil.__CLASS_NAME__, ndt.__NAME__,
                   "can't resize a Mil Image"))
            return None

        image_new = self.class_d.resize(max_dim, replace)

        if image_new is None:
            print("Error: %s (line: %s) %s::%s: %s resize error" %
                  (__FILE__, ndt.__LINE__, Nil.__CLASS_NAME__, ndt.__NAME__,
                   self.class_d.__CLASS_NAME__))
            return None

        # exit gracefully
        #
        return image_new
    #
    # end of method

    def open(self, ifname):
        """
        method: open

        arguments:
         ifname: input file name

        returns:
         a Boolean value indicating status

        description:
         This method opens an image. It essentially abstracts
         the type of image.
        """

        # display informational message
        #
        if self.dbgl_d > ndt.BRIEF:
            print("%s (line: %s) %s::%s: opening file (%s)" %
                  (__FILE__, ndt.__LINE__, Nil.__CLASS_NAME__, ndt.__NAME__,
                   ifname))

        # get the type of image
        #
        file_type = self.is_image(ifname)
        if not file_type:
            print("%s (line: %s) %s::%s: not a supported type (%s)" %
                  (__FILE__, ndt.__LINE__, Nil.__CLASS_NAME__, ndt.__NAME__,
                   ifname))
            sys.exit(os.EX_SOFTWARE)

        if self.dbgl_d > ndt.BRIEF:
            print("%s (line: %s) %s::%s: image type (%d)" %
                  (__FILE__, ndt.__LINE__, Nil.__CLASS_NAME__, ndt.__NAME__,
                   file_type))

        # convert the class based on the indication status
        #
        if file_type == PIL:
            self.class_d = IMG_CLASS[PIL_NAME]

        elif file_type == SVS:
            self.class_d = IMG_CLASS[MIL_NAME]

        if not self.class_d.open(ifname):
            print("Error: %s (line: %s) %s::%s: %s unable to open file" %
                  (__FILE__, ndt.__LINE__, Nil.__CLASS_NAME__, ndt.__NAME__,
                   self.class_d.__CLASS_NAME__))
            return False

        # exit gracefully
        #
        return True
    #
    # end of method

    def close(self):
        """
        method: close

        arguments:
         none

        returns:
         a Boolean indication status

        description:
         This method closes the image files and frees memory.
        """

        # display informational message
        #
        if self.dbgl_d > ndt.BRIEF:
            print("%s (line: %s) %s::%s: closing file" %
                  (__FILE__, ndt.__LINE__, Nil.__CLASS_NAME__, ndt.__NAME__))

        if self.class_d is None:
            print("Error: %s (line: %s) %s::%s: %s" %
                  (__FILE__, ndt.__LINE__, Nil.__CLASS_NAME__, ndt.__NAME__,
                   "can't close file because the Image class is not set"))
            return False

        # branch on the type of image
        #
        if not self.class_d.close():
            print("Error: %s (line: %s) %s::%s: %s fail to close file" %
                  (__FILE__, ndt.__LINE__, Nil.__CLASS_NAME__, ndt.__NAME__,
                   self.class_d.__CLASS_NAME__))
            return False

        # exit gracefully
        #
        return True
    #
    # end of method

    def read_data(self, coordinate = (DEF_CENTX, DEF_CENTY),
                  npixx = DEF_NPIXX, npixy = DEF_NPIXY,
                  color_mode = DEF_RGB_MODE):
        """
        method: read_data

        arguments:
         coordinate: a tuple of coordinate (x, y) (0, 0)
         npixx : window length (0)
         npixy : window height (0)
         color_mode: the color mode to convert the data (RGB)
          color_mode options: RGB, RGBA, L (grayscale), CMYK, etc...

        returns:
         a numpy array containing the image data

        description:
         Read a region of the image and returns a numpy array containing the
         corresponding data
        """

        # display informational message
        #
        if self.dbgl_d > ndt.BRIEF:
            print("%s (line: %s) %s::%s: reading the image rgb value" %
                  (__FILE__, ndt.__LINE__, Nil.__CLASS_NAME__, ndt.__NAME__))

        if self.class_d is None:
            print("Error: %s (line: %s) %s::%s: %s" %
                  (__FILE__, ndt.__LINE__, Nil.__CLASS_NAME__, ndt.__NAME__,
                   "can't close file because the Image class is not set"))
            return None

        data = self.class_d.read_data(coordinate, npixx, npixy, color_mode)

        # exit gracefully: return the numpy matrix
        #
        return data
    #
    # end of method

    def read_data_multithread(self, coordinates,
                              npixx = DEF_NPIXX, npixy = DEF_NPIXY,
                              color_mode = DEF_RGB_MODE,
                              num_threads = DEF_NUM_READ_THREADS):
        """
        method: read_data_multithread

        arguments:
         coordinates: a list of tuple of coordinate (x, y)
          Ex: [(x0, y0), (x1, y1), ..., (xn, yn)]
         npixx : window length (0)
         npixy : window height (0)
         color_mode: the color mode to convert the data (RGB)
         num_threads: number of thread to use (6)

        returns:
         a list of numpy arrays

        description:
         Returns a numpy array containing the read region of the image.
        """

        # display informational message
        #
        if self.dbgl_d > ndt.BRIEF:
            print("%s (line: %s) %s::%s: %s" %
                (__FILE__, ndt.__LINE__, Nil.__CLASS_NAME__, ndt.__NAME__,
                 "reading the image rgb value with multithread"))

        if self.class_d is None:
            print("Error: %s (line: %s) %s::%s:" %
                  (__FILE__, ndt.__LINE__, Nil.__CLASS_NAME__, ndt.__NAME__,
                   "can't close file because the Image class is not set"))
            return None

        # start multithread
        #
        data = self.class_d.read_data_multithread(coordinates, npixx, npixy,
                                                  color_mode, num_threads)

        # exit gracefully
        #
        return data
    #
    # end of method

    def read_metadata(self):
        """
        method: read_metadata

        arguments:
         none

        returns:
         a dictionary containing the image properties

        description:
         Returns the image information and properties.
        """

        # display informational message
        #
        if self.dbgl_d > ndt.BRIEF:
            print("%s (line: %s) %s::%s: reading the image metadate" %
                  (__FILE__, ndt.__LINE__, Nil.__CLASS_NAME__, ndt.__NAME__))

        if self.class_d is None:
            print("Error: %s (line: %s) %s::%s: %s" %
                  (__FILE__, ndt.__LINE__, Nil.__CLASS_NAME__, ndt.__NAME__,
                   "can't close file because the Image class is not set"))
            return None

        data = self.class_d.read_metadata()

        # exit gracefully: return a dictionary of the image metadata
        #
        return data
    #
    # end of method

    def write(self, fname, ftype):
        """
        method: write

        arguments:
         fname: file name
         ftype: image type to write as (e.g.: BMP, GIF, PNG, JPEG)

        returns:
         a boolean value indicating status

        description:
         This method writes an image to a file.
        """

        # display informational message
        #
        if self.dbgl_d > ndt.BRIEF:
            print("%s (line: %s) %s::%s: writing image" %
                  (__FILE__, ndt.__LINE__, Nil.__CLASS_NAME__, ndt.__NAME__))

        if self.class_d is None:
            print("Error: %s (line: %s) %s::%s: Image class not set" %
                  (__FILE__, ndt.__LINE__, Nil.__CLASS_NAME__, ndt.__NAME__))
            return False

        # branch on the type of image and exit gracefully
        #
        if not self.class_d.write(fname, ftype):
            print("Error: %s (line: %s) %s::%s: %s write error" %
                  (__FILE__, ndt.__LINE__, Nil.__CLASS_NAME__, ndt.__NAME__,
                   self.class_d.__CLASS_NAME__))
            return False

        # exit gracefully
        #
        return True
    #
    # end of method

    def print_header(self, fname, fp = sys.stdout, prefix = nft.DELIM_TAB):
        """
        method: print_header

        arguments:
         fname: input file
         fp: stream to be used for printing
         prefix: a prefix to use before each field is printed

        returns:
         a boolean value indicating status

        description:
         This method prints all necessary
         image header infromation.
        """

        # display debug information
        #
        if ndt.Dbgl.level_d > ndt.BRIEF:
            print("%s (line %d) %s: printing header\n" %
                  (__FILE__, __LINE__, __PRETTY_FUNCTION__))

        # open the image via openslide's API
        #
        img_info = osl.open_slide(fname)

        # create variables to hold frequently used openslide data
        #
        img_props = img_info.properties
        img_associated = img_info.associated_images
        img_num_levels = img_info.level_count
        img_dimensions = img_info.dimensions

        # find the length of the longest key in openslides.properties
        # dictionary to use for formating the print to the fp stream
        #
        max_klen = int(0)
        for key in img_info.properties.keys():
            if len(key) > max_klen:
                max_klen = len(key)
        max_klen += KEY_LEN_PAD

        # (1) Version Information: pretty print the first four bytes of
        # the fname file
        #
        fp.write("%sBlock 1: Version Information\n" % (prefix))

        # create file pointer to first four bytes of the input file
        #
        fname_ptr = open(fname,nft.MODE_READ_BINARY)
        if fp is None:
            fp.write("Error: %s (line: %s) %s::%s: error opening file (%s)" %
                  (__FILE__, ndt.__LINE__, Edf.__CLASS_NAME__, ndt.__NAME__,
                   fname))
            return False
        fname_ptr.seek(0, os.SEEK_SET)

        # declare and initalize byte array to contain the first
        # four bytes
        #
        barray = fname_ptr.read(IMG_VERS_BSIZE_LEN)
        tstr = barray.decode().split(nft.DELIM_BLANK)[0]
        fp.write("%s%s = [%s][%d %d %d %d]\n\n" %
                 (prefix, 'version'.rjust(max_klen), tstr,
                  barray[0], barray[1], barray[2], barray[3]))
        fname_ptr.close()

        # print the vendor-specific fields/metadata contained in
        # openslide's propertiies dictionary
        #
        fp.write("%sBlock 2: Raw Header Information\n" % (prefix))
        for prop_name, prop_value in img_props.items():
            fp.write("%s%s = [%s]\n" %
                     (prefix, prop_name.rjust(max_klen), prop_value))
        fp.write(nft.DELIM_NEWLINE)

        # print the standard properties:
        #  a list of these can be found here: https://openslide.org/api/python/
        # the standard properties are stored in macros
        #
        fp.write("%sBlock 3: Standard Properties\n" % (prefix))

        # this implementation only exists becaue the openslide
        # get_property method fails if the property
        # doesn't exist.
        #
        tstr = img_props.get(osl.PROPERTY_NAME_COMMENT)
        if tstr is None:
            tstr = nft.DELIM_NULL
        fp.write("%s%s = [%s]\n" % \
                 (prefix, osl.PROPERTY_NAME_COMMENT.rjust(max_klen), tstr))

        tstr = img_props.get(osl.PROPERTY_NAME_VENDOR)
        if tstr is None:
            tstr = nft.DELIM_NULL
        fp.write("%s%s = [%s]\n" % \
                 (prefix, osl.PROPERTY_NAME_VENDOR.rjust(max_klen), tstr))

        tstr = img_props.get(osl.PROPERTY_NAME_QUICKHASH1)
        if tstr is None:
            tstr = nft.DELIM_NULL
        fp.write("%s%s = [%s]\n" % \
                 (prefix, osl.PROPERTY_NAME_QUICKHASH1.rjust(max_klen), tstr))

        tstr = img_props.get(osl.PROPERTY_NAME_BACKGROUND_COLOR)
        if tstr is None:
            tstr = nft.DELIM_NULL
        fp.write("%s%s = [%s]\n" % \
                 (prefix, osl.PROPERTY_NAME_BACKGROUND_COLOR.rjust(max_klen),
                  tstr))

        tval = img_props.get(osl.PROPERTY_NAME_OBJECTIVE_POWER)
        if tval is not None:
            tval = int(tval)
        else:
            tval = INT_NO_INFO
        fp.write("%s%s = [%d]\n" % \
                     (prefix,
                      osl.PROPERTY_NAME_OBJECTIVE_POWER.rjust(max_klen),
                      tval))

        tval = img_props.get(osl.PROPERTY_NAME_MPP_X)
        if tval is not None:
            tval = float(tval)
        else:
            tval = FLOAT_NO_INFO
        fp.write("%s%s = [%15.6f]\n" % \
                     (prefix, osl.PROPERTY_NAME_MPP_X.rjust(max_klen), tval))

        tval = img_props.get(osl.PROPERTY_NAME_MPP_Y)
        if tval is not None:
            tval = float(tval)
        else:
            tval = FLOAT_NO_INFO
        fp.write("%s%s = [%15.6f]\n" %
                 (prefix, osl.PROPERTY_NAME_MPP_Y.rjust(max_klen), tval))

        tval = img_props.get(osl.PROPERTY_NAME_BOUNDS_X)
        if tval is not None:
            tval = int(tval)
        else:
            tval = INT_NO_INFO
        fp.write("%s%s = [%d]\n" % \
                 (prefix, osl.PROPERTY_NAME_BOUNDS_X.rjust(max_klen), tval))

        tval = img_props.get(osl.PROPERTY_NAME_BOUNDS_Y)
        if tval is not None:
            tval = int(tval)
        else:
            tval = INT_NO_INFO
        fp.write("%s%s = [%d]\n" % \
                 (prefix, osl.PROPERTY_NAME_BOUNDS_Y.rjust(max_klen), tval))

        tval = img_props.get(osl.PROPERTY_NAME_BOUNDS_WIDTH)
        if tval is not None:
            tval = int(tval)
        else:
            tval = INT_NO_INFO
        fp.write("%s%s = [%d]\n" % \
                 (prefix, osl.PROPERTY_NAME_BOUNDS_WIDTH.rjust(max_klen),
                  tval))

        tval = img_props.get(osl.PROPERTY_NAME_BOUNDS_HEIGHT)
        if tval is not None:
            tval = int(tval)
        else:
            tval = INT_NO_INFO
        fp.write("%s%s = [%d]\n" % \
                 (prefix, osl.PROPERTY_NAME_BOUNDS_HEIGHT.rjust(max_klen),
                  tval))

        fp.write(nft.DELIM_NEWLINE)

        # print the derived properties:
        #  a list of these can be found here: https://openslide.org/api/python/
        #
        fp.write("%sBlock 4: Derived Properties\n" % (prefix))

        # print the derived properties located in openslide's
        # associated_images field. a list of these can be found here:
        #  https://openslide.org/api/python/
        #
        index = int(0)
        for image_names,dimensions in img_associated.items():
            fp.write("%s%s = [%s]\n" %
                     (prefix,
                      ('associated image names[  %d]'
                       % (index)).rjust(max_klen),
                      image_names ))
            fp.write("%s%s = [%d pixels by %d pixels]\n" %
                     (prefix,
                      ('image dimensions[  %d]' %
                       (index)).rjust(max_klen),
                      dimensions.size[1], dimensions.size[0]))
            index += 1

        # print the derived properties located in opernslide's
        # level_count field
        #
        fp.write("%s%s = [%d]\n" %
                 (prefix, 'num levels'.rjust(max_klen),
                  img_num_levels))
        fp.write("\n")

        # print aditional properties:
        #  print image height and width
        #
        fp.write("%sBlock 5: Additional Properties\n" % (prefix))
        width, height = img_dimensions
        fp.write("%s%s = [%d]\n" %
                 (prefix, 'max_height'.rjust(max_klen), height))
        fp.write("%s%s = [%d]\n" %
                 (prefix, 'max_width'.rjust(max_klen), width))
        fp.write("\n")

        # close the image
        #
        img_info.close()

        # exit gracefully
        #
        return True
    #
    # end of method


#
# end of class

#-------------------------------------------------------------------------------
#
# definitions dependent on the above classes go here
#
#-------------------------------------------------------------------------------

# define variables to configure the machine learning algorithms
#
IMG_CLASS = {PIL_NAME: Pil(), MIL_NAME: Mil()}

#
# end of file
