#!/usr/bin/env python
#
# file: $NEDC_NFC/class/python/nedc_dpath_ann_tools/nedc_dpath_pproc_tools.py
#
# revision history:
#
# 20250624 (DH): Consolidated helper routines and multiple region post-processors.
#
# This module houses the common utilities (bitmap creation, hole filling,
# component merging) plus five concrete post-processing strategies
# (confidence, dilation, KNN, mode, priority-fill)
#-------------------------------------------------------------------------------

# import sys modules
#
import os
import sys
import copy

# import misc modules
#
from collections import deque, Counter
from enum import IntEnum

# import numpy/shapely modules
#
import numpy as np
import shapely.geometry as geom
import shapely.ops as ops

# import scipy modules
#
from scipy.ndimage import generic_filter, maximum_filter, distance_transform_edt
from scipy.stats import mode

# import nedc modules
#
import nedc_dpath_ann_tools as nda
import nedc_debug_tools as ndt
import nedc_image_tools as nit
import nedc_file_tools as nft
import nedc_dpath_image_tools as ndi
import nedc_dpath_tissue_mask_tools as ndtm

#-------------------------------------------------------------------------------
#
# constants defined here
#
#-------------------------------------------------------------------------------

# declare a global debug object so we can use it in functions
#
dbgl = ndt.Dbgl()

# set the filename using base name
#
__FILE__ = os.path.basename(__file__)

# define a constant that defines tool version
#
PPROC_VER = "1.0.0"

# define a constant for heat map creation (generate_sparse_matricies)
#
CKEY_CONFIDENCES = "Confidences"

# define a constant to determine numpy padding rules
#
DEF_NUMPY_PAD_MODE = "constant"

# define a constant to determine if an annotation has only
# four coordinates
#
DEF_NUM_FRAME_COORDS = int(4)

# provide keys for PPROC OBJECTS
#
CKEY_ALG_CONFIDENCE = "confidence"
CKEY_ALG_DILATION = "dilation"
CKEY_ALG_KNN = "knn"
CKEY_ALG_PRIORITY = "priority"
CKEY_ALG_MODE = "mode"
CKEY_ALG_SLIDING_WINDOW = "sliding_window"
CKEY_ALG_TISSUE_MASK = "tissue_mask"

#-------------------------------------------------------------------------------
#
# functions defined here
#
#-------------------------------------------------------------------------------

def unpack_graph(ffile):
    """
    function: unpack_graph

    arguments:
     ann: a dpath annotation graph annotation

    return:
     coords: list of coordinates (only top-left coordinates)
     probs: list of probabilities
     label_names: list of label names
     frame_size: the calculated and constant frame size
    
    description:
     unpacks a graph into several lists all
     sorted so each coordinate has its respective
     probability and label name. will also return
     None if an error is encountered
    """

    # create ann tool
    #
    ann = nda.AnnDpath()

    # load the 
    #
    ann.load(ffile)

    # fetch the graph
    #
    agraph = ann.get_graph()

    # fetch the header
    #
    hdr = ann.get_header()

    # ensure graph and header are not empty
    #
    if nda.validate_AnnGrDpath(agraph, hdr) == False:
        print("Error: %s (line: %s) %s: priority map doesn't exist (%s)" %
              (__FILE__, ndt.__LINE__, ndt.__NAME__, ffile))
        return None

    # create lists to store info
    #
    coords = list()
    probs = list()
    labels = list()

    # find frame width/height it is assumed each annotation is
    # a square
    #
    first_key = next(iter(agraph))
    first_coord_x = int(agraph[first_key][nda.CKEY_COORDINATES][0][0])
    second_coord_x = int(agraph[first_key][nda.CKEY_COORDINATES][1][0])
    frmsize = abs(first_coord_x - second_coord_x)

    # ensure frame size is greator than zero 
    #
    if frmsize < 0:
        print("Error: %s (line: %s) %s: non-positive frame size" %
              (__FILE__, ndt.__LINE__, ndt.__NAME__))
        return None

    # iterate over the idx keys of the graph
    #
    for idx in list(agraph.keys()):

        # ensure annotation has the exact amount of
        # coordinates to be square
        #
        if len(agraph[idx][nda.CKEY_COORDINATES]) != DEF_NUM_FRAME_COORDS:
            print("Error: %s (line: %s) %s: %s %s (%s)" %
                  (__FILE__, ndt.__LINE__, ndt.__NAME__,
                   "non square annotation detected,",
                   "insufficient coordinate count",
                   agraph[idx][nda.CKEY_COORDINATES]))
            return None
        
        # fetch all five coordinates
        #
        first_coord_y = int(agraph[idx][nda.CKEY_COORDINATES][0][1])
        first_coord_x = int(agraph[idx][nda.CKEY_COORDINATES][0][0])
        second_coord_y = int(agraph[idx][nda.CKEY_COORDINATES][1][1])
        second_coord_x = int(agraph[idx][nda.CKEY_COORDINATES][1][0])
        third_coord_y = int(agraph[idx][nda.CKEY_COORDINATES][2][1])
        third_coord_x = int(agraph[idx][nda.CKEY_COORDINATES][2][0])
        fourth_coord_y = int(agraph[idx][nda.CKEY_COORDINATES][3][1])
        fourth_coord_x = int(agraph[idx][nda.CKEY_COORDINATES][3][0])

        # check that ann is square 
        #
        xs_ = [first_coord_x, second_coord_x, third_coord_x, fourth_coord_x]
        ys_ = [first_coord_y, second_coord_y, third_coord_y, fourth_coord_y]
        w = max(xs_) - min(xs_)
        h = max(ys_) - min(ys_)
        if w != h or w != frmsize:
            print("Error: %s (line: %s) %s: %s (%s)" %
                  (__FILE__, ndt.__LINE__, ndt.__NAME__,
                   "non square or inconsistent frame size",
                   agraph[idx][nda.CKEY_COORDINATES]))
            return None
        
        # fetch the first (top - left) coordinate and append it
        #
        left_x = int(agraph[idx][nda.CKEY_COORDINATES][0][0])
        top_y = int(agraph[idx][nda.CKEY_COORDINATES][0][1])
        coords.append((left_x, top_y))

        # append probability/confidence
        #
        probs.append(float(agraph[idx][nda.CKEY_CONFIDENCE]))

        # append annotation label name
        #
        labels.append(str(agraph[idx][nda.CKEY_TEXT]))

    # exit gracefully
    #  return needed values
    #
    return (coords, probs, labels, frmsize)
#
# end of function

def _make_label_enum(priority):
    """
    function: _make_label_enum

    arguments:
     priority: a label priority map

    return:
     return an Enum whose name/value pairs encode the requested order.

    description:
     return an custom enumerable object defined by the user
    """

    # ensure ints and strings
    #
    p = {str(k): int(v) for k, v in priority.items()}
    
    # define members in highlow order but keep the provided values
    #
    members = [(lbl, p[lbl]) for lbl, _ in sorted(p.items(),
                                                  key=lambda kv: kv[1],
                                                  reverse=True)]
    
    # exit gracefully
    #  return label order
    #
    return IntEnum("local_label_order", members)
#
# end of function

def _merge_label_grid(label_grid, confidence_grid, frmsize, label_order):
    """
    function: _merge_label_grid

    arguments:
     label_grid: 2D array containing integer class identifiers
     confidence_grid: 2D float array of equal shape with cell confidences
     frmsize: the frame size of each annotation
     label_order: the priority label order
    
    return: an annotation graph dictionary where each entry describes a region

    description:
     This helper walks the label grid, extracts connected components for
     each class, translates them into 2D polygons, and
     records the mean confidence over the component.
    """

    # create a dictionary to accumulate output regions
    #
    annotations = {}

    # initialize a unique identifier for each output region
    #
    region_id = int(1)

    # capture the grid dimensions for boundary checks
    #
    grid_height, grid_width = label_grid.shape

    # allocate a boolean array to record visited grid cells
    #
    visited = np.zeros((grid_height, grid_width), dtype = bool)

    # define all possible 4-connected neighbor offsets
    #
    neighbor_offsets = [(-1, 0), (1, 0), (0, -1), (0, 1)]

    # iterate over every cell in the label grid using nested loops
    #
    for row_idx in range(grid_height):

        # iterate over the column index for the current row
        #
        for col_idx in range(grid_width):

            # skip the cell if it has already been processed
            #
            if visited[row_idx, col_idx]:

                # advance to next column if current cell is visited
                #
                continue

            # fetch the integer class identifier stored in the current cell
            #
            class_value = int(label_grid[row_idx, col_idx])

            # initialise a queue for breadthfirst search of the component
            #
            bfs_queue = [(row_idx, col_idx)]

            # create a list to collect grid indices belonging to the component
            #
            component_cells = []

            # set the visited flag for the seed cell of the component
            #
            visited[row_idx, col_idx] = True

            # perform the breadthfirst traversal until the queue is empty
            #
            while bfs_queue:

                # pop the first element in the queue (FIFO order)
                #
                current_row, current_col = bfs_queue.pop(0)

                # append the popped cell to the component collection
                #
                component_cells.append((current_row, current_col))

                # iterate over each neighbor offset defined earlier
                #
                for off_row, off_col in neighbor_offsets:

                    # compute the neighbor's absolute grid position
                    #
                    neighbor_row = current_row + off_row
                    neighbor_col = current_col + off_col

                    # check that the neighbor lies inside grid bounds
                    #
                    within_bounds = (0 <= neighbor_row < grid_height \
                                     and 0 <= neighbor_col < grid_width)

                    # proceed only if the neighbor is inside the grid
                    #
                    if within_bounds:

                        # skip processing if the neighbor is already visited
                        #
                        if visited[neighbor_row, neighbor_col]:

                            # continue with next neighbor offset
                            #
                            continue

                        # verify that the neighbor holds the same class value
                        #
                        if label_grid[neighbor_row, neighbor_col] == class_value:

                            # mark the neighbor as visited to avoid requeuing
                            #
                            visited[neighbor_row, neighbor_col] = True

                            # add the neighbor to the bfs queue for further expansion
                            #
                            bfs_queue.append((neighbor_row, neighbor_col))

            # create a list to hold unit polygons for the collected component cells
            #
            unit_polygons = []

            # iterate over each component cell to build its corresponding polygon
            #
            for comp_row, comp_col in component_cells:

                # build a square polygon representing the patch area in grid space
                #
                square_polygon: geom.Polygon = geom.Polygon([
                    (comp_col, comp_row),
                    (comp_col + 1, comp_row),
                    (comp_col + 1, comp_row + 1),
                    (comp_col, comp_row + 1)
                ])

                # append the square polygon to the accumulator list
                #
                unit_polygons.append(square_polygon)

            # merge all unit polygons into a single (possibly multipart) geometry
            #
            merged_geometry = ops.unary_union(unit_polygons)

            # obtain an iterable over individual polygons for multigeometries
            #
            geometry_iter = merged_geometry.geoms if \
                hasattr(merged_geometry, "geoms") else [merged_geometry]

            # compute the average confidence across all cells in the component
            #
            average_confidence = float(
                np.mean([confidence_grid[r, c] for r, c in component_cells])
            )

            # iterate over each polygon produced by the unary union
            # operation
            #
            for polygon in geometry_iter:

                # convert every vertex from grid units to pixel units
                #
                pixel_vertices = [
                    (int(x_coord) * frmsize, int(y_coord) * frmsize, 0)
                    for x_coord, y_coord in polygon.exterior.coords
                ]

                # construct the annotation entry for the current region id
                #
                annotations[region_id] = {
                    nda.CKEY_REGION_ID: region_id,
                    nda.CKEY_TEXT: label_order(class_value).name,
                    nda.CKEY_COORDINATES: pixel_vertices,
                    nda.CKEY_CONFIDENCE: average_confidence,
                    nda.CKEY_TISSUE_TYPE: nda.DEF_GRAPH_TISSUE,
                    nda.CKEY_GEOM_PROPS: {
                        nda.CKEY_LENGTH: 0.0,
                        nda.CKEY_AREA: 0.0,
                        nda.CKEY_MICRON_LENGTH: 0.0,
                        nda.CKEY_MICRON_AREA: 0.0
                    }
                }

                # increment the region identifier for the next region
                #
                region_id += 1

    # return the filled annotations dictionary to the caller
    #
    return annotations

#
# end of function

def _build_conf_grid(labels, coords, confs,
                     label_grid, frmsize, label_order):
    """
    function: _build_conf_grid
    
    arguments:
     labels: list of patch-level label names
     coords: list of (x, y) patch positions
     confs: list of confidence values per patch
     label_grid: mode-filtered integer grid
     frmsize: the frame size of each frame
     label_order: the label (priority) order

    return: a 2-D numpy array of float confidences

    description:
     Fills confidence where the filtered grid matches the original
     patch label id.
    """

    # retrieve grid dimensions for bound checking
    #
    rows, cols = label_grid.shape
    
    # allocate a zero-initialized confidence grid of identical shape
    #
    conf_grid = np.zeros((rows, cols), dtype = np.float32)
    
    # iterate over every patch prediction
    #
    for lbl, conf_val, (x_pos, y_pos) in zip(labels, confs, coords):
        
        # compute the grid row index for the patch
        #
        c = x_pos // frmsize
        
        # compute the grid column index for the patch
        #
        r = y_pos // frmsize
        
        # proceed only if the indices are valid
        #
        if 0 <= r < rows and 0 <= c < cols:
            
            # record confidence when the mode id equals the patch id
            #
            if label_grid[r, c] == label_order[lbl].value:
                conf_grid[r, c] = conf_val

    # return the completed confidence grid
    #
    return conf_grid
#
# end of function

def generate_sparse_matrices(labels,
                             top_left_coords,
                             confidence_values,
                             label_order):
    """
    function: generate_sparse_matrices

    arguments:
     labels: list of label names
     top_left_coords: list of (x, y) tuples
     confidence_values: list of confidence floats
     label_order: the label order

    return:
     dictionary mapping label to coordinates and confidences

    description:
     This function groups coordinates and confidences by label.
    """

    # Initialize mapping from label to its data
    #
    sparse_map = {lbl.name: {nda.CKEY_COORDINATES: [], CKEY_CONFIDENCES: []}
                  for lbl in label_order}

    # Populate mapping with each label's points and confidences
    #
    for lbl, coord, conf in zip(labels, top_left_coords, confidence_values):

        # Append coordinate to the corresponding label list
        #
        sparse_map[lbl][nda.CKEY_COORDINATES].append(coord)

        # Append confidence to the corresponding label list
        #
        sparse_map[lbl][CKEY_CONFIDENCES].append(conf)

    # Return the structured dictionary
    #
    return sparse_map

#
# end of function

def is_within_bounds(point, bottom_right, top_left = (0, 0)):
    """
    function: is_within_bounds

    arguments:
     point: (x, y) tuple
     bottom_right: maximum allowed (x, y)
     top_left: minimum allowed (x, y)

    return:
     True if point is within rectangle bounds

    description:
     Checks whether a point lies within the given bounds.
    """

    # Compare x coordinate within range
    #
    if top_left[0] <= point[0] <= bottom_right[0]:

        # Compare y coordinate within range
        #
        if top_left[1] <= point[1] <= bottom_right[1]:
            return True

    # Return False if any check fails
    #
    return False

#
# end of function

def fill_region(matrix, start):
    """
    function: fill_region

    arguments:
     matrix: 2D numpy array
     start: starting (x, y) tuple for flood fill

    return:
     None (in-place fill)

    description:
     Performs flood-fill to set all connected zeros to one.
    """

    # Compute bottom-right bound index
    #
    max_idx = (matrix.shape[0] - 1, matrix.shape[1] - 1)

    # Use deque for efficient pops from left
    #
    queue = deque([start])

    # Loop until no more points to process
    #
    while queue:

        # Dequeue current point
        #
        x, y = queue.popleft()

        # Skip if already set
        #
        if matrix[x, y] == 1:
            continue

        # Mark the cell
        #
        matrix[x, y] = 1

        # Gather neighbors
        #
        neighbors = [(x-1, y), (x+1, y), (x, y-1), (x, y+1)]

        # Enqueue valid zero-valued neighbors
        #
        for nx, ny in neighbors:
            if is_within_bounds((nx, ny), max_idx) and matrix[nx, ny] == 0:
                queue.append((nx, ny))

#
# end of function

def pad_and_fill(matrix):
    """
    function: pad_and_fill

    arguments:
     matrix: 2D numpy array

    return:
     matrix with holes filled

    description:
     Pads the matrix, flood-fills the pad, and removes the pad.
    """

    # Copy original
    #
    mask = matrix.copy()

    # Pad edges with zeros
    #
    mask = np.pad(mask, pad_width=1, mode=DEF_NUMPY_PAD_MODE)

    # Flood-fill from top-left of padded mask
    #
    fill_region(mask, (0, 0))

    # Invert fill mask
    #
    inv_mask = 1 - mask

    # Remove padding
    #
    inv_mask = inv_mask[1:-1, 1:-1]

    # Combine original and inverted mask
    #
    return matrix + inv_mask
#
# end of function

def align_matrix_sizes(mat1, mat2):
    """
    function: align_matrix_sizes

    arguments:
     mat1: first matrix
     mat2: second matrix

    return:
     tuple of matrices padded to same size

    description:
     Pads the smaller matrix along each dimension to match the larger.
    """

    # Calculate size differences
    #
    diff = np.array(mat1.shape) - np.array(mat2.shape)

    # Pad each matrix accordingly
    #
    for axis, delta in enumerate(diff):

        # Skip if same size on this axis
        #
        if delta == 0:
            continue

        # Define pad widths ((before, after), ...) per axis
        #
        pad_width = [(0, 0), (0, 0)]
        pad_width[axis] = (0, abs(delta))

        # Apply padding to the smaller matrix
        #
        if delta > 0:
            mat2 = np.pad(mat2, pad_width, mode=DEF_NUMPY_PAD_MODE)
        else:
            mat1 = np.pad(mat1, pad_width, mode=DEF_NUMPY_PAD_MODE)

    # Return aligned matrices
    #
    return mat1, mat2

#
# end of function

def build_bitmap(coordinate_list, frmsize):
    """
    function: build_bitmap
    
    arguments:
     coordinate_list: list of patch top left coordinates
     frmsize: the frame size of each annotation

    return:
     a 2D numpy array where foreground cells (value 1) correspond to
     patches for the current label

    description:
     This helper converts the sparse list of patch coordinates into a
     mask.
    """
    
    # handle the edge case of an empty coordinate list by returning a
    # zerosized array to the caller
    #
    if not coordinate_list:
        return np.zeros((0, 0), dtype = np.uint8)
       
    # convert the coordinate list into a NumPy array for vectorised math
    #
    numpy_coords = np.asarray(coordinate_list, dtype = int)
    
    # compute the row indices in grid space by integer division of y by dy
    #
    row_indices = numpy_coords[:, 1] // frmsize
    
    # compute the column indices in grid space by integer division of x by dx
    #
    col_indices = numpy_coords[:, 0] // frmsize
     
    # determine the required bitmap dimensions by inspecting max indices
    #
    bitmap_height = row_indices.max() + 1
    bitmap_width = col_indices.max() + 1
    
    # initialise a binary bitmap with zeros of the computed dimensions
    #
    bitmap = np.zeros((bitmap_height, bitmap_width), dtype = np.uint8)
    
    # set bitmap cells to one for every patch position recorded earlier
    #
    bitmap[row_indices, col_indices] = 1
    
    # return the populated bitmap to the caller
    #
    return bitmap
#
# end of function

def _build_label_grid(labels, coords, frmsize, label_order):
    """
    function: _build_label_grid
    
    arguments:
     labels: list of patch-level label names
     coords: list of (x, y) patch top-left positions
     frmsize: the frame size of each frame
    
    return: a 2-D numpy array of integer class ids

    description:
     Converts sparse patch predictions into a dense grid.
    """

    # handle the edge case of zero patches by returning an empty grid
    #
    if not labels:
        return np.zeros((0, 0), dtype = np.uint8)
    
    # unpack patch dimensions into scalar variables
    #
    dx, dy = frmsize, frmsize
    
    # compute grid indices for each patch
    #
    grid_tuples = \
        [(y // dy, x // dx, lbl) for lbl, (x, y) in zip(labels, coords)]
    
    # determine grid height by finding the maximum row index
    #
    height = max(t[0] for t in grid_tuples) + 1
    
    # determine grid width by finding the maximum column index
    #
    width = max(t[1] for t in grid_tuples) + 1
    
    # allocate a zero-initialized integer grid of the computed dimensions
    #
    grid = np.zeros((height, width), dtype = np.uint8)
    
    # write the integer class id into the appropriate grid locations
    #
    for row, col, lbl in grid_tuples:
        grid[row, col] = label_order[lbl].value
        
    # return the populated integer label grid to the caller
    #
    return grid

#
# end of function

#-------------------------------------------------------------------------------
#
# classes defined here
#
#-------------------------------------------------------------------------------

class ConfidencePostProcessor:
    """
    class: ConfidencePostProcessor

    description:
     This concrete class generates region predictions by expanding patch
     indicators into solid regions using a flood fill algorithm.  When
     multiple classes overlap a cell, the class with the higher average
     confidence takes precedence.
    """

    def __init__(self, pmap = None):
        """
        method: __init__

        arguments:
         label_order: a label priority pmap

        return: None

        description:
         this is the class constructor method
        """

        # create label order variable
        #
        self.label_order = None
        
        # set label_order if pmap present
        #
        if pmap is not None:
            self.label_order = _make_label_enum(pmap)
        
    #
    # end of method

    def load(self, ffile):
        """
        method: __init__

        arguments:
         ffile: a dpath annotation file

        return: boolean value indicating status

        description:
         this method loads the data in a way the post-processing
         algorithm can use
        """

        # get and save necessary information
        #
        out = unpack_graph(ffile)

        # unpack graph can return None if an error
        # is encountered or the actual data return
        # 
        #
        if out is None:
            return False
        else:
            self.coords, self.confs, self.labels, self.frmsize = out
 
        # exit gracefully
        #  return status
        #
        return True
    #
    # end of method

    def set_label_order(self, label_pmap):
        """
        method: set_label_order

        arguments:
         label_pmap: the raw priority map to load

        return: None

        description:
         sets the label order
        """

        # set label_order if pmap present
        #
        if label_pmap is not None:
            self.label_order = _make_label_enum(label_pmap)
    #
    # end of method
    
    def create_heatmaps(self, sparse_map):
        """
        method: create_heatmaps

        arguments:
         sparse_map: dictionary produced by generate_sparse_matrices

        return:
         two 2D numpy arrays: the label heatmap and the confidence map

        description:
         For each label, build a hole-filled bitmap and assign a per-cell
         confidence by propagating the nearest patch's confidence to all
         filled interior cells. Then merge labels by choosing, at each
         cell, the label with the higher propagated confidence. This
         implements confidence-priority behavior that mirrors label
         priority merging while using per-patch local confidence.
        """

        # initialize the integer label heatmap with a single background cell
        #
        heatmap = np.zeros((1, 1), dtype = np.uint8)

        # initialize the confidence map with the same shape as the heatmap
        #
        confidence_map = np.zeros_like(heatmap, dtype = np.float32)

        # iterate over each label entry in the sparse map
        #
        for label_name, label_data in sparse_map.items():

            # fetch the list of patch coordinates for this label
            #
            coord_list = label_data[nda.CKEY_COORDINATES]

            # fetch the list of patch confidences for this label
            #
            conf_list = label_data[CKEY_CONFIDENCES]

            # continue to next label if there are no patches present
            #
            if not coord_list:
                continue

            # build a binary bitmap representing the current label patches
            #
            patch_bitmap = build_bitmap(coord_list, self.frmsize)

            # fill interior holes within the bitmap
            #
            filled_bitmap = pad_and_fill(patch_bitmap)

            # create a boolean seed mask for patch cells only
            # (same shape as patch_bitmap)
            #
            seed_mask = np.zeros_like(patch_bitmap, dtype = bool)

            # create a float array to hold per-seed confidences at patch cells
            #
            seed_conf = np.zeros_like(patch_bitmap, dtype = np.float32)

            # iterate over patches to place seeds and their confidences
            #
            for (px, py), pconf in zip(coord_list, conf_list):

                # compute the row index from the x coordinate
                #
                c = int(px) // int(self.frmsize)

                # compute the column index from the y coordinate
                #
                r = int(py) // int(self.frmsize)

                # guard against out-of-bounds indices
                #
                if 0 <= r < seed_mask.shape[0] and 0 <= c < seed_mask.shape[1]:

                    # mark this grid cell as a seed location
                    #
                    seed_mask[r, c] = True

                    # store the patch confidence at the seed location
                    #
                    seed_conf[r, c] = float(pconf)

            # skip label if no valid seeds were placed
            #
            if not seed_mask.any():
                continue

            # compute nearest-seed indices for every cell using a distance
            # transform
            #
            # note: passing the logical negation of seed_mask makes seed pixels
            # be zeros, and the transform returns the indices of the nearest
            # zero (nearest seed)
            #
            idx_rows, idx_cols = \
                distance_transform_edt(
                    ~seed_mask, return_indices = True,
                    return_distances=False
                )

            # gather the propagated confidence for every cell from its nearest seed
            #
            propagated_conf = seed_conf[idx_rows, idx_cols]

            # zero-out any cells that are not inside the filled bitmap for this label
            #
            propagated_conf = propagated_conf * (filled_bitmap.astype(np.float32))

            # align per-label confidence to the current global arrays
            #
            propagated_conf, confidence_map = \
                align_matrix_sizes(propagated_conf, confidence_map)

            # align the heatmap to match the (possibly grown) confidence_map
            #
            heatmap, _ = align_matrix_sizes(heatmap, confidence_map)

            # compute the boolean mask where this label's confidence wins locally
            #
            overwrite_mask = propagated_conf > confidence_map

            # write the integer class value into the heatmap wherever this label wins
            #
            heatmap[overwrite_mask] = self.label_order[label_name].value

            # write the winning confidence values into the confidence map
            #
            confidence_map[overwrite_mask] = propagated_conf[overwrite_mask]

        # return the completed heatmap and confidence map to the caller
        #
        return heatmap, confidence_map
    #
    # end of method
    
    def predict_regions(self):
        """
        method: predict_regions

        arguments: None

        return:
         a dictionary in annotation graph format describing the predicted
         tissue regions

        description:
         Entry point required by the BasePostProcessor interface.  It chains
         together sparse map construction, heatmap generation, and region
         merging to produce the final result.
        """

        # build the sparse representation grouping patches by label name
        #
        sparse_map = generate_sparse_matrices(self.labels,
                                              self.coords,
                                              self.confs,
                                              self.label_order)

        # convert sparse patches into dense label and confidence heatmaps
        #
        label_heatmap, confidence_map = \
            self.create_heatmaps(sparse_map)

        # merge the heatmap into polygons and return the annotations
        #
        result = _merge_label_grid(
            label_heatmap,
            confidence_map,
            self.frmsize,
            self.label_order
        )

        # exit gracefully
        #  return result
        #
        return result
    #
    # end of method

#
# end of class

class DilationPostProcessor:
    """
    class: DilationPostProcessor

    description:
     Applies a morphological dilation to the integer label grid.
     The pipeline is:
       1) load(): unpack the dpath ann into patch coords/labels/confs
       2) predict_regions(): build the label grid, dilate, build per-cell
          confidences aligned to the dilated labels, then merge to regions.
    """

    def __init__(self, pmap = None, kernel_size = 2):
        """
        method: __init__

        arguments:
         pmap: a raw label priority map
         kernel_size: the square window size for the max filter

        return: None

        description:
         initializes instance variables
        """

        # save the dilation window size
        #
        self.kernel_size = int(kernel_size)

        # set label order
        #
        self.label_order = None
        if pmap is not None:
            self.label_order = _make_label_enum(pmap)

        # initialize state members
        #
        self.labels = None
        self.coords = None
        self.confs = None
        self.frmsize = None
    #
    # end of method

    def set_label_order(self, pmap):
        """
        method: set_label_order

        arguments:
         pmap: label priority map (string -> int)

        return: None

        description:
         sets the label order enum used everywhere else
        """

        # set label order
        #
        if pmap is not None:
            self.label_order = _make_label_enum(pmap)
    #
    # end of method

    def load(self, ffile):
        """
        method: load

        arguments:
         ffile: a dpath annotation file

        return:
         boolean value indicating status

        description:
         unpacks a dpath ann into internal arrays
        """

        # attempt to unpack dpath ann file
        #
        out = unpack_graph(ffile)
        if out is None:
            return False

        # fetch the coords
        #
        self.coords, self.confs, self.labels, self.frmsize = out

        # exit gracefully
        # return status
        #
        return True
    #
    # end of method

    def _apply_dilation(self, grid):
        """
        method: _apply_dilation

        arguments:
         grid: 2-D numpy array of integer label ids

        return:
         a dilated label grid

        description:
         expands label ids via a maximum filter
        """

        # return the grid unchanged if it is empty
        #
        if grid.size == 0:
            return grid

        # apply the maximum filter with the stored kernel size
        #
        dilated = maximum_filter(grid,
                                 size = self.kernel_size,
                                 mode = DEF_NUMPY_PAD_MODE,
                                 cval = 0)

        # cast the dilated grid back to unsigned 8-bit integers
        #
        return dilated.astype(np.uint8)

    #
    # end of method

    def predict_regions(self):
        """
        method: predict_regions

        arguments: none

        return:
         a dictionary in annotation graph format

        description:
         executes dilation + merge using the shared helpers
        """

        # build integer grid from sparse patches
        #
        raw_grid = _build_label_grid(self.labels,
                                     self.coords,
                                     self.frmsize,
                                     self.label_order)

        # expand regions
        #
        dil_grid = self._apply_dilation(raw_grid)

        # per-cell confidences aligned to the dilated labels
        #
        conf_grid = _build_conf_grid(self.labels,
                                     self.coords,
                                     self.confs,
                                     dil_grid,
                                     self.frmsize,
                                     self.label_order)

        # merge into polygons
        #
        return _merge_label_grid(dil_grid,
                                 conf_grid,
                                 self.frmsize,
                                 self.label_order)
    #
    # end of method
#
# end of class

class KnnPostProcessor:
    """
    class: KnnPostProcessor

    description:
     Applies a value-space KNN vote in a 3x3 window to the label grid.
    """

    def __init__(self, k_neighbours = 3, pmap = None):
        """
        method: __init__

        arguments:
         k_neighbours: number of neighbors to include in the vote
         pmap: optional label priority map

        return:
         none

        description:
         constructor that stores K and optional label order
        """

        # store K for neighbor voting
        #
        self.k_neighbours = int(k_neighbours)

        # initialize label order to None
        #
        self.label_order = None

        # set label order if provided
        #
        if pmap is not None:
            self.label_order = _make_label_enum(pmap)
    #
    # end of method

    def set_label_order(self, pmap):
        """
        method: set_label_order

        arguments:
         pmap: a dict mapping label string -> priority integer

        return:
         none

        description:
         sets the internal label order enum used by this processor
        """

        # update label order enum using the provided map
        #
        if pmap is not None:
            self.label_order = _make_label_enum(pmap)
    #
    # end of method

    def load(self, ffile):
        """
        method: load

        arguments:
         ffile: path to a dpath annotation file

        return:
         boolean value indicating status

        description:
         loads patch coordinates, confidences, labels, and frame size
        """

        # call shared unpack routine to fetch inputs
        #
        out = unpack_graph(ffile)

        # handle load failure
        #
        if out is None:
            return False

        # unpack the tuple into instance fields
        #
        self.coords, self.confs, self.labels, self.frmsize = out

        # return success
        #
        return True
    #
    # end of method

    def _reducer(self, values):
        """
        method: _reducer

        arguments:
         values: flattened 3x3 neighborhood as 1-D numpy array

        return:
         integer label id after KNN vote

        description:
         chooses the most frequent among the K closest in absolute value
        """

        # fetch the center element
        #
        cval = int(values[len(values) // 2])

        # typecast values
        #
        vals = values.astype(np.int16)
        
        # return background when all are zero
        #
        if np.all(values == 0):
            return 0

        # compute absolute differences to the center value
        #
        diffs = np.abs(vals - cval)

        # find indices of the K smallest differences
        #
        idx = np.argsort(diffs)[: self.k_neighbours]

        # gather the K nearest values
        #
        kvals = vals[idx].astype(np.uint8)

        # select the most common value among the K
        #
        return Counter(kvals).most_common(1)[0][0]
    #
    # end of method

    def _knn_filter(self, grid):
        """
        method: _knn_filter

        arguments:
         grid: 2-D numpy array of integer label ids

        return:
         a filtered 2-D numpy array of integer label ids

        description:
         runs a 3x3 generic_filter with the KNN reducer
        """

        # short circuit on empty input
        #
        if grid.size == 0:
            return grid

        # apply generic_filter with the reducer
        #
        flt = generic_filter(
            grid, function = self._reducer,
            size = 3, mode = DEF_NUMPY_PAD_MODE, cval = 0
        )

        # cast result back to uint8
        #
        return flt.astype(np.uint8)
    #
    # end of method

    def predict_regions(self):
        """
        method: predict_regions

        arguments:
         none

        return:
         annotation graph dictionary after KNN and merge

        description:
         builds the label grid, applies KNN filter, aligns confidences, and merges
        """

        # build integer label grid from sparse patches
        #
        lbl_grid = _build_label_grid(
            self.labels, self.coords,
            self.frmsize, self.label_order
        )

        # apply KNN value-space smoothing to the grid
        #
        knn_grid = self._knn_filter(lbl_grid)

        # build per-cell confidence grid consistent with the filtered labels
        #
        conf_grid = _build_conf_grid(
            self.labels, self.coords, self.confs,
            knn_grid, self.frmsize, self.label_order
        )

        # merge the grid into polygons and return annotations
        #
        return _merge_label_grid(
            knn_grid, conf_grid, self.frmsize, self.label_order
        )
    #
    # end of method
#
# end of class

class ModePostProcessor:
    """
    class: ModePostProcessor

    description:
     Applies a 3x3 mode filter to the label grid and merges regions.
    """

    def __init__(self, pmap = None):
        """
        method: __init__

        arguments:
         pmap: optional label priority map

        return:
         none

        description:
         constructor that stores optional label order
        """

        # initialize label order to None
        #
        self.label_order = None

        # set label order if provided
        #
        if pmap is not None:
            self.label_order = _make_label_enum(pmap)
    #
    # end of method

    def set_label_order(self, pmap):
        """
        method: set_label_order

        arguments:
         pmap: a dict mapping label string -> priority integer

        return:
         none

        description:
         sets the internal label order enum used by this processor
        """

        # update label order enum using the provided map
        #
        if pmap is not None:
            self.label_order = _make_label_enum(pmap)
    #
    # end of method

    def load(self, ffile):
        """
        method: load

        arguments:
         ffile: path to a dpath annotation file

        return:
         boolean value indicating status

        description:
         loads patch coordinates, confidences, labels, and frame size
        """

        # call shared unpack routine to fetch inputs
        #
        out = unpack_graph(ffile)

        # handle load failure
        #
        if out is None:
            return False

        # unpack the tuple into instance fields
        #
        self.coords, self.confs, self.labels, self.frmsize = out

        # return success
        #
        return True
    #
    # end of method

    def _numpy_mode(self, values):
        """
        method: _numpy_mode

        arguments:
         values: flattened neighborhood values

        return:
         integer label id equal to the mode of values

        description:
         computes a robust statistical mode using numpy operations
        """

        # compute unique values and their counts
        #
        uniq, cnt = np.unique(values, return_counts = True)

        # select the value with the maximum count
        #
        return int(uniq[np.argmax(cnt)])
    #
    # end of method

    def _mode_filter(self, grid):
        """
        method: _mode_filter

        arguments:
         grid: 2-D numpy array of integer label ids

        return:
         a filtered 2-D numpy array after mode filtering

        description:
         runs a 3x3 generic_filter that computes the statistical mode
        """

        # short circuit on empty input
        #
        if grid.size == 0:
            return grid

        # apply generic_filter with the mode reducer
        #
        flt = generic_filter(
            grid, function = self._numpy_mode,
            size = 3, mode = DEF_NUMPY_PAD_MODE, cval = 0
        )

        # cast result back to uint8
        #
        return flt.astype(np.uint8)
    #
    # end of method

    def predict_regions(self):
        """
        method: predict_regions

        arguments:
         none

        return:
         annotation graph dictionary after mode filter and merge

        description:
         builds the label grid, applies mode filter, aligns confidences, and merges
        """

        # build integer label grid from sparse patches
        #
        lbl_grid = _build_label_grid(
            self.labels, self.coords,
            self.frmsize, self.label_order
        )

        # apply the 3x3 statistical mode filter
        #
        md_grid = self._mode_filter(lbl_grid)

        # build per-cell confidence grid consistent with the filtered labels
        #
        conf_grid = _build_conf_grid(
            self.labels, self.coords, self.confs,
            md_grid, self.frmsize, self.label_order
        )

        # merge the grid into polygons and return annotations
        #
        return _merge_label_grid(
            md_grid, conf_grid,
            self.frmsize, self.label_order
        )
    #
    # end of method
#
# end of class

class PriorityFloodFillPostProcessor:
    """
    class: PriorityFloodFillPostProcessor

    description:
     Expands label bitmaps by hole-filling and writes them in a fixed
     priority order into the cumulative heatmap.
    """

    def __init__(self, pmap = None):
        """
        method: __init__

        arguments:
         pmap: optional label priority map

        return:
         none

        description:
         constructor that stores optional label order
        """

        # initialize label order to None
        #
        self.label_order = None

        # set label order if provided
        #
        if pmap is not None:
            self.label_order = _make_label_enum(pmap)
    #
    # end of method

    def set_label_order(self, pmap):
        """
        method: set_label_order

        arguments:
         pmap: a dict mapping label string -> priority integer

        return:
         none

        description:
         sets the internal label order enum used by this processor
        """

        # update label order enum using the provided map
        #
        if pmap is not None:
            self.label_order = _make_label_enum(pmap)
    #
    # end of method

    def load(self, ffile):
        """
        method: load

        arguments:
         ffile: path to a dpath annotation file

        return:
         boolean value indicating status

        description:
         loads patch coordinates, confidences, labels, and frame size
        """

        # call shared unpack routine to fetch inputs
        #
        out = unpack_graph(ffile)

        # handle load failure
        #
        if out is None:
            return False

        # unpack the tuple into instance fields
        #
        self.coords, self.confs, self.labels, self.frmsize = out

        # return success
        #
        return True
    #
    # end of method

    def predict_regions(self):
        """
        method: predict_regions

        arguments:
         none

        return:
         annotation graph dictionary after priority fill and merge

        description:
         hole-fills per label, writes to background in priority order,
         then merges
        """

        # group sparse data by label name
        #
        sparse_map = generate_sparse_matrices(
            self.labels, self.coords,
            self.confs, self.label_order
        )

        # initialize cumulative heatmap as 1x1 background
        #
        heatmap = np.zeros((1, 1), dtype = np.uint8)

        # initialize parallel confidence map
        #
        conf_map = np.zeros_like(heatmap, dtype = np.float32)

        # iterate labels by priority order
        #
        for lbl in self.label_order:

            # fetch current label name
            #
            name = lbl.name

            # fetch coordinate list for this label
            #
            coords = sparse_map[name][nda.CKEY_COORDINATES]

            # continue if there are no patches
            #
            if not coords:
                continue

            # build a binary bitmap for this label
            #
            bmp = build_bitmap(coords, self.frmsize)

            # fill internal holes in the bitmap
            #
            bmp = pad_and_fill(bmp)

            # align the bitmap with the cumulative heatmap
            #
            bmp, heatmap = align_matrix_sizes(bmp, heatmap)

            # align the confidence map shape to the heatmap
            #
            conf_map, _ = align_matrix_sizes(conf_map, heatmap)

            # compute the label mean confidence for write bookkeeping
            #
            mean_conf = float(np.mean(sparse_map[name][CKEY_CONFIDENCES]))

            # compute write mask for background cells only
            #
            write_mask = (bmp == 1) & (heatmap == 0)

            # write the integer label value where allowed by the mask
            #
            heatmap[write_mask] = lbl.value

            # write the bookkeeping confidence in the same cells
            #
            conf_map[write_mask] = mean_conf

        # merge the final heatmap to polygons and return
        #
        return _merge_label_grid(
            heatmap, conf_map,
            self.frmsize, self.label_order
        )
    #
    # end of method
#
# end of class

class PostProcessor:
    """
    class: PostProcessor

    description:
     Lightweight facade that forwards all API calls to a concrete post
     processing implementation chosen at instantiation time.
    """

    def __init__(self, alg, pmap):
        """
        method: __init__

        arguments:
         alg: post processing algorithm
         pmap: label priority map

        return: None

        description:
         constructor
        """

        # store the requested processor type in lower case form
        #
        alg = alg.lower()

        # validate that the requested processor is available in
        # the dictionary
        #
        if alg not in PPROC_OBJECTS:
            print("Error: %s (line: %s) %s: %s (%s)" %
                  (__FILE__, ndt.__LINE__, ndt.__NAME__,
                   "unkown processor type", alg))
            sys.exit(os.EX_SOFTWARE)

        # keep a reference to the singleton implementation
        # for future calls
        #
        self.impl = copy.deepcopy(PPROC_OBJECTS[alg][1])
        self.pmap = pmap
        self.set_pmap(pmap) 
    #
    # end of method

    def load(self, ffile):
        """
        method: load

        arguments:
         ffile: a dpath annotation file

        return: boolean value indicating status

        description:
         calls a specific algorithms load method
        """

        # check to ensure an algorithm has been set
        #
        if not hasattr(self, "impl") or self.impl is None:
            print("Error: %s (line: %s) %s: %s" %
                  (__FILE__, ndt.__LINE__, ndt.__NAME__,
                   "algorithm not defined"))
            sys.exit(os.EX_SOFTWARE)

        # call algorithm classes load method
        #
        status = self.impl.load(ffile)

        # exit gracefully
        #  return status
        #
        return status
    #
    # end of method
    
    def set_alg(self, alg):
        """
        method: set_alg

        arguments:
         alg: post-processing algorithm to set

        return: none

        description:
         post-processing algorithm setter method
        """

        alg = alg.lower()
        
        # validate that the requested processor is available
        # in the dictionary
        #
        if alg not in PPROC_OBJECTS:
            print("Error: %s (line: %s) %s: %s (%s)" %
                  (__FILE__, ndt.__LINE__, ndt.__NAME__,
                   "unkown processor type", alg))
            sys.exit(os.EX_SOFTWARE)
            
        self.impl = copy.deepcopy(PPROC_OBJECTS[alg][1])
        self.impl.set_label_order(self.pmap)

    #
    # end of method

    def set_pmap(self, pmap):
        """
        method: set_pmap

        arguments:
         pmap: post-processing algorithm priority map to set

        return: none

        description:
         post-processing algorithm setter method
        """
        self.pmap = pmap
        self.impl.set_label_order(self.pmap)
    #
    # end of method

    def get_alg(self):
        """
        method: get_alg

        arguments: none

        return: specified algorithm

        description:
         post-processing algorithm getter method
        """
        return self.impl

    #
    # end of method

    def get_pmap(self):
        """
        method: get_pmap

        arguments: none

        return: none

        description:
         post-processing algorithm getter method
        """
        return self.pmap
    #
    # end of method

    def predict_regions(self):
        """
        method: predict_regions

        arguments: None

        return:
         annotation graph dictionary produced by the selected algorithm

        description:
         Delegates the call to the concrete post processor chosen during
         construction.
        """

        # dispatch the predict_regions call to the concrete implementation
        #
        return self.impl.predict_regions()

    #
    # end of method

#
# end of class

class SlidingWindowPreProcessor:
    """
    class: SlidingWindowPreProcessor

    description:
     Implement a sliding window based patch extraction preprocessing algorithm.
    """
        
    def get_image_dataset(self, fpath):
        """
        method: get_image_dataset

        arguments:
         fpath: img path to get data from
        
        return:
         dataset: the created image dataset
         coords: all top left coordinates of all the patches
        
        description:
         This function will store img signals into an image dataset
        """

        # display debug information
        #
        if dbgl > ndt.BRIEF:
            print("%s (line: %s) %s:: fetching image dataset (%s)" %
                  (__FILE__, ndt.__LINE__, ndt.__NAME__, fname))

        # set img reader to mil if svs else pil
        #
        if nit.Mil().is_mil(fpath):            
            img = nit.Mil()
            
        else:
            img = nit.Pil()

        # open fpath
        #
        img.open(fpath)
        
        # extract image dimensions
        #
        self.img_width, self.img_height = img.get_dimension()
        img.close()

        # generate grid coordinates
        #
        coords_frms = [
            (x, y)
            for x in range(0, self.img_width - self.frmsize + 1, self.frmsize)
            for y in range(0, self.img_height - self.frmsize + 1, self.frmsize)
        ]

        coords_windows = [
            (
                x + self.frmsize // 2 - self.window_size // 2,
                y + self.frmsize // 2 - self.window_size // 2
            )
            for (x, y) in coords_frms
        ]

        # create the dataset
        #
        dataset = \
            ndi.DecodeImage(
                fpath,
                coords_windows,
                self.window_size,
                self.transforms
            )
        
        # exit gracefully
        # return dataset
        #
        return dataset, coords_frms
    #
    # end of method
#
# end of class

class TissueMaskPreProcessor:
    """
    class: TissueMaskPreProcessor

    description:
     This class implements a tissue mask based patch extraction preprocessing
     algorithm. Only patches containing tissue are extracted based on a
     morphological operation.
    """
        
    def get_image_dataset(self, fpath):
        """
        method: get_image_dataset

        arguments:
         fpath: img path to get data from
        
        return:
         dataset: the created image dataset
         coords: all top left coordinates of all the patches
        
        description:
         This function will store img signals into an image dataset
        """

        # display debug information
        #
        if dbgl > ndt.BRIEF:
            print("%s (line: %s) %s:: fetching image dataset (%s)" %
                  (__FILE__, ndt.__LINE__, ndt.__NAME__, fname))

        # set img reader to mil if svs else pil
        #
        if nit.Mil().is_mil(fpath):            
            img = nit.Mil()
            
        else:
            img = nit.Pil()

        # open fpath
        #
        img.open(fpath)
        
        # extract image dimensions
        #
        self.img_width, self.img_height = img.get_dimension()
        img.close()
        
        if dbgl > ndt.BRIEF:
            print(f"Image dimensions: {self.img_width}x{self.img_height}")
        
        # create a tissue mask
        #
        status, masks = ndtm.run([fpath])
        
        # as we are extracting masks from only one image, we only need the
        # first one
        #
        mask = masks[0]
        
        # initialize coordinates
        #
        coords_windows = []
        
        # status is a boolean indicating if the mask was created successfully
        # if it's true, we can proceed
        
        if status:
            # extract the mask dimensions
            #
            mask_h, mask_w = mask.shape
        
            # the mask is the downsampled version of the original image. So, we
            # compute scale factors from original to mask dimensions
            #
            scale_x = int(self.img_width  / mask_w)
            scale_y = int(self.img_height / mask_h)
            
            # compute frame and window sizes in mask pixels
            #
            mask_frm = max(1, int(self.frmsize / scale_x))
            mask_win = max(1, int(self.window_size / scale_x))
            
            # generate frame coordinates in mask pixels. This is the same
            # coordinates generated in get_image_dataset (without 
            # preprocessing), but now in mask
            # pixels instead of original image pixels
            #
            coords_frms_m = [
                (mx, my)
                for mx in range(0, mask_w - mask_frm + 1, mask_frm)
                for my in range(0, mask_h - mask_frm + 1, mask_frm)
            ]
            
            # generate window coordinates in mask pixels. This is the same
            # coordinates generated in get_image_dataset (without 
            # preprocessing), but now in mask
            # pixels instead of original image pixels
            #
            coords_win_m = [
                (mx + mask_frm // 2 - mask_win // 2,
                my + mask_frm // 2 - mask_win // 2)
                for mx, my in coords_frms_m
            ]
            # filter only those mask‐windows containing tissue
            #
            valid_frms_m = []
            valid_win_m  = []
            
            for (mx, my), (wx, wy) in zip(coords_frms_m, coords_win_m):
                
                # extract the patch from the mask based on the window
                # size in mask pixels
                # 
                patch = mask[wy:wy + mask_win, wx:wx + mask_win]
                
                # if the patch contains pixel values other than 0
                # (background), it means it's not empty and is a valid patch
                #
                if patch.any():
                    valid_frms_m.append((mx, my))
                    valid_win_m.append((wx, wy))
            
            # if no valid frames or windows were found, we can use the
            # original coordinates generated in mask pixels
            #
            # this can happen if the mask is empty or if the image has no
            # tissue regions at all
            #
            if not valid_frms_m and not valid_win_m:
                if dbgl > ndt.BRIEF:
                    print(f"Warning: No valid frames or windows found in {fpath}.")
                # use the original coordinates in mask pixels as valid
                # frames and windows
                #
                valid_frms_m = coords_frms_m
                valid_win_m = coords_win_m
            
            # map back to original pixel coordinates from mask coordinates by
            # multiplying the scale factors
            #
            coords_frms = [
                (int(fx*scale_x), int(fy*scale_y))
                for fx, fy in valid_frms_m
            ]
            coords_windows = [
                (int(wx*scale_x), int(wy*scale_y))
                for wx, wy in valid_win_m
            ]
            print(f"Number of valid windows: {len(coords_windows)}")
            print(f"Number of valid frames:  {len(coords_frms)}")

        # create the dataset
        #
        dataset = \
            ndi.DecodeImage(
                fpath,
                coords_windows,
                self.window_size,
                self.transforms
            )
        
        # exit gracefully
        #  return dataset
        #
        return dataset, coords_frms
    #
    # end of method
#
# end of class

class PreProcessor:
    """
    class: PreProcessor

    arguments:
     processor_type: string selecting one of the keys in PPROC_OBJECTS

    description:
     Lightweight facade that forwards all API calls to a concrete pre-processing
     implementation chosen at instantiation time.
    """

    def __init__(self, alg, window_size, frmsize, transforms):
        """
        method: __init__

        arguments:
         alg: pre processing algorithm name
         window_size: size of the patches to extract
         frmsize: size of the frames to extract patches from
         transforms: image transforms to apply to each patch

        return: None

        description:
         constructor
        """

        # store the requested processor type in lower case form
        #
        alg = alg.lower()

        # validate that the requested processor is available in the dictionary
        #
        if alg not in PPROC_OBJECTS:
            print("Error: %s (line: %s) %s: %s (%s)" %
                  (__FILE__, ndt.__LINE__, ndt.__NAME__,
                   "unkown processor type", alg))
            sys.exit(os.EX_SOFTWARE)

        # keep a reference to the singleton implementation for future calls
        #
        self.impl = copy.deepcopy(PPROC_OBJECTS[alg][1])
        
        # store parameters
        #
        self.impl.window_size = window_size
        self.impl.frmsize = frmsize
        self.impl.transforms = transforms
    #
    # end of method
    
    def set_alg(self, alg):
        """
        method: set_alg

        arguments:
         alg: post-processing algorithm to set

        return: none

        description:
         post-processing algorithm setter method
        """

        alg = alg.lower()
        
        # validate that the requested processor is available in the dictionary
        #
        if alg not in PPROC_OBJECTS:
            print("Error: %s (line: %s) %s: %s (%s)" %
                  (__FILE__, ndt.__LINE__, ndt.__NAME__,
                   "unkown processor type", alg))
            sys.exit(os.EX_SOFTWARE)
            
        self.impl = copy.deepcopy(PPROC_OBJECTS[alg][1])

    #
    # end of method

    def get_alg(self):
        """
        method: get_alg

        arguments: none

        return: specified algorithm

        description:
         post-processing algorithm getter method
        """
        return self.impl

    #
    # end of method

    def get_image_dataset(self, fpath):
        """
        method: get_image_dataset

        arguments:
         fpath: img path to get data from
        
        return:
         dataset: the created image dataset
         coords: all top left coordinates of all the patches
        
        description:
         This function will store img signals into an image dataset
        """

        # dispatch the get_image_dataset call to the concrete implementation
        #
        return self.impl.get_image_dataset(fpath)

    #
    # end of method

#
# end of class

#-------------------------------------------------------------------------------
#
# define dictionary to use for post processor wrapper
#
#-------------------------------------------------------------------------------

PPROC_OBJECTS = {
    CKEY_ALG_CONFIDENCE: [PPROC_VER, ConfidencePostProcessor()],
    CKEY_ALG_DILATION : [PPROC_VER, DilationPostProcessor()],
    CKEY_ALG_KNN: [PPROC_VER, KnnPostProcessor()],
    CKEY_ALG_MODE : [PPROC_VER, ModePostProcessor()],
    CKEY_ALG_PRIORITY : [PPROC_VER, PriorityFloodFillPostProcessor()],
    CKEY_ALG_SLIDING_WINDOW : [PPROC_VER, SlidingWindowPreProcessor()],
    CKEY_ALG_TISSUE_MASK : [PPROC_VER, TissueMaskPreProcessor()]
}

#
# end of file
