#!/usr/bin/env python
#
# file: $NEDC_NFC/class/python/nedc_dpath_demo_tools/nedc_dpath_demo_shared.py
#
# revision history:
# 20251217 (DH): initial version
#
# description:
#  This file defines shared UI widgets used by the NEDC DPATH demo tool.
#  Widgets include a navigator thumbnail view, an RGB histogram plot, a
#  logging widget, a photo viewer with zoom support, and a heatmap canvas.
#------------------------------------------------------------------------------

# import system modules
#
import logging

# import external modules
#
import numpy as np
from scipy.ndimage import gaussian_filter1d

# import matplotlib modules
#
from matplotlib.backends.backend_qtagg import FigureCanvasQTAgg
from matplotlib.figure import Figure
import math

# import PyQt modules
#
from PyQt6.QtCore import (
    Qt,
    QRectF,
    QPointF,
    pyqtSignal,
    QObject,
    QTimer,
    QEvent
)
from PyQt6.QtGui import (
    QPixmap,
    QBrush,
    QPen,
    QColor,
    QPainter,
    QTransform,
    QTextOption,
)
from PyQt6.QtWidgets import (
    QGraphicsView,
    QGraphicsScene,
    QGraphicsPixmapItem,
    QGraphicsRectItem,
    QPlainTextEdit,
    QVBoxLayout,
    QSizePolicy,
    QFrame,
    QWidget,
    QGraphicsItem,
)

# import NEDC modules
#
from nedc_dpath_demo_config import AppConfig

#------------------------------------------------------------------------------
#
# global constants are listed below
#
#------------------------------------------------------------------------------

# define configuration keys: zoom and viewport behavior
#
CFG_KEY_DEF_SCALE_FACTOR = 'Behavior/DEF_SCALE_FACTOR'
CFG_KEY_INIT_VIEWPORT_W = 'Dimensions/INIT_VIEWPORT_W'
CFG_KEY_INIT_VIEWPORT_H = 'Dimensions/INIT_VIEWPORT_H'
CFG_KEY_WHEEL_DEGREES_PER_STEP = 'Behavior/WHEEL_DEGREES_PER_STEP'
CFG_KEY_DIM_DENOM_FLOOR = 'Behavior/DIM_DENOM_FLOOR'
CFG_KEY_FOLLOW_TIMER_MS = 'Behavior/FOLLOW_TIMER_MS'
CFG_KEY_PRUNE_SCHEDULE_DELAY_MS = 'Behavior/PRUNE_SCHEDULE_DELAY_MS'
CFG_KEY_MOUSE_DRAG_PRUNE_DELAY_MS = 'Behavior/MOUSE_DRAG_PRUNE_DELAY_MS'

# define configuration keys: rendering and caching
#
CFG_KEY_OVERLAY_Z_VALUE = 'Rendering/OVERLAY_Z_VALUE'
CFG_KEY_TILE_Z_VALUE = 'Rendering/TILE_Z_VALUE'
CFG_KEY_MIN_VIEWPORT_EDGE_PX = 'Rendering/MIN_VIEWPORT_EDGE_PX'
CFG_KEY_FOLLOW_TOLERANCE_FRACTION = 'Rendering/FOLLOW_TOLERANCE_FRACTION'
CFG_KEY_LRU_LIMIT_DEFAULT = 'Rendering/LRU_LIMIT_DEFAULT'
CFG_KEY_GUARD_BAND_SCENE_PX_DEFAULT = 'Rendering/GUARD_BAND_SCENE_PX_DEFAULT'
CFG_KEY_TILE_UNLOAD_BUDGET = 'Stream/TILE_UNLOAD_BUDGET'
CFG_KEY_NAVIGATOR_RECT_Z = 'Rendering/NAVIGATOR_RECT_Z'

# define configuration keys: histogram plot settings
#
CFG_KEY_LOGGER_FORMAT = 'Formatting/LOGGER_FORMAT'
CFG_KEY_LOGGER_BORDER_CSS = 'Styles/LOGGER_BORDER_CSS'
CFG_KEY_LOGGER_MIN_HEIGHT_PX = 'Dimensions/LOGGER_MIN_HEIGHT_PX'
CFG_KEY_HIST_NUM_BINS = 'Rendering/HIST_NUM_BINS'
CFG_KEY_HIST_X_MIN = 'Rendering/HIST_X_MIN'
CFG_KEY_HIST_X_MAX = 'Rendering/HIST_X_MAX'
CFG_KEY_HIST_SMOOTH_SIGMA = 'Rendering/HIST_SMOOTH_SIGMA'
CFG_KEY_AXIS_LABEL_X = 'Labels/AXIS_X'
CFG_KEY_AXIS_LABEL_Y = 'Labels/AXIS_Y'
CFG_KEY_HIST_LEGEND_LOC = 'Behavior/HIST_LEGEND_LOC'
CFG_KEY_HIST_COLOR_RED = 'Colors/HIST_COLOR_RED'
CFG_KEY_HIST_COLOR_GREEN = 'Colors/HIST_COLOR_GREEN'
CFG_KEY_HIST_COLOR_BLUE = 'Colors/HIST_COLOR_BLUE'
CFG_KEY_LINE_LABEL_R = 'Labels/LINE_R'
CFG_KEY_LINE_LABEL_G = 'Labels/LINE_G'
CFG_KEY_LINE_LABEL_B = 'Labels/LINE_B'
CFG_KEY_HIST_NORMALIZE_EPS = 'Behavior/HIST_NORMALIZE_EPS'

# define configuration keys: navigator overlay pen and brush styling
#
CFG_KEY_NAVIGATOR_RECT_PEN_R = 'Colors/NAVIGATOR_RECT_PEN_R'
CFG_KEY_NAVIGATOR_RECT_PEN_G = 'Colors/NAVIGATOR_RECT_PEN_G'
CFG_KEY_NAVIGATOR_RECT_PEN_B = 'Colors/NAVIGATOR_RECT_PEN_B'
CFG_KEY_NAVIGATOR_RECT_PEN_WIDTH = 'Dimensions/NAVIGATOR_RECT_PEN_WIDTH'
CFG_KEY_NAVIGATOR_RECT_BRUSH_R = 'Colors/NAVIGATOR_RECT_BRUSH_R'
CFG_KEY_NAVIGATOR_RECT_BRUSH_G = 'Colors/NAVIGATOR_RECT_BRUSH_G'
CFG_KEY_NAVIGATOR_RECT_BRUSH_B = 'Colors/NAVIGATOR_RECT_BRUSH_B'
CFG_KEY_NAVIGATOR_RECT_BRUSH_A = 'Colors/NAVIGATOR_RECT_BRUSH_A'

# define configuration keys: viewer background colors
#
CFG_KEY_BG_WHITE_R = 'Colors/BG_WHITE_R'
CFG_KEY_BG_WHITE_G = 'Colors/BG_WHITE_G'
CFG_KEY_BG_WHITE_B = 'Colors/BG_WHITE_B'

# define generic dict metadata keys
#
META_ITEM_KEY = 'item'
META_LAST_KEY = 'last'

# define axis display mode strings
#
AXIS_MODE_OFF = 'off'
AXIS_MODE_ON = 'on'

# define histogram legend defaults
#
DEF_HIST_LEGEND_FONT_SIZE = 8
DEF_HIST_LEGEND_BBOX_ANCHOR = (0, 1)

#------------------------------------------------------------------------------
#
# classes are listed below
#
#------------------------------------------------------------------------------

class MapNavigator(QGraphicsView):
    """
    class: MapNavigator

    arguments:
      parent: optional parent widget

    description:
      This class provides a small thumbnail view of the full slide and a
      rectangle overlay indicating the current viewport of a linked viewer.
    """

    #--------------------------------------------------------------------------
    #
    # constructor methods
    #
    #--------------------------------------------------------------------------

    def __init__(self, parent=None):
        """
        method: constructor

        arguments:
          parent: optional parent widget

        return:
          none

        description:
          This method initializes the MapNavigator widget and its overlay.
        """

        # call the base class constructor
        #
        super().__init__(parent)

        # create and cache the application configuration accessor
        #
        self.config = AppConfig()

        # create the graphics scene used by this view
        #
        self._scene = QGraphicsScene(self)

        # attach the scene to the view
        #
        self.setScene(self._scene)

        # create the background thumbnail pixmap item
        #
        self._photo = QGraphicsPixmapItem()

        # add the thumbnail item to the scene
        #
        self._scene.addItem(self._photo)

        # create a rectangle item to show the viewer viewport
        #
        self._viewport_rect = QGraphicsRectItem(self._photo)

        # fetch pen color components from configuration
        #
        pen_r = int(self.config.get(CFG_KEY_NAVIGATOR_RECT_PEN_R))
        pen_g = int(self.config.get(CFG_KEY_NAVIGATOR_RECT_PEN_G))
        pen_b = int(self.config.get(CFG_KEY_NAVIGATOR_RECT_PEN_B))

        # fetch pen width from configuration
        #
        pen_w = int(self.config.get(CFG_KEY_NAVIGATOR_RECT_PEN_WIDTH))

        # create the viewport rectangle pen
        #
        pen = QPen(QColor(pen_r, pen_g, pen_b))

        # set the pen width
        #
        pen.setWidth(pen_w)

        # apply the pen to the rectangle
        #
        self._viewport_rect.setPen(pen)

        # fetch brush color components from configuration
        #
        br_r = int(self.config.get(CFG_KEY_NAVIGATOR_RECT_BRUSH_R))
        br_g = int(self.config.get(CFG_KEY_NAVIGATOR_RECT_BRUSH_G))
        br_b = int(self.config.get(CFG_KEY_NAVIGATOR_RECT_BRUSH_B))
        br_a = int(self.config.get(CFG_KEY_NAVIGATOR_RECT_BRUSH_A))

        # create and apply the rectangle brush
        #
        brush = QBrush(QColor(br_r, br_g, br_b, br_a))
        self._viewport_rect.setBrush(brush)

        # set the viewport rectangle z-order
        #
        z_val = float(self.config.get(CFG_KEY_NAVIGATOR_RECT_Z))
        self._viewport_rect.setZValue(z_val)

        # hide the rectangle until a viewer is attached
        #
        self._viewport_rect.setVisible(False)

        # disable interactions in this view
        #
        self.setInteractive(False)

        # disable drag mode
        #
        self.setDragMode(QGraphicsView.DragMode.NoDrag)

        # hide scroll bars
        #
        self.setVerticalScrollBarPolicy(
            Qt.ScrollBarPolicy.ScrollBarAlwaysOff
        )
        self.setHorizontalScrollBarPolicy(
            Qt.ScrollBarPolicy.ScrollBarAlwaysOff
        )

        # do not auto-adjust transforms on input events
        #
        self.setTransformationAnchor(QGraphicsView.ViewportAnchor.NoAnchor)
        self.setResizeAnchor(QGraphicsView.ViewportAnchor.NoAnchor)

        # prevent focus stealing
        #
        self.setFocusPolicy(Qt.FocusPolicy.NoFocus)

        # store the current thumbnail pixmap
        #
        self._thumb = QPixmap()

        # track whether external transforms are locked out
        #
        self._transform_locked = False

        # allow internal transform changes when fitting
        #
        self._allow_internal_transform = False

        # store the viewer this navigator is attached to
        #
        self._attached_viewer = None

        # expand to available space
        #
        self.setSizePolicy(
            QSizePolicy.Policy.Expanding,
            QSizePolicy.Policy.Expanding,
        )

        # exit gracefully
        #
        return None
    #
    # end of method

    #--------------------------------------------------------------------------
    #
    # transform control methods
    #
    #--------------------------------------------------------------------------

    def setTransform(self, transform, combine=False):
        """
        method: setTransform

        arguments:
          transform: new transform to apply
          combine: combine with the current transform when True

        return:
          object: return value from base class, or None when blocked

        description:
          This method prevents callers from changing the view transform after
          the navigator has fitted the thumbnail to the widget.
        """

        # allow transforms when unlocked or when running internal code paths
        #
        if (not self._transform_locked) or self._allow_internal_transform:
            return super().setTransform(transform, combine)

        # block external transform changes
        #
        return None
    #
    # end of method

    #--------------------------------------------------------------------------
    #
    # Qt event handler methods
    #
    #--------------------------------------------------------------------------

    def resizeEvent(self, event):
        """
        method: resizeEvent

        arguments:
          event: Qt resize event

        return:
          none

        description:
          This method refits the thumbnail to the widget on resize.
        """

        # call the base class implementation
        #
        super().resizeEvent(event)

        # refit the thumbnail if present
        #
        if not self._thumb.isNull():

            # temporarily allow internal transform changes
            #
            self._allow_internal_transform = True

            # reset and refit the view
            #
            super().resetTransform()
            super().fitInView(
                self._photo,
                Qt.AspectRatioMode.KeepAspectRatio,
            )

            # re-lock transforms after fitting
            #
            self._allow_internal_transform = False
            self._transform_locked = True

        # update the viewport rectangle from the attached viewer
        #
        if self._attached_viewer is not None:
            self.update_viewport_from_viewer(self._attached_viewer)

        # exit gracefully
        #
        return None
    #
    # end of method

    #--------------------------------------------------------------------------
    #
    # thumbnail and viewport methods
    #
    #--------------------------------------------------------------------------

    def load_thumbnail(self, thumbnail_pixmap):
        """
        method: load_thumbnail

        arguments:
          thumbnail_pixmap: QPixmap to display

        return:
          none

        description:
          This method loads a thumbnail pixmap and fits it to the widget.
        """

        # check for null input
        #
        if thumbnail_pixmap is None or thumbnail_pixmap.isNull():
            return None

        # store the thumbnail
        #
        self._thumb = thumbnail_pixmap

        # display the thumbnail
        #
        self._photo.setPixmap(self._thumb)

        # set the scene rect to the pixmap bounds
        #
        self.setSceneRect(QRectF(self._thumb.rect()))

        # hide the viewport rectangle until updated
        #
        self._viewport_rect.setVisible(False)

        # unlock transforms for the internal fit operation
        #
        self._transform_locked = False
        self._allow_internal_transform = True

        # fit the view to the thumbnail
        #
        super().resetTransform()
        super().fitInView(
            self._photo,
            Qt.AspectRatioMode.KeepAspectRatio,
        )

        # re-lock transforms after fitting
        #
        self._allow_internal_transform = False
        self._transform_locked = True

        # exit gracefully
        #
        return None
    #
    # end of method

    def update_viewport_from_viewer(self, photo_viewer):
        """
        method: update_viewport_from_viewer

        arguments:
          photo_viewer: viewer to sample the visible scene rect from

        return:
          none

        description:
          This method computes a thumbnail-space rectangle for the viewer's
          visible scene region and updates the navigator overlay.
        """

        # check for missing thumbnail or viewer
        #
        if self._thumb.isNull() or photo_viewer is None:
            self._viewport_rect.setVisible(False)
            return None

        # compute the visible scene rectangle of the viewer
        #
        vis = photo_viewer.mapToScene(
            photo_viewer.viewport().rect()
        ).boundingRect()

        # fetch the full scene rectangle
        #
        scene_rect = photo_viewer.sceneRect()

        # check for invalid rectangles
        #
        if vis.isNull() or scene_rect.isNull():
            self._viewport_rect.setVisible(False)
            return None

        # fetch denominator floor for safe scaling
        #
        dim_denom = float(self.config.get(CFG_KEY_DIM_DENOM_FLOOR))

        # convert thumbnail dimensions to floats
        #
        tw = float(self._thumb.width())
        th = float(self._thumb.height())

        # compute the scene dimensions with a floor
        #
        dw = float(scene_rect.width())
        dh = float(scene_rect.height())
        dw = dw if dw > dim_denom else dim_denom
        dh = dh if dh > dim_denom else dim_denom

        # compute scale factors from scene to thumbnail
        #
        sx = tw / dw
        sy = th / dh

        # map the viewer rectangle into thumbnail coordinates
        #
        x0 = (vis.x() - scene_rect.x()) * sx
        y0 = (vis.y() - scene_rect.y()) * sy
        ww = vis.width() * sx
        hh = vis.height() * sy

        # clamp the rectangle to the thumbnail bounds
        #
        img_rect = QRectF(self._thumb.rect())
        thumb_rect = QRectF(x0, y0, ww, hh).intersected(img_rect)

        # enforce a minimal visible rectangle size
        #
        min_edge = float(self.config.get(CFG_KEY_MIN_VIEWPORT_EDGE_PX))

        # create a minimal rect if the mapped rect is too small
        #
        if thumb_rect.width() < 1 or thumb_rect.height() < 1:
            c = thumb_rect.center()
            thumb_rect = QRectF(
                c.x() - min_edge / 2.0,
                c.y() - min_edge / 2.0,
                min_edge,
                min_edge,
            ).intersected(img_rect)

        # update the rectangle geometry
        #
        self._viewport_rect.setRect(thumb_rect)

        # show the rectangle
        #
        self._viewport_rect.setVisible(True)

        # exit gracefully
        #
        return None
    #
    # end of method

    def attach_to_viewer(self, photo_viewer):
        """
        method: attach_to_viewer

        arguments:
          photo_viewer: viewer that drives the navigator overlay updates

        return:
          none

        description:
          This method attaches to the viewer's signals so the navigator stays
          synchronized with pans and zooms.
        """

        # check for missing or already-attached viewer
        #
        if photo_viewer is None or self._attached_viewer is photo_viewer:
            return None

        # store the attached viewer reference
        #
        self._attached_viewer = photo_viewer

        # perform an initial viewport update
        #
        self.update_viewport_from_viewer(photo_viewer)

        # define a small helper to update using the stored viewer
        #
        def _sync_viewport(*_args):
            self.update_viewport_from_viewer(photo_viewer)

        # connect to transform changes
        #
        photo_viewer.transformChanged.connect(_sync_viewport)

        # connect to scroll bar changes
        #
        photo_viewer.horizontalScrollBar().valueChanged.connect(_sync_viewport)
        photo_viewer.verticalScrollBar().valueChanged.connect(_sync_viewport)

        # connect to view rect change signal
        #
        photo_viewer.viewRectChanged.connect(_sync_viewport)

        # exit gracefully
        #
        return None
    #
    # end of method
#
# end of class

class HistogramCanvas(FigureCanvasQTAgg):
    """
    class: HistogramCanvas

    arguments:
      parent: parent widget
      width: figure width in inches
      height: figure height in inches
      dpi: figure dots-per-inch

    description:
      This class renders an RGB histogram using matplotlib, and is intended to
      be embedded into a Qt layout.
    """

    #--------------------------------------------------------------------------
    #
    # constructor methods
    #
    #--------------------------------------------------------------------------

    def __init__(self, parent, width=5.0, height=4.0, dpi=100):
        """
        method: constructor

        arguments:
          parent: parent widget
          width: figure width in inches
          height: figure height in inches
          dpi: figure dots-per-inch

        return:
          none

        description:
          This method initializes the matplotlib figure and its histogram axes.
        """

        # create the matplotlib figure
        #
        self.figure = Figure(
            figsize=(width, height),
            dpi=dpi,
            constrained_layout=True,
        )

        # initialize the canvas with the figure
        #
        super().__init__(self.figure)

        # cache configuration accessor
        #
        self.config = AppConfig()

        # allow the widget to expand
        #
        self.setSizePolicy(
            QSizePolicy.Policy.Expanding,
            QSizePolicy.Policy.Expanding,
        )

        # update size hints
        #
        self.updateGeometry()

        # create a single axes for the plot
        #
        self.axes = self.figure.add_subplot(111)

        # hide axes until data is drawn
        #
        self.axes.axis(AXIS_MODE_OFF)

        # fetch histogram bin range
        #
        x_min = int(self.config.get(CFG_KEY_HIST_X_MIN))
        x_max = int(self.config.get(CFG_KEY_HIST_X_MAX))

        # fetch histogram bin count
        #
        num_bins = int(self.config.get(CFG_KEY_HIST_NUM_BINS))

        # create evenly-spaced bins
        #
        self.bins = np.linspace(x_min, x_max, num_bins)

        # apply the border css styling
        #
        css = self.config.get(CFG_KEY_LOGGER_BORDER_CSS)
        self.setStyleSheet(css)

        # draw the empty canvas
        #
        self.draw()

        # exit gracefully
        #
        return None
    #
    # end of method

    #--------------------------------------------------------------------------
    #
    # histogram helper methods
    #
    #--------------------------------------------------------------------------

    def clear_histogram(self):
        """
        method: clear_histogram

        arguments:
          none

        return:
          none

        description:
          This method clears the plot and reinitializes labels and axis limits.
        """

        # fetch axis labels
        #
        x_lbl = self.config.get(CFG_KEY_AXIS_LABEL_X)
        y_lbl = self.config.get(CFG_KEY_AXIS_LABEL_Y)

        # fetch x-axis limits
        #
        x_min = int(self.config.get(CFG_KEY_HIST_X_MIN))
        x_max = int(self.config.get(CFG_KEY_HIST_X_MAX))

        # clear axes
        #
        self.axes.clear()

        # set axis labels and limits
        #
        self.axes.set_xlabel(x_lbl)
        self.axes.set_ylabel(y_lbl)
        self.axes.set_xlim(x_min, x_max)

        # show axes
        #
        self.axes.axis(AXIS_MODE_ON)

        # exit gracefully
        #
        return None
    #
    # end of method

    def update_histogram(
        self,
        red_channel_values,
        green_channel_values,
        blue_channel_values,
    ):
        """
        method: update_histogram

        arguments:
          red_channel_values: iterable of red channel values
          green_channel_values: iterable of green channel values
          blue_channel_values: iterable of blue channel values

        return:
          none

        description:
          This method computes and plots smoothed, normalized RGB histograms.
        """

        # reset the plot
        #
        self.clear_histogram()

        # fetch smoothing and normalization parameters
        #
        sigma = float(self.config.get(CFG_KEY_HIST_SMOOTH_SIGMA))
        eps = float(self.config.get(CFG_KEY_HIST_NORMALIZE_EPS))

        # compute raw histogram counts
        #
        histogram_counts_red, _ = np.histogram(
            red_channel_values,
            bins=self.bins,
        )
        histogram_counts_green, _ = np.histogram(
            green_channel_values,
            bins=self.bins,
        )
        histogram_counts_blue, _ = np.histogram(
            blue_channel_values,
            bins=self.bins,
        )

        # apply gaussian smoothing
        #
        smoothed_red = gaussian_filter1d(
            histogram_counts_red.astype(float),
            sigma=sigma,
        )
        smoothed_green = gaussian_filter1d(
            histogram_counts_green.astype(float),
            sigma=sigma,
        )
        smoothed_blue = gaussian_filter1d(
            histogram_counts_blue.astype(float),
            sigma=sigma,
        )

        # compute safe denominators
        #
        denom_red = max(float(smoothed_red.sum()), eps)
        denom_green = max(float(smoothed_green.sum()), eps)
        denom_blue = max(float(smoothed_blue.sum()), eps)

        # normalize curves to sum to 1
        #
        smoothed_red /= denom_red
        smoothed_green /= denom_green
        smoothed_blue /= denom_blue

        # compute bin centers for plotting
        #
        bin_centers = (self.bins[:-1] + self.bins[1:]) / 2

        # fetch plot colors
        #
        c_r = self.config.get(CFG_KEY_HIST_COLOR_RED)
        c_g = self.config.get(CFG_KEY_HIST_COLOR_GREEN)
        c_b = self.config.get(CFG_KEY_HIST_COLOR_BLUE)

        # fetch plot labels
        #
        l_r = self.config.get(CFG_KEY_LINE_LABEL_R)
        l_g = self.config.get(CFG_KEY_LINE_LABEL_G)
        l_b = self.config.get(CFG_KEY_LINE_LABEL_B)

        # fetch legend placement
        #
        loc = self.config.get(CFG_KEY_HIST_LEGEND_LOC)

        # plot the curves
        #
        self.axes.plot(
            bin_centers,
            smoothed_red,
            color=c_r,
            label=l_r,
        )
        self.axes.plot(
            bin_centers,
            smoothed_green,
            color=c_g,
            label=l_g,
        )
        self.axes.plot(
            bin_centers,
            smoothed_blue,
            color=c_b,
            label=l_b,
        )

        # add legend
        #
        self.axes.legend(
            loc=loc,
            fontsize=DEF_HIST_LEGEND_FONT_SIZE,
            bbox_to_anchor=DEF_HIST_LEGEND_BBOX_ANCHOR,
        )

        # schedule a redraw
        #
        self.draw_idle()

        # exit gracefully
        #
        return None
    #
    # end of method
#
# end of class

class QTextEditLogger(logging.Handler, QObject):
    """
    class: QTextEditLogger

    arguments:
      parent: parent widget

    description:
      This class is a logging handler that emits log records to a
      QPlainTextEdit.
    """

    # declare a Qt signal for thread-safe text appends
    #
    appendPlainText = pyqtSignal(str)

    def __init__(self, parent):
        """
        method: constructor

        arguments:
          parent: parent widget

        return:
          none

        description:
          This method initializes the handler and its internal QPlainTextEdit.
        """

        # initialize the logging handler base
        #
        logging.Handler.__init__(self)

        # initialize QObject base
        #
        QObject.__init__(self)

        # cache configuration accessor
        #
        self.config = AppConfig()

        # create the text widget
        #
        self.widget = QPlainTextEdit(parent)

        # set the widget to read-only
        #
        self.widget.setReadOnly(True)

        # apply the configured minimum height
        #
        min_h = int(self.config.get(CFG_KEY_LOGGER_MIN_HEIGHT_PX))
        self.widget.setMinimumHeight(min_h)

        # allow the widget to expand
        #
        self.widget.setSizePolicy(
            QSizePolicy.Policy.Expanding,
            QSizePolicy.Policy.Expanding,
        )

        # enable word wrapping
        #
        self.widget.setWordWrapMode(QTextOption.WrapMode.WrapAnywhere)

        # connect signal to widget append slot
        #
        self.appendPlainText.connect(self.widget.appendPlainText)

        # exit gracefully
        #
        return None
    #
    # end of method

    def emit(self, record):
        """
        method: emit

        arguments:
          record: logging record instance

        return:
          none

        description:
          This method formats the record and emits the resulting text.
        """

        # format the record to a message string
        #
        try:
            msg = self.format(record)
        except Exception:
            msg = str(record.getMessage())

        # emit the message through the Qt signal
        #
        self.appendPlainText.emit(msg)

        # exit gracefully
        #
        return None
    #
    # end of method
#
# end of class

class Logger(QWidget):
    """
    class: Logger

    arguments:
      parent: optional parent widget

    description:
      This class wraps a QTextEditLogger into a QWidget for layout embedding.
    """

    def __init__(self, parent=None):
        """
        method: constructor

        arguments:
          parent: optional parent widget

        return:
          none

        description:
          This method installs a QTextEditLogger on the root logger.
        """

        # call base class constructor
        #
        super().__init__(parent)

        # cache configuration accessor
        #
        self.config = AppConfig()

        # create the text logger handler
        #
        text_handler = QTextEditLogger(self)

        # configure the handler formatter
        #
        fmt = self.config.get(CFG_KEY_LOGGER_FORMAT)
        text_handler.setFormatter(logging.Formatter(fmt))

        # install handler on the root logger
        #
        root = logging.getLogger()
        if root.level > logging.INFO:
            root.setLevel(logging.INFO)
        text_handler.setLevel(logging.INFO)
        root.addHandler(text_handler)

        # build the widget layout
        #
        layout = QVBoxLayout()

        # remove margins and spacing
        #
        layout.setContentsMargins(0, 0, 0, 0)
        layout.setSpacing(0)

        # add the log widget
        #
        layout.addWidget(text_handler.widget)

        # set the layout on this widget
        #
        self.setLayout(layout)

        # exit gracefully
        #
        return None
    #
    # end of method
#
# end of class

class PhotoViewer(QGraphicsView):
    """
    class: PhotoViewer

    arguments:
      parent: parent widget

    description:
      This class displays a pixmap in a QGraphicsView and supports wheel zoom.
      It emits signals for transform changes, visible scene rect changes, and
      desired resolution changes for pyramid-based slide rendering.
    """

    # define Qt signals used by outside components
    #
    transformChanged = pyqtSignal(QTransform)
    viewRectChanged = pyqtSignal(QRectF)
    zoomLevelChanged = pyqtSignal(int)
    resolutionChangeRequested = pyqtSignal(int)

    # define how many wheel steps correspond to one pyramid level
    #
    ZOOM_STEPS_PER_LEVEL = 4

    def __init__(self, parent):
        """
        method: constructor

        arguments:
          parent: parent widget

        return:
          none

        description:
          This method initializes the graphics scene, photo item, and
          zoom state.
        """

        # call base class constructor
        #
        super().__init__(parent)

        # cache configuration accessor
        #
        self.config = AppConfig()

        # create the graphics scene
        #
        self._scene = QGraphicsScene(self)

        # fetch initial scene dimensions
        #
        w0 = int(self.config.get(CFG_KEY_INIT_VIEWPORT_W))
        h0 = int(self.config.get(CFG_KEY_INIT_VIEWPORT_H))

        # cache initial scene rect
        #
        self.initSize = QRectF(0, 0, w0, h0)

        # apply initial scene rect
        #
        self._scene.setSceneRect(self.initSize)

        # create the pixmap item
        #
        self._photo = QGraphicsPixmapItem()

        # set fast shape mode
        #
        self._photo.setShapeMode(
            QGraphicsPixmapItem.ShapeMode.BoundingRectShape
        )

        # add item to scene
        #
        self._scene.addItem(self._photo)

        # attach scene to view
        #
        self.setScene(self._scene)

        # set zoom anchors under mouse
        #
        self.setTransformationAnchor(
            QGraphicsView.ViewportAnchor.AnchorUnderMouse
        )
        self.setResizeAnchor(
            QGraphicsView.ViewportAnchor.AnchorUnderMouse
        )

        # hide scroll bars
        #
        self.setVerticalScrollBarPolicy(
            Qt.ScrollBarPolicy.ScrollBarAlwaysOff
        )
        self.setHorizontalScrollBarPolicy(
            Qt.ScrollBarPolicy.ScrollBarAlwaysOff
        )

        # fetch background color components
        #
        br = int(self.config.get(CFG_KEY_BG_WHITE_R))
        bg = int(self.config.get(CFG_KEY_BG_WHITE_G))
        bb = int(self.config.get(CFG_KEY_BG_WHITE_B))

        # apply the background brush
        #
        self.setBackgroundBrush(QBrush(QColor(br, bg, bb)))

        # set a box frame
        #
        self.setFrameShape(QFrame.Shape.Box)

        # enable mouse tracking for follow behavior
        #
        self.setMouseTracking(True)

        # initialize empty state flag
        #
        self._empty = True

        # initialize zoom counter
        #
        self._zoom = 0

        # initialize a wheel delta accumulator for high-resolution devices (mac trackpads)
        #
        self._wheel_delta_accum = 0.0
        
        # enable touch events on the viewport so Qt delivers gesture events
        #
        self.viewport().setAttribute(Qt.WidgetAttribute.WA_AcceptTouchEvents, True)
        
        # register pinch gesture on the viewport for pinch-to-zoom on macOS
        #
        self.viewport().grabGesture(Qt.GestureType.PinchGesture)

        # initialize current pyramid level
        #
        self._current_pyramid_level = 0

        # initialize list of downsample values
        #
        self._downs = []

        # initialize prune callback
        #
        self._tile_prune_cb = None

        # track the last emitted desired level
        #
        self._last_emitted_level = None

        # create a timer used to batch prune requests
        #
        self._prune_timer = QTimer(self)

        # configure timer as single-shot
        #
        self._prune_timer.setSingleShot(True)

        # connect the timer to the prune hook
        #
        self._prune_timer.timeout.connect(self._perform_stream_prune)

        # exit gracefully
        #
        return None
    #
    # end of method

    #--------------------------------------------------------------------------
    #
    # configuration and state methods
    #
    #--------------------------------------------------------------------------

    def set_tile_prune_callback(self, callback):
        """
        method: set_tile_prune_callback

        arguments:
          callback: callable used to prune stream tiles

        return:
          none

        description:
          This method sets the callback invoked when the prune timer fires.
        """

        # store the callback
        #
        self._tile_prune_cb = callback

        # exit gracefully
        #
        return None
    #
    # end of method

    def set_downsamples(self, downs):
        """
        method: set_downsamples

        arguments:
          downs: iterable of downsample values

        return:
          none

        description:
          This method stores the per-level downsample factors as floats.
        """

        # normalize downsample values
        #
        self._downs = [float(d) for d in (downs or [])]

        # exit gracefully
        #
        return None
    #
    # end of method

    def current_pyramid_level(self):
        """
        method: current_pyramid_level

        arguments:
          none

        return:
          int: current pyramid level used for tile generation

        description:
          This method returns the currently selected pyramid level.
        """

        # return the current level
        #
        return int(self._current_pyramid_level)
    #
    # end of method

    def set_current_pyramid_level(self, level):
        """
        method: set_current_pyramid_level

        arguments:
          level: new pyramid level

        return:
          none

        description:
          This method updates the current level and schedules a prune.
        """

        # store the new level
        #
        self._current_pyramid_level = int(level)

        # set last emitted to match current
        #
        self._last_emitted_level = self._current_pyramid_level

        # schedule a stream prune after an update delay
        #
        ms = int(self.config.get(CFG_KEY_PRUNE_SCHEDULE_DELAY_MS))
        self._schedule_stream_prune(ms)

        # exit gracefully
        #
        return None
    #
    # end of method

    #--------------------------------------------------------------------------
    #
    # pixmap and view methods
    #
    #--------------------------------------------------------------------------

    def viewportEvent(self, event):
        """
        method: viewportEvent

        arguments:
          event: the viewport event object

        return:
          bool: true if the event was handled

        description:
          This method intercepts viewport gesture events (e.g., pinch-to-zoom).
        """

        # check if this is a Qt gesture event
        #
        if event.type() == QEvent.Type.Gesture:

            # delegate gesture handling
            #
            handled = self._on_viewport_gesture(event)

            # return handled state
            #
            return bool(handled)

        # defer to base implementation for non-gesture events
        #
        return super().viewportEvent(event)
    #
    # end of method

    def _on_viewport_gesture(self, event):
        """
        method: _on_viewport_gesture

        arguments:
          event: the gesture event object

        return:
          bool: true if a gesture was handled

        description:
          This method handles pinch gestures and maps them to view zoom.
        """

        # fetch pinch gesture (if present)
        #
        pinch = event.gesture(Qt.GestureType.PinchGesture)

        # return false if this gesture event is not a pinch
        #
        if pinch is None:
            return False

        # fetch incremental scale factor since last gesture update
        #
        factor = float(pinch.scaleFactor())

        # ignore invalid factors
        #
        if factor <= 0.0:
            return False

        # fetch configured scale factor (one wheel "step" zoom multiplier)
        #
        scale_fac = float(self.config.get(CFG_KEY_DEF_SCALE_FACTOR))

        # guard against degenerate configuration
        #
        if scale_fac <= 1.0:
            scale_fac = 1.25

        # convert pinch factor into zoom-step units (continuous)
        #
        dz = math.log(factor) / math.log(scale_fac)

        # compute new zoom counter (continuous)
        #
        new_zoom = float(self._zoom) + dz

        # clamp zoom to minimum of zero (fit-to-view)
        #
        if new_zoom <= 0.0:

            # reset zoom counter
            #
            self._zoom = 0.0

            # reset view transform to fit
            #
            self.resetView(1)

            # emit zoom level changed signal (integer view)
            #
            self.zoomLevelChanged.emit(0)

            # run standard post-change logic
            #
            self._after_view_change()

            # accept this gesture
            #
            event.accept(pinch)

            # report handled
            #
            return True

        # apply incremental scaling to the view
        #
        self.scale(factor, factor)

        # store updated zoom counter
        #
        self._zoom = new_zoom

        # emit zoom level changed signal (integer view)
        #
        self.zoomLevelChanged.emit(int(self._zoom))

        # run standard post-change logic
        #
        self._after_view_change()

        # accept this gesture
        #
        event.accept(pinch)

        # report handled
        #
        return True
    #
    # end of method

    def setPhoto(self, pixmap=None):
        """
        method: setPhoto

        arguments:
          pixmap: optional QPixmap to display

        return:
          none

        description:
          This method sets the photo pixmap and resets the view when empty.
        """

        # check for a valid pixmap
        #
        if pixmap and (not pixmap.isNull()):

            # mark as non-empty
            #
            self._empty = False

            # enable panning
            #
            self.setDragMode(QGraphicsView.DragMode.ScrollHandDrag)

            # set the pixmap on the item
            #
            self._photo.setPixmap(pixmap)

        else:

            # mark as empty
            #
            self._empty = True

            # disable panning
            #
            self.setDragMode(QGraphicsView.DragMode.NoDrag)

            # clear the pixmap
            #
            self._photo.setPixmap(QPixmap())

        # reset or update after a new photo is set
        #
        if self._empty:
            self.resetView(1)
        else:
            self._after_view_change(
                clamp=False,
                request_resolution=False,
            )

        # exit gracefully
        #
        return None
    #
    # end of method

    def resetView(self, scale=1):
        """
        method: resetView

        arguments:
          scale: integer scaling multiplier

        return:
          none

        description:
          This method resets the view to fit the scene rect at a given scale.
        """

        # fetch current scene rect
        #
        rect = self.sceneRect()

        # check for null scene rect
        #
        if rect.isNull():
            return None

        # normalize scale to a positive integer
        #
        s = max(1, int(scale))

        # compute the current unit transform mapping
        #
        unity = self.transform().mapRect(QRectF(0, 0, 1, 1))

        # reset scaling if unit is valid
        #
        if unity.width() and unity.height():
            self.scale(1 / unity.width(), 1 / unity.height())

        # fetch viewport and scene mapped rectangles
        #
        viewrect = self.viewport().rect()
        scenerect = self.transform().mapRect(rect)

        # fetch denominator floor
        #
        dim_denom = float(self.config.get(CFG_KEY_DIM_DENOM_FLOOR))

        # compute safe scene dimensions
        #
        d_w = scenerect.width()
        d_h = scenerect.height()
        d_w = d_w if d_w > dim_denom else dim_denom
        d_h = d_h if d_h > dim_denom else dim_denom

        # compute fit factors
        #
        fx = viewrect.width() / d_w
        fy = viewrect.height() / d_h

        # compute the applied scale factor
        #
        factor = min(fx, fy) * float(s)

        # apply scaling and center
        #
        self.scale(factor, factor)
        self.centerOn(rect.center())

        # reset zoom counter at unity scale
        #
        if s == 1:
            self._zoom = 0

        # update and request resolution change
        #
        self._after_view_change(
            clamp=False,
            request_resolution=True,
        )

        # exit gracefully
        #
        return None
    #
    # end of method

    #--------------------------------------------------------------------------
    #
    # stream prune scheduling methods
    #
    #--------------------------------------------------------------------------

    def _schedule_stream_prune(self, delay_ms):
        """
        method: _schedule_stream_prune

        arguments:
          delay_ms: delay in milliseconds before prune callback fires

        return:
          none

        description:
          This method schedules a single-shot prune callback.
        """

        # check for missing prune callback
        #
        if self._tile_prune_cb is None:
            return None

        # start the timer
        #
        self._prune_timer.start(int(delay_ms))

        # exit gracefully
        #
        return None
    #
    # end of method

    def _perform_stream_prune(self):
        """
        method: _perform_stream_prune

        arguments:
          none

        return:
          none

        description:
          This method invokes the prune callback if one is registered.
        """

        # cache callback reference
        #
        cb = self._tile_prune_cb

        # invoke callback if present
        #
        if cb is not None:
            cb()

        # exit gracefully
        #
        return None
    #
    # end of method

    #--------------------------------------------------------------------------
    #
    # view change helper methods
    #
    #--------------------------------------------------------------------------

    def _after_view_change(
        self,
        *,
        clamp=True,
        request_resolution=True,
        prune_delay=None,
    ):
        """
        method: _after_view_change

        arguments:
          clamp: clamp viewport to scene bounds when True
          request_resolution: emit desired pyramid level when True
          prune_delay: optional override delay for prune scheduling

        return:
          none

        description:
          This method emits view-related signals and schedules pruning.
        """

        # clamp the view to the scene rect if requested
        #
        if clamp:
            self._clamp_to_scene()

        # emit the current transform
        #
        self.transformChanged.emit(self.transform())

        # compute and emit the visible scene rectangle
        #
        rect = self.mapToScene(self.viewport().rect()).boundingRect()
        self.viewRectChanged.emit(rect)

        # request a resolution change when needed
        #
        if request_resolution:
            self._maybe_request_resolution_change()

        # choose prune delay
        #
        if prune_delay is None:
            ms = int(self.config.get(CFG_KEY_PRUNE_SCHEDULE_DELAY_MS))
        else:
            ms = int(prune_delay)

        # schedule pruning
        #
        self._schedule_stream_prune(ms)

        # exit gracefully
        #
        return None
    #
    # end of method

    def _desired_level_from_zoom(self):
        """
        method: _desired_level_from_zoom

        arguments:
          none

        return:
          int: desired pyramid level based on current zoom counter

        description:
          This method maps the current zoom step counter to a pyramid level.
        """

        # return current level when no downsample data exists
        #
        if not self._downs:
            return int(self._current_pyramid_level)

        # compute number of levels
        #
        n = len(self._downs)

        # normalize steps per level
        #
        steps = max(1, int(self.ZOOM_STEPS_PER_LEVEL))

        # compute desired level such that zooming in increases resolution
        #
        desired = n - 1 - int(self._zoom) // steps

        # clamp to valid range
        #
        desired = max(0, min(n - 1, int(desired)))

        # return desired level
        #
        return int(desired)
    #
    # end of method

    def _maybe_request_resolution_change(self):
        """
        method: _maybe_request_resolution_change

        arguments:
          none

        return:
          none

        description:
          This method emits a resolutionChangeRequested signal when the desired
          pyramid level differs from the current level.
        """

        # compute desired level from zoom
        #
        desired = self._desired_level_from_zoom()

        # skip when desired equals last emitted
        #
        if desired == self._last_emitted_level:
            return None

        # record the emitted level
        #
        self._last_emitted_level = desired

        # emit request only when it differs from the applied level
        #
        if desired != int(self._current_pyramid_level):
            self.resolutionChangeRequested.emit(int(desired))

        # exit gracefully
        #
        return None
    #
    # end of method

    #--------------------------------------------------------------------------
    #
    # Qt event handler methods
    #
    #--------------------------------------------------------------------------

    def wheelEvent(self, event):
        """
        method: wheelEvent

        arguments:
          event: the wheel event object

        return:
          none

        description:
          This method handles zooming (mouse wheel / trackpad) with support for
          high-resolution deltas on macOS by accumulating partial deltas.
        """

        # fetch wheel delta (angle-based, common for physical wheels)
        #
        delta_angle = int(event.angleDelta().y())

        # fetch wheel delta (pixel-based, common for trackpads / Magic Mouse)
        #
        delta_pixel = int(event.pixelDelta().y())

        # fetch degrees-per-step configuration
        #
        deg = float(self.config.get(CFG_KEY_WHEEL_DEGREES_PER_STEP))

        # guard against invalid configuration
        #
        if deg <= 0.0:
            deg = 120.0

        # define an approximate pixel-to-step scale for pixelDelta fallback
        #
        pixels_per_step = 50.0

        # select a delta expressed in "degree-units"
        #
        if delta_angle != 0:

            # use angle delta directly (already in degree-units)
            #
            delta = float(delta_angle)

        elif delta_pixel != 0:

            # convert pixel delta into degree-units so it works with the same threshold
            #
            delta = float(delta_pixel) * (deg / pixels_per_step)

        else:

            # no usable delta
            #
            delta = 0.0

        # accumulate partial deltas so small macOS deltas eventually produce a step
        #
        self._wheel_delta_accum += delta

        # compute integer zoom steps from the accumulated delta
        #
        steps = int(self._wheel_delta_accum / deg)

        # keep the remainder for future wheel events
        #
        self._wheel_delta_accum -= float(steps) * deg

        # if we still do not have a full step, do not drop the event
        #
        if steps == 0:

            # pass event to base class so trackpad scrolling can still pan
            #
            super().wheelEvent(event)

            # exit gracefully
            #
            return None

        # fetch scale factor
        #
        scale_fac = float(self.config.get(CFG_KEY_DEF_SCALE_FACTOR))

        # compute new zoom counter (continuous-safe)
        #
        new_zoom = max(0.0, float(self._zoom) + float(steps))

        # if zoom counter goes to zero, reset view
        #
        if new_zoom <= 0.0:

            # reset zoom counter
            #
            self._zoom = 0.0

            # reset view transform
            #
            self.resetView(1)

        else:

            # compute scale multiplier for these steps
            #
            mult = scale_fac ** steps

            # apply scaling
            #
            self.scale(mult, mult)

            # store new zoom counter
            #
            self._zoom = new_zoom

        # emit zoom level changed signal (integer view)
        #
        self.zoomLevelChanged.emit(int(self._zoom))

        # accept this event
        #
        event.accept()

        # handle resolution request and view change signals
        #
        self._after_view_change()

        # exit gracefully
        #
        return None
    #
    # end of method

    def scrollContentsBy(self, dx, dy):
        """
        method: scrollContentsBy

        arguments:
          dx: horizontal scroll delta
          dy: vertical scroll delta

        return:
          none

        description:
          This method forwards scroll updates and emits view rect changes.
        """

        # call base class implementation
        #
        super().scrollContentsBy(dx, dy)

        # update view state without requesting resolution changes
        #
        self._after_view_change(
            clamp=True,
            request_resolution=False,
        )

        # exit gracefully
        #
        return None
    #
    # end of method

    def mouseMoveEvent(self, event):
        """
        method: mouseMoveEvent

        arguments:
          event: Qt mouse move event

        return:
          none

        description:
          This method schedules a delayed prune while dragging.
        """

        # forward to base class
        #
        super().mouseMoveEvent(event)

        # adjust prune delay while mouse buttons are pressed
        #
        if event.buttons():

            # fetch drag prune delay
            #
            ms = int(self.config.get(CFG_KEY_MOUSE_DRAG_PRUNE_DELAY_MS))

            # update view without clamping
            #
            self._after_view_change(
                clamp=False,
                request_resolution=False,
                prune_delay=ms,
            )

        # exit gracefully
        #
        return None
    #
    # end of method

    def mouseReleaseEvent(self, event):
        """
        method: mouseReleaseEvent

        arguments:
          event: Qt mouse release event

        return:
          none

        description:
          This method clamps the view and requests a resolution check
          on release.
        """

        # forward to base class
        #
        super().mouseReleaseEvent(event)

        # update view and request resolution changes
        #
        self._after_view_change(
            clamp=True,
            request_resolution=True,
        )

        # exit gracefully
        #
        return None
    #
    # end of method

    def resizeEvent(self, event):
        """
        method: resizeEvent

        arguments:
          event: Qt resize event

        return:
          none

        description:
          This method refits the view when at unity zoom.
        """

        # forward to base class
        #
        super().resizeEvent(event)

        # refit at unity zoom
        #
        if self._zoom == 0:
            self.fitInView(
                self.sceneRect(),
                Qt.AspectRatioMode.KeepAspectRatio,
            )

        # update view and request resolution
        #
        self._after_view_change(
            clamp=True,
            request_resolution=True,
        )

        # exit gracefully
        #
        return None
    #
    # end of method

    #--------------------------------------------------------------------------
    #
    # scene clamp and initialization methods
    #
    #--------------------------------------------------------------------------

    def _clamp_to_scene(self):
        """
        method: _clamp_to_scene

        arguments:
          none

        return:
          none

        description:
          This method clamps the view center so the viewport stays within scene
          bounds.
        """

        # fetch scene rect
        #
        scene_rect = self.sceneRect()

        # check for null scene rect
        #
        if scene_rect.isNull():
            return None

        # compute visible scene rect
        #
        vis = self.mapToScene(self.viewport().rect()).boundingRect()

        # check for invalid visible rect
        #
        if vis.isNull():
            return None

        # compute half extents
        #
        half_w = vis.width() / 2.0
        half_h = vis.height() / 2.0

        # clamp center x
        #
        cx = vis.center().x()
        lo_x = scene_rect.left() + half_w
        hi_x = scene_rect.right() - half_w
        cx = min(max(cx, lo_x), hi_x)

        # clamp center y
        #
        cy = vis.center().y()
        lo_y = scene_rect.top() + half_h
        hi_y = scene_rect.bottom() - half_h
        cy = min(max(cy, lo_y), hi_y)

        # center view on clamped point
        #
        self.centerOn(QPointF(cx, cy))

        # exit gracefully
        #
        return None
    #
    # end of method

    def initialize_slide_thumbnail(
        self,
        base_w,
        base_h,
        thumbnail_pm,
        downsamples,
        preview_level,
    ):
        """
        method: initialize_slide_thumbnail

        arguments:
          base_w: base level width in pixels
          base_h: base level height in pixels
          thumbnail_pm: preview pixmap to show in the viewer
          downsamples: list of downsample factors, indexed by level
          preview_level: level index used for thumbnail scaling

        return:
          none

        description:
          This method configures the scene to base dimensions and installs a
          preview pixmap scaled by the preview level downsample.
        """

        # fetch the scene object
        #
        scene = self.scene()

        # update scene rect to base dimensions
        #
        if scene is not None:
            scene.setSceneRect(0, 0, float(base_w), float(base_h))

        # store pyramid downsample factors
        #
        self.set_downsamples(downsamples)

        # set the current pyramid level
        #
        self.set_current_pyramid_level(preview_level)

        # enable caching for the photo item
        #
        self._photo.setCacheMode(
            QGraphicsItem.CacheMode.DeviceCoordinateCache
        )

        # set the preview pixmap
        #
        self.setPhoto(thumbnail_pm)

        # place pixmap at origin
        #
        self._photo.setPos(0.0, 0.0)

        # create the item transform for the preview scaling
        #
        transform = QTransform()
        scale_val = float(downsamples[preview_level])
        transform.scale(scale_val, scale_val)

        # apply transform to photo item
        #
        self._photo.setTransform(transform, combine=False)

        # reset view transform and fit to scene
        #
        self.resetTransform()
        self._zoom = 0
        self.fitInView(
            self.sceneRect(),
            Qt.AspectRatioMode.KeepAspectRatio,
        )

        # emit zoom level reset
        #
        self.zoomLevelChanged.emit(0)

        # exit gracefully
        #
        return None
    #
    # end of method

    def clear_base_photo(self):
        """
        method: clear_base_photo

        arguments:
          none

        return:
          none

        description:
          This method clears the base photo pixmap while keeping the
          item visible.
        """

        # ensure the photo item is visible
        #
        self._photo.setVisible(True)

        # clear pixmap
        #
        self._photo.setPixmap(QPixmap())

        # exit gracefully
        #
        return None
    #
    # end of method

    def base_photo_z(self):
        """
        method: base_photo_z

        arguments:
          none

        return:
          float: z-value of the base photo item

        description:
          This method returns the z-value of the base photo pixmap item.
        """

        # return z value
        #
        return float(self._photo.zValue())
    #
    # end of method

    def set_base_photo_visible(self, visible):
        """
        method: set_base_photo_visible

        arguments:
          visible: visibility flag

        return:
          none

        description:
          This method sets whether the base photo pixmap item is visible.
        """

        # update visibility
        #
        self._photo.setVisible(bool(visible))

        # exit gracefully
        #
        return None
    #
    # end of method
#
# end of class

class HeatMapCanvas(QGraphicsView):
    """
    class: HeatMapCanvas

    arguments:
      parent: optional parent widget
      config: optional AppConfig instance

    description:
      This class displays a thumbnail and a set of stream-loaded tiles in a
      separate scene. It also supports an overlay pixmap and auto-following
      a requested viewport rectangle.
    """

    def __init__(self, parent=None, config=None):
        """
        method: constructor

        arguments:
          parent: optional parent widget
          config: optional AppConfig accessor

        return:
          none

        description:
          This method initializes the heatmap scene and its state trackers.
        """

        # call base class constructor
        #
        super().__init__(parent)

        # store config accessor
        #
        self.config = config or AppConfig()

        # create and attach the scene
        #
        self._scene = QGraphicsScene(self)
        self.setScene(self._scene)

        # create background pixmap item
        #
        self._bg_item = QGraphicsPixmapItem()

        # place background at the bottom
        #
        self._bg_item.setZValue(0.0)

        # use fast transformation mode
        #
        self._bg_item.setTransformationMode(
            Qt.TransformationMode.FastTransformation
        )

        # add background item to the scene
        #
        self._scene.addItem(self._bg_item)

        # initialize overlay item
        #
        self._overlay_item = None

        # initialize tile tracking maps
        #
        self._tile_items_by_level = {}
        self._tile_meta_by_level = {}

        # initialize tile LRU clock
        #
        self._tile_clock = 0

        # initialize base level dimensions
        #
        self._lvl0_w = 1.0
        self._lvl0_h = 1.0

        # initialize scene mapping factors
        #
        self._scene_scale = 1.0
        self._scene_offset = QPointF(0.0, 0.0)

        # initialize follow tracking state
        #
        self._last_follow_rect = QRectF()
        self._last_prune_ms = 0

        # fetch LRU limit from config
        #
        lru = self.config.get(CFG_KEY_LRU_LIMIT_DEFAULT)
        self._lru_limit = int(lru) if lru else 400

        # fetch guard band size from config
        #
        gb = self.config.get(CFG_KEY_GUARD_BAND_SCENE_PX_DEFAULT)
        self._tile_guard_band_scene = float(gb)

        # initialize pending follow rectangle
        #
        self._pending_follow = None

        # create follow timer
        #
        self._follow_timer = QTimer(self)
        self._follow_timer.setSingleShot(True)
        self._follow_timer.timeout.connect(self._apply_follow_viewport)

        # disable expensive render hints
        #
        self.setRenderHint(QPainter.RenderHint.SmoothPixmapTransform, False)
        self.setRenderHint(QPainter.RenderHint.Antialiasing, False)

        # disable caching for dynamic tile updates
        #
        self.setCacheMode(QGraphicsView.CacheModeFlag.CacheNone)

        # minimize viewport updates
        #
        self.setViewportUpdateMode(
            QGraphicsView.ViewportUpdateMode.MinimalViewportUpdate
        )

        # disable interactions
        #
        self.setInteractive(False)
        self.setDragMode(QGraphicsView.DragMode.NoDrag)
        self.setVerticalScrollBarPolicy(
            Qt.ScrollBarPolicy.ScrollBarAlwaysOff
        )
        self.setHorizontalScrollBarPolicy(
            Qt.ScrollBarPolicy.ScrollBarAlwaysOff
        )
        self.setTransformationAnchor(QGraphicsView.ViewportAnchor.NoAnchor)
        self.setResizeAnchor(QGraphicsView.ViewportAnchor.NoAnchor)

        # exit gracefully
        #
        return None
    #
    # end of method

    #--------------------------------------------------------------------------
    #
    # overlay management methods
    #
    #--------------------------------------------------------------------------

    def clear_overlay(self):
        """
        method: clear_overlay

        arguments:
          none

        return:
          none

        description:
          This method removes the overlay pixmap item from the scene.
        """

        # check for an existing overlay item
        #
        if self._overlay_item is not None:

            # clear the pixmap
            #
            self._overlay_item.setPixmap(QPixmap())

            # remove item from scene
            #
            self._scene.removeItem(self._overlay_item)

            # drop reference
            #
            self._overlay_item = None

        # exit gracefully
        #
        return None
    #
    # end of method

    def set_overlay_pixmap(self, overlay_pixmap, visible=True):
        """
        method: set_overlay_pixmap

        arguments:
          overlay_pixmap: QPixmap overlay to display
          visible: visibility flag

        return:
          none

        description:
          This method creates or updates the overlay pixmap item.
        """

        # clear overlay when input is null
        #
        if overlay_pixmap is None or overlay_pixmap.isNull():
            self.clear_overlay()
            return None

        # create overlay item if needed
        #
        if self._overlay_item is None:
            self._overlay_item = self._scene.addPixmap(overlay_pixmap)
        else:
            self._overlay_item.setPixmap(overlay_pixmap)

        # position overlay using the scene offset
        #
        self._overlay_item.setPos(QPointF(self._scene_offset))

        # set overlay z-order
        #
        z = float(self.config.get(CFG_KEY_OVERLAY_Z_VALUE))
        self._overlay_item.setZValue(z)

        # set overlay visibility
        #
        self._overlay_item.setVisible(bool(visible))

        # exit gracefully
        #
        return None
    #
    # end of method

    def enable_stream_tiles(self, enable=True):
        """
        method: enable_stream_tiles

        arguments:
          enable: visibility flag

        return:
          none

        description:
          This method toggles overlay visibility during tile streaming.
        """

        # toggle overlay visibility if present
        #
        if self._overlay_item is not None:
            self._overlay_item.setVisible(bool(enable))

            # restore overlay z-order when enabled
            #
            if enable:
                z = float(self.config.get(CFG_KEY_OVERLAY_Z_VALUE))
                self._overlay_item.setZValue(z)

        # exit gracefully
        #
        return None
    #
    # end of method

    #--------------------------------------------------------------------------
    #
    # thumbnail and scene initialization methods
    #
    #--------------------------------------------------------------------------

    def load_thumbnail_only(self, thumbnail_pixmap):
        """
        method: load_thumbnail_only

        arguments:
          thumbnail_pixmap: QPixmap thumbnail

        return:
          none

        description:
          This method loads a thumbnail without initializing a base size.
        """

        # check for null pixmap
        #
        if thumbnail_pixmap is None or thumbnail_pixmap.isNull():
            return None

        # reset mapping factors
        #
        self._scene_scale = 1.0
        self._scene_offset = QPointF(0.0, 0.0)

        # store base size from thumbnail
        #
        self._lvl0_w = float(thumbnail_pixmap.width())
        self._lvl0_h = float(thumbnail_pixmap.height())

        # set pixmap on background item
        #
        self._bg_item.setPixmap(thumbnail_pixmap)
        self._bg_item.setPos(self._scene_offset)

        # set scene rect to thumbnail size
        #
        self.setSceneRect(QRectF(0, 0, self._lvl0_w, self._lvl0_h))

        # reset transforms and fit to view
        #
        self.resetTransform()
        self.fitInView(
            self.sceneRect(),
            Qt.AspectRatioMode.KeepAspectRatio,
        )

        # exit gracefully
        #
        return None
    #
    # end of method

    def initialize_heatmap_scene(
        self,
        thumb_pixmap,
        level0_size,
        pad_x=0,
        pad_y=0,
    ):
        """
        method: initialize_heatmap_scene

        arguments:
          thumb_pixmap: QPixmap thumbnail
          level0_size: tuple of base dimensions (width, height)
          pad_x: scene x-offset padding
          pad_y: scene y-offset padding

        return:
          none

        description:
          This method initializes the scene scaling based on a base image size.
        """

        # extract base dimensions
        #
        width0 = int(level0_size[0]) if level0_size else 0
        height0 = int(level0_size[1]) if level0_size else 0

        # validate inputs
        #
        if (
            width0 <= 0
            or height0 <= 0
            or thumb_pixmap is None
            or thumb_pixmap.isNull()
        ):
            return None

        # store base level dimensions
        #
        self._lvl0_w = float(width0)
        self._lvl0_h = float(height0)

        # compute safe denominator
        #
        denom = float(self.config.get(CFG_KEY_DIM_DENOM_FLOOR))
        d = self._lvl0_w if self._lvl0_w > denom else denom

        # compute scene scale from thumbnail width
        #
        self._scene_scale = float(thumb_pixmap.width()) / d

        # store the scene offset
        #
        self._scene_offset = QPointF(float(pad_x), float(pad_y))

        # build the scene rect in scene units
        #
        scene_rect = QRectF(
            self._scene_offset.x(),
            self._scene_offset.y(),
            self._lvl0_w * self._scene_scale,
            self._lvl0_h * self._scene_scale,
        )

        # apply scene rect
        #
        self._scene.setSceneRect(scene_rect)

        # set the background thumbnail
        #
        self._bg_item.setPixmap(thumb_pixmap)
        self._bg_item.setPos(self._scene_offset)

        # exit gracefully
        #
        return None
    #
    # end of method

    #--------------------------------------------------------------------------
    #
    # tile management methods
    #
    #--------------------------------------------------------------------------

    def clear_tiles(self, level=None):
        """
        method: clear_tiles

        arguments:
          level: optional pyramid level to clear; clears all when None

        return:
          none

        description:
          This method removes tile items from the scene and clears
          tracking maps.
        """

        # choose levels to clear
        #
        if level is None:
            levels = list(self._tile_items_by_level.keys())
        else:
            levels = [int(level)]

        # iterate levels and remove items
        #
        for lvl in levels:
            items_map = self._tile_items_by_level.get(lvl, {})

            # remove each item from the scene
            #
            for item in list(items_map.values()):
                self._scene.removeItem(item)

            # reset maps for this level
            #
            self._tile_items_by_level[lvl] = {}
            self._tile_meta_by_level[lvl] = {}

        # exit gracefully
        #
        return None
    #
    # end of method

    def prune_to_level(self, keep_level):
        """
        method: prune_to_level

        arguments:
          keep_level: pyramid level to keep

        return:
          none

        description:
          This method clears all tile levels except the requested keep level.
        """

        # normalize keep level
        #
        keep = int(keep_level)

        # clear all other levels
        #
        for lvl in list(self._tile_items_by_level.keys()):
            if int(lvl) == keep:
                continue
            self.clear_tiles(level=int(lvl))

        # exit gracefully
        #
        return None
    #
    # end of method

    def remove_tile(self, level, tile_key):
        """
        method: remove_tile

        arguments:
          level: pyramid level index
          tile_key: key identifying a tile in that level

        return:
          none

        description:
          This method removes a single tile item from the scene and tracking.
        """

        # normalize level
        #
        lvl = int(level)

        # fetch map for this level
        #
        items = self._tile_items_by_level.get(lvl)

        # check for empty level
        #
        if not items:
            return None

        # remove item reference
        #
        item = items.pop(tile_key, None)

        # drop metadata for the key
        #
        self._tile_meta_by_level.get(lvl, {}).pop(tile_key, None)

        # ignore missing item
        #
        if item is None:
            return None

        # remove from scene
        #
        self._scene.removeItem(item)

        # exit gracefully
        #
        return None
    #
    # end of method

    def mirror_tile(self, lvl, rect_level, pixmap, ds_level):
        """
        method: mirror_tile

        arguments:
          lvl: pyramid level index
          rect_level: (x, y, w, h) in level coordinates
          pixmap: QPixmap tile at that level
          ds_level: downsample factor for the level

        return:
          none

        description:
          This method installs a tile pixmap into the scene with proper scaling.
        """

        # check for a valid pixmap
        #
        if pixmap is None or pixmap.isNull():
            return None

        # normalize level and key
        #
        level = int(lvl)
        key = tuple(int(v) for v in rect_level)

        # ensure level dictionaries exist
        #
        self._tile_items_by_level.setdefault(level, {})
        self._tile_meta_by_level.setdefault(level, {})

        # remove any existing tile with the same key
        #
        old = self._tile_items_by_level[level].pop(key, None)
        if old is not None:
            self._scene.removeItem(old)
        self._tile_meta_by_level[level].pop(key, None)

        # unpack rectangle and downsample
        #
        x_l, y_l, _w_l, _h_l = [float(v) for v in rect_level]
        ds_f = float(ds_level)

        # convert level coords to base coords
        #
        x0 = x_l * ds_f
        y0 = y_l * ds_f

        # cache scale and offsets
        #
        s = float(self._scene_scale)
        offx = self._scene_offset.x()
        offy = self._scene_offset.y()

        # compute tile position in scene coordinates
        #
        pos = QPointF(offx + x0 * s, offy + y0 * s)

        # compute tile scale in scene coordinates
        #
        scale = ds_f * s

        # create the scene item for this pixmap
        #
        item = self._scene.addPixmap(pixmap)

        # set z-order from config
        #
        tz = float(self.config.get(CFG_KEY_TILE_Z_VALUE))
        item.setZValue(tz)

        # set item position
        #
        item.setPos(pos)

        # set item transform scaling
        #
        t = QTransform()
        t.scale(scale, scale)
        item.setTransform(t)

        # store item and metadata
        #
        self._tile_items_by_level[level][key] = item
        self._tile_meta_by_level[level][key] = {
            META_ITEM_KEY: item,
            META_LAST_KEY: int(self._tile_clock),
            'ds': float(ds_f),
        }

        # increment LRU clock
        #
        self._tile_clock += 1

        # exit gracefully
        #
        return None
    #
    # end of method

    def _rect0_from_key_ds(self, key, ds_f):
        """
        method: _rect0_from_key_ds

        arguments:
          key: (x, y, w, h) tile key in level units
          ds_f: downsample factor for the level

        return:
          QRectF: tile rectangle in level-0 coordinates

        description:
          This method converts a level tile key to a level-0 QRectF.
        """

        # unpack key values
        #
        x_l, y_l, w_l, h_l = [float(v) for v in key]

        # normalize downsample
        #
        ds = float(ds_f)

        # return rect in level-0 coordinates
        #
        return QRectF(x_l * ds, y_l * ds, w_l * ds, h_l * ds)
    #
    # end of method

    def tick_prune_stream_tiles(self, visible_rect_level0):
        """
        method: tick_prune_stream_tiles

        arguments:
          visible_rect_level0: QRectF visible region in level-0 coordinates

        return:
          list: list of (level, key) tuples pruned from the scene

        description:
          This method prunes tile items outside the visible region and enforces
          a LRU cap.
        """

        # initialize return list
        #
        pruned = []

        # check for invalid visible rect
        #
        if visible_rect_level0 is None or visible_rect_level0.isNull():
            return pruned

        # expand visible rect by a guard band
        #
        pad = float(self._tile_guard_band_scene)
        padded_rect = QRectF(
            visible_rect_level0.x() - pad,
            visible_rect_level0.y() - pad,
            visible_rect_level0.width() + 2.0 * pad,
            visible_rect_level0.height() + 2.0 * pad,
        )

        # fetch prune budget
        #
        max_remove = int(self.config.get(CFG_KEY_TILE_UNLOAD_BUDGET))
        max_remove = max(0, max_remove)

        # return early when budget is zero
        #
        if max_remove == 0:
            return pruned

        # gather prune candidates
        #
        candidates = []
        total_items = 0

        # iterate over all tile items
        #
        for lvl, items_map in self._tile_items_by_level.items():
            meta_map = self._tile_meta_by_level.get(lvl, {})

            # iterate keys in this level
            #
            for key, item in list(items_map.items()):

                # read metadata
                #
                meta = meta_map.get(key, {})
                last = int(meta.get(META_LAST_KEY, 0))
                ds_f = float(meta.get('ds', 1.0))

                # compute tile rect in level-0
                #
                rect0 = self._rect0_from_key_ds(key, ds_f)

                # compute visibility against padded region
                #
                visible = rect0.isValid() and rect0.intersects(padded_rect)

                # add candidate tuple
                #
                candidates.append((visible, last, int(lvl), key, item))
                total_items += 1

        # return when there are no candidates
        #
        if not candidates:
            return pruned

        # sort by visibility then last used clock
        #
        candidates.sort(key=lambda c: (c[0], c[1]))

        # remove invisible items first
        #
        removed = 0
        for visible, _last, lvl, key, item in candidates:

            # stop when budget is reached
            #
            if removed >= max_remove:
                break

            # skip visible tiles
            #
            if visible:
                continue

            # remove from scene and maps
            #
            self._scene.removeItem(item)
            self._tile_items_by_level.get(lvl, {}).pop(key, None)
            self._tile_meta_by_level.get(lvl, {}).pop(key, None)

            # record removal
            #
            pruned.append((lvl, key))
            removed += 1
            total_items -= 1

        # enforce LRU cap if still over budget
        #
        cap = max(0, int(self._lru_limit))
        if cap > 0 and total_items > cap and removed < max_remove:

            # collect remaining candidates that still exist
            #
            remaining = []
            for c in candidates:
                lvl = c[2]
                key = c[3]
                if key in self._tile_items_by_level.get(lvl, {}):
                    remaining.append(c)

            # sort remaining by visibility then last
            #
            remaining.sort(key=lambda c: (c[0], c[1]))

            # prune until under cap or budget
            #
            for visible, _last, lvl, key, item in remaining:
                if removed >= max_remove or total_items <= cap:
                    break

                # remove from scene and maps
                #
                self._scene.removeItem(item)
                self._tile_items_by_level.get(lvl, {}).pop(key, None)
                self._tile_meta_by_level.get(lvl, {}).pop(key, None)

                # record removal
                #
                pruned.append((lvl, key))
                removed += 1
                total_items -= 1

        # return pruned list
        #
        return pruned
    #
    # end of method

    #--------------------------------------------------------------------------
    #
    # follow viewport methods
    #
    #--------------------------------------------------------------------------

    def follow_viewport(self, visible_rect):
        """
        method: follow_viewport

        arguments:
          visible_rect: QRectF in level-0 coordinates

        return:
          none

        description:
          This method schedules a fitInView call around a requested rectangle.
        """

        # check for missing input
        #
        if visible_rect is None:
            return None

        # check that background exists
        #
        if self._bg_item.pixmap().isNull():
            return None

        # copy the rect
        #
        rect = QRectF(visible_rect)

        # map rect to scene coordinates
        #
        s = float(self._scene_scale)
        ox = self._scene_offset.x()
        oy = self._scene_offset.y()
        rect = QRectF(
            ox + rect.x() * s,
            oy + rect.y() * s,
            rect.width() * s,
            rect.height() * s,
        )

        # store as pending follow rect
        #
        self._pending_follow = rect

        # start timer if not active
        #
        if not self._follow_timer.isActive():
            ms = int(self.config.get(CFG_KEY_FOLLOW_TIMER_MS))
            self._follow_timer.start(ms)

        # exit gracefully
        #
        return None
    #
    # end of method

    def _apply_follow_viewport(self):
        """
        method: _apply_follow_viewport

        arguments:
          none

        return:
          none

        description:
          This method applies the pending follow rectangle by fitting the view.
        """

        # check for pending rect
        #
        if self._pending_follow is None:
            return None

        # check background state
        #
        if self._bg_item.pixmap().isNull():
            self._pending_follow = None
            return None

        # pop pending rect
        #
        rect_to_fit = self._pending_follow
        self._pending_follow = None

        # check scene rect
        #
        scene_rect = self.sceneRect()
        if scene_rect.isNull():
            return None

        # clip to scene rect
        #
        rect_to_fit = rect_to_fit.intersected(scene_rect)

        # enforce a minimal viewport size
        #
        min_edge = float(self.config.get(CFG_KEY_MIN_VIEWPORT_EDGE_PX))
        if rect_to_fit.width() < 1 or rect_to_fit.height() < 1:
            c = rect_to_fit.center()
            rect_to_fit = QRectF(
                c.x() - min_edge / 2.0,
                c.y() - min_edge / 2.0,
                min_edge,
                min_edge,
            ).intersected(scene_rect)

        # check tolerance against last follow rect
        #
        last = self._last_follow_rect
        if not last.isNull():

            # compute center and size deltas
            #
            dcx = abs(rect_to_fit.center().x() - last.center().x())
            dcy = abs(rect_to_fit.center().y() - last.center().y())
            dsw = abs(rect_to_fit.width() - last.width())
            dsh = abs(rect_to_fit.height() - last.height())

            # compute tolerance from rect size
            #
            longest_edge = max(rect_to_fit.width(), rect_to_fit.height())
            tol_frac = float(self.config.get(CFG_KEY_FOLLOW_TOLERANCE_FRACTION))
            tol = max(1.0, tol_frac * longest_edge)

            # skip updates within tolerance
            #
            if dcx < tol and dcy < tol and dsw < tol and dsh < tol:
                return None

        # fit view to rect
        #
        self.resetTransform()
        self.fitInView(rect_to_fit, Qt.AspectRatioMode.KeepAspectRatio)

        # store last rect
        #
        self._last_follow_rect = QRectF(rect_to_fit)

        # exit gracefully
        #
        return None
    #
    # end of method

    #--------------------------------------------------------------------------
    #
    # Qt event handler methods
    #
    #--------------------------------------------------------------------------

    def resizeEvent(self, event):
        """
        method: resizeEvent

        arguments:
          event: Qt resize event

        return:
          none

        description:
          This method refits the background to the window on resize.
        """

        # forward to base class
        #
        super().resizeEvent(event)

        # fit the view
        #
        self._fit_to_window()

        # exit gracefully
        #
        return None
    #
    # end of method

    #--------------------------------------------------------------------------
    #
    # internal helper methods
    #
    #--------------------------------------------------------------------------

    def _fit_to_window(self):
        """
        method: _fit_to_window

        arguments:
          none

        return:
          none

        description:
          This method fits the scene rect into the view when a
          background exists.
        """

        # check for missing background
        #
        if self._bg_item.pixmap().isNull():
            return None

        # disable drag mode
        #
        self.setDragMode(QGraphicsView.DragMode.NoDrag)

        # fit scene into view
        #
        self.fitInView(
            self.sceneRect(),
            Qt.AspectRatioMode.KeepAspectRatio,
        )

        # exit gracefully
        #
        return None
    #
    # end of method
#
# end of class

#
# end of file
