#!/usr/bin/env python
#
# file: $NEDC_NFC/class/python/nedc_dpath_demo_tools/nedc_dpath_demo_menus.py
#
# revision history:
#  20251217 (DH): initial version
#
# description:
#  This file contains the implementation of menus and the preferences
#  dialog for the DPATH demo application.
#------------------------------------------------------------------------------

# import system modules
#
from pathlib import Path

# import PyQt modules
#
from PyQt6.QtCore import Qt, pyqtSignal, QDir
from PyQt6.QtGui import QColor, QAction
from PyQt6.QtWidgets import (
    QCheckBox,
    QComboBox,
    QColorDialog,
    QDialog,
    QFileDialog,
    QDoubleSpinBox,
    QFormLayout,
    QHBoxLayout,
    QLayout,
    QLineEdit,
    QMenu,
    QMessageBox,
    QPushButton,
    QSpinBox,
    QSizePolicy,
    QTabWidget,
    QVBoxLayout,
    QWidget,
)

# import NEDC modules
#
from nedc_dpath_demo_config import AppConfig, SUPPORTED_MODELS
import nedc_file_tools as nft

#------------------------------------------------------------------------------
#
# global constants are listed here
#
#------------------------------------------------------------------------------

# define menu and UI string keys
#
CFG_KEY_TITLE_PREFERENCES = 'MenuStrings/TITLE_PREFERENCES'
CFG_KEY_TITLE_EDIT_MENU = 'MenuStrings/TITLE_EDIT_MENU'
CFG_KEY_TITLE_FILE_MENU = 'MenuStrings/TITLE_FILE_MENU'
CFG_KEY_TITLE_VIEW_MENU = 'MenuStrings/TITLE_VIEW_MENU'
CFG_KEY_TITLE_PROCESS_MENU = 'MenuStrings/TITLE_PROCESS_MENU'
CFG_KEY_TITLE_PABCC_MENU = 'MenuStrings/TITLE_PABCC_MENU'
CFG_KEY_TITLE_HELP_MENU = 'MenuStrings/TITLE_HELP_MENU'

# define action text keys
#
CFG_KEY_ACTION_OPEN_TXT = 'MenuActions/ACTION_OPEN'
CFG_KEY_ACTION_EXIT_TXT = 'MenuActions/ACTION_EXIT'
CFG_KEY_ACTION_PREFERENCES_TXT = 'MenuActions/ACTION_PREFERENCES'
CFG_KEY_ACTION_SAVE_SETTINGS_TXT = 'MenuActions/ACTION_SAVE_SETTINGS'
CFG_KEY_ACTION_RESET_SETTINGS_TXT = 'MenuActions/ACTION_RESET_SETTINGS'
CFG_KEY_ACTION_TOGGLE_LEGEND_TXT = 'MenuActions/ACTION_TOGGLE_LEGEND'
CFG_KEY_ACTION_TOGGLE_HISTOGRAM_TXT = 'MenuActions/ACTION_TOGGLE_HISTOGRAM'
CFG_KEY_ACTION_TOGGLE_NAVIGATOR_TXT = 'MenuActions/ACTION_TOGGLE_NAVIGATOR'
CFG_KEY_ACTION_TOGGLE_HEATMAP_TXT = 'MenuActions/ACTION_TOGGLE_HEATMAP'
CFG_KEY_ACTION_TOGGLE_LOGGING_TXT = 'MenuActions/ACTION_TOGGLE_LOGGING'
CFG_KEY_ACTION_DECODE_TXT = 'MenuActions/ACTION_DECODE'
CFG_KEY_ACTION_POSTPROCESS_TXT = 'MenuActions/ACTION_POSTPROCESS'
CFG_KEY_ACTION_ABOUT_TXT = 'MenuActions/ACTION_ABOUT'
CFG_KEY_ACTION_HELP_TXT = 'MenuActions/ACTION_HELP'

# define tab and button text keys
#
CFG_KEY_TAB_GENERAL_TXT = 'MenuTabs/TAB_GENERAL'
CFG_KEY_TAB_PROCESS_TXT = 'MenuTabs/TAB_PROCESS'
CFG_KEY_TAB_COLORS_TXT = 'MenuTabs/TAB_COLORS'
CFG_KEY_TAB_CACHE_TXT = 'MenuTabs/TAB_CACHE'
CFG_KEY_BTN_RESET_TXT = 'MenuButtons/BTN_RESET'
CFG_KEY_BTN_OK_TXT = 'MenuButtons/BTN_OK'
CFG_KEY_BTN_CANCEL_TXT = 'MenuButtons/BTN_CANCEL'
CFG_KEY_BTN_BROWSE_TXT = 'MenuButtons/BTN_BROWSE'

# define label keys
#
CFG_KEY_LBL_DEFAULT_FOLDER = 'MenuLabels/LBL_DEFAULT_FOLDER'
CFG_KEY_LBL_REF_EDGE = 'MenuLabels/LBL_REF_EDGE'
CFG_KEY_LBL_HYP_EDGE = 'MenuLabels/LBL_HYP_EDGE'
CFG_KEY_LBL_HYP_OPACITY = 'MenuLabels/LBL_HYP_OPACITY'
CFG_KEY_LBL_TILES_PER_TICK = 'MenuLabels/LBL_TILES_PER_TICK'
CFG_KEY_LBL_STREAM_SETTLE_MS = 'MenuLabels/LBL_STREAM_SETTLE_MS'
CFG_KEY_LBL_TILE_LOAD_BUDGET = 'MenuLabels/LBL_TILE_LOAD_BUDGET'
CFG_KEY_LBL_TILE_UNLOAD_BUDGET = 'MenuLabels/LBL_TILE_UNLOAD_BUDGET'
CFG_KEY_LBL_HEAT_INDEX_CELL = 'MenuLabels/LBL_HEAT_INDEX_CELL'
CFG_KEY_LBL_TILE_WORKERS = 'MenuLabels/LBL_TILE_WORKERS'
CFG_KEY_LBL_ACTIVE_MODEL = 'MenuLabels/LBL_ACTIVE_MODEL'
CFG_KEY_LBL_DECODER_EXEC = 'MenuLabels/LBL_DECODER_EXEC'
CFG_KEY_LBL_DECODER_PFILE = 'MenuLabels/LBL_DECODER_PFILE'
CFG_KEY_LBL_POSTPROC_PFILE = 'MenuLabels/LBL_POSTPROC_PFILE'
CFG_KEY_LBL_CONF_THRESHOLD = 'MenuLabels/LBL_CONF_THRESHOLD'
CFG_KEY_LBL_CACHE_ENABLE = 'MenuLabels/LBL_CACHE_ENABLE'
CFG_KEY_LBL_QT_PIXMAP_CACHE = 'MenuLabels/LBL_QT_PIXMAP_CACHE'
CFG_KEY_LBL_READY_QUEUE_MAX = 'MenuLabels/LBL_READY_QUEUE_MAX'
CFG_KEY_LBL_TILE_CACHE_CAP = 'MenuLabels/LBL_TILE_CACHE_CAP'
CFG_KEY_LBL_COLOR_VIS_SHOW = 'MenuLabels/LBL_COLOR_VIS_SHOW'
CFG_KEY_SUFFIX_MB = 'MenuLabels/SUFFIX_MB'
CFG_KEY_SUFFIX_TILES = 'MenuLabels/SUFFIX_TILES'

# define dialog and error keys
#
CFG_KEY_DIALOG_TITLE_DECODER_PFILE = 'MenuDialogs/TITLE_DECODER_PFILE'
CFG_KEY_DIALOG_TITLE_POSTPROC_PFILE = 'MenuDialogs/TITLE_POSTPROC_PFILE'
CFG_KEY_ERR_INVALID_PATH_TITLE = 'MenuDialogs/ERR_INVALID_PATH_TITLE'
CFG_KEY_ERR_INVALID_FILE_TITLE = 'MenuDialogs/ERR_INVALID_FILE_TITLE'
CFG_KEY_ERR_DIR_DOES_NOT_EXIST = 'MenuDialogs/ERR_DIR_DOES_NOT_EXIST'
CFG_KEY_ERR_POSTPROC_PFILE_MISSING = 'MenuDialogs/ERR_POSTPROC_PFILE_MISSING'

# define file extensions and constraints
#
CFG_KEY_FILTER_TOML = 'FileExt/FILTER_TOML'
CFG_KEY_EDGE_WIDTH_MIN = 'WidgetConstraints/EDGE_WIDTH_MIN'
CFG_KEY_EDGE_WIDTH_MAX = 'WidgetConstraints/EDGE_WIDTH_MAX'
CFG_KEY_OPACITY_MIN = 'WidgetConstraints/OPACITY_MIN'
CFG_KEY_OPACITY_MAX = 'WidgetConstraints/OPACITY_MAX'
CFG_KEY_OPACITY_STEP = 'WidgetConstraints/OPACITY_STEP'
CFG_KEY_OPACITY_DECIMALS = 'WidgetConstraints/OPACITY_DECIMALS'
CFG_KEY_INT_SPIN_MIN = 'WidgetConstraints/INT_SPIN_MIN'
CFG_KEY_INT_SPIN_MAX_LARGE = 'WidgetConstraints/INT_SPIN_MAX_LARGE'
CFG_KEY_SETTLE_MS_MIN = 'WidgetConstraints/SETTLE_MS_MIN'
CFG_KEY_SETTLE_MS_MAX = 'WidgetConstraints/SETTLE_MS_MAX'
CFG_KEY_SETTLE_MS_STEP = 'WidgetConstraints/SETTLE_MS_STEP'
CFG_KEY_CACHE_PIXMAP_MB_MIN = 'WidgetConstraints/CACHE_PIXMAP_MB_MIN'
CFG_KEY_CACHE_PIXMAP_MB_MAX = 'WidgetConstraints/CACHE_PIXMAP_MB_MAX'
CFG_KEY_CACHE_PIXMAP_MB_STEP = 'WidgetConstraints/CACHE_PIXMAP_MB_STEP'
CFG_KEY_READY_QUEUE_MIN = 'WidgetConstraints/READY_QUEUE_MIN'
CFG_KEY_READY_QUEUE_MAX = 'WidgetConstraints/READY_QUEUE_MAX'
CFG_KEY_READY_QUEUE_STEP = 'WidgetConstraints/READY_QUEUE_STEP'
CFG_KEY_TILE_CACHE_CAP_MIN = 'WidgetConstraints/TILE_CACHE_CAP_MIN'
CFG_KEY_TILE_CACHE_CAP_MAX = 'WidgetConstraints/TILE_CACHE_CAP_MAX'
CFG_KEY_TILE_CACHE_CAP_STEP = 'WidgetConstraints/TILE_CACHE_CAP_STEP'

# define default model name
#
CFG_KEY_DEFAULT_MODEL_NAME = 'Models/DEFAULT_MODEL_NAME'

# define internal configuration keys
#
CFG_KEY_DEFAULT_SEARCH_DIR = 'DEFAULT_SEARCH_DIR'
CFG_KEY_REF_EDGE_WIDTH = 'REF_EDGE_WIDTH'
CFG_KEY_HYP_EDGE_WIDTH = 'HYP_EDGE_WIDTH'
CFG_KEY_HYP_OPACITY = 'HYP_OPACITY'
CFG_KEY_STREAM_UI_TILES_PER_TICK = 'Stream/UI_TILES_PER_TICK'
CFG_KEY_STREAM_SETTLE_MS = 'Stream/STREAM_SETTLE_MS'
CFG_KEY_TILE_SUBMIT_BUDGET = 'Stream/TILE_SUBMIT_BUDGET'
CFG_KEY_TILE_UNLOAD_BUDGET = 'Stream/TILE_UNLOAD_BUDGET'
CFG_KEY_HEAT_INDEX_CELL = 'Stream/HEAT_INDEX_CELL'
CFG_KEY_NUM_TILE_THREADS = 'Stream/NUM_TILE_THREADS'
CFG_KEY_ACTIVE_MODEL = 'ACTIVE_MODEL'
CFG_KEY_MODEL_DECODER_EXECUTABLE_FMT = 'Model/{}/decoder_executable'
CFG_KEY_MODEL_DECODER_PARAMS_FMT = 'Model/{}/decoder_params'
CFG_KEY_POSTPROC_PARAM_FILE = 'POSTPROC_PARAM_FILE'
CFG_KEY_CONFIDENCE_THRESHOLD = 'CONFIDENCE_THRESHOLD'
CFG_KEY_LEGACY_CONFIDENCE_THRESHOD = 'CONFIDENCE_THRESHOD'
CFG_KEY_ANN_COLOR_FMT = 'AnnotationsColor/{}'
CFG_KEY_VIS_REF_FMT = 'AnnotationsVisibleRef/{}'
CFG_KEY_VIS_HYP_FMT = 'AnnotationsVisibleHyp/{}'
CFG_KEY_CACHE_LIMITS_ENABLED = 'Cache/LIMITS_ENABLED'
CFG_KEY_CACHE_QT_PIXMAP_MB = 'Cache/QT_PIXMAP_MB'
CFG_KEY_CACHE_READY_QUEUE_MAX = 'Cache/READY_QUEUE_MAX'
CFG_KEY_CACHE_TILE_CACHE_CAP = 'Cache/TILE_CACHE_CAP'

# define color mapping keys
#
DEF_COLOUR_KEYS = [
    ('bckg_ref', 'Background (ref)'),
    ('norm_ref', 'Normal (ref)'),
    ('nneo_ref', 'NNeo (ref)'),
    ('indc_ref', 'Indc (ref)'),
    ('dcis_ref', 'DCIS (ref)'),
    ('artf_ref', 'Artifact (ref)'),
    ('null_ref', 'Null (ref)'),
    ('infl_ref', 'Inflammation (ref)'),
    ('susp_ref', 'Suspicious (ref)'),
    ('bckg_hyp', 'Background (hyp)'),
    ('norm_hyp', 'Normal (hyp)'),
    ('nneo_hyp', 'NNeo (hyp)'),
    ('indc_hyp', 'Indc (hyp)'),
    ('dcis_hyp', 'DCIS (hyp)'),
    ('artf_hyp', 'Artifact (hyp)'),
    ('null_hyp', 'Null (hyp)'),
    ('infl_hyp', 'Inflammation (hyp)'),
    ('susp_hyp', 'Suspicious (hyp)'),
]

#------------------------------------------------------------------------------
#
# classes are listed here
#
#------------------------------------------------------------------------------

class PreferencesDialog(QDialog):
    """
    class: PreferencesDialog

    arguments:
      parent: parent widget
      config: application configuration object

    description:
      A modal dialog for managing application settings and preferences.
    """

    # define signal for preferences saved
    #
    preferences_saved = pyqtSignal()

    def __init__(self, parent=None, config=None):
        """
        method: constructor

        arguments:
          parent: the parent widget
          config: the configuration singleton

        return:
          none

        description:
          Initializes the dialog and builds the user interface.
        """

        # call parent constructor
        #
        super().__init__(parent)

        # set configuration object
        #
        self.cfg = config or AppConfig()

        # initialize widget tracking dictionaries
        #
        self._colour_btns = {}
        self._colour_vis_checks = {}

        # set window properties
        #
        self.setModal(True)

        # set window title from config
        #
        self.setWindowTitle(self.cfg.get(CFG_KEY_TITLE_PREFERENCES))

        # enable window size grip
        #
        self.setSizeGripEnabled(True)

        # set expanding size policy
        #
        self.setSizePolicy(
            QSizePolicy.Policy.Expanding,
            QSizePolicy.Policy.Expanding,
        )

        # build the user interface
        #
        self._build_ui()

        # load settings from configuration into widgets
        #
        self._load_from_config()

        # exit gracefully
        #
        return None
    #
    # end of method

    #--------------------------------------------------------------------------
    #
    # widget helper methods
    #
    #--------------------------------------------------------------------------

    def _make_int_spin(self, *, min_v, max_v, step=1, suffix=nft.DELIM_NULL):
        """
        method: _make_int_spin

        arguments:
          min_v: minimum value
          max_v: maximum value
          step: increment step
          suffix: text suffix for the box

        return:
          sp: the configured QSpinBox

        description:
          Creates and configures an integer spin box.
        """

        # create spin box
        #
        sp = QSpinBox()

        # set range
        #
        sp.setRange(int(min_v), int(max_v))

        # set single-step increment
        #
        sp.setSingleStep(int(step))

        # set suffix if provided
        #
        if suffix:
            sp.setSuffix(str(suffix))

        # exit gracefully
        # return spin box
        #
        return sp
    #
    # end of method

    def _make_double_spin(self, *, min_v, max_v, step, decimals):
        """
        method: _make_double_spin

        arguments:
          min_v: minimum value
          max_v: maximum value
          step: increment step
          decimals: number of decimal places

        return:
          sp: the configured QDoubleSpinBox

        description:
          Creates and configures a double-precision spin box.
        """

        # create double spin box
        #
        sp = QDoubleSpinBox()

        # set numeric range
        #
        sp.setRange(float(min_v), float(max_v))

        # set step size
        #
        sp.setSingleStep(float(step))

        # set number of decimals
        #
        sp.setDecimals(int(decimals))

        # exit gracefully
        # return spin box
        #
        return sp
    #
    # end of method

    #--------------------------------------------------------------------------
    #
    # UI construction methods
    #
    #--------------------------------------------------------------------------

    def _build_ui(self):
        """
        method: _build_ui

        arguments:
          none

        return:
          none

        description:
          Constructs the layout and tabs for the preferences dialog.
        """

        # create main vertical layout
        #
        main_column = QVBoxLayout(self)

        # create tab widget
        #
        self.tabs = QTabWidget()

        # set tabs to expand with dialog
        #
        self.tabs.setSizePolicy(
            QSizePolicy.Policy.Expanding,
            QSizePolicy.Policy.Expanding,
        )

        # add tabs widget to the main layout
        #
        main_column.addWidget(self.tabs)

        # add configuration tabs
        #
        self._add_general_tab()
        self._add_process_tab()
        self._add_colour_tab()
        self._add_cache_tab()

        # create bottom button row
        #
        button_row = QHBoxLayout()

        # push buttons to the right
        #
        button_row.addStretch(1)

        # create control buttons
        #
        self.btn_reset = QPushButton(self.cfg.get(CFG_KEY_BTN_RESET_TXT))
        self.btn_ok = QPushButton(self.cfg.get(CFG_KEY_BTN_OK_TXT))
        self.btn_cancel = QPushButton(self.cfg.get(CFG_KEY_BTN_CANCEL_TXT))

        # add buttons to row
        #
        button_row.addWidget(self.btn_reset)
        button_row.addWidget(self.btn_ok)
        button_row.addWidget(self.btn_cancel)

        # add row to main column
        #
        main_column.addLayout(button_row)

        # connect OK button to save-and-close
        #
        self.btn_ok.clicked.connect(self._save_and_close)

        # connect Cancel button to reject/close
        #
        self.btn_cancel.clicked.connect(self.reject)

        # connect Reset button to factory defaults
        #
        self.btn_reset.clicked.connect(self._reset_defaults)

        # do not constrain layout size
        #
        main_column.setSizeConstraint(QLayout.SizeConstraint.SetNoConstraint)

        # set minimum dialog size for usability
        #
        self.setMinimumSize(800, 600)

        # exit gracefully
        #
        return None
    #
    # end of method

    def _add_general_tab(self):
        """
        method: _add_general_tab

        arguments:
          none

        return:
          none

        description:
          Constructs and adds the general settings tab.
        """

        # create tab container
        #
        tab = QWidget()

        # create form layout
        #
        form = QFormLayout(tab)

        # align labels to the right
        #
        form.setLabelAlignment(Qt.AlignmentFlag.AlignRight)

        # create directory line-edit
        #
        self.le_dir = QLineEdit()

        # create browse button
        #
        browse_btn = QPushButton(self.cfg.get(CFG_KEY_BTN_BROWSE_TXT))

        # fix browse button width to match style
        #
        browse_btn.setFixedWidth(26)

        # connect browse button to directory picker
        #
        browse_btn.clicked.connect(self._browse_dir)

        # create directory row layout
        #
        dir_row = QHBoxLayout()

        # add widgets to row
        #
        dir_row.addWidget(self.le_dir, 1)
        dir_row.addWidget(browse_btn)

        # add row to form
        #
        form.addRow(self.cfg.get(CFG_KEY_LBL_DEFAULT_FOLDER), dir_row)

        # create ref edge spinbox
        #
        self.sp_ref_edge = self._make_int_spin(
            min_v=self.cfg.get(CFG_KEY_EDGE_WIDTH_MIN),
            max_v=self.cfg.get(CFG_KEY_EDGE_WIDTH_MAX),
        )

        # add ref edge row
        #
        form.addRow(self.cfg.get(CFG_KEY_LBL_REF_EDGE), self.sp_ref_edge)

        # create hyp edge spinbox
        #
        self.sp_hyp_edge = self._make_int_spin(
            min_v=self.cfg.get(CFG_KEY_EDGE_WIDTH_MIN),
            max_v=self.cfg.get(CFG_KEY_EDGE_WIDTH_MAX),
        )

        # add hyp edge row
        #
        form.addRow(self.cfg.get(CFG_KEY_LBL_HYP_EDGE), self.sp_hyp_edge)

        # create opacity spinbox
        #
        self.sp_hyp_opacity = self._make_double_spin(
            min_v=self.cfg.get(CFG_KEY_OPACITY_MIN),
            max_v=self.cfg.get(CFG_KEY_OPACITY_MAX),
            step=self.cfg.get(CFG_KEY_OPACITY_STEP),
            decimals=self.cfg.get(CFG_KEY_OPACITY_DECIMALS),
        )

        # add opacity row
        #
        form.addRow(self.cfg.get(CFG_KEY_LBL_HYP_OPACITY), self.sp_hyp_opacity)

        # create tiles-per-tick spinbox
        #
        self.sp_ui_tiles_per_tick = self._make_int_spin(
            min_v=self.cfg.get(CFG_KEY_INT_SPIN_MIN),
            max_v=self.cfg.get(CFG_KEY_INT_SPIN_MAX_LARGE),
        )

        # add tiles-per-tick row
        #
        form.addRow(
            self.cfg.get(CFG_KEY_LBL_TILES_PER_TICK),
            self.sp_ui_tiles_per_tick,
        )

        # create stream settle spinbox
        #
        self.sp_stream_settle_ms = self._make_int_spin(
            min_v=self.cfg.get(CFG_KEY_SETTLE_MS_MIN),
            max_v=self.cfg.get(CFG_KEY_SETTLE_MS_MAX),
            step=self.cfg.get(CFG_KEY_SETTLE_MS_STEP),
        )

        # add settle row
        #
        form.addRow(
            self.cfg.get(CFG_KEY_LBL_STREAM_SETTLE_MS),
            self.sp_stream_settle_ms,
        )

        # create submit budget spinbox
        #
        self.sp_tile_submit_budget = self._make_int_spin(
            min_v=self.cfg.get(CFG_KEY_INT_SPIN_MIN),
            max_v=self.cfg.get(CFG_KEY_INT_SPIN_MAX_LARGE),
        )

        # add submit budget row
        #
        form.addRow(
            self.cfg.get(CFG_KEY_LBL_TILE_LOAD_BUDGET),
            self.sp_tile_submit_budget,
        )

        # create unload budget spinbox
        #
        self.sp_tile_unload_budget = self._make_int_spin(
            min_v=self.cfg.get(CFG_KEY_INT_SPIN_MIN),
            max_v=self.cfg.get(CFG_KEY_INT_SPIN_MAX_LARGE),
        )

        # add unload budget row
        #
        form.addRow(
            self.cfg.get(CFG_KEY_LBL_TILE_UNLOAD_BUDGET),
            self.sp_tile_unload_budget,
        )

        # create heat index cell spinbox
        #
        self.sp_heat_index_cell = self._make_int_spin(
            min_v=self.cfg.get(CFG_KEY_INT_SPIN_MIN),
            max_v=self.cfg.get(CFG_KEY_INT_SPIN_MAX_LARGE),
        )

        # add heat index cell row
        #
        form.addRow(
            self.cfg.get(CFG_KEY_LBL_HEAT_INDEX_CELL),
            self.sp_heat_index_cell,
        )

        # create worker threads spinbox
        #
        self.sp_num_tile_threads = self._make_int_spin(
            min_v=self.cfg.get(CFG_KEY_INT_SPIN_MIN),
            max_v=self.cfg.get(CFG_KEY_INT_SPIN_MAX_LARGE),
        )

        # add worker threads row
        #
        form.addRow(
            self.cfg.get(CFG_KEY_LBL_TILE_WORKERS),
            self.sp_num_tile_threads,
        )

        # register tab in the tab widget
        #
        self.tabs.addTab(tab, self.cfg.get(CFG_KEY_TAB_GENERAL_TXT))

        # exit gracefully
        #
        return None
    #
    # end of method

    def _add_process_tab(self):
        """
        method: _add_process_tab

        arguments:
          none

        return:
          none

        description:
          Constructs and adds the model processing tab.
        """

        # create tab container
        #
        tab = QWidget()

        # create form layout
        #
        form = QFormLayout(tab)

        # align labels to the right
        #
        form.setLabelAlignment(Qt.AlignmentFlag.AlignRight)

        # create model selection combo box
        #
        self.model_combo = QComboBox()

        # populate combo box with supported models
        #
        for model_name in SUPPORTED_MODELS:
            self.model_combo.addItem(model_name)

        # add combo to form
        #
        form.addRow(self.cfg.get(CFG_KEY_LBL_ACTIVE_MODEL), self.model_combo)

        # create decoder executable line edit
        #
        self.le_decoder_exec = QLineEdit()

        # add decoder executable row
        #
        form.addRow(self.cfg.get(CFG_KEY_LBL_DECODER_EXEC), self.le_decoder_exec)

        # create decoder parameter file widgets
        #
        self.le_decoder_pfile = QLineEdit()

        # create browse button
        #
        btn_decoder_pfile = QPushButton(self.cfg.get(CFG_KEY_BTN_BROWSE_TXT))

        # fix browse button width
        #
        btn_decoder_pfile.setFixedWidth(26)

        # connect browse button to file chooser
        #
        btn_decoder_pfile.clicked.connect(
            lambda: self._choose_file_into(
                self.le_decoder_pfile,
                self.cfg.get(CFG_KEY_DIALOG_TITLE_DECODER_PFILE),
            )
        )

        # layout parameter file row
        #
        row_decoder_pfile = QHBoxLayout()
        row_decoder_pfile.addWidget(self.le_decoder_pfile, 1)
        row_decoder_pfile.addWidget(btn_decoder_pfile)

        # add row to form
        #
        form.addRow(self.cfg.get(CFG_KEY_LBL_DECODER_PFILE), row_decoder_pfile)

        # create post-processing file widgets
        #
        self.le_pproc_pfile = QLineEdit()

        # create browse button
        #
        btn_pproc_pfile = QPushButton(self.cfg.get(CFG_KEY_BTN_BROWSE_TXT))

        # fix browse button width
        #
        btn_pproc_pfile.setFixedWidth(26)

        # connect browse button to file chooser
        #
        btn_pproc_pfile.clicked.connect(
            lambda: self._choose_file_into(
                self.le_pproc_pfile,
                self.cfg.get(CFG_KEY_DIALOG_TITLE_POSTPROC_PFILE),
            )
        )

        # layout postproc row
        #
        row_pproc_pfile = QHBoxLayout()
        row_pproc_pfile.addWidget(self.le_pproc_pfile, 1)
        row_pproc_pfile.addWidget(btn_pproc_pfile)

        # add row to form
        #
        form.addRow(self.cfg.get(CFG_KEY_LBL_POSTPROC_PFILE), row_pproc_pfile)

        # create confidence threshold spin box
        #
        self.sp_conf = self._make_double_spin(
            min_v=0.0,
            max_v=1.0,
            step=0.05,
            decimals=2,
        )

        # add confidence row
        #
        form.addRow(self.cfg.get(CFG_KEY_LBL_CONF_THRESHOLD), self.sp_conf)

        # connect selection change to update model-specific fields
        #
        self.model_combo.currentTextChanged.connect(
            self._update_model_fields_for_selection
        )

        # register tab
        #
        self.tabs.addTab(tab, self.cfg.get(CFG_KEY_TAB_PROCESS_TXT))

        # exit gracefully
        #
        return None
    #
    # end of method

    def _add_cache_tab(self):
        """
        method: _add_cache_tab

        arguments:
          none

        return:
          none

        description:
          Constructs and adds the cache settings tab.
        """

        # create tab container
        #
        tab = QWidget()

        # create form layout
        #
        form = QFormLayout(tab)

        # align labels to the right
        #
        form.setLabelAlignment(Qt.AlignmentFlag.AlignRight)

        # create cache enablement checkbox
        #
        self.chk_cache_limits = QCheckBox(self.cfg.get(CFG_KEY_LBL_CACHE_ENABLE))

        # add checkbox row with empty label column
        #
        form.addRow(nft.DELIM_NULL, self.chk_cache_limits)

        # create pixmap cache spinbox
        #
        self.sp_qt_pixmap_mb = self._make_int_spin(
            min_v=self.cfg.get(CFG_KEY_CACHE_PIXMAP_MB_MIN),
            max_v=self.cfg.get(CFG_KEY_CACHE_PIXMAP_MB_MAX),
            step=self.cfg.get(CFG_KEY_CACHE_PIXMAP_MB_STEP),
            suffix=self.cfg.get(CFG_KEY_SUFFIX_MB),
        )

        # add pixmap cache row
        #
        form.addRow(self.cfg.get(CFG_KEY_LBL_QT_PIXMAP_CACHE), self.sp_qt_pixmap_mb)

        # create ready queue spinbox
        #
        self.sp_ready_queue_max = self._make_int_spin(
            min_v=self.cfg.get(CFG_KEY_READY_QUEUE_MIN),
            max_v=self.cfg.get(CFG_KEY_READY_QUEUE_MAX),
            step=self.cfg.get(CFG_KEY_READY_QUEUE_STEP),
        )

        # add ready queue row
        #
        form.addRow(
            self.cfg.get(CFG_KEY_LBL_READY_QUEUE_MAX),
            self.sp_ready_queue_max,
        )

        # create tile cache capacity spinbox
        #
        self.sp_tile_cache_cap = self._make_int_spin(
            min_v=self.cfg.get(CFG_KEY_TILE_CACHE_CAP_MIN),
            max_v=self.cfg.get(CFG_KEY_TILE_CACHE_CAP_MAX),
            step=self.cfg.get(CFG_KEY_TILE_CACHE_CAP_STEP),
            suffix=self.cfg.get(CFG_KEY_SUFFIX_TILES),
        )

        # add tile cache row
        #
        form.addRow(
            self.cfg.get(CFG_KEY_LBL_TILE_CACHE_CAP),
            self.sp_tile_cache_cap,
        )

        # connect checkbox toggle to enable/disable cache widgets
        #
        self.chk_cache_limits.toggled.connect(
            self._update_cache_widgets_enabled
        )

        # register tab
        #
        self.tabs.addTab(tab, self.cfg.get(CFG_KEY_TAB_CACHE_TXT))

        # exit gracefully
        #
        return None
    #
    # end of method

    def _add_colour_tab(self):
        """
        method: _add_colour_tab

        arguments:
          none

        return:
          none

        description:
          Constructs and adds the annotation color settings tab.
        """

        # create tab container
        #
        tab = QWidget()

        # create form layout
        #
        form = QFormLayout(tab)

        # align labels to the right
        #
        form.setLabelAlignment(Qt.AlignmentFlag.AlignRight)

        # iterate through color keys to build widget rows
        #
        for key, label in DEF_COLOUR_KEYS:

            # build full setting path for this key
            #
            setting_path = CFG_KEY_ANN_COLOR_FMT.format(key)

            # create color pick button
            #
            btn = QPushButton()

            # store object name for debugging / styling
            #
            btn.setObjectName(key)

            # allow width expansion but fixed height
            #
            btn.setSizePolicy(
                QSizePolicy.Policy.Expanding,
                QSizePolicy.Policy.Fixed,
            )

            # connect click to color picker
            #
            btn.clicked.connect(
                lambda _checked=False, s=setting_path: self._pick_colour(s)
            )

            # track button by setting path
            #
            self._colour_btns[setting_path] = btn

            # compute base label (strip _ref/_hyp)
            #
            base_lbl = key[:-4]

            # compute visibility key based on ref/hyp suffix
            #
            if key.endswith('_ref'):
                vis_key = CFG_KEY_VIS_REF_FMT.format(base_lbl)
            else:
                vis_key = CFG_KEY_VIS_HYP_FMT.format(base_lbl)

            # create visibility checkbox
            #
            chk = QCheckBox(self.cfg.get(CFG_KEY_LBL_COLOR_VIS_SHOW))

            # track checkbox by visibility key
            #
            self._colour_vis_checks[vis_key] = chk

            # build row layout
            #
            row = QHBoxLayout()

            # add widgets to row
            #
            row.addWidget(btn, 1)
            row.addWidget(chk)

            # add row to form
            #
            form.addRow(f'{label}:', row)

        # register tab
        #
        self.tabs.addTab(tab, self.cfg.get(CFG_KEY_TAB_COLORS_TXT))

        # exit gracefully
        #
        return None
    #
    # end of method

    #--------------------------------------------------------------------------
    #
    # logic and data methods
    #
    #--------------------------------------------------------------------------

    def _load_from_config(self):
        """
        method: _load_from_config

        arguments:
          none

        return:
          none

        description:
          Populates the user interface with stored configuration data.
        """

        # load search directory (expand variables)
        #
        search_dir = self.cfg.get(CFG_KEY_DEFAULT_SEARCH_DIR)

        # normalize to full path string
        #
        self.le_dir.setText(nft.get_fullpath(search_dir))

        # load edge and opacity numeric values
        #
        self.sp_ref_edge.setValue(int(self.cfg.get(CFG_KEY_REF_EDGE_WIDTH)))
        self.sp_hyp_edge.setValue(int(self.cfg.get(CFG_KEY_HYP_EDGE_WIDTH)))
        self.sp_hyp_opacity.setValue(float(self.cfg.get(CFG_KEY_HYP_OPACITY)))

        # load streaming parameters
        #
        self.sp_ui_tiles_per_tick.setValue(
            int(self.cfg.get(CFG_KEY_STREAM_UI_TILES_PER_TICK))
        )
        self.sp_stream_settle_ms.setValue(int(self.cfg.get(CFG_KEY_STREAM_SETTLE_MS)))
        self.sp_tile_submit_budget.setValue(
            int(self.cfg.get(CFG_KEY_TILE_SUBMIT_BUDGET))
        )
        self.sp_tile_unload_budget.setValue(
            int(self.cfg.get(CFG_KEY_TILE_UNLOAD_BUDGET))
        )
        self.sp_heat_index_cell.setValue(int(self.cfg.get(CFG_KEY_HEAT_INDEX_CELL)))
        self.sp_num_tile_threads.setValue(int(self.cfg.get(CFG_KEY_NUM_TILE_THREADS)))

        # load active model selection
        #
        active = self.cfg.get(CFG_KEY_ACTIVE_MODEL)

        # load default model fallback
        #
        default_model = self.cfg.get(CFG_KEY_DEFAULT_MODEL_NAME)

        # validate model name
        #
        if active not in SUPPORTED_MODELS:
            active = default_model

        # update model combo and associated fields
        #
        self.model_combo.blockSignals(True)
        self.model_combo.setCurrentText(active)
        self.model_combo.blockSignals(False)
        self._update_model_fields_for_selection(active)

        # load post-processing path
        #
        pproc_path = self.cfg.get(CFG_KEY_POSTPROC_PARAM_FILE)

        # normalize and set
        #
        self.le_pproc_pfile.setText(nft.get_fullpath(pproc_path))

        # load confidence threshold (with legacy fallback)
        #
        thr_val = self.cfg.get(CFG_KEY_CONFIDENCE_THRESHOLD)

        # fallback if missing
        #
        if thr_val is None:
            thr_val = self.cfg.get(CFG_KEY_LEGACY_CONFIDENCE_THRESHOD)

        # write threshold into spin box
        #
        if thr_val is not None:
            self.sp_conf.setValue(float(thr_val))
        else:
            self.sp_conf.setValue(0.0)

        # load cache parameters
        #
        enabled = bool(self.cfg.get(CFG_KEY_CACHE_LIMITS_ENABLED))

        # update checkbox
        #
        self.chk_cache_limits.setChecked(enabled)

        # load cache numeric values
        #
        self.sp_qt_pixmap_mb.setValue(int(self.cfg.get(CFG_KEY_CACHE_QT_PIXMAP_MB)))
        self.sp_ready_queue_max.setValue(
            int(self.cfg.get(CFG_KEY_CACHE_READY_QUEUE_MAX))
        )
        self.sp_tile_cache_cap.setValue(int(self.cfg.get(CFG_KEY_CACHE_TILE_CACHE_CAP)))

        # enable/disable widgets to match cache toggle
        #
        self._update_cache_widgets_enabled(enabled)

        # load color button styles
        #
        for setting_path, btn in self._colour_btns.items():

            # fetch stored hex value
            #
            hex_value = self.cfg.get(setting_path)

            # update button text
            #
            btn.setText(hex_value)

            # update background style
            #
            btn.setStyleSheet(f'background:{hex_value};')

        # load visibility checkbox states
        #
        for vis_key, chk in self._colour_vis_checks.items():

            # read stored state
            #
            state = bool(self.cfg.get(vis_key))

            # apply to checkbox
            #
            chk.setChecked(state)

        # exit gracefully
        #
        return None
    #
    # end of method

    def _commit_to_config(self):
        """
        method: _commit_to_config

        arguments:
          none

        return:
          bool: boolean success flag

        description:
          Validates current inputs and commits them to the configuration object.
        """

        # validate search directory path if provided
        #
        default_dir = self.le_dir.text().strip()

        # show error if non-empty path does not exist
        #
        if default_dir and (not Path(default_dir).exists()):

            # display critical error
            #
            QMessageBox.critical(
                self,
                self.cfg.get(CFG_KEY_ERR_INVALID_PATH_TITLE),
                self.cfg.get(CFG_KEY_ERR_DIR_DOES_NOT_EXIST),
            )

            # indicate failure
            #
            return False

        # commit default search directory
        #
        self.cfg.set(CFG_KEY_DEFAULT_SEARCH_DIR, default_dir)

        # commit visual parameters
        #
        self.cfg.set(CFG_KEY_REF_EDGE_WIDTH, int(self.sp_ref_edge.value()))
        self.cfg.set(CFG_KEY_HYP_EDGE_WIDTH, int(self.sp_hyp_edge.value()))
        self.cfg.set(CFG_KEY_HYP_OPACITY, float(self.sp_hyp_opacity.value()))

        # commit streaming parameters
        #
        self.cfg.set(
            CFG_KEY_STREAM_UI_TILES_PER_TICK,
            int(self.sp_ui_tiles_per_tick.value()),
        )
        self.cfg.set(
            CFG_KEY_STREAM_SETTLE_MS,
            int(self.sp_stream_settle_ms.value()),
        )
        self.cfg.set(
            CFG_KEY_TILE_SUBMIT_BUDGET,
            int(self.sp_tile_submit_budget.value()),
        )
        self.cfg.set(
            CFG_KEY_TILE_UNLOAD_BUDGET,
            int(self.sp_tile_unload_budget.value()),
        )
        self.cfg.set(
            CFG_KEY_HEAT_INDEX_CELL,
            int(self.sp_heat_index_cell.value()),
        )
        self.cfg.set(
            CFG_KEY_NUM_TILE_THREADS,
            int(self.sp_num_tile_threads.value()),
        )

        # select model name (fallback to default)
        #
        model = self.model_combo.currentText().strip()

        # apply default model if empty
        #
        if not model:
            model = self.cfg.get(CFG_KEY_DEFAULT_MODEL_NAME)

        # commit active model
        #
        self.cfg.set(CFG_KEY_ACTIVE_MODEL, model)

        # commit decoder executable for selected model
        #
        self.cfg.set(
            CFG_KEY_MODEL_DECODER_EXECUTABLE_FMT.format(model),
            self.le_decoder_exec.text().strip(),
        )

        # commit decoder parameter file for selected model
        #
        self.cfg.set(
            CFG_KEY_MODEL_DECODER_PARAMS_FMT.format(model),
            self.le_decoder_pfile.text().strip(),
        )

        # validate post-processing parameter file if provided
        #
        pproc_path = self.le_pproc_pfile.text().strip()

        # show error if non-empty path does not exist
        #
        if pproc_path and (not Path(pproc_path).exists()):

            # display critical error
            #
            QMessageBox.critical(
                self,
                self.cfg.get(CFG_KEY_ERR_INVALID_FILE_TITLE),
                self.cfg.get(CFG_KEY_ERR_POSTPROC_PFILE_MISSING),
            )

            # indicate failure
            #
            return False

        # commit post-processing file
        #
        self.cfg.set(CFG_KEY_POSTPROC_PARAM_FILE, pproc_path)

        # commit confidence threshold (and legacy key for backward support)
        #
        thr_val = float(self.sp_conf.value())
        self.cfg.set(CFG_KEY_CONFIDENCE_THRESHOLD, thr_val)
        self.cfg.set(CFG_KEY_LEGACY_CONFIDENCE_THRESHOD, thr_val)

        # commit cache toggles and sizes
        #
        self.cfg.set(
            CFG_KEY_CACHE_LIMITS_ENABLED,
            bool(self.chk_cache_limits.isChecked()),
        )
        self.cfg.set(CFG_KEY_CACHE_QT_PIXMAP_MB, int(self.sp_qt_pixmap_mb.value()))
        self.cfg.set(
            CFG_KEY_CACHE_READY_QUEUE_MAX,
            int(self.sp_ready_queue_max.value()),
        )
        self.cfg.set(
            CFG_KEY_CACHE_TILE_CACHE_CAP,
            int(self.sp_tile_cache_cap.value()),
        )

        # commit color values from buttons
        #
        for setting_path, btn in self._colour_btns.items():
            self.cfg.set(setting_path, btn.text().strip())

        # commit visibility values from checkboxes
        #
        for vis_key, chk in self._colour_vis_checks.items():
            self.cfg.set(vis_key, bool(chk.isChecked()))

        # exit gracefully
        # return success
        #
        return True
    #
    # end of method

    def _save_and_close(self):
        """
        method: _save_and_close

        arguments:
          none

        return:
          none

        description:
          Saves current settings and closes the dialog.
        """

        # commit UI values to configuration
        #
        if not self._commit_to_config():
            return None

        # sync configuration to persistent storage
        #
        self.cfg.sync()

        # notify observers that preferences were saved
        #
        self.preferences_saved.emit()

        # close dialog with accept
        #
        self.accept()

        # exit gracefully
        #
        return None
    #
    # end of method

    def _reset_defaults(self):
        """
        method: _reset_defaults

        arguments:
          none

        return:
          none

        description:
          Resets configuration and UI to factory default states.
        """

        # reset configuration to defaults
        #
        self.cfg.reset()

        # sync configuration
        #
        self.cfg.sync()

        # reload user interface from config
        #
        self._load_from_config()

        # notify observers
        #
        self.preferences_saved.emit()

        # exit gracefully
        #
        return None
    #
    # end of method

    def _choose_file_into(self, line_edit, title):
        """
        method: _choose_file_into

        arguments:
          line_edit: widget to receive file path
          title: dialog window title

        return:
          none

        description:
          Opens a file dialog and writes the selection into a QLineEdit.
        """

        # determine starting directory
        #
        start_dir = line_edit.text() or QDir.homePath()

        # fetch filter string from config
        #
        filt = self.cfg.get(CFG_KEY_FILTER_TOML)

        # open file dialog and take the selected filename
        #
        chosen = QFileDialog.getOpenFileName(
            self,
            title,
            start_dir,
            filt,
        )[0]

        # write result to widget
        #
        if chosen:
            line_edit.setText(QDir.toNativeSeparators(chosen))

        # exit gracefully
        #
        return None
    #
    # end of method

    def _browse_dir(self):
        """
        method: _browse_dir

        arguments:
          none

        return:
          none

        description:
          Opens a directory browser and updates the search path widget.
        """

        # determine starting directory
        #
        start_dir = self.le_dir.text() or QDir.homePath()

        # choose directory
        #
        chosen = QFileDialog.getExistingDirectory(
            self,
            self.cfg.get(CFG_KEY_LBL_DEFAULT_FOLDER),
            start_dir,
        )

        # update line edit
        #
        if chosen:
            self.le_dir.setText(QDir.toNativeSeparators(chosen))

        # exit gracefully
        #
        return None
    #
    # end of method

    def _update_cache_widgets_enabled(self, checked):
        """
        method: _update_cache_widgets_enabled

        arguments:
          checked: boolean enablement flag

        return:
          none

        description:
          Enables or disables cache-related numeric input widgets.
        """

        # normalize checked flag
        #
        enabled = bool(checked)

        # enable/disable numeric cache widgets
        #
        self.sp_qt_pixmap_mb.setEnabled(enabled)
        self.sp_ready_queue_max.setEnabled(enabled)
        self.sp_tile_cache_cap.setEnabled(enabled)

        # exit gracefully
        #
        return None
    #
    # end of method

    def _update_model_fields_for_selection(self, model_name):
        """
        method: _update_model_fields_for_selection

        arguments:
          model_name: name of the selected model

        return:
          none

        description:
          Updates the executable and parameter widgets based on model selection.
        """

        # build model-specific configuration keys
        #
        exec_key = CFG_KEY_MODEL_DECODER_EXECUTABLE_FMT.format(model_name)
        param_key = CFG_KEY_MODEL_DECODER_PARAMS_FMT.format(model_name)

        # fetch values from configuration
        #
        exec_path = self.cfg.get(exec_key)
        param_path = self.cfg.get(param_key)

        # update widgets (use empty delimiter if missing)
        #
        self.le_decoder_exec.setText(str(exec_path or nft.DELIM_NULL))
        self.le_decoder_pfile.setText(str(param_path or nft.DELIM_NULL))

        # exit gracefully
        #
        return None
    #
    # end of method

    def _pick_colour(self, setting_path):
        """
        method: _pick_colour

        arguments:
          setting_path: configuration path for the color key

        return:
          none

        description:
          Opens a color dialog and updates button style and tracking data.
        """

        # retrieve button for this setting path
        #
        btn = self._colour_btns[setting_path]

        # parse current color from button text
        #
        current = QColor(btn.text())

        # open color picker dialog
        #
        chosen = QColorDialog.getColor(current, self, 'Select colour')

        # update button if selection is valid
        #
        if chosen.isValid():

            # convert to hex name
            #
            hex_value = chosen.name()

            # update button label
            #
            btn.setText(hex_value)

            # update button background
            #
            btn.setStyleSheet(f'background:{hex_value};')

        # exit gracefully
        #
        return None
    #
    # end of method

    def showEvent(self, event):
        """
        method: showEvent

        arguments:
          event: the show event object

        return:
          none

        description:
          Overrides show event to ensure data is fresh when dialog appears.
        """

        # call base implementation
        #
        super().showEvent(event)

        # reload from configuration for freshness
        #
        self._load_from_config()

        # exit gracefully
        #
        return None
    #
    # end of method
#
# end of class


class FileMenu(QMenu):
    """
    class: FileMenu

    arguments:
      parent: parent widget

    description:
      Implementation of the application File menu.
    """

    def __init__(self, parent=None):
        """
        method: constructor

        arguments:
          parent: the parent widget

        return:
          none

        description:
          Initializes the menu and its associated actions.
        """

        # fetch application configuration
        #
        config = AppConfig()

        # call parent constructor with title
        #
        super().__init__(config.get(CFG_KEY_TITLE_FILE_MENU), parent)

        # create open action
        #
        self.action_open = QAction(config.get(CFG_KEY_ACTION_OPEN_TXT), self)

        # create exit action
        #
        self.action_exit = QAction(config.get(CFG_KEY_ACTION_EXIT_TXT), self)

        # add actions to menu
        #
        self.addAction(self.action_open)
        self.addSeparator()
        self.addAction(self.action_exit)

        # exit gracefully
        #
        return None
    #
    # end of method
#
# end of class


class EditMenu(QMenu):
    """
    class: EditMenu

    arguments:
      parent: parent widget

    description:
      Implementation of the application Edit menu for preferences and settings.
    """

    def __init__(self, parent=None):
        """
        method: constructor

        arguments:
          parent: the parent widget

        return:
          none

        description:
          Initializes the menu and configuration management actions.
        """

        # fetch application configuration
        #
        config = AppConfig()

        # call parent constructor with title
        #
        super().__init__(config.get(CFG_KEY_TITLE_EDIT_MENU), parent)

        # create preferences action
        #
        self.action_preferences = QAction(
            config.get(CFG_KEY_ACTION_PREFERENCES_TXT),
            self,
        )

        # create reset settings action
        #
        self.action_reset_settings = QAction(
            config.get(CFG_KEY_ACTION_RESET_SETTINGS_TXT),
            self,
        )

        # create save settings action
        #
        self.action_save_settings = QAction(
            config.get(CFG_KEY_ACTION_SAVE_SETTINGS_TXT),
            self,
        )

        # layout menu items
        #
        self.addAction(self.action_preferences)
        self.addAction(self.action_save_settings)
        self.addSeparator()
        self.addAction(self.action_reset_settings)

        # exit gracefully
        #
        return None
    #
    # end of method
#
# end of class


class ViewMenu(QMenu):
    """
    class: ViewMenu

    arguments:
      parent: parent widget

    description:
      Implementation of the View menu for toggling user interface panels.
    """

    def __init__(self, parent=None):
        """
        method: constructor

        arguments:
          parent: the parent widget

        return:
          none

        description:
          Initializes toggle actions for legends, heatmaps, and logging.
        """

        # fetch application configuration
        #
        config = AppConfig()

        # call parent constructor with title
        #
        super().__init__(config.get(CFG_KEY_TITLE_VIEW_MENU), parent)

        # create toggle actions
        #
        self.action_toggle_legend = QAction(
            config.get(CFG_KEY_ACTION_TOGGLE_LEGEND_TXT),
            self,
        )
        self.action_toggle_histogram = QAction(
            config.get(CFG_KEY_ACTION_TOGGLE_HISTOGRAM_TXT),
            self,
        )
        self.action_toggle_navigator_overlay = QAction(
            config.get(CFG_KEY_ACTION_TOGGLE_NAVIGATOR_TXT),
            self,
        )
        self.action_toggle_heatmap = QAction(
            config.get(CFG_KEY_ACTION_TOGGLE_HEATMAP_TXT),
            self,
        )
        self.action_toggle_logging = QAction(
            config.get(CFG_KEY_ACTION_TOGGLE_LOGGING_TXT),
            self,
        )

        # layout menu items
        #
        self.addAction(self.action_toggle_legend)
        self.addAction(self.action_toggle_histogram)
        self.addAction(self.action_toggle_navigator_overlay)
        self.addAction(self.action_toggle_heatmap)
        self.addSeparator()
        self.addAction(self.action_toggle_logging)

        # exit gracefully
        #
        return None
    #
    # end of method
#
# end of class


class ProcessMenu(QMenu):
    """
    class: ProcessMenu

    arguments:
      parent: parent widget

    description:
      Implementation of the Process menu for model decoding operations.
    """

    def __init__(self, parent=None):
        """
        method: constructor

        arguments:
          parent: the parent widget

        return:
          none

        description:
          Initializes model decoding and post-processing actions.
        """

        # fetch application configuration
        #
        config = AppConfig()

        # call parent constructor with title
        #
        super().__init__(config.get(CFG_KEY_TITLE_PROCESS_MENU), parent)

        # create decode action
        #
        self.action_decode = QAction(config.get(CFG_KEY_ACTION_DECODE_TXT), self)

        # create post-process action
        #
        self.action_postprocess = QAction(
            config.get(CFG_KEY_ACTION_POSTPROCESS_TXT),
            self,
        )

        # add actions
        #
        self.addAction(self.action_decode)
        self.addAction(self.action_postprocess)

        # exit gracefully
        #
        return None
    #
    # end of method
#
# end of class


class PBCCMenu(QMenu):
    """
    class: PBCCMenu

    arguments:
      parent: parent widget

    description:
      Implementation of the application info and metadata menu.
    """

    def __init__(self, parent=None):
        """
        method: constructor

        arguments:
          parent: the parent widget

        return:
          none

        description:
          Initializes the About action for software metadata.
        """

        # fetch application configuration
        #
        config = AppConfig()

        # call parent constructor with title
        #
        super().__init__(config.get(CFG_KEY_TITLE_PABCC_MENU), parent)

        # create and add about action
        #
        self.action_about = QAction(config.get(CFG_KEY_ACTION_ABOUT_TXT), self)
        self.addAction(self.action_about)

        # exit gracefully
        #
        return None
    #
    # end of method
#
# end of class


class HelpMenu(QMenu):
    """
    class: HelpMenu

    arguments:
      parent: parent widget

    description:
      Implementation of the application Help and documentation menu.
    """

    def __init__(self, parent=None):
        """
        method: constructor

        arguments:
          parent: the parent widget

        return:
          none

        description:
          Initializes help and documentation actions.
        """

        # fetch application configuration
        #
        config = AppConfig()

        # call parent constructor with title
        #
        super().__init__(config.get(CFG_KEY_TITLE_HELP_MENU), parent)

        # create and add help action
        #
        self.action_help = QAction(config.get(CFG_KEY_ACTION_HELP_TXT), self)
        self.addAction(self.action_help)

        # exit gracefully
        #
        return None
    #
    # end of method
#
# end of class

#
# end of file
