#!/usr/bin/env python
#
# file: $NEDC_NFC/class/python/nedc_dpath_demo_tools/nedc_dpath_demo_core.py
#
# revision history:
#  20251217 (DH): initial version
#
# description:
#  This file contains the core logic, classes, and UI components for the
#  NEDC Digital Pathology Demo. It includes session management,
#  overlay control, tile streaming, and the main window assembly.
#------------------------------------------------------------------------------

# import system modules
#
import os
import sys
import math
import copy
import time
import logging
import threading
from collections import deque, defaultdict
from concurrent.futures import ThreadPoolExecutor

# import external modules
#
import numpy as np
from PyQt6 import QtCore
from PyQt6.QtCore import (
    Qt, QObject, pyqtSignal, QPointF,
    QTimer, QRectF, QDir, QProcess
)
from PyQt6.QtGui import (
    QPixmap, QColor, QPolygonF, QPen, QImage, QBrush,
    QPainter, QTransform, QPixmapCache
)
from PyQt6.QtWidgets import (
    QMainWindow, QWidget, QVBoxLayout, QHBoxLayout,
    QGridLayout, QFileDialog, QLabel, QMessageBox,
    QGroupBox, QProgressBar, QSizePolicy, QMenuBar,
    QGraphicsPixmapItem, QGraphicsPolygonItem,
    QGraphicsItem, QGraphicsView, QSplitter
)
# import nedc modules
#
from nedc_dpath_demo_config import AppConfig
from nedc_dpath_demo_menus import (
    PBCCMenu, ViewMenu, ProcessMenu, FileMenu,
    EditMenu, PreferencesDialog, HelpMenu
)
from nedc_dpath_demo_shared import (
    HeatMapCanvas, HistogramCanvas, Logger, PhotoViewer, MapNavigator
)
import nedc_dpath_ann_tools as nad
import nedc_image_tools as nit
import nedc_file_tools as nft
import nedc_debug_tools as ndt

#------------------------------------------------------------------------------
#
# global constants are listed below
#
#------------------------------------------------------------------------------

# define application and window configuration keys
#
CFG_KEY_ACTIVE_MODEL = 'ACTIVE_MODEL'
CFG_KEY_APP_NAME = 'APP_NAME'
CFG_KEY_DEFAULT_SEARCH_DIR = 'DEFAULT_SEARCH_DIR'
CFG_KEY_HEIGHT = 'HEIGHT'
CFG_KEY_MAIN_WIN_POS_X = 'MainWConf/MAIN_WIN_POS_X'
CFG_KEY_MAIN_WIN_POS_Y = 'MainWConf/MAIN_WIN_POS_Y'
CFG_KEY_WIDTH = 'WIDTH'

# define stream and tile configuration keys
#
CFG_KEY_HEAT_INDEX_CELL = 'Stream/HEAT_INDEX_CELL'
CFG_KEY_HIDE_BG_AT_OR_BELOW = 'HIDE_BG_AT_OR_BELOW'
CFG_KEY_NUM_TILE_THREADS = 'Stream/NUM_TILE_THREADS'
CFG_KEY_STREAM_FORCE_CATCHUP_MS = 'Stream/STREAM_FORCE_CATCHUP_MS'
CFG_KEY_STREAM_SETTLE_MS = 'Stream/STREAM_SETTLE_MS'
CFG_KEY_STREAM_TICK_MS = 'Stream/STREAM_TICK_MS'
CFG_KEY_STREAM_UI_TILES_PER_TICK = 'Stream/UI_TILES_PER_TICK'
CFG_KEY_TILE_SIZE = 'TILE_SIZE'
CFG_KEY_TILE_SUBMIT_BUDGET = 'Stream/TILE_SUBMIT_BUDGET'
CFG_KEY_TILE_UNLOAD_BUDGET = 'Stream/TILE_UNLOAD_BUDGET'
CFG_KEY_VIEWPORT_MARGIN = 'VIEWPORT_MARGIN'

# define ui panel and widget configuration keys
#
CFG_KEY_PANEL_MIN_W = 'UI/PANEL_MIN_W'
CFG_KEY_PBAR_WIDTH = 'UI/PBAR_WIDTH'
CFG_KEY_SHOW_HEATMAP = 'UIPanels/SHOW_HEATMAP'
CFG_KEY_SHOW_HISTOGRAM = 'UIPanels/SHOW_HISTOGRAM'
CFG_KEY_SHOW_LEGEND = 'UIPanels/SHOW_LEGEND'
CFG_KEY_SHOW_LOGGING = 'UIPanels/SHOW_LOGGING'
CFG_KEY_SHOW_NAVIGATOR_OVERLAY = 'UIPanels/SHOW_NAVIGATOR_OVERLAY'
CFG_KEY_SINGLESHOT_INIT_DELAY_MS = 'UI/SINGLESHOT_INIT_DELAY_MS'
CFG_KEY_SPLITTER_HANDLE_W = 'UI/SPLITTER_HANDLE_W'
CFG_KEY_SPLITTER_INIT_LOG_H = 'UI/SPLITTER_INIT_LOG_H'
CFG_KEY_SPLITTER_INIT_SIDE_H = 'UI/SPLITTER_INIT_SIDE_H'
CFG_KEY_SWATCH_MIN_H = 'UI/SWATCH_MIN_H'
CFG_KEY_SWATCH_MIN_W = 'UI/SWATCH_MIN_W'

# define scene, overlay, and annotation configuration keys
#
CFG_KEY_CONFIDENCE_THRESHOLD = 'CONFIDENCE_THRESHOLD'
CFG_KEY_HEAT_STROKE_MIN_W = 'Heatmap/HEAT_STROKE_MIN_W'
CFG_KEY_HYP_EDGE_WIDTH = 'HYP_EDGE_WIDTH'
CFG_KEY_HYP_OPACITY = 'HYP_OPACITY'
CFG_KEY_REF_EDGE_WIDTH = 'REF_EDGE_WIDTH'
CFG_KEY_SCALE_EPSILON = 'SCALE_EPSILON'
CFG_KEY_SHOW_HEAT_ON_PHOTO = 'SHOW_HEAT_ON_PHOTO'
CFG_KEY_Z_ANNOT_OFFSET = 'Scene/Z_ANNOT_OFFSET'
CFG_KEY_Z_TILES_OFFSET = 'Scene/Z_TILES_OFFSET'

# define process, model, and cache configuration keys
#
CFG_KEY_CACHE_ENFORCER_INTERVAL_MS = 'Cache/ENFORCER_INTERVAL_MS'
CFG_KEY_CACHE_LIMITS_ENABLED = 'Cache/LIMITS_ENABLED'
CFG_KEY_CACHE_QT_PIXMAP_MB = 'Cache/QT_PIXMAP_MB'
CFG_KEY_CACHE_READY_QUEUE_MAX = 'Cache/READY_QUEUE_MAX'
CFG_KEY_CACHE_TILE_CACHE_CAP = 'Cache/TILE_CACHE_CAP'
CFG_KEY_MODEL_DECODER_EXECUTABLE_FMT = 'Model/{}/decoder_executable'
CFG_KEY_MODEL_DECODER_PARAMS_FMT = 'Model/{}/decoder_params'
CFG_KEY_POSTPROC_PARAM_FILE = 'POSTPROC_PARAM_FILE'
CFG_KEY_WAIT_FOR_STARTED_MS = 'Process/WAIT_FOR_STARTED_MS'

# define message title, body, and format keys
#
CFG_KEY_MSG_ABOUT_BODY = 'Messages/ABOUT_BODY'
CFG_KEY_MSG_ABOUT_TITLE = 'Messages/ABOUT_TITLE'
CFG_KEY_MSG_BUSY_TITLE = 'Messages/BUSY_TITLE'
CFG_KEY_MSG_CONFIG_ERROR_TITLE = 'Messages/CONFIG_ERROR_TITLE'
CFG_KEY_MSG_DECODE_FAILED_TITLE = 'Messages/DECODE_FAILED_TITLE'
CFG_KEY_MSG_DECODE_LABEL = 'Messages/DECODE_LABEL'
CFG_KEY_MSG_FAILED_COULD_NOT_START = 'Messages/FAILED_COULD_NOT_START'
CFG_KEY_MSG_FAILED_EXIT_FMT = 'Messages/FAILED_EXIT_FMT'
CFG_KEY_MSG_FAILED_NO_OUTPUT_FMT = 'Messages/FAILED_NO_OUTPUT_FMT'
CFG_KEY_MSG_GENERIC_FAILED_TITLE_FMT = 'Messages/GENERIC_FAILED_TITLE_FMT'
CFG_KEY_MSG_HELP_BODY = 'Messages/HELP_BODY'
CFG_KEY_MSG_HELP_TITLE = 'Messages/HELP_TITLE'
CFG_KEY_MSG_MISSING_INPUT_BODY_PREFIX = 'Messages/MISSING_INPUT_BODY_PREFIX'
CFG_KEY_MSG_MISSING_INPUT_TITLE = 'Messages/MISSING_INPUT_TITLE'
CFG_KEY_MSG_MISSING_MODEL_CONFIG_FMT = 'Messages/MISSING_MODEL_CONFIG_FMT'
CFG_KEY_MSG_POSTPROC_FAILED_TITLE = 'Messages/POSTPROC_FAILED_TITLE'
CFG_KEY_MSG_POSTPROC_LABEL = 'Messages/POSTPROC_LABEL'

# define label and text configuration keys
#
CFG_KEY_FILTER_SVS = 'Labels/FILTER_SVS'
CFG_KEY_LBL_FILE_PREFIX = 'Labels/FILE_PREFIX'
CFG_KEY_LBL_HEAT = 'Labels/HEAT'
CFG_KEY_LBL_HIST = 'Labels/HIST'
CFG_KEY_LBL_LEGEND = 'Labels/LEGEND'
CFG_KEY_LBL_LOGGING = 'Labels/LOGGING'
CFG_KEY_LBL_NAVIGATOR = 'Labels/NAVIGATOR'
CFG_KEY_LEGEND_HEADER_HYP = 'Labels/LEGEND_HEADER_HYP'
CFG_KEY_LEGEND_HEADER_REF = 'Labels/LEGEND_HEADER_REF'
CFG_KEY_TITLE_OPEN_SVS = 'Labels/TITLE_OPEN_SVS'

# define ui object name keys
#
CFG_KEY_OBJ_CONF_PBAR = 'UIObjects/OBJ_CONF_PBAR'
CFG_KEY_OBJ_FILE_NAME_GROUP = 'UIObjects/OBJ_FILE_NAME_GROUP'
CFG_KEY_OBJ_FILE_NAME_LABEL = 'UIObjects/OBJ_FILE_NAME_LABEL'
CFG_KEY_OBJ_HEAT_LABEL = 'UIObjects/OBJ_HEAT_LABEL'
CFG_KEY_OBJ_HIST_LABEL = 'UIObjects/OBJ_HIST_LABEL'
CFG_KEY_OBJ_LEGEND_CONTAINER = 'UIObjects/OBJ_LEGEND_CONTAINER'
CFG_KEY_OBJ_LOG_LABEL = 'UIObjects/OBJ_LOG_LABEL'
CFG_KEY_OBJ_MAIN_WIN = 'UIObjects/OBJ_MAIN_WIN'
CFG_KEY_OBJ_NAVIGATOR_LABEL = 'UIObjects/OBJ_NAVIGATOR_LABEL'

# define style, color, and formatting keys
#
CFG_KEY_PBAR_FORMAT_PERCENT = 'Formatting/PBAR_FORMAT_PERCENT'
CFG_KEY_STYLE_CONF_PBAR_TEMPLATE = 'Styles/CONF_PBAR_TEMPLATE'
CFG_KEY_STYLE_CONF_PBAR_TOKEN_TOP = 'Styles/CONF_PBAR_TOKEN_TOP'
CFG_KEY_STYLE_LEGEND_GRID = 'Styles/LEGEND_GRID'
CFG_KEY_STYLE_SWATCH_TEMPLATE = 'Styles/SWATCH_TEMPLATE'

# define dimensions, values, and rendering logic keys
#
CFG_KEY_ALPHA_SCALE_255 = 'Rendering/ALPHA_SCALE_255'
CFG_KEY_CONF_DIVISOR = 'Behavior/CONF_DIVISOR'
CFG_KEY_CONF_HSV_S = 'Rendering/CONF_HSV_S'
CFG_KEY_CONF_HSV_V = 'Rendering/CONF_HSV_V'
CFG_KEY_CONF_HUE_MAX = 'Rendering/CONF_HUE_MAX'
CFG_KEY_CONF_OFFSET_DCIS = 'Behavior/CONF_OFFSET_DCIS'
CFG_KEY_CONF_OFFSET_INDC = 'Behavior/CONF_OFFSET_INDC'
CFG_KEY_CONF_SCALE_LOW = 'Behavior/CONF_SCALE_LOW'
CFG_KEY_HIST_FIG_DPI = 'Dimensions/HIST_FIG_DPI'
CFG_KEY_HIST_FIG_HEIGHT = 'Dimensions/HIST_FIG_HEIGHT'
CFG_KEY_HIST_FIG_WIDTH = 'Dimensions/HIST_FIG_WIDTH'
CFG_KEY_HSV_SAT_MAX = 'Rendering/HSV_SAT_MAX'
CFG_KEY_HSV_VAL_MAX = 'Rendering/HSV_VAL_MAX'
CFG_KEY_HUE_RANGE_BLUE_TO_RED_MAX = 'Rendering/HUE_RANGE_BLUE_TO_RED_MAX'
CFG_KEY_LEGEND_GRID_SPACING = 'Dimensions/LEGEND_GRID_SPACING'
CFG_KEY_LEGEND_MARGIN_LR = 'Dimensions/LEGEND_MARGIN_LR'
CFG_KEY_LEGEND_MARGIN_TB = 'Dimensions/LEGEND_MARGIN_TB'
CFG_KEY_OPEN_PBAR_STAGE1 = 'Rendering/OPEN_PBAR_STAGE1'
CFG_KEY_OPEN_PBAR_STAGE2 = 'Rendering/OPEN_PBAR_STAGE2'
CFG_KEY_OPEN_PBAR_STAGE3 = 'Rendering/OPEN_PBAR_STAGE3'
CFG_KEY_OPEN_PBAR_STAGE4 = 'Rendering/OPEN_PBAR_STAGE4'
CFG_KEY_OVERLAY_MIN_STROKE_PX = 'Rendering/OVERLAY_MIN_STROKE_PX'

# define file extension and program constants
#
CFG_KEY_EXT_CSVF = 'FileExt/EXT_CSVF'
CFG_KEY_EXT_CSVH = 'FileExt/EXT_CSVH'
CFG_KEY_EXT_CSV = 'FileExt/EXT_CSV'
CFG_KEY_POSTPROC_PROG = 'FileExt/POSTPROC_PROG'

# define ui text, messages, and title constants
#

# define ui object names and style templates
#

# define numeric constants and logic values
#
DEF_CACHE_ENFORCE_GUARD_MS = 100
DEF_CONF_HSV_S = 255
DEF_CONF_HSV_V = 255
DEF_HEAT_HUE_MAX = 240
DEF_JOB_PBAR_COMPLETE_HOLD_MS = 1000
DEF_JOB_PBAR_STEP_INTERVAL_MS = 150
DEF_POST_DISPLAY_IMAGE_DELAY_MS = 300
DEF_QPIX_CACHE_KB_PER_MB = 1024
DEF_SCALE_REL_EPSILON = 1e-09
DEF_TILE_BACKOFF_MAX_MS = 2000
DEF_TILE_SIZE_MAX_PX = 1536
DEF_TILE_SIZE_MIN_PX = 512
DEF_Z_TILE_LEVEL_SPACING = 0.01

# define internal keys, flags, and tuple constants
#
ALL_CLASS_KEYS = (
    'bckg', 'norm', 'artf', 'null', 'susp', 'infl', 'nneo', 'indc', 'dcis'
)
ANN_CONFS_KEY = 'conf'
ANN_LABELS_KEY = 'label'
ANN_PARSE_DATA_KEY = 'parse_data'
CACHE_CONF = 'conf'
CACHE_CSV_PATH = 'csv_path'
CACHE_ITEMS = 'items'
CACHE_LABEL = 'label'
ERRORS_IGNORE = 'ignore'
FMT_ANNOT_COLOR_HYP = 'AnnotationsColor/{}_hyp'
FMT_ANNOT_COLOR_REF = 'AnnotationsColor/{}_ref'
FMT_VISIBLE_FLAG = '{}/{}'
JOB_PBAR_PHASE1_VALUES = (25, 50)
JOB_PBAR_PHASE2_VALUES = (75, 100)
PREF_VISIBLE_HYP_PREFIX = 'AnnotationsVisibleHyp'
PREF_VISIBLE_REF_PREFIX = 'AnnotationsVisibleRef'
TILEMETA_ITEM_KEY = 'item'
TILEMETA_LAST_KEY = 'last'

#------------------------------------------------------------------------------
#
# helper functions are listed below
#
#------------------------------------------------------------------------------

def _patch_to_qimage(patch):
    """
    function: _patch_to_qimage

    arguments:
      patch: the input image patch (numpy array)

    return:
      QImage: the converted QImage object

    description:
      This function converts a numpy array patch to a QImage.
    """

    # get the numpy array
    #
    img_np = np.asarray(patch)

    # remove the alpha channel if present
    #
    if img_np.ndim == 3 and img_np.shape[2] > 3:
        img_np = img_np[:, :, :3]

    # ensure the dtype is uint8
    #
    if img_np.dtype != np.uint8:
        img_np = img_np.astype(np.uint8, copy=False)

    # ensure the array is contiguous
    #
    if not img_np.flags['C_CONTIGUOUS']:
        img_np = np.ascontiguousarray(img_np)

    # get the dimensions
    #
    img_h = int(img_np.shape[0])
    img_w = int(img_np.shape[1])

    # create and return the qimage
    #
    return QImage(img_np.data, img_w, img_h, int(3 * img_w),
                  QImage.Format.Format_RGB888).copy()
#
# end of function

def _compute_uniform_overlay_scale(base_w, base_h, ann_w, ann_h, eps):
    """
    function: _compute_uniform_overlay_scale

    arguments:
      base_w: base width
      base_h: base height
      ann_w: annotation width
      ann_h: annotation height
      eps: epsilon for comparison

    return:
      float: the computed scale factor or None

    description:
      This function computes the uniform overlay scale factor.
    """

    # convert inputs to int
    #
    base_w = int(base_w or 0)
    base_h = int(base_h or 0)
    ann_w = int(ann_w or 0)
    ann_h = int(ann_h or 0)

    # check for invalid dimensions
    #
    if base_w <= 0 or base_h <= 0 or ann_w <= 0 or (ann_h <= 0):
        return 1.0

    # compute scaling factors
    #
    sx = float(base_w) / float(ann_w)
    sy = float(base_h) / float(ann_h)

    # check if aspect ratios match within epsilon
    #
    if abs(sx - sy) <= float(eps) * max(sx, sy):
        return 0.5 * (sx + sy)

    # return none if mismatch
    #
    return None
#
# end of function

#------------------------------------------------------------------------------
#
# classes are listed below
#
#------------------------------------------------------------------------------

class MenuBar(QMenuBar):
    """
    class: MenuBar

    arguments:
      parent: the parent widget

    description:
      This class implements the main menu bar for the application.
    """

    # define the debug level
    #
    dbgl_d = ndt.Dbgl()

    def __init__(self, parent):
        """
        method: constructor

        arguments:
          parent: the parent widget

        return:
          none

        description:
          This method initializes the MenuBar class.
        """

        # call the super constructor
        #
        super().__init__(parent)

        # disable native menu bar
        #
        self.setNativeMenuBar(False)

        # create the pbcc menu
        #
        self.pbcc_menu = PBCCMenu(self)
        self.addMenu(self.pbcc_menu)

        # create the file menu
        #
        self.file_menu = FileMenu(self)
        self.addMenu(self.file_menu)

        # create the edit menu
        #
        self.edit_menu = EditMenu(self)
        self.addMenu(self.edit_menu)

        # create the view menu
        #
        self.view_menu = ViewMenu(self)
        self.addMenu(self.view_menu)

        # create the process menu
        #
        self.process_menu = ProcessMenu(self)
        self.addMenu(self.process_menu)

        # create the help menu
        #
        self.help_menu = HelpMenu(self)
        self.addMenu(self.help_menu)

        # initialize the preferences dialog
        #
        self._prefs_dialog = None

        # display debug information
        #
        if self.dbgl_d > ndt.BRIEF:
            logging.debug('%s (line: %s) %s: menu bar initialized',
                          __FILE__, ndt.__LINE__, self.__class__.__name__)
    #
    # end of method

    def connect_actions_to_event_loop(self, parent):
        """
        method: connect_actions_to_event_loop

        arguments:
          parent: the parent widget containing slots

        return:
          none

        description:
          This method connects menu actions to the parent's slots.
        """

        # connect pbcc menu actions
        #
        self.pbcc_menu.action_about.triggered.connect(parent.about_pabcc)

        # connect file menu actions
        #
        self.file_menu.action_open.triggered.connect(parent.open_file)
        self.file_menu.action_exit.triggered.connect(parent.close)

        # connect edit menu actions
        #
        self.edit_menu.action_save_settings.triggered.connect(parent.save_settings)
        self._prefs_dialog = PreferencesDialog(parent, parent.config)
        self._prefs_dialog.preferences_saved.connect(parent.on_preferences_saved)
        self.edit_menu.action_preferences.triggered.connect(self._prefs_dialog.exec)
        self.edit_menu.action_save_settings.triggered.connect(parent.config.sync)
        self.edit_menu.action_reset_settings.triggered.connect(parent.reset_settings)

        # connect view menu actions
        #
        self.view_menu.action_toggle_legend.triggered.connect(parent.toggle_legend)
        self.view_menu.action_toggle_histogram.triggered.connect(parent.toggle_histogram)
        self.view_menu.action_toggle_navigator_overlay.triggered.connect(parent.toggle_navigator_overlay)
        self.view_menu.action_toggle_heatmap.triggered.connect(parent.toggle_heatmap)
        self.view_menu.action_toggle_logging.triggered.connect(parent.toggle_logging)

        # connect help menu actions
        #
        self.help_menu.action_help.triggered.connect(parent.show_help)

        # connect process menu actions
        #
        self.process_menu.action_decode.triggered.connect(parent.decode)
        self.process_menu.action_postprocess.triggered.connect(parent.postprocess)

        # display debug information
        #
        if self.dbgl_d > ndt.BRIEF:
            logging.debug('%s (line: %s) %s: menu actions connected',
                          __FILE__, ndt.__LINE__, self.__class__.__name__)
    #
    # end of method
#
# end of class

class OverlayController:
    """
    class: OverlayController

    arguments:
      parent: the parent widget
      config: the configuration object
      session: the session object

    description:
      This class manages the overlay logic, including annotations (reference
      and hypothesis), heatmap streaming, and histogram data updates.
    """

    def __init__(self, parent, config, session):
        """
        method: constructor

        arguments:
          parent: the parent widget
          config: the configuration object
          session: the session object

        return:
          none

        description:
          This method initializes the OverlayController class.
        """

        # set the parent
        #
        self.parent = parent

        # set the configuration
        #
        self.config = config

        # set the session
        #
        self.session = session

        # initialize viewer references
        #
        self.image_viewer = None
        self.heatmap = None
        self.navigator = None
        self.hist_widget = None

        # initialize heatmap streaming state
        #
        self.heatmap_stream_enabled = False
        self.heat_graph_for_stream = {}
        self.heat_spatial_index = None

        # initialize reference annotations container
        #
        self.ref_annotations = {
            ANN_LABELS_KEY: [],
            ANN_PARSE_DATA_KEY: [],
            ANN_CONFS_KEY: []
        }

        # initialize cache for reference items
        #
        self._ref_cache = {
            CACHE_CSV_PATH: nft.DELIM_NULL,
            CACHE_ITEMS: [],
            CACHE_LABEL: [],
            CACHE_CONF: [],
            ANN_PARSE_DATA_KEY: []
        }

        # initialize cache for hypothesis items
        #
        self._hyp_cache = {
            CACHE_CSV_PATH: nft.DELIM_NULL,
            CACHE_ITEMS: [],
            CACHE_LABEL: [],
            CACHE_CONF: [],
            ANN_PARSE_DATA_KEY: []
        }
    #
    # end of method

    def attach_viewer(self, viewer):
        """
        method: attach_viewer

        arguments:
          viewer: the photo viewer object

        return:
          none

        description:
          This method attaches the photo viewer to the controller.
        """

        # set the viewer
        #
        self.image_viewer = viewer
    #
    # end of method

    def attach_heatmap(self, heatmap):
        """
        method: attach_heatmap

        arguments:
          heatmap: the heatmap canvas object

        return:
          none

        description:
          This method attaches the heatmap canvas to the controller.
        """

        # set the heatmap
        #
        self.heatmap = heatmap


        # enable stream tiles if required
        #
        if self.heatmap is not None:
            self.heatmap.enable_stream_tiles(bool(self.heatmap_stream_enabled))
    #
    # end of method

    def attach_navigator(self, navigator):
        """
        method: attach_navigator

        arguments:
          navigator: the map navigator object

        return:
          none

        description:
          This method attaches the map navigator to the controller.
        """

        # set the navigator
        #
        self.navigator = navigator
    #
    # end of method

    def attach_hist_widget(self, hist_widget):
        """
        method: attach_hist_widget

        arguments:
          hist_widget: the histogram widget object

        return:
          none

        description:
          This method attaches the histogram widget to the controller.
        """

        # set the histogram widget
        #
        self.hist_widget = hist_widget
    #
    # end of method

    def _label_is_visible(self, label):
        """
        method: _label_is_visible

        arguments:
          label: the class label

        return:
          bool: true if the label is visible

        description:
          This method checks if a label is visible in either ref or hyp.
        """

        # format the keys
        #
        key_ref = FMT_VISIBLE_FLAG.format(PREF_VISIBLE_REF_PREFIX, label)
        key_hyp = FMT_VISIBLE_FLAG.format(PREF_VISIBLE_HYP_PREFIX, label)

        # check preferences
        #
        return self.config.get(key_ref) or self.config.get(key_hyp)
    #
    # end of method

    def _build_items(self, csv_path):
        """
        method: _build_items

        arguments:
          csv_path: the path to the annotation csv file

        return:
          dict: a cache dictionary containing items and metadata

        description:
          This method loads annotations from a file and creates graphics items.
        """

        # check if file exists
        #
        # check for empty path
        #
        if not csv_path:
            return {
                CACHE_CSV_PATH: nft.DELIM_NULL,
                CACHE_ITEMS: [],
                CACHE_LABEL: [],
                CACHE_CONF: [],
                ANN_PARSE_DATA_KEY: []
             }

        # get the absolute path
        #
        abs_path = nft.get_fullpath(str(csv_path))

        # check existence
        #
        if not os.path.exists(abs_path):
            return {
                CACHE_CSV_PATH: nft.DELIM_NULL,
                CACHE_ITEMS: [],
                CACHE_LABEL: [],
                CACHE_CONF: [],
                ANN_PARSE_DATA_KEY: []
            }

        # load the annotations
        #
        ann = nad.AnnDpath()
        ok = ann.load(abs_path)

        # get base dimensions
        #
        base_w = int(self.session.base_width)
        base_h = int(self.session.base_height)

        # check for load failure
        #
        if not ok:
            logging.error('AnnDpath.load failed (%s)', csv_path)
            return {
                CACHE_CSV_PATH: csv_path,
                CACHE_ITEMS: [],
                CACHE_LABEL: [],
                CACHE_CONF: [],
                ANN_PARSE_DATA_KEY: []
            }

        # get header and graph
        #
        header = ann.get_header()
        graph_raw = ann.get_graph()

        # get annotation dimensions
        #
        ann_w = int(header[nad.CKEY_WIDTH])
        ann_h = int(header[nad.CKEY_HEIGHT])

        # get scale epsilon
        #
        eps = float(self.config.get(CFG_KEY_SCALE_EPSILON))

        # compute scale factor
        #
        scale_use = _compute_uniform_overlay_scale(
            base_w, base_h, ann_w, ann_h, eps
        )

        # check for scale mismatch
        #
        if scale_use is None:
            logging.warning('annotation aspect ratio mismatch; overlay disabled')
            return {
                CACHE_CSV_PATH: csv_path,
                CACHE_ITEMS: [],
                CACHE_LABEL: [],
                CACHE_CONF: [],
                ANN_PARSE_DATA_KEY: []
            }

        # check if scaling is needed
        #
        if abs(scale_use - 1.0) > DEF_SCALE_REL_EPSILON:
            graph = {}
            for k, entry in graph_raw.items():
                coords = []
                for xy in entry.get(nad.CKEY_COORDINATES, []):
                    xs = int(round(float(xy[0]) * scale_use))
                    ys = int(round(float(xy[1]) * scale_use))
                    rest = list(xy[2:])
                    coords.append((xs, ys, *rest))
                e2 = copy.deepcopy(entry)
                e2[nad.CKEY_COORDINATES] = coords
                graph[k] = e2
        else:
            graph = graph_raw

        # check viewer and scene availability
        #
        if self.image_viewer is None:
            logging.warning('scene not ready for items')
            return {
                CACHE_CSV_PATH: csv_path,
                CACHE_ITEMS: [],
                CACHE_LABEL: [],
                CACHE_CONF: [],
                ANN_PARSE_DATA_KEY: []
            }

        # get the scene
        #
        scene = self.image_viewer.scene()
        if scene is None:
            logging.warning('scene not ready for items')
            return {
                CACHE_CSV_PATH: csv_path,
                CACHE_ITEMS: [],
                CACHE_LABEL: [],
                CACHE_CONF: [],
                ANN_PARSE_DATA_KEY: []
            }

        # calculate z-value
        #
        z_above = self.image_viewer.base_photo_z() +\
            float(self.config.get(CFG_KEY_Z_ANNOT_OFFSET))

        # initialize lists
        #
        items = []
        labels = []
        confs = []
        parse_data = []
        pending_items = []

        # iterate through graph entries
        #
        for entry in graph.values():
            coords = entry.get(nad.CKEY_COORDINATES)

            pts = []
            region = []

            # build polygon points
            #
            for xy in coords:
                xi = int(xy[0])
                yi = int(xy[1])
                region.append((xi, yi))
                pts.append(QPointF(float(xi), float(yi)))

            # create graphics item
            #
            poly = QPolygonF(pts)
            item = QGraphicsPolygonItem(poly)
            item.setZValue(z_above)

            # extract metadata
            #
            label = str(entry.get(nad.CKEY_TEXT, nft.DELIM_NULL)).strip().lower()
            cval = float(entry[nad.CKEY_CONFIDENCE])

            # add to pending list
            #
            pending_items.append((item, label, cval, region))

        # batch add items to scene
        #
        if pending_items:
            self.image_viewer.setUpdatesEnabled(False)
            for item, label, cval, region in pending_items:
                scene.addItem(item)
                items.append(item)
                labels.append(label)
                confs.append(cval)
                parse_data.append(region)
            self.image_viewer.setUpdatesEnabled(True)

        # return cache dictionary
        #
        return {
            CACHE_CSV_PATH: csv_path,
            CACHE_ITEMS: items,
            CACHE_LABEL: labels,
            CACHE_CONF: confs,
            ANN_PARSE_DATA_KEY: parse_data
        }
    #
    # end of method

    def _restyle(self, cache, colour_map, pref_prefix,
                 edge_px, fill_alpha, conf_thr):
        """
        method: _restyle

        arguments:
          cache: the item cache dictionary
          colour_map: the mapping of labels to colors
          pref_prefix: the preference prefix string
          edge_px: the edge width in pixels
          fill_alpha: the fill alpha value
          conf_thr: the confidence threshold

        return:
          none

        description:
          This method updates the style and visibility of annotation items.
        """

        # check for empty cache
        #
        if not cache or not cache.get(CACHE_ITEMS):
            return

        # iterate items
        #
        for item, label, conf in\
                zip(cache[CACHE_ITEMS], cache[CACHE_LABEL], cache[CACHE_CONF]):

            # determine visibility
            #
            visible_key = FMT_VISIBLE_FLAG.format(pref_prefix, label)
            visible = self.config.get(visible_key)
            ok = visible and float(conf) >= float(conf_thr)

            # set visibility
            #
            item.setVisible(ok)
            if not ok:
                continue

            # get color
            #
            color_hex = colour_map.get(label)

            # set pen
            #
            pen = QPen(QColor(color_hex), int(edge_px))
            pen.setCosmetic(True)
            item.setPen(pen)

            # set brush
            #
            alpha = int(float(fill_alpha) * float(conf) * 255)
            r, g, b = QColor(color_hex).getRgb()[:3]
            item.setBrush(QBrush(QColor(r, g, b, alpha)))
    #
    # end of method

    def load_ref_annotations(self, ref_file):
        """
        method: load_ref_annotations

        arguments:
          ref_file: the path to the reference annotation file

        return:
          none

        description:
          This method loads and displays reference annotations.
        """

        # check for empty file
        #
        if not ref_file:
            return

        # check if rebuild is needed
        #
        if self._ref_cache[CACHE_CSV_PATH] != ref_file:
            self._ref_cache = self._build_items(ref_file)
            self.ref_annotations = {
                ANN_LABELS_KEY: list(self._ref_cache[CACHE_LABEL]),
                ANN_PARSE_DATA_KEY: list(self._ref_cache[ANN_PARSE_DATA_KEY]),
                ANN_CONFS_KEY: list(self._ref_cache[CACHE_CONF])
            }

        # restyle items
        #
        self._restyle(
            cache=self._ref_cache,
            colour_map=self.config.ref_color_map,
            pref_prefix=PREF_VISIBLE_REF_PREFIX,
            edge_px=int(self.config.get(CFG_KEY_REF_EDGE_WIDTH)),
            fill_alpha=0.0,
            conf_thr=0.0
        )
    #
    # end of method

    def draw_hyp_annotations(self, hyp_file):
        """
        method: draw_hyp_annotations

        arguments:
          hyp_file: the path to the hypothesis annotation file

        return:
          none

        description:
          This method loads and displays hypothesis annotations.
        """

        # check for empty file
        #
        if not hyp_file:
            return

        # check if rebuild is needed
        #
        if self._hyp_cache[CACHE_CSV_PATH] != hyp_file:
            self._hyp_cache = self._build_items(hyp_file)

        # restyle items
        #
        self._restyle(
            cache=self._hyp_cache,
            colour_map=self.config.hyp_color_map,
            pref_prefix=PREF_VISIBLE_HYP_PREFIX,
            edge_px=int(self.config.get(CFG_KEY_HYP_EDGE_WIDTH)),
            fill_alpha=float(self.config.get(CFG_KEY_HYP_OPACITY)),
            conf_thr=float(self.config.get(CFG_KEY_CONFIDENCE_THRESHOLD))
        )
    #
    # end of method

    def restyle_all(self):
        """
        method: restyle_all

        arguments:
          none

        return:
          none

        description:
          This method restyles both reference and hypothesis annotations.
        """

        # restyle reference items
        #
        self._restyle(
            self._ref_cache,
            self.config.ref_color_map,
            PREF_VISIBLE_REF_PREFIX,
            int(self.config.get(CFG_KEY_REF_EDGE_WIDTH)),
            0.0,
            0.0
        )

        # restyle hypothesis items
        #
        self._restyle(
            self._hyp_cache,
            self.config.hyp_color_map,
            PREF_VISIBLE_HYP_PREFIX,
            int(self.config.get(CFG_KEY_HYP_EDGE_WIDTH)),
            float(self.config.get(CFG_KEY_HYP_OPACITY)),
            float(self.config.get(CFG_KEY_CONFIDENCE_THRESHOLD))
        )
    #
    # end of method

    def update_histogram_data(self):
        """
        method: update_histogram_data

        arguments:
          none

        return:
          none

        description:
          This method updates the histogram widget with pixel data from
          annotated regions.
        """

        # check for histogram widget
        #
        if self.hist_widget is None:
            return

        # check for existing annotations
        #
        if not self.ref_annotations[ANN_LABELS_KEY]:
            self.hist_widget.clear_histogram()
            return

        # check for mil handle
        #
        if self.session.mil_full_res is None:
            logging.warning('no MIL handle available for histogram')
            return

        # get mil handle
        #
        mil = self.session.mil_full_res
        mil.set_level(0)

        # process data
        #
        all_r, all_g, all_b = self._process_histogram_data(mil)

        # update widget
        #
        self.hist_widget.update_histogram(all_r, all_g, all_b)
    #
    # end of method

    def _process_histogram_data(self, mil):
        """
        method: _process_histogram_data

        arguments:
          mil: the multi-resolution image library handle

        return:
          tuple: lists of red, green, and blue pixel values

        description:
          This method extracts pixel data from annotated regions.
        """

        # initialize lists
        #
        all_r, all_g, all_b = ([], [], [])
        labels = self.ref_annotations[ANN_LABELS_KEY]
        parse_data = self.ref_annotations[ANN_PARSE_DATA_KEY]

        # iterate labels and regions
        #
        for _label, region in zip(labels, parse_data):

            # check region validity
            #
            if not region:
                continue

            # calculate bounding box
            #
            xs = [int(pt[0]) for pt in region]
            ys = [int(pt[1]) for pt in region]
            min_x, max_x = (min(xs), max(xs))
            min_y, max_y = (min(ys), max(ys))
            width = max(1, max_x - min_x)
            height = max(1, max_y - min_y)

            # read image patch
            #
            patch = mil.read_data((min_x, min_y), npixx=width, npixy=height)
            if patch is None:
                continue

            # extract channels
            #
            arr = np.asarray(patch)
            if arr.ndim == 3 and arr.shape[2] == 3:
                all_r.append(arr[:, :, 0].ravel())
                all_g.append(arr[:, :, 1].ravel())
                all_b.append(arr[:, :, 2].ravel())

        # handle empty result
        #
        if not all_r or not all_g or (not all_b):
            empty = np.array([], dtype=np.uint8)
            return (empty, empty.copy(), empty.copy())

        # return concatenated arrays
        #
        return (np.concatenate(all_r),
                np.concatenate(all_g),
                np.concatenate(all_b))
    #
    # end of method

    def set_heatmap_streaming(self, enabled):
        """
        method: set_heatmap_streaming

        arguments:
        enabled: boolean flag to enable/disable streaming

        return:
        none

        description:
        This method enables or disables heatmap streaming.
        """

        # set enabled flag
        #
        self.heatmap_stream_enabled = bool(enabled)

        # notify heatmap widget
        #
        if self.heatmap is not None:
            self.heatmap.enable_stream_tiles(self.heatmap_stream_enabled)

        # clear indices if disabled
        #
        if not self.heatmap_stream_enabled:
            self.heat_graph_for_stream = {}
            self.heat_spatial_index = None
    #
    # end of method

    def _resync_heatmap_tiles(self, tile_items_by_level):
        """
        method: _resync_heatmap_tiles

        arguments:
          tile_items_by_level: dictionary of tile items keyed by level

        return:
          none

        description:
          This method updates the heatmap widget with current tile images.
        """

        # check if streaming is enabled
        #
        if not self.heatmap_stream_enabled:
            return

        # check dependencies
        #
        if self.heatmap is None or self.image_viewer is None:
            return

        # get current level
        #
        level = int(self.image_viewer.current_pyramid_level())
        downs = self.session.level_downsamples

        # check level validity
        #
        if not downs or level < 0 or level >= len(downs):
            return

        # get downsample factor
        #
        level_downsample = float(downs[level])

        # get items for level
        #
        by_level = tile_items_by_level.get(level, {})

        # mirror tiles to heatmap
        #
        for (tx, ty, tw, th), item in list(by_level.items()):
            pix = item.pixmap()
            if not pix.isNull():
                self.heatmap.mirror_tile(
                    level, (tx, ty, tw, th), pix, level_downsample
                )
    #
    # end of method

    def sync_heatmap_viewport(self, rect=None):
        """
        method: sync_heatmap_viewport

        arguments:
          rect: optional rectangle override

        return:
          none

        description:
          This method synchronizes the heatmap viewport
          with the photo viewer.
        """

        # get references
        #
        hm = self.heatmap
        viewer = self.image_viewer

        # check references
        #
        if hm is None or viewer is None:
            return

        # get viewport rect if needed
        #
        if rect is None:
            rect = viewer.mapToScene(
                viewer.viewport().rect()
            ).boundingRect()

        # check for null rect
        #
        if rect.isNull():
            return

        # follow viewport
        #
        hm.follow_viewport(rect)
    #
    # end of method



    def render_heat_overlay_tile(self, level, tile_rect):
        """
        method: render_heat_overlay_tile

        arguments:
        level: the pyramid level
        tile_rect: the tile rectangle (x, y, w, h)

        return:
        QImage: the rendered overlay image or None

        description:
        This method renders the heatmap overlay for a specific tile.
        """

        # check if overlay is enabled
        #
        show_on_photo = bool(self.config.get(CFG_KEY_SHOW_HEAT_ON_PHOTO))
        if not show_on_photo or not self.heatmap_stream_enabled:
            return None

        # check for graph data
        #
        graph = self.heat_graph_for_stream
        if not graph:
            return None

        # check spatial index
        #
        index_pack = self.heat_spatial_index
        if index_pack is None:
            return None

        # unpack tile rect
        #
        tile_x, tile_y, tile_w, tile_h = [int(v) for v in tile_rect]
        if tile_w <= 0 or tile_h <= 0:
            return None

        # check downsamples
        #
        downs = self.session.level_downsamples
        if not downs or int(level) < 0 or int(level) >= len(downs):
            return None
        level_downsample = float(downs[int(level)])

        # calculate scene rect
        #
        tile_rect_scene = QRectF(
            float(tile_x) * level_downsample,
            float(tile_y) * level_downsample,
            float(tile_w) * level_downsample,
            float(tile_h) * level_downsample
        )

        # create overlay image
        #
        overlay = QImage(
            int(tile_w), int(tile_h), QImage.Format.Format_ARGB32_Premultiplied
        )
        overlay.fill(0)

        # initialize painter
        #
        painter = QPainter(overlay)
        painter.setRenderHint(QPainter.RenderHint.Antialiasing, False)

        # get configuration
        #
        hyp_opacity = float(self.config.get(CFG_KEY_HYP_OPACITY))
        stroke_px = int(self.config.get(CFG_KEY_HYP_EDGE_WIDTH))
        min_stroke = int(self.config.get(CFG_KEY_HEAT_STROKE_MIN_W))

        # unpack index
        #
        index_map, bboxes, cell = index_pack

        # determine grid bounds
        #
        cx0 = int(tile_rect_scene.left() // cell)
        cy0 = int(tile_rect_scene.top() // cell)
        cx1 = int((tile_rect_scene.right() - 1) // cell)
        cy1 = int((tile_rect_scene.bottom() - 1) // cell)

        # collect candidate keys
        #
        candidate_keys = set()
        for cx in range(cx0, cx1 + 1):
            for cy in range(cy0, cy1 + 1):
                keys = index_map.get((cx, cy))
                if keys:
                    candidate_keys.update(keys)

        # filter candidates
        #
        candidates = [(k, bboxes.get(k)) for k in candidate_keys if k in graph]

        # draw candidates
        #
        for k, bbox in candidates:
            entry = graph[k]
            coords = entry.get(nad.CKEY_COORDINATES, [])
            if not coords or bbox is None:
                continue

            x0, y0, x1, y1 = bbox
            rect_scene =\
                QRectF(float(x0), float(y0), float(x1 - x0), float(y1 - y0))

            # check intersection
            #
            if not rect_scene.intersects(tile_rect_scene):
                continue

            # calculate color properties
            #
            conf = float(entry.get(nad.CKEY_CONFIDENCE, 0.0))
            hue = int((1.0 - conf) * DEF_HEAT_HUE_MAX)
            alpha = int(hyp_opacity * conf * 255)

            # calculate drawing coordinates
            #
            draw_x = float(x0) / level_downsample - float(tile_x)
            draw_y = float(y0) / level_downsample - float(tile_y)
            draw_w = float(x1 - x0) / level_downsample
            draw_h = float(y1 - y0) / level_downsample

            if draw_w <= 0.0 or draw_h <= 0.0:
                continue

            # set pen and brush
            #
            fill_qcolor = QColor.fromHsv(hue, DEF_CONF_HSV_S, DEF_CONF_HSV_V, alpha)
            pen_qcolor = QColor.fromHsv(hue, DEF_CONF_HSV_S, DEF_CONF_HSV_V, 255)
            pen = QPen(pen_qcolor)
            pen.setWidth(
                max(min_stroke, int(stroke_px / max(1.0, level_downsample)))
            )
            painter.setPen(pen)

            # draw rectangle
            #
            r = QRectF(draw_x, draw_y, draw_w, draw_h)
            painter.fillRect(r, fill_qcolor)
            painter.drawRect(r)

        # end painting
        #
        painter.end()

        # return overlay
        #
        return overlay
    #
    # end of method

    def _build_heat_index(self, graph, cell_size=None):
        """
        method: _build_heat_index

        arguments:
         graph: the graph data structure
         cell_size: the grid cell size

        return:
         tuple: index map, bounding boxes, and cell size

        description:
         This method builds a spatial index for heatmap rendering.
        """

        # check for graph data
        #
        if not graph:
            return None

        # set cell size
        #
        if cell_size is None:
            cell_size = int(self.config.get(CFG_KEY_HEAT_INDEX_CELL))

        # initialize index and boxes
        #
        index = defaultdict(set)
        bboxes = {}

        # iterate graph entries
        #
        for key, entry in graph.items():
            coords = entry.get(nad.CKEY_COORDINATES, [])
            if not coords:
                continue

            # calculate bounding box
            #
            xs = [int(pt[0]) for pt in coords]
            ys = [int(pt[1]) for pt in coords]
            x0, x1 = (min(xs), max(xs))
            y0, y1 = (min(ys), max(ys))

            # validate dimensions
            #
            if x1 <= x0 or y1 <= y0:
                continue

            # store bbox
            #
            bboxes[key] = (x0, y0, x1, y1)

            # index grid cells
            #
            cx0 = x0 // cell_size
            cy0 = y0 // cell_size
            cx1 = max(x0, x1 - 1) // cell_size
            cy1 = max(y0, y1 - 1) // cell_size

            for cx in range(cx0, cx1 + 1):
                for cy in range(cy0, cy1 + 1):
                    index[cx, cy].add(key)

        # return index bundle
        #
        return (index, bboxes, int(cell_size))
    #
    # end of method

    def _load_heat_graph_scaled_to_base(self, csv_path):
        """
        method: _load_heat_graph_scaled_to_base

        arguments:
         csv_path: path to the heat CSV for the current frame

        return:
         dict: heat graph with coordinates scaled into base-image coordinates

        description:
         This method loads a DPATH annotation CSV and scales the graph coordinates
         into the current session's base image coordinate system using a uniform
         scale derived from the annotation header.
        """

        # check for empty path
        #
        if not csv_path:
            return {}

        # expand to absolute path
        #
        abs_path = nft.get_fullpath(str(csv_path))

        # validate path existence
        #
        if (not abs_path) or (not os.path.exists(abs_path)):

            # log error for missing csv
            #
            logging.error('heat csv missing: %s', csv_path)

            # return empty graph
            #
            return {}

        # create annotation loader
        #
        ann = nad.AnnDpath()

        # load the CSV file
        #
        if not ann.load(str(abs_path)):

            # log loader failure
            #
            logging.error('AnnDpath.load failed (%s)', csv_path)

            # return empty graph
            #
            return {}

        # get header dictionary
        #
        header = ann.get_header() or {}

        # get raw graph dictionary
        #
        graph_raw = ann.get_graph() or {}

        # read base image dimensions from session
        #
        base_w = int(self.session.base_width)

        # read base image dimensions from session
        #
        base_h = int(self.session.base_height)

        # read annotation width from header
        #
        ann_w = int(header.get(nad.CKEY_WIDTH, 0) or 0)

        # read annotation height from header
        #
        ann_h = int(header.get(nad.CKEY_HEIGHT, 0) or 0)

        # read epsilon for scale computations
        #
        eps = float(self.config.get(CFG_KEY_SCALE_EPSILON))

        # compute uniform scale to map annotation coords to base coords
        #
        scale_use = _compute_uniform_overlay_scale(
            base_w,
            base_h,
            ann_w,
            ann_h,
            eps,
        )

        # check for aspect mismatch
        #
        if scale_use is None:

            # warn and disable overlay
            #
            logging.warning(
                'annotation aspect ratio mismatch; heat overlay disabled'
            )

            # return empty graph
            #
            return {}

        # if scale is effectively identity, return a deep copy of the raw graph
        #
        if abs(scale_use - 1.0) <= DEF_SCALE_REL_EPSILON:

            # return deep-copied dict
            #
            return {k: copy.deepcopy(v) for k, v in graph_raw.items()}

        # allocate output graph
        #
        graph = {}

        # iterate regions in the raw graph
        #
        for k, entry in graph_raw.items():

            # allocate scaled coordinate list
            #
            coords = []

            # fetch coordinate list safely
            #
            coord_list = entry.get(nad.CKEY_COORDINATES, []) or []

            # iterate coordinate tuples
            #
            for xy in coord_list:

                # compute scaled x coordinate
                #
                xs = int(round(float(xy[0]) * scale_use))

                # compute scaled y coordinate
                #
                ys = int(round(float(xy[1]) * scale_use))

                # preserve remaining fields
                #
                rest = list(xy[2:])

                # append scaled coordinate tuple
                #
                coords.append((xs, ys, *rest))

            # deep-copy entry to avoid mutating the source
            #
            e2 = copy.deepcopy(entry)

            # store scaled coordinates in the copy
            #
            e2[nad.CKEY_COORDINATES] = coords

            # store into output graph
            #
            graph[k] = e2

        # return scaled graph
        #
        return graph
    #
    # end of method


    def _postprocess_heat_graph(self, dpath_graph):
        """
        method: _postprocess_heat_graph

        arguments:
         dpath_graph: heat graph in base coordinates

        return:
         dict: processed heat graph with remapped confidence + threshold applied

        description:
         This method remaps per-region confidence based on label semantics and
         applies a threshold filter to remove low-confidence regions.
        """

        # read threshold
        #
        thr = float(self.config.get(CFG_KEY_CONFIDENCE_THRESHOLD))

        # read INDC offset
        #
        off_indc = float(self.config.get(CFG_KEY_CONF_OFFSET_INDC))

        # read DCIS offset
        #
        off_dcis = float(self.config.get(CFG_KEY_CONF_OFFSET_DCIS))

        # read divisor for high-confidence scaling
        #
        div = float(self.config.get(CFG_KEY_CONF_DIVISOR))

        # read low-confidence scale
        #
        scale_low = float(self.config.get(CFG_KEY_CONF_SCALE_LOW))

        # deep copy input graph
        #
        src = (dpath_graph or {}).items()

        # create result dict
        #
        result = {k: copy.deepcopy(v) for k, v in src}

        # iterate over a stable snapshot for safe popping
        #
        for key, entry in list(result.items()):

            # normalize label
            #
            label = str(entry.get(nad.CKEY_TEXT, nft.DELIM_NULL)).strip().lower()

            # read raw probability
            #
            prob = float(entry.get(nad.CKEY_CONFIDENCE, 0.0))

            # compute confidence based on label
            #
            if label == 'indc':
                conf = off_indc + prob / div

            # compute confidence based on label
            #
            elif label == 'dcis':
                conf = off_dcis + prob / div

            # compute confidence based on label
            #
            elif label in ('nneo', 'infl'):
                conf = prob * scale_low

            # default confidence is zero
            #
            else:
                conf = 0.0

            # clamp confidence to [0, 1]
            #
            conf = min(1.0, max(0.0, conf))

            # remove entries below threshold
            #
            if conf < thr:
                result.pop(key, None)

            # otherwise, store remapped confidence
            #
            else:
                entry[nad.CKEY_CONFIDENCE] = conf

        # return processed result
        #
        return result
    #
    # end of method


    def _build_heatmap_thumbnail_overlay_pixmap(self, graph_base_coords, stroke_px):
        """
        method: _build_heatmap_thumbnail_overlay_pixmap

        arguments:
         graph_base_coords: processed heat graph in base coordinates
         stroke_px: stroke width (pixels) used for rectangle edges

        return:
         QPixmap: overlay pixmap sized to the thumbnail

        description:
         This method renders an ARGB overlay for the heatmap panel (thumbnail
         space) by scaling from base coordinates into thumbnail coordinates and
         drawing per-region colored rectangles.
        """

        # fetch thumbnail pixmap
        #
        thumb = self.session.thumbnail

        # validate thumbnail pixmap
        #
        if thumb is None or thumb.isNull():
            return QPixmap()

        # read thumbnail dimensions
        #
        thumb_w = int(thumb.width())

        # read thumbnail dimensions
        #
        thumb_h = int(thumb.height())

        # read base dimensions
        #
        base_w = int(self.session.base_width)

        # read base dimensions
        #
        base_h = int(self.session.base_height)

        # read epsilon for scale computations
        #
        eps = float(self.config.get(CFG_KEY_SCALE_EPSILON))

        # compute uniform scale from base -> thumbnail
        #
        scale_thumb = _compute_uniform_overlay_scale(
            thumb_w,
            thumb_h,
            base_w,
            base_h,
            eps,
        )

        # abort if aspect mismatch
        #
        if scale_thumb is None:

            # warn and skip overlay
            #
            logging.warning(
                'heat overlay: thumbnail/base aspect mismatch; overlay skipped'
            )

            # return empty pixmap
            #
            return QPixmap()

        # create an ARGB overlay image
        #
        overlay = QImage(
            int(thumb_w),
            int(thumb_h),
            QImage.Format.Format_ARGB32_Premultiplied,
        )

        # clear overlay to transparent
        #
        overlay.fill(Qt.GlobalColor.transparent)

        # return early if no graph provided
        #
        if not graph_base_coords:
            return QPixmap.fromImage(overlay)

        # read opacity scaling factor
        #
        hyp_opacity = float(self.config.get(CFG_KEY_HYP_OPACITY))

        # read minimum stroke width
        #
        min_stroke = int(self.config.get(CFG_KEY_OVERLAY_MIN_STROKE_PX))

        # read alpha scaling
        #
        alpha_scale = int(self.config.get(CFG_KEY_ALPHA_SCALE_255))

        # read hue range max
        #
        hue_max = int(self.config.get(CFG_KEY_HUE_RANGE_BLUE_TO_RED_MAX))

        # read HSV saturation max
        #
        sat_max = int(self.config.get(CFG_KEY_HSV_SAT_MAX))

        # read HSV value max
        #
        val_max = int(self.config.get(CFG_KEY_HSV_VAL_MAX))

        # create painter
        #
        painter = QPainter(overlay)

        # disable anti aliasing for speed and crisp pixels
        #
        painter.setRenderHint(QPainter.RenderHint.Antialiasing, False)

        # scale painter so base coords map into thumbnail coords
        #
        painter.scale(float(scale_thumb), float(scale_thumb))

        # create pen placeholder
        #
        pen_q = QPen(QColor.fromHsv(0, 0, 0))

        # set pen width with minimum
        #
        pen_q.setWidth(max(min_stroke, int(stroke_px)))

        # iterate regions
        #
        for region in graph_base_coords.values():

            # read region confidence
            #
            conf = float(region.get(nad.CKEY_CONFIDENCE, 0.0))

            # clamp confidence
            #
            conf = min(1.0, max(0.0, conf))

            # compute alpha for this region
            #
            alpha = int(float(hyp_opacity) * conf * alpha_scale)

            # compute hue (blue->red mapping)
            #
            hue = int((1.0 - conf) * hue_max)

            # build fill color
            #
            fill = QColor.fromHsv(hue, sat_max, val_max, alpha)

            # build pen color
            #
            pen_color = QColor.fromHsv(hue, sat_max, val_max, alpha)

            # update pen color
            #
            pen_q.setColor(pen_color)

            # set pen on painter
            #
            painter.setPen(pen_q)

            # set brush on painter
            #
            painter.setBrush(fill)

            # fetch region coords
            #
            coords = region.get(nad.CKEY_COORDINATES, []) or []

            # skip empty coords
            #
            if not coords:
                continue

            # build x list
            #
            xs = [int(xy[0]) for xy in coords]

            # build y list
            #
            ys = [int(xy[1]) for xy in coords]

            # validate lists
            #
            if not xs or not ys:
                continue

            # compute bbox min/max
            #
            x0, x1 = (min(xs), max(xs))

            # compute bbox min/max
            #
            y0, y1 = (min(ys), max(ys))

            # compute bbox width with floor
            #
            rw = max(1, int(x1 - x0))

            # compute bbox height with floor
            #
            rh = max(1, int(y1 - y0))

            # create rectangle in base space (painter is scaled)
            #
            rect = QRectF(float(x0), float(y0), float(rw), float(rh))

            # fill rectangle
            #
            painter.fillRect(rect, fill)

            # draw rectangle edge
            #
            painter.drawRect(rect)

        # finalize painter
        #
        painter.end()

        # return pixmap from image
        #
        return QPixmap.fromImage(overlay)
    #
    # end of method


    def load_heatmap_overlay(self, frame_csv_path, tile_items_by_level):
        """
        method: load_heatmap_overlay

        arguments:
         frame_csv_path: CSV path for the current frame's heat overlay
         tile_items_by_level: existing tile state used for resync

        return:
         float: maximum confidence among retained regions

        description:
         This method owns the end-to-end CSV->graph->overlay pipeline:
         load the CSV, scale into base coordinates, remap confidence, build the
         thumbnail overlay pixmap, and enable streaming overlay tiles.
        """

        # check for null thumbnail
        #
        if self.session.thumbnail.isNull():

            # warn and return no confidence
            #
            logging.warning('thumbnail is null; cannot load heat overlay')

            # return max confidence
            #
            return 0.0

        # check heatmap widget presence
        #
        if self.heatmap is None:

            # warn and return no confidence
            #
            logging.warning('heatmap widget not initialized')

            # return max confidence
            #
            return 0.0

        # handle empty path by clearing overlay and disabling streaming
        #
        if not frame_csv_path:

            # clear overlay
            #
            self.heatmap.clear_overlay()

            # disable streaming mode
            #
            self.set_heatmap_streaming(False)

            # return max confidence
            #
            return 0.0

        # load raw graph and scale into base coords
        #
        raw_graph = self._load_heat_graph_scaled_to_base(frame_csv_path)

        # handle empty graph by clearing overlay and disabling streaming
        #
        if not raw_graph:

            # clear overlay
            #
            self.heatmap.clear_overlay()

            # disable streaming mode
            #
            self.set_heatmap_streaming(False)

            # return max confidence
            #
            return 0.0

        # apply confidence remap and threshold
        #
        graph = self._postprocess_heat_graph(raw_graph)

        # read stroke width for overlay edges
        #
        stroke_px = int(self.config.get(CFG_KEY_HYP_EDGE_WIDTH))

        # build overlay pixmap for heatmap panel
        #
        overlay_pm = self._build_heatmap_thumbnail_overlay_pixmap(
            graph_base_coords=graph,
            stroke_px=stroke_px,
        )

        # set overlay into the widget
        #
        self.heatmap.set_overlay_pixmap(overlay_pm, visible=True)

        # store processed graph for streaming overlay
        #
        self.heat_graph_for_stream = graph

        # build spatial index for streaming overlay tiles
        #
        self.heat_spatial_index = self._build_heat_index(
            self.heat_graph_for_stream
        )

        # enable streaming mode
        #
        self.set_heatmap_streaming(True)

        # resync tiles to current view using existing tile state
        #
        self._resync_heatmap_tiles(tile_items_by_level)

        # sync heatmap viewport to match viewer
        #
        self.sync_heatmap_viewport()

        # compute max confidence among retained regions
        #
        max_conf = 0.0

        # iterate retained regions
        #
        for entry in self.heat_graph_for_stream.values():

            # read confidence
            #
            c = float(entry.get(nad.CKEY_CONFIDENCE, 0.0))

            # update max
            #
            if c > max_conf:
                max_conf = c

        # return max confidence
        #
        return float(max_conf)
    #
    # end of method

    def reset_overlays(self):
        """
        method: reset_overlays

        arguments:
         none

        return:
         none

        description:
         This method resets all overlay data and caches.
        """

        # get viewer
        #
        viewer = self.image_viewer

        # clear existing items from scene
        #
        if viewer is None:
            self._ref_cache = {
                CACHE_CSV_PATH: nft.DELIM_NULL,
                CACHE_ITEMS: [],
                CACHE_LABEL: [],
                CACHE_CONF: [],
                ANN_PARSE_DATA_KEY: []
            }
            self._hyp_cache = {
                CACHE_CSV_PATH: nft.DELIM_NULL,
                CACHE_ITEMS: [],
                CACHE_LABEL: [],
                CACHE_CONF: [],
                ANN_PARSE_DATA_KEY: []
            }
            self.ref_annotations = {
                ANN_LABELS_KEY: [],
                ANN_PARSE_DATA_KEY: [],
                ANN_CONFS_KEY: []
            }
        else:
            scene = viewer.scene()
            if scene is not None:
                for item in list(self._ref_cache[CACHE_ITEMS]):
                    scene.removeItem(item)
                for item in list(self._hyp_cache[CACHE_ITEMS]):
                    scene.removeItem(item)

            self._ref_cache = {
                CACHE_CSV_PATH: nft.DELIM_NULL,
                CACHE_ITEMS: [],
                CACHE_LABEL: [],
                CACHE_CONF: [],
                ANN_PARSE_DATA_KEY: []
            }
            self._hyp_cache = {
                CACHE_CSV_PATH: nft.DELIM_NULL,
                CACHE_ITEMS: [],
                CACHE_LABEL: [],
                CACHE_CONF: [],
                ANN_PARSE_DATA_KEY: []
            }
            self.ref_annotations = {
                ANN_LABELS_KEY: [],
                ANN_PARSE_DATA_KEY: [],
                ANN_CONFS_KEY: []
            }

        # clear heatmap
        #
        if self.heatmap is not None:
            self.heatmap.clear_tiles()
            self.heatmap.clear_overlay()

        # disable streaming
        #
        self.set_heatmap_streaming(False)
    #
    # end of method

#
# end of class

class MainWindow(QMainWindow):
    """
    class: MainWindow

    arguments:
      none

    description:
      This class implements the main application window, managing the top-level
      UI, menu actions, file operations, and external processing jobs.
    """

    def __init__(self):
        """
        method: constructor

        arguments:
          none

        return:
          none

        description:
          This method initializes the MainWindow class.
        """

        # call the super constructor
        #
        super().__init__()

        # initialize file state
        #
        self.svs_file = nft.DELIM_NULL

        # initialize first show flag
        #
        self._first_show = True

        # initialize configuration
        #
        self.config = AppConfig(first_run=True)

        # set window title
        #
        self.setWindowTitle(self.config.get(CFG_KEY_APP_NAME))

        # set object name
        #
        self.setObjectName(self.config.get(CFG_KEY_OBJ_MAIN_WIN))

        # get geometry configuration
        #
        x = int(self.config.get(CFG_KEY_MAIN_WIN_POS_X))
        y = int(self.config.get(CFG_KEY_MAIN_WIN_POS_Y))
        w = int(self.config.get(CFG_KEY_WIDTH))
        h = int(self.config.get(CFG_KEY_HEIGHT))

        # set geometry
        #
        self.setGeometry(x, y, w, h)

        # create central widget
        #
        central_widget = QWidget(self)
        self.setCentralWidget(central_widget)

        # create menu bar
        #
        self.menu_bar = MenuBar(self)
        self.setMenuBar(self.menu_bar)

        # connect menu actions
        #
        self.menu_bar.connect_actions_to_event_loop(self)

        # create main display
        #
        self.main_display = MainDisplay(self, self.config)

        # get display layout block
        #
        display_block = self.main_display.main_block()

        # create main layout
        #
        layout = QVBoxLayout(central_widget)
        layout.addLayout(display_block, stretch=1)

        # connect components
        #
        self.connect_photoviewer_to_heatmap()

        # initialize job progress variables
        #
        self._job_pbar_timer = None
        self._job_pbar_values = []
        self._job_pbar_index = 0
        self._job_pbar_on_done = None
        self._job_proc = None
        self._job_kind = nft.DELIM_NULL
        self._job_on_success = None
        self._job_expected = nft.DELIM_NULL
        self._last_postproc_out = nft.DELIM_NULL
        self._menu_state_snapshot = None

        # apply cache limits
        #
        self.apply_cache_limits_from_config()

        # update menu actions
        #
        self._update_process_actions()
    #
    # end of method

    def save_settings(self):
        """
        method: save_settings

        arguments:
          none

        return:
          none

        description:
          This method saves the current configuration to disk.
        """

        # sync configuration
        #
        self.config.sync()

        # log action
        #
        logging.info('settings saved')
    #
    # end of method

    def _freeze_menus(self):
        """
        method: _freeze_menus

        arguments:
          none

        return:
          none

        description:
          This method disables the menu bar to prevent interaction during jobs.
        """

        # check if already frozen
        #
        if self._menu_state_snapshot is not None:
            return

        # get menu bar
        #
        mb = self.menu_bar
        if mb is None:
            return

        # define menus to freeze
        #
        menus = [mb.pbcc_menu, mb.file_menu, mb.edit_menu,
                 mb.view_menu, mb.process_menu, mb.help_menu]

        # snapshot state and disable
        #
        states = []
        for m in menus:
            states.append((m, bool(m.isEnabled())))
            m.setEnabled(False)

        # disable main bar
        #
        mb.setEnabled(False)

        # store snapshot
        #
        self._menu_state_snapshot = states
    #
    # end of method

    def _thaw_menus(self):
        """
        method: _thaw_menus

        arguments:
          none

        return:
          none

        description:
          This method restores the menu bar state after a job completes.
        """

        # check snapshot
        #
        if self._menu_state_snapshot is None:
            return

        # enable main bar
        #
        if self.menu_bar is not None:
            self.menu_bar.setEnabled(True)

        # restore menu states
        #
        for m, was_enabled in self._menu_state_snapshot:
            m.setEnabled(bool(was_enabled))

        # clear snapshot
        #
        self._menu_state_snapshot = None
    #
    # end of method

    def apply_cache_limits_from_config(self):
        """
        method: apply_cache_limits_from_config

        arguments:
          none

        return:
          none

        description:
          This method applies cache limits from the configuration
          to the display.
        """

        # get configuration
        #
        cfg = self.config

        # get display
        #
        disp = self.main_display
        if disp is None:
            return

        # check if limits enabled
        #
        enabled = bool(cfg.get(CFG_KEY_CACHE_LIMITS_ENABLED))

        # disable limits if not enabled
        #
        if not enabled:
            disp.set_runtime_cache_limits(tile_cache_cap=0)
            return

        # get values
        #
        qt_mb = int(cfg.get(CFG_KEY_CACHE_QT_PIXMAP_MB))
        rq_max = int(cfg.get(CFG_KEY_CACHE_READY_QUEUE_MAX))
        cap_tiles = int(cfg.get(CFG_KEY_CACHE_TILE_CACHE_CAP))

        # apply limits
        #
        disp.set_runtime_cache_limits(qt_pixmap_cache_mb=max(0, qt_mb),
                                      ready_queue_max=max(1, rq_max),
                                      tile_cache_cap=max(0, cap_tiles))
    #
    # end of method

    def _set_job_ui_state(self, busy):
        """
        method: _set_job_ui_state

        arguments:
          busy: boolean flag indicating busy state

        return:
          none

        description:
          This method updates the UI state based on whether a job is running.
        """

        # freeze or thaw menus
        #
        if busy:
            self._freeze_menus()
        else:
            self._thaw_menus()
    #
    # end of method

    def _clear_job_state(self):
        """
        method: _clear_job_state

        arguments:
          none

        return:
          none

        description:
          This method clears internal job tracking variables.
        """

        # reset variables
        #
        self._job_proc = None
        self._job_kind = nft.DELIM_NULL
        self._job_on_success = None
        self._job_expected = nft.DELIM_NULL
    #
    # end of method

    def _reset_job_progress_bar(self):
        """
        method: _reset_job_progress_bar

        arguments:
          none

        return:
          none

        description:
          This method resets the main progress bar to its initial state.
        """

        # get progress bar
        #
        pb = self.main_display.progress_bar if self.main_display else None
        if pb is None:
            return

        # reset properties
        #
        pb.setRange(0, 100)
        pb.setFormat(self.config.get(CFG_KEY_PBAR_FORMAT_PERCENT))
        pb.setTextVisible(True)
        pb.setValue(0)
    #
    # end of method

    def _finalize_job_ui(self):
        """
        method: _finalize_job_ui

        arguments:
          none

        return:
          none

        description:
          This method cleans up the UI after a job completes.
        """

        # stop progress sequence
        #
        self._stop_job_pbar_sequence()

        # reset ui state
        #
        self._set_job_ui_state(False)

        # reset progress bar
        #
        self._reset_job_progress_bar()

        # update actions
        #
        self._update_process_actions()
    #
    # end of method

    def _stop_job_pbar_sequence(self):
        """
        method: _stop_job_pbar_sequence

        arguments:
          none

        return:
          none

        description:
          This method stops any running progress bar animation sequence.
        """

        # stop timer
        #
        if self._job_pbar_timer is not None:
            self._job_pbar_timer.stop()
            self._job_pbar_timer = None

        # reset variables
        #
        self._job_pbar_values = []
        self._job_pbar_index = 0
        self._job_pbar_on_done = None
    #
    # end of method

    def _start_job_pbar_sequence(self, values, on_done=None):
        """
        method: _start_job_pbar_sequence

        arguments:
          values: list of progress values
          on_done: callback function when complete

        return:
          none

        description:
          This method starts a timed sequence of progress bar updates.
        """

        # get progress bar
        #
        pb = self.main_display.progress_bar if self.main_display else None
        if pb is None:
            return

        # stop existing sequence
        #
        self._stop_job_pbar_sequence()

        # set values
        #
        self._job_pbar_values = list(values or [])
        self._job_pbar_index = 0
        self._job_pbar_on_done = on_done

        # check for empty values
        #
        if not self._job_pbar_values:
            return

        # initialize progress bar
        #
        pb.setRange(0, 100)
        pb.setFormat(self.config.get(CFG_KEY_PBAR_FORMAT_PERCENT))
        pb.setTextVisible(True)
        pb.setValue(int(self._job_pbar_values[0]))

        # check for single value
        #
        if len(self._job_pbar_values) == 1:
            if callable(self._job_pbar_on_done):
                self._job_pbar_on_done()
            return

        # create timer
        #
        self._job_pbar_timer = QTimer(self)
        self._job_pbar_timer.setSingleShot(False)
        self._job_pbar_timer.setInterval(DEF_JOB_PBAR_STEP_INTERVAL_MS)
        self._job_pbar_timer.timeout.connect(self._on_job_pbar_tick)
        self._job_pbar_timer.start()
    #
    # end of method

    def _on_job_pbar_tick(self):
        """
        method: _on_job_pbar_tick

        arguments:
          none

        return:
          none

        description:
          This method handles timer ticks for the progress bar sequence.
        """

        # get progress bar
        #
        pb = self.main_display.progress_bar if self.main_display else None

        # check existence
        #
        if pb is None:
            self._stop_job_pbar_sequence()
            return

        # get values
        #
        vals = self._job_pbar_values
        if not vals:
            self._stop_job_pbar_sequence()
            return

        # increment index
        #
        self._job_pbar_index += 1

        # check completion
        #
        if self._job_pbar_index > len(vals) - 1:
            done = self._job_pbar_on_done
            self._stop_job_pbar_sequence()
            if callable(done):
                done()
            return

        # set value
        #
        pb.setValue(int(vals[self._job_pbar_index]))

        # check final value
        #
        if self._job_pbar_index == len(vals) - 1:
            done = self._job_pbar_on_done
            self._stop_job_pbar_sequence()
            if callable(done):
                done()
    #
    # end of method

    def _start_job_pbar_phase1(self):
        """
        method: _start_job_pbar_phase1

        arguments:
          none

        return:
          none

        description:
          This method starts the first phase of job progress.
        """

        # define values
        #
        vals = (0, *JOB_PBAR_PHASE1_VALUES)

        # start sequence
        #
        self._start_job_pbar_sequence(vals)
    #
    # end of method

    def _start_job_pbar_phase2(self):
        """
        method: _start_job_pbar_phase2

        arguments:
          none

        return:
          none

        description:
          This method starts the second phase of job progress.
        """

        # define values
        #
        vals = (JOB_PBAR_PHASE1_VALUES[-1], *JOB_PBAR_PHASE2_VALUES)

        # define completion callback
        #
        def after_reach_100():
            QTimer.singleShot(DEF_JOB_PBAR_COMPLETE_HOLD_MS, self._finalize_job_ui)

        # start sequence
        #
        self._start_job_pbar_sequence(vals, on_done=after_reach_100)
    #
    # end of method

    def _start_external_job_with_progress(self, kind, program, args,
                                          expected_output_path, on_success):
        """
        method: _start_external_job_with_progress

        arguments:
          kind: the type of job (e.g., decode)
          program: the executable path
          args: list of arguments
          expected_output_path: expected output file path
          on_success: callback on success

        return:
          bool: true if started successfully

        description:
          This method prepares the UI and starts an external job.
        """

        # set busy state
        #
        self._set_job_ui_state(True)

        # start progress
        #
        self._start_job_pbar_phase1()

        # run job
        #
        ok = self._run_external_job(
            kind, program, args, expected_output_path, on_success
        )

        # handle failure
        #
        if not ok:
            self._finalize_job_ui()

        # return status
        #
        return ok
    #
    # end of method

    def _get_install_root(self):
        """
        method: _get_install_root

        arguments:
          none

        return:
          str: absolute path to installation root directory

        description:
          This method returns the installation root directory used to locate
          the lib/ and bin/ directories. It first checks the environment
          variable NEDC_NFC, and if unset, derives the root from the location
          of this module (the parent directory of the lib/ folder).
        """

        # try the environment variable first
        #
        root = os.environ.get('NEDC_NFC', nft.DELIM_NULL).strip()

        # fall back to the parent directory of this file's directory
        #
        if not root:
            try:
                lib_dir = os.path.dirname(os.path.abspath(__file__))
                root = os.path.dirname(lib_dir)
            except Exception:
                root = os.getcwd()

        # normalize and return
        #
        return str(nft.get_fullpath(root))
    #
    # end of method

    def _qprocess_environment(self):
        """
        method: _qprocess_environment

        arguments:
          none

        return:
          QProcessEnvironment: an environment to use for child jobs

        description:
          This method constructs an environment for child QProcess jobs that:
            - sets NEDC_NFC to the installation root
            - prepends <root>/bin and <root>/lib to PATH
            - prepends <root>/lib to PYTHONPATH

          This removes the need for users to manually set PYTHONPATH and helps
          the tool run consistently on Windows, macOS, and Linux.
        """

        # start from the current process environment
        #
        env = QtCore.QProcessEnvironment.systemEnvironment()

        # compute installation paths
        #
        root = self._get_install_root()
        bin_dir = os.path.join(root, 'bin')
        lib_dir = os.path.join(root, 'lib')

        # set NEDC_NFC for child processes
        #
        env.insert('NEDC_NFC', root)

        # helper to prepend paths without duplication
        #
        def _prepend_path(var_name, new_paths):
            cur = env.value(var_name) or nft.DELIM_NULL
            cur_list = [p for p in cur.split(os.pathsep) if p]
            out = []
            for p in new_paths + cur_list:
                if p and p not in out:
                    out.append(p)
            env.insert(var_name, os.pathsep.join(out))

        # update PATH and PYTHONPATH
        #
        _prepend_path('PATH', [bin_dir, lib_dir])
        _prepend_path('PYTHONPATH', [lib_dir])

        # return environment
        #
        return env
    #
    # end of method

    def _qprocess_program_and_args(self, program, args):
        """
        method: _qprocess_program_and_args

        arguments:
          program: the configured program path (may be a script)
          args: list of arguments

        return:
          str: program to pass to QProcess
          list: argument list to pass to QProcess

        description:
          QProcess uses platform APIs similar to CreateProcess on Windows, which
          cannot directly execute non-.exe scripts (e.g., a Python script with no
          .exe wrapper). This method detects Python scripts and runs them via
          the current interpreter (sys.executable), which works on Windows,
          macOS, and Linux.
        """

        # normalize arguments
        #
        if args is None:
            args = []
        args = [str(a) for a in args]

        # normalize program path
        #
        prog_abs = nft.get_fullpath(str(program))
        prog_str = str(prog_abs)

        # decide if this looks like a Python script
        #
        ext = os.path.splitext(prog_str)[1].lower()
        is_py = ext in ('.py', '.pyw')

        # attempt shebang detection for extension-less scripts
        #
        if not is_py and ext == nft.DELIM_NULL and os.path.isfile(prog_str):
            try:
                with open(prog_str, nft.MODE_READ_BINARY) as fp:
                    first = fp.readline(256)
                first_txt = \
                    first.decode(
                        DEF_CHAR_ENCODING, errors='ignore'
                    ).lower().strip()
                if first_txt.startswith('#!') and 'python' in first_txt:
                    is_py = True
                elif os.name == 'nt':

                    # On Windows, an extension-less file is almost never a valid
                    # executable; if it looks like text, treat it as a script.
                    #
                    if b'\x00' not in first:
                        is_py = True
            except Exception:
                pass

        # run Python scripts via interpreter
        #
        if is_py:
            py_exe = sys.executable or 'python'
            py_exe = str(nft.get_fullpath(py_exe))

            # use -u so stdout/stderr are not buffered
            #
            return py_exe, ['-u', prog_str] + args

        # otherwise, run the program directly
        #
        return prog_str, args
    #
    # end of method

    def _run_external_job(self, kind, program, args, expected_output_path,
                          on_success):
        """
        method: _run_external_job

        arguments:
          kind: job type
          program: executable path (or script path)
          args: arguments list
          expected_output_path: output path
          on_success: success callback

        return:
          bool: true if process started

        description:
          This method configures and starts a QProcess for the job. On Windows,
          if the target program is a Python script (including extension-less
          scripts installed in /bin), it will be executed via the current
          Python interpreter to avoid "%1 is not a valid Win32 application".
        """

        # check for existing job
        #
        if self._job_proc is not None:
            logging.warning('another job is already running')
            msg_text = 'Another %s is running.' %\
                (self._job_kind)
            QMessageBox.warning(
                self, self.config.get(CFG_KEY_MSG_BUSY_TITLE), msg_text
            )
            return False

        # check program path
        #
        if not program:
            logging.error('empty program path')
            return False

        # set job parameters
        #
        self._job_kind = str(kind)
        self._job_expected = str(expected_output_path)
        self._job_on_success = on_success

        # create process
        #
        proc = QProcess(self)

        # set environment so child scripts can import from /lib without the
        # user having to manually set PYTHONPATH
        #
        proc.setProcessEnvironment(self._qprocess_environment())

        # normalize program/args for QProcess across platforms
        #
        q_prog, q_args = self._qprocess_program_and_args(program, args)
        proc.setProgram(str(q_prog))
        proc.setArguments([str(a) for a in q_args])

        # set channel mode
        #
        proc.setProcessChannelMode(QProcess.ProcessChannelMode.SeparateChannels)

        # connect signals
        #
        proc.readyReadStandardOutput.connect(
            lambda p=proc: self._log_process_stream(
                p, p.readAllStandardOutput, logging.info)
        )
        proc.readyReadStandardError.connect(
            lambda p=proc: self._log_process_stream(
                p, p.readAllStandardError, logging.warning)
        )
        proc.finished.connect(self._on_job_finished)

        # set process reference
        #
        self._job_proc = proc

        # log start
        #
        logging.info('%s started', self._job_kind)

        # start process
        #
        proc.start()

        # wait for start
        #
        wait_ms = int(self.config.get(CFG_KEY_WAIT_FOR_STARTED_MS))
        if not proc.waitForStarted(wait_ms):
            self._job_proc = None

            # Grab Qt's best explanation of why it couldn't start
            #
            err_str = proc.errorString() or "Unknown QProcess error"
            prog = proc.program() or "<unknown program>"
            pargs = proc.arguments() or []

            logging.error(
                "%s failed to start. program=%s args=%s error=%s",
                self._job_kind, prog, pargs, err_str
            )

            # Config message is "Could not start {}." so format with job kind
            #
            base_msg = self.config.get(CFG_KEY_MSG_FAILED_COULD_NOT_START)
            try:
                base_msg = base_msg.format(self._job_kind)
            except Exception:
                pass

            QMessageBox.critical(
                self,
                self._failed_title_for_kind(self._job_kind),
                f"{base_msg}\n\n"
                f"Reason: {err_str}\n\n"
                f"Program: {prog}\n"
                f"Args: {' '.join(map(str, pargs))}"
            )

            self._clear_job_state()
            return False

        # return success
        #
        return True
    #
    # end of method

    def _log_process_stream(self, proc, read_fn, log_fn):
        """
        method: _log_process_stream

        arguments:
          proc: the process object
          read_fn: function to read data
          log_fn: function to log data

        return:
          none

        description:
          This method reads output from a process and logs it.
        """

        # check process
        #
        if proc is None:
            return

        # read and decode data
        #
        data = bytes(read_fn()).decode(
            nft.DEF_CHAR_ENCODING, errors=ERRORS_IGNORE
        )

        # log lines
        #
        for line in data.splitlines():
            s = line.strip()
            if s:
                log_fn(s)
    #
    # end of method

    def _on_job_finished(self, exit_code, exit_status):
        """
        method: _on_job_finished

        arguments:
          exit_code: process exit code
          exit_status: process exit status

        return:
          none

        description:
          This method handles the completion of an external job.
        """

        # get context
        #
        kind = self._job_kind
        expected = self._job_expected
        proc = self._job_proc

        # flush logs
        #
        if proc is not None:
            self._log_process_stream(
                proc, proc.readAllStandardOutput, logging.info
            )
            self._log_process_stream(
                proc, proc.readAllStandardError, logging.warning
            )

        # get progress bar
        #
        pb = self.main_display.progress_bar if self.main_display else None

        # stop animation
        #
        self._stop_job_pbar_sequence()

        # reset progress bar properties
        #
        if pb is not None:
            pb.setRange(0, 100)
            pb.setFormat(self.config.get(CFG_KEY_PBAR_FORMAT_PERCENT))
            pb.setTextVisible(True)

        # check for failure
        #
        if exit_status != QProcess.ExitStatus.NormalExit or int(exit_code) != 0:
            logging.error('%s failed (exit=%d)', kind, int(exit_code))
            fmt = self.config.get(CFG_KEY_MSG_FAILED_EXIT_FMT)
            QMessageBox.critical(self, self._failed_title_for_kind(kind),
                                 fmt.format(kind, int(exit_code)))
            self._clear_job_state()
            self._finalize_job_ui()
            return

        # check output file
        #
        if expected and str(expected) != nft.DELIM_NULL and\
           (not os.path.exists(expected)):
            logging.error('%s success but file not found: %s', kind, expected)
            fmt = self.config.get(CFG_KEY_MSG_FAILED_NO_OUTPUT_FMT)
            QMessageBox.critical(self, self._failed_title_for_kind(kind),
                                 fmt.format(kind, expected))
            self._clear_job_state()
            self._finalize_job_ui()
            return

        # execute success callback
        #
        if callable(self._job_on_success):
            self._job_on_success()

        # log success
        #
        logging.info('%s finished successfully', kind)

        # set progress
        #
        if pb is not None:
            pb.setValue(int(JOB_PBAR_PHASE1_VALUES[-1]))

        # clear state and start phase 2
        #
        self._clear_job_state()
        self._start_job_pbar_phase2()
    #
    # end of method

    def _failed_title_for_kind(self, kind):
        """
        method: _failed_title_for_kind

        arguments:
          kind: job type string

        return:
          str: failure message title

        description:
          This method returns an appropriate title for a failed job.
        """

        # normalize kind
        #
        k = str(kind).lower()

        # check for decode
        #
        if k == str(self.config.get(CFG_KEY_MSG_DECODE_LABEL)).lower():
            return self.config.get(CFG_KEY_MSG_DECODE_FAILED_TITLE)

        # check for postproc
        #
        if k == str(self.config.get(CFG_KEY_MSG_POSTPROC_LABEL)).lower():
            return self.config.get(CFG_KEY_MSG_POSTPROC_FAILED_TITLE)

        # return generic format
        #
        fmt = self.config.get(CFG_KEY_MSG_GENERIC_FAILED_TITLE_FMT)
        return fmt.format(str(kind).capitalize() if kind else KIND_DEFAULT_JOB)
    #
    # end of method

    def _svs_dir_and_base(self):
        """
        method: _svs_dir_and_base

        arguments:
          none

        return:
          tuple: directory and base filename

        description:
          This method extracts the directory and base name from the SVS file path.
        """

        # check file
        #
        if not self.svs_file or self.svs_file == nft.DELIM_NULL:
            return (None, None)

        # extract components
        #
        svs_dir = os.path.dirname(self.svs_file)
        base = os.path.splitext(os.path.basename(self.svs_file))[0]

        # return tuple
        #
        return (svs_dir, base)
    #
    # end of method

    def _update_process_actions(self):
        """
        method: _update_process_actions

        arguments:
          none

        return:
          none

        description:
          This method enables or disables process menu actions based on state.
        """

        # get menu bar
        #
        mb = self.menu_bar
        if mb is None:
            return

        # get process menu
        #
        pm = mb.process_menu

        # get display
        #
        disp = self.main_display

        # enable decode
        #
        pm.action_decode.setEnabled(bool(
            self.svs_file and self.svs_file != nft.DELIM_NULL
        ))

        # check decoded status
        #
        decoded_ok = False
        if disp is not None:
            csvf_path = disp.frame_csv_path
            if csvf_path and csvf_path != nft.DELIM_NULL:
                decoded_ok = os.path.exists(str(csvf_path))

        # enable postprocess
        #
        pm.action_postprocess.setEnabled(bool(decoded_ok))
    #
    # end of method

    def decode(self):
        """
        method: decode

        arguments:
          none

        return:
          bool: true if job started

        description:
          This method initiates the decoding process for the current SVS file.
        """

        # check file
        #
        if not self.svs_file or self.svs_file == nft.DELIM_NULL:
            logging.warning('no SVS file selected for decode')
            return False

        # get model config
        #
        active = str(self.config.get(CFG_KEY_ACTIVE_MODEL))
        dec_exe = nft.get_fullpath(
            self.config.get(CFG_KEY_MODEL_DECODER_EXECUTABLE_FMT.format(active))
        )
        dec_pfile = nft.get_fullpath(
            self.config.get(CFG_KEY_MODEL_DECODER_PARAMS_FMT.format(active))
        )

        # validate config
        #
        if not dec_exe or not dec_pfile:
            msg = self.config.get(
                CFG_KEY_MSG_MISSING_MODEL_CONFIG_FMT
            ).format(active, active, active)
            logging.error(msg.replace(nft.DELIM_NULL, ' | '))
            QMessageBox.critical(
                self, self.config.get(CFG_KEY_MSG_CONFIG_ERROR_TITLE), msg
            )
            return False

        # validate files exist
        #
        if not os.path.exists(dec_exe) or not os.path.exists(dec_pfile):
            msg_text = 'Decoder or params missing:\n\n%s\n%s' %\
                (dec_exe, dec_pfile)
            QMessageBox.critical(
                self, self.config.get(CFG_KEY_MSG_CONFIG_ERROR_TITLE), msg_text
            )
            return False

        # get paths
        #
        svs_dir, base = self._svs_dir_and_base()
        if not svs_dir or not base:
            logging.error('could not resolve svs base/dir')
            return False

        # output directory
        #
        out_dir = svs_dir

        # output file
        #
        ext_csvf = self.config.get(CFG_KEY_EXT_CSVF)
        produced_csvf = os.path.join(out_dir, '%s.%s' % (base, ext_csvf))

        # define callback
        #
        def on_decode_success():
            self.main_display.frame_csv_path = produced_csvf
            self.main_display.load_heatmap_overlay(produced_csvf)
            logging.info('overlay heatmap loaded')
            self._update_process_actions()

        # setup arguments
        #
        args = ['-p', dec_pfile, '-o', out_dir, '-e', ext_csvf, self.svs_file]

        # log start
        #
        logging.info('decode started')

        # start job
        #
        return self._start_external_job_with_progress(
            self.config.get(CFG_KEY_MSG_DECODE_LABEL),
            dec_exe,
            args,
            produced_csvf,
            on_decode_success
        )
    #
    # end of method

    def postprocess(self):
        """
        method: postprocess

        arguments:
          none

        return:
          bool: true if job started

        description:
          This method initiates the post-processing job.
        """

        # check file
        #
        if not self.svs_file or self.svs_file == nft.DELIM_NULL:
            logging.warning('no SVS file selected for postprocess')
            return False

        # get paths
        #
        svs_dir, base = self._svs_dir_and_base()
        if not svs_dir or not base:
            logging.error('could not resolve svs base/dir')
            return False

        # check source file
        #
        ext_csvf = self.config.get(CFG_KEY_EXT_CSVF)
        csvf_src = os.path.join(svs_dir, '%s.%s' % (base, ext_csvf))
        if not os.path.exists(csvf_src):
            logging.error('decode output not found: %s', csvf_src)
            QMessageBox.critical(
                self, self.config.get(CFG_KEY_MSG_MISSING_INPUT_TITLE),
                self.config.get(CFG_KEY_MSG_MISSING_INPUT_BODY_PREFIX) + csvf_src
            )
            return False

        # get params
        #
        pproc_pfile = nft.get_fullpath(self.config.get(CFG_KEY_POSTPROC_PARAM_FILE))
        if not pproc_pfile or not os.path.exists(pproc_pfile):
            msg_text = 'Postproc params not found:\n\n%s' %\
                (pproc_pfile or nft.DELIM_NULL)
            QMessageBox.critical(
                self, self.config.get(CFG_KEY_MSG_CONFIG_ERROR_TITLE), msg_text
            )
            return False

        # output file
        #
        ext_csvh = self.config.get(CFG_KEY_EXT_CSVH)
        produced_csvh = os.path.join(svs_dir, '%s.%s' % (base, ext_csvh))

        # get program
        #
        pproc_prog = self.config.get(CFG_KEY_POSTPROC_PROG)

        # setup arguments
        #
        args = ['-p', pproc_pfile, '-o', svs_dir, '-e', ext_csvh, csvf_src]

        # set last output
        #
        self._last_postproc_out = produced_csvh

        # log start
        #
        logging.info('post-processing started')

        # start job
        #
        return self._start_external_job_with_progress(
            self.config.get(CFG_KEY_MSG_POSTPROC_LABEL),
            pproc_prog,
            args,
            produced_csvh,
            self._on_postproc_success
        )
    #
    # end of method

    def _on_postproc_success(self):
        """
        method: _on_postproc_success

        arguments:
          none

        return:
          none

        description:
          This method handles successful post-processing completion.
        """

        # load annotations
        #
        if self._last_postproc_out and self._last_postproc_out != nft.DELIM_NULL:
            self.main_display.draw_hyp_annotations(self._last_postproc_out)
            logging.info('hyp overlay loaded')
            self._update_process_actions()
    #
    # end of method

    def open_file(self):
        """
        method: open_file

        arguments:
          none

        return:
          none

        description:
          This method opens a file dialog to select an SVS file.
        """

        # get start directory
        #
        start_dir = self.config.get(CFG_KEY_DEFAULT_SEARCH_DIR)

        # get file name
        #
        new_svs_file, _ = QFileDialog.getOpenFileName(
            self,
            self.config.get(CFG_KEY_TITLE_OPEN_SVS),
            start_dir,
            self.config.get(CFG_KEY_FILTER_SVS)
        )

        # normalize path
        #
        new_svs_file = str(new_svs_file or nft.DELIM_NULL)
        self.svs_file = str(QDir.toNativeSeparators(new_svs_file))\
            if new_svs_file else nft.DELIM_NULL

        # open file if valid
        #
        if self.svs_file and self.svs_file != nft.DELIM_NULL:
            self.open_svs_file(self.svs_file)
    #
    # end of method

    def _prime_open_progress_bar(self, pb):
        """
        method: _prime_open_progress_bar

        arguments:
          pb: progress bar widget

        return:
          none

        description:
          This method resets the progress bar for the file opening sequence.
        """

        # check widget
        #
        if pb is None:
            return

        # setup progress bar
        #
        pb.setRange(0, 100)
        pb.setFormat(self.config.get(CFG_KEY_PBAR_FORMAT_PERCENT))
        pb.setTextVisible(True)
        pb.setValue(0)
    #
    # end of method

    def _set_open_progress(self, pb, value):
        """
        method: _set_open_progress

        arguments:
          pb: progress bar widget
          value: progress value

        return:
          none

        description:
          This method updates the progress bar and processes events.
        """

        # check widget
        #
        if pb is None:
            return

        # set value
        #
        pb.setValue(int(value))

        # process events
        #
        QtCore.QCoreApplication.processEvents(
            QtCore.QEventLoop.ProcessEventsFlag.AllEvents
        )
    #
    # end of method

    def open_svs_file(self, svs_file):
        """
        method: open_svs_file

        arguments:
          svs_file: path to the SVS file

        return:
          none

        description:
          This method performs the complete sequence to open
          and display an SVS file.
        """

        # get full path
        #
        full_path = QDir.toNativeSeparators(os.path.abspath(str(svs_file)))

        # check existence
        #
        if not os.path.exists(full_path):
            logging.error('SVS file not found: %s', full_path)
            return

        # update state
        #
        self.svs_file = full_path

        # get display
        #
        disp = self.main_display
        if disp is None:
            logging.error('MainDisplay not initialized')
            return

        # update title
        #
        disp.update_file_name(full_path)
        logging.info('opening %s', full_path)

        # get progress bar
        #
        pb = disp.progress_bar

        # prime progress
        #
        self._prime_open_progress_bar(pb)
        self._set_open_progress(pb, 0)

        # reset display
        #
        disp.reset_for_new_image()

        # update progress
        #
        self._set_open_progress(pb, self.config.get(CFG_KEY_OPEN_PBAR_STAGE1))

        # open mil file
        #
        probe0 = nit.Mil()
        if not probe0.open(full_path):
            logging.error('Mil.open failed (%s)', full_path)
            self._set_open_progress(pb, 0)
            return

        # set level
        #
        probe0.set_level(0)

        # get dimensions
        #
        img = probe0.image_d
        if hasattr(img, 'level_dimensions'):
            level_dims = img.level_dimensions
            level_downsamples = img.level_downsamples
        else:
            res = getattr(img, 'resolutions')
            level_dims = res.get('level_dimensions')
            level_downsamples = res.get('level_downsamples')

        # convert dimensions
        #
        level_dims_list = [tuple(map(int, wh)) for wh in level_dims or []]
        level_downsamples_list = [float(d) for d in level_downsamples or []]

        # validate dimensions
        #
        if not level_dims_list:
            logging.error('no level dimensions available from MIL')
            self._set_open_progress(pb, 0)
            return

        # validate downsamples
        #
        if not level_downsamples_list or\
           len(level_downsamples_list) != len(level_dims_list):
            logging.error('MIL level_downsamples mismatch; aborting open')
            self._set_open_progress(pb, 0)
            return

        # get base info
        #
        base_w, base_h = level_dims_list[0]
        max_level = len(level_dims_list) - 1
        preview_level = max_level
        prev_w, prev_h = level_dims_list[preview_level]

        # setup preview mil
        #
        if preview_level == 0:
            mil_preview = probe0
        else:
            mil_preview = nit.Mil()
            if not mil_preview.open(full_path):
                logging.error('Mil.open failed for preview (%s)', full_path)
                self._set_open_progress(pb, 0)
                return
            mil_preview.set_level(preview_level)

        # read preview data
        #
        patch = mil_preview.read_data((0, 0), npixx=prev_w, npixy=prev_h)
        if patch is None:
            logging.error('preview read_data returned None')
            self._set_open_progress(pb, 0)
            return

        # convert to image
        #
        qimg = _patch_to_qimage(patch)
        if qimg.isNull():
            logging.error('failed to build base thumbnail QImage')
            self._set_open_progress(pb, 0)
            return

        # convert to pixmap
        #
        pm = QPixmap.fromImage(qimg)
        if pm.isNull():
            logging.error('base thumbnail QPixmap isNull')
            self._set_open_progress(pb, 0)
            return

        # update progress
        #
        self._set_open_progress(pb, self.config.get(CFG_KEY_OPEN_PBAR_STAGE2))

        # set slide state
        #
        disp.set_slide_state(
            full_path=full_path, base_w=int(base_w), base_h=int(base_h),
            thumbnail_pm=pm, level_dims_list=level_dims_list,
            level_downsamples_list=level_downsamples_list,
            mil_full_res=probe0
        )

        # initialize viewer
        #
        viewer = disp.image_viewer
        if viewer is None:
            logging.error('PhotoViewer not initialized')
            self._set_open_progress(pb, 0)
            return
        viewer.initialize_slide_thumbnail(
            base_w=int(base_w), base_h=int(base_h),
            thumbnail_pm=pm,
            downsamples=level_downsamples_list,
            preview_level=int(preview_level)
        )

        # initialize heatmap
        #
        if disp.heatmap is not None:
            disp.heatmap.load_thumbnail_only(pm)
            self.heatmap_after_thumbnail_ready(pm)
            vis = viewer.mapToScene(viewer.viewport().rect()).boundingRect()
            disp.heatmap.follow_viewport(vis)
            disp.set_heatmap_streaming(True)
            disp.heatmap.clear_tiles()

        # initialize navigator
        #
        if disp.navigator is not None:
            disp.navigator.attach_to_viewer(viewer)
            disp.navigator.load_thumbnail(pm)
            disp.navigator.update_viewport_from_viewer(viewer)

        # update progress
        #
        self._set_open_progress(pb, self.config.get(CFG_KEY_OPEN_PBAR_STAGE3))

        # install hooks and display
        #
        disp.install_stream_hooks()
        disp.display_image_at_level(preview_level)

        # load overlays
        #
        self.autoload_overlays_on_open()

        # update progress
        #
        self._set_open_progress(pb, self.config.get(CFG_KEY_OPEN_PBAR_STAGE4))

        # define idle reset
        #
        def _reset_if_idle():
            if self._job_proc is None and pb is not None:
                pb.setValue(0)

        # schedule reset
        #
        QTimer.singleShot(DEF_JOB_PBAR_COMPLETE_HOLD_MS, _reset_if_idle)

        # update menu actions
        #
        self._update_process_actions()
    #
    # end of method

    def heatmap_after_thumbnail_ready(self, thumb_pixmap):
        """
        method: heatmap_after_thumbnail_ready

        arguments:
          thumb_pixmap: thumbnail pixmap

        return:
          none

        description:
          This method initializes the heatmap scene after the thumbnail
          is ready.
        """

        # get display
        #
        disp = self.main_display
        if disp is None or disp.heatmap is None:
            return

        # get dimensions
        #
        base_w = int(disp.session.base_width)
        base_h = int(disp.session.base_height)
        if base_w <= 0 or base_h <= 0:
            logging.warning('invalid base size for heatmap init')
            return

        # check pixmap
        #
        if thumb_pixmap is None or thumb_pixmap.isNull():
            logging.warning('thumb pixmap invalid for heatmap init')
            return

        # calculate padding
        #
        view_w = disp.heatmap.viewport().width()
        view_h = disp.heatmap.viewport().height()
        pad_x = max(0, (view_w - thumb_pixmap.width()) // 2)
        pad_y = max(0, (view_h - thumb_pixmap.height()) // 2)

        # initialize scene
        #
        disp.heatmap.initialize_heatmap_scene(thumb_pixmap, (base_w, base_h),
                                              pad_x=pad_x, pad_y=pad_y)
    #
    # end of method

    def autoload_overlays_on_open(self):
        """
        method: autoload_overlays_on_open

        arguments:
          none

        return:
          none

        description:
          This method attempts to load associated annotation files.
        """

        # check file
        #
        if not self.svs_file or self.svs_file == nft.DELIM_NULL:
            return

        # get paths
        #
        svs_dir, base = self._svs_dir_and_base()
        if not svs_dir or not base:
            return

        # get display
        #
        disp = self.main_display
        if disp is None:
            return

        # load csvf (heatmap)
        #
        ext_csvf = self.config.get(CFG_KEY_EXT_CSVF)
        csvf_path = os.path.join(svs_dir, '%s.%s' % (base, ext_csvf))
        disp.frame_csv_path = csvf_path
        if os.path.exists(csvf_path):
            disp.load_heatmap_overlay(csvf_path)

        # load csvh (hypotheses)
        #
        ext_csvh = self.config.get(CFG_KEY_EXT_CSVH)
        csvh_path = os.path.join(svs_dir, '%s.%s' % (base, ext_csvh))
        if os.path.exists(csvh_path):
            disp.draw_hyp_annotations(csvh_path)

        # load csv (references)
        #
        ext_csv = self.config.get(CFG_KEY_EXT_CSV)
        ref_path = os.path.join(svs_dir, '%s.%s' % (base, ext_csv))
        if os.path.exists(ref_path):
            disp.load_ref_annotations(ref_path)

        # update histogram
        #
        disp.update_histogram_data()

        # update actions
        #
        self._update_process_actions()
    #
    # end of method

    def connect_photoviewer_to_heatmap(self):
        """
        method: connect_photoviewer_to_heatmap

        arguments:
          none

        return:
          none

        description:
          This method connects the photo viewer to the heatmap controller.
        """

        # get display
        #
        disp = self.main_display
        if disp is None:
            return

        # get viewer
        #
        pv = disp.image_viewer
        if pv is None:
            return

        # set update mode
        #
        pv.setViewportUpdateMode(
            QGraphicsView.ViewportUpdateMode.MinimalViewportUpdate
        )

        # set callback
        #
        pv.set_tile_prune_callback(disp.stream.prune_main_tiles)
    #
    # end of method

    def on_preferences_saved(self):
        """
        method: on_preferences_saved

        arguments:
          none

        return:
          none

        description:
          This method applies preferences after they have been saved.
        """

        # get configuration
        #
        cfg = self.config

        # get display
        #
        disp = self.main_display
        if disp is None:
            return

        # log action
        #
        logging.info('preferences saved; applying')

        # update color map
        #
        cfg.create_color_map_dict()

        # apply cache limits
        #
        self.apply_cache_limits_from_config()

        # update display components
        #
        disp.update_legend_colors()
        disp.apply_overlay_preferences()

        # reload heatmap overlay if available
        #
        ann_file = disp.frame_csv_path
        if ann_file and ann_file != nft.DELIM_NULL and\
           (disp.heatmap is not None) and (not disp.session.thumbnail.isNull()):
            if os.path.exists(str(ann_file)):
                disp.load_heatmap_overlay(str(ann_file))

        # apply stream preferences
        #
        disp.apply_stream_preferences()

        # update actions
        #
        self._update_process_actions()
    #
    # end of method

    def reset_settings(self):
        """
        method: reset_settings

        arguments:
          none

        return:
          none

        description:
          This method resets the application settings to defaults.
        """

        # reset config
        #
        self.config.reset()

        # update color map
        #
        self.config.create_color_map_dict()

        # apply settings
        #
        self.on_preferences_saved()

        # log action
        #
        logging.info('settings reset to defaults')
    #
    # end of method

    def toggle_legend(self):
        """
        method: toggle_legend

        arguments:
          none

        return:
          none

        description:
          This method toggles the visibility of the legend panel.
        """

        # get display
        #
        disp = self.main_display
        if disp is None or disp.legend_panel is None:
            return

        # toggle visibility
        #
        disp.legend_panel.setVisible(not disp.legend_panel.isVisible())

        # rebalance layout
        #
        self._rebalance_side_splitter()
    #
    # end of method

    def toggle_histogram(self):
        """
        method: toggle_histogram

        arguments:
          none

        return:
          none

        description:
          This method toggles the visibility of the histogram panel.
        """

        # get display
        #
        disp = self.main_display
        if disp is None or disp.hist_panel is None:
            return

        # toggle visibility
        #
        disp.hist_panel.setVisible(not disp.hist_panel.isVisible())

        # rebalance layout
        #
        self._rebalance_side_splitter()
    #
    # end of method

    def toggle_navigator_overlay(self):
        """
        method: toggle_navigator_overlay

        arguments:
          none

        return:
          none

        description:
          This method toggles the visibility of the NAVIGATOR overlay panel.
        """

        # get display
        #
        disp = self.main_display
        if disp is None or disp.navigator_panel is None:
            return

        # toggle visibility
        #
        disp.navigator_panel.setVisible(not disp.navigator_panel.isVisible())

        # rebalance layout
        #
        self._rebalance_side_splitter()
    #
    # end of method

    def toggle_heatmap(self):
        """
        method: toggle_heatmap

        arguments:
          none

        return:
          none

        description:
          This method toggles the visibility of the heatmap panel.
        """

        # get display
        #
        disp = self.main_display
        if disp is None or disp.heat_panel is None:
            return

        # toggle visibility
        #
        disp.heat_panel.setVisible(not disp.heat_panel.isVisible())

        # rebalance layout
        #
        self._rebalance_side_splitter()
    #
    # end of method

    def toggle_logging(self):
        """
        method: toggle_logging

        arguments:
          none

        return:
          none

        description:
          This method toggles the visibility of the logging panel.
        """

        # get display
        #
        disp = self.main_display
        if disp is None or disp.logging_panel is None:
            return

        # toggle visibility
        #
        disp.logging_panel.setVisible(not disp.logging_panel.isVisible())

        # rebalance layout
        #
        self._rebalance_side_splitter()
    #
    # end of method

    def _rebalance_side_splitter(self):
        """
        method: _rebalance_side_splitter

        arguments:
          none

        return:
          none

        description:
          This method adjusts the splitter sizes based on visible panels.
        """

        # get display
        #
        disp = self.main_display
        if disp is None:
            return

        # get splitter
        #
        splitter = disp.side_splitter
        if splitter is None:
            return

        # get panels
        #
        legend_slice = disp.legend_panel
        hist_slice = disp.hist_panel
        navigator_slice = disp.navigator_panel
        heat_slice = disp.heat_panel

        # determine fixed width
        #
        legend_width = disp.legend_default_width if legend_slice\
            and legend_slice.isVisible() else 0

        # identify resizable panels
        #
        resizable = []
        if hist_slice and hist_slice.isVisible():
            resizable.append(hist_slice)
        if navigator_slice and navigator_slice.isVisible():
            resizable.append(navigator_slice)
        if heat_slice and heat_slice.isVisible():
            resizable.append(heat_slice)

        # calculate dimensions
        #
        total_w = splitter.size().width()
        remaining_w = max(0, total_w - legend_width)
        n_resizable = len(resizable)

        # handle all hidden
        #
        if n_resizable == 0:
            splitter.setSizes([legend_width, 0, 0, 0])
            return

        # calculate share
        #
        ideal_share = remaining_w // n_resizable
        panel_min = int(self.config.get(CFG_KEY_PANEL_MIN_W))
        panel_w = max(panel_min, ideal_share)

        # adjust for constraint
        #
        if panel_min * n_resizable > remaining_w:
            panel_w = max(0, remaining_w // max(1, n_resizable))

        # set sizes
        #
        sizes = [
            legend_width,
            panel_w if hist_slice and hist_slice.isVisible() else 0,
            panel_w if navigator_slice and navigator_slice.isVisible() else 0,
            panel_w if heat_slice and heat_slice.isVisible() else 0
        ]
        splitter.setSizes(sizes)
    #
    # end of method

    def about_pabcc(self):
        """
        method: about_pabcc

        arguments:
          none

        return:
          none

        description:
          This method displays the About dialog box.
        """

        # show message box
        #
        QMessageBox.information(
            self,
            self.config.get(CFG_KEY_MSG_ABOUT_TITLE),
            self.config.get(CFG_KEY_MSG_ABOUT_BODY),
            QMessageBox.StandardButton.Ok
        )
    #
    # end of method

    def show_help(self):
        """
        method: show_help

        arguments:
          none

        return:
          none

        description:
          This method displays the Help dialog box.
        """

        # show message box
        #
        QMessageBox.information(
            self,
            self.config.get(CFG_KEY_MSG_HELP_TITLE),
            self.config.get(CFG_KEY_MSG_HELP_BODY),
            QMessageBox.StandardButton.Ok
        )
    #
    # end of method

    def showEvent(self, event):
        """
        method: showEvent

        arguments:
          event: the show event

        return:
          none

        description:
          This method handles the window show event to
          perform initial layout.
        """

        # call super
        #
        super().showEvent(event)

        # check first show
        #
        if self._first_show:
            self._first_show = False
            self._rebalance_side_splitter()
    #
    # end of method
#
# end of class

class MainDisplay:
    """
    class: MainDisplay

    arguments:
      parent: the parent widget
      config: the configuration object

    description:
      This class assembles the main UI components of the application,
      managing the layout, panels, viewers, and connecting the logic
      controllers to the visual elements.
    """

    def __init__(self, parent, config):
        """
        method: constructor

        arguments:
          parent: the parent widget
          config: the configuration object

        return:
          none

        description:
          This method initializes the MainDisplay class.
        """

        # set the parent
        #
        self.parent = parent

        # set the configuration
        #
        self.config = config

        # initialize the session
        #
        self.session = SlideSession()

        # initialize the overlay controller
        #
        self.overlays = OverlayController(parent, config, self.session)

        # initialize the stream controller
        #
        self.stream =\
            TileStreamController(parent, config, self.session, self.overlays)

        # initialize widget references
        #
        self.image_viewer = None
        self.heatmap = None
        self.navigator = None
        self.logger = None
        self.image_conf_bar = None
        self.legend_layout = None
        self.legend_panel = None
        self.hist_panel = None
        self.navigator_panel = None
        self.heat_panel = None
        self.logging_panel = None
        self.progress_bar = None
        self.side_splitter = None
        self.main_splitter = None
        self.legend_default_width = 0
        self.hist_widget = None
        self.file_name_label = None

        # initialize state flags
        #
        self.stream_hook_installed = False
    #
    # end of method

    def main_block(self):
        """
        method: main_block

        arguments:
          none

        return:
          QVBoxLayout: the main layout containing the assembled UI

        description:
          This method constructs the primary layout of the central widget,
          including the top bar, image viewer, side panels, and logging area.
        """

        # create the top widget
        #
        top_widget = QWidget()
        top_box = QVBoxLayout(top_widget)
        top_box.setContentsMargins(0, 0, 0, 0)

        # add file info block
        #
        top_box.addWidget(self.file_info_block(), stretch=0)

        # create image row layout
        #
        image_row = QHBoxLayout()
        image_row.addLayout(self.image_block(), stretch=1)
        top_box.addLayout(image_row, stretch=1)

        # create panel layouts
        #
        self.legend_layout = self._create_legend_layout()
        hist_layout = self._create_histogram_layout()
        navigator_layout = self._create_navigator_layout()
        heat_layout = self._create_heatmap_layout()

        # create legend panel
        #
        self.legend_panel = QWidget()
        self.legend_panel.setLayout(self.legend_layout)
        self.legend_panel.setSizePolicy(QSizePolicy.Policy.Minimum,
                                        QSizePolicy.Policy.Preferred)

        # create histogram panel
        #
        self.hist_panel = QWidget()
        self.hist_panel.setLayout(hist_layout)
        self.hist_panel.setSizePolicy(QSizePolicy.Policy.Expanding,
                                      QSizePolicy.Policy.Preferred)

        # create navigator panel
        #
        self.navigator_panel = QWidget()
        self.navigator_panel.setLayout(navigator_layout)
        self.navigator_panel.setSizePolicy(QSizePolicy.Policy.Expanding,
                                     QSizePolicy.Policy.Preferred)

        # create heatmap panel
        #
        self.heat_panel = QWidget()
        self.heat_panel.setLayout(heat_layout)
        self.heat_panel.setSizePolicy(QSizePolicy.Policy.Expanding,
                                      QSizePolicy.Policy.Preferred)

        # set legend default width
        #
        self.legend_default_width = self.legend_panel.sizeHint().width()
        self.legend_panel.setMinimumWidth(self.legend_default_width)

        # create side splitter
        #
        self.side_splitter = QSplitter(Qt.Orientation.Horizontal)
        self.side_splitter.setHandleWidth(
            int(self.config.get(CFG_KEY_SPLITTER_HANDLE_W))
        )
        self.side_splitter.setOpaqueResize(False)

        # add panels to side splitter
        #
        self.side_splitter.addWidget(self.legend_panel)
        self.side_splitter.addWidget(self.hist_panel)
        self.side_splitter.addWidget(self.navigator_panel)
        self.side_splitter.addWidget(self.heat_panel)

        # disable collapsing for panels
        #
        for idx in range(4):
            self.side_splitter.setCollapsible(idx, False)

        # create logging layout
        #
        logging_layout = self._create_logging_layout()
        log_panel = QWidget()
        log_box = QVBoxLayout(log_panel)
        log_box.setContentsMargins(0, 0, 0, 0)
        log_box.addLayout(logging_layout)

        # create progress bar
        #
        self.progress_bar = QProgressBar()
        self.progress_bar.setRange(0, 100)
        self.progress_bar.setTextVisible(True)
        log_box.addWidget(self.progress_bar)
        self.logging_panel = log_panel

        # create main splitter
        #
        self.main_splitter = QSplitter(Qt.Orientation.Vertical)
        self.main_splitter.setHandleWidth(
            int(self.config.get(CFG_KEY_SPLITTER_HANDLE_W))
        )
        self.main_splitter.setOpaqueResize(False)

        # add widgets to main splitter
        #
        self.main_splitter.addWidget(top_widget)
        self.main_splitter.addWidget(self.side_splitter)
        self.main_splitter.addWidget(log_panel)

        # disable collapsing for main sections
        #
        for idx in range(3):
            self.main_splitter.setCollapsible(idx, False)

        # set stretch factors
        #
        self.main_splitter.setStretchFactor(0, 3)
        self.main_splitter.setStretchFactor(1, 1)
        self.main_splitter.setStretchFactor(2, 0)

        # apply initial visibility
        #
        self._apply_initial_panel_visibility()

        # schedule initial sizing
        #
        QTimer.singleShot(int(self.config.get(CFG_KEY_SINGLESHOT_INIT_DELAY_MS)),
                          self._apply_initial_sizes)

        # attach viewer to stream
        #
        if self.image_viewer is not None:
            self.stream.attach_viewer(self.image_viewer)

        # attach heatmap to stream
        #
        if self.heatmap is not None:
            self.stream.attach_heatmap(self.heatmap)

        # attach navigator to stream
        #
        if self.navigator is not None:
            self.stream.attach_navigator(self.navigator)

        # attach histogram to overlays
        #
        if self.hist_widget is not None:
            self.overlays.attach_hist_widget(self.hist_widget)

        # create outer layout
        #
        outer = QVBoxLayout()
        outer.setContentsMargins(0, 0, 0, 0)
        outer.addWidget(self.main_splitter)

        # return the layout
        #
        return outer
    #
    # end of method

    def _apply_initial_panel_visibility(self):
        """
        method: _apply_initial_panel_visibility

        arguments:
          none

        return:
          none

        description:
          This method sets the initial visibility of the UI panels based
          on configuration settings.
        """

        # set legend visibility
        #
        if self.legend_panel is not None:
            self.legend_panel.setVisible(self.config.get(CFG_KEY_SHOW_LEGEND))

        # set histogram visibility
        #
        if self.hist_panel is not None:
            self.hist_panel.setVisible(self.config.get(CFG_KEY_SHOW_HISTOGRAM))

        # set navigator visibility
        #
        if self.navigator_panel is not None:
            self.navigator_panel.setVisible(
                self.config.get(CFG_KEY_SHOW_NAVIGATOR_OVERLAY)
            )

        # set heatmap visibility
        #
        if self.heat_panel is not None:
            self.heat_panel.setVisible(self.config.get(CFG_KEY_SHOW_HEATMAP))

        # set logging visibility
        #
        if self.logging_panel is not None:
            self.logging_panel.setVisible(self.config.get(CFG_KEY_SHOW_LOGGING))
    #
    # end of method

    def _apply_initial_sizes(self):
        """
        method: _apply_initial_sizes

        arguments:
          none

        return:
          none

        description:
          This method sets the initial sizes of the splitter sections.
        """

        # get main splitter
        #
        main_splitter = self.main_splitter
        if main_splitter is None:
            return

        # calculate sizes
        #
        total_h = max(0, main_splitter.size().height())
        side_h_target = int(self.config.get(CFG_KEY_SPLITTER_INIT_SIDE_H))
        log_h = int(self.config.get(CFG_KEY_SPLITTER_INIT_LOG_H))
        top_h = max(1, total_h - side_h_target - log_h)

        # set sizes
        #
        main_splitter.setSizes([top_h, side_h_target, log_h])
    #
    # end of method

    def _create_legend_layout(self):
        """
        method: _create_legend_layout

        arguments:
          none

        return:
          QVBoxLayout: the layout for the legend panel

        description:
          This method creates the layout and widgets for the class legend.
        """

        # create vbox layout
        #
        legend_vbox = QVBoxLayout()

        # create legend label
        #
        legend_label = QLabel(self.config.get(CFG_KEY_LBL_LEGEND))
        label_font = legend_label.font()
        label_font.setBold(True)
        legend_label.setFont(label_font)

        # create legend grid
        #
        legend_grid = self._build_legend_swatch_widget()
        legend_grid.setStyleSheet(self.config.get(CFG_KEY_STYLE_LEGEND_GRID))

        # add widgets to layout
        #
        legend_vbox.addWidget(legend_label)
        legend_vbox.addWidget(legend_grid, 1)

        # return layout
        #
        return legend_vbox
    #
    # end of method

    def _label_is_visible(self, label):
        """
        method: _label_is_visible

        arguments:
          label: the class label

        return:
          bool: true if the label is visible

        description:
          This method checks the visibility preference for a label via the
          overlay controller.
        """

        # check visibility
        #
        return self.overlays._label_is_visible(label)
    #
    # end of method

    def _build_legend_swatch_widget(self):
        """
        method: _build_legend_swatch_widget

        arguments:
          none

        return:
          QWidget: the widget containing the color swatches

        description:
          This method builds the grid of color swatches for the legend.
        """

        # create container widget
        #
        legend_widget = QWidget()
        legend_widget.setObjectName(self.config.get(CFG_KEY_OBJ_LEGEND_CONTAINER))

        # create grid layout
        #
        grid_layout = QGridLayout(legend_widget)
        lr = int(self.config.get(CFG_KEY_LEGEND_MARGIN_LR))
        tb = int(self.config.get(CFG_KEY_LEGEND_MARGIN_TB))
        sp = int(self.config.get(CFG_KEY_LEGEND_GRID_SPACING))

        # set margins and spacing
        #
        grid_layout.setContentsMargins(lr, tb, lr, tb)
        grid_layout.setSpacing(sp)
        grid_layout.setAlignment(Qt.AlignmentFlag.AlignCenter)

        # get visible keys
        #
        class_keys =\
            [label for label in ALL_CLASS_KEYS if self._label_is_visible(label)]

        # create headers
        #
        header_label_ref = QLabel(self.config.get(CFG_KEY_LEGEND_HEADER_REF))
        header_label_ref.setAlignment(Qt.AlignmentFlag.AlignCenter)
        header_label_ref.setSizePolicy(
            QSizePolicy.Policy.Expanding, QSizePolicy.Policy.Fixed
        )

        header_label_hyp = QLabel(self.config.get(CFG_KEY_LEGEND_HEADER_HYP))
        header_label_hyp.setAlignment(Qt.AlignmentFlag.AlignCenter)
        header_label_hyp.setSizePolicy(
            QSizePolicy.Policy.Expanding, QSizePolicy.Policy.Fixed
        )

        # add headers
        #
        grid_layout.addWidget(header_label_ref, 0, 1)
        grid_layout.addWidget(header_label_hyp, 0, 2)

        # get style template and dimensions
        #
        swatch_tpl = self.config.get(CFG_KEY_STYLE_SWATCH_TEMPLATE)
        min_h = int(self.config.get(CFG_KEY_SWATCH_MIN_H))
        min_w = int(self.config.get(CFG_KEY_SWATCH_MIN_W))

        # create rows for classes
        #
        for row_index, cls_key in enumerate(class_keys, start=1):

            # create class label
            #
            class_label = QLabel(cls_key)
            class_label.setAlignment(Qt.AlignmentFlag.AlignCenter)
            class_label.setSizePolicy(
                QSizePolicy.Policy.Expanding, QSizePolicy.Policy.Fixed
            )
            grid_layout.addWidget(class_label, row_index, 0)

            # get hex colors
            #
            ref_hex = self.config.get(FMT_ANNOT_COLOR_REF.format(cls_key))
            hyp_hex = self.config.get(FMT_ANNOT_COLOR_HYP.format(cls_key))

            # create ref swatch
            #
            ref_square = QLabel()
            ref_square.setMinimumHeight(min_h)
            ref_square.setMinimumWidth(min_w)
            ref_square.setSizePolicy(
                QSizePolicy.Policy.Expanding, QSizePolicy.Policy.Expanding
            )
            ref_square.setStyleSheet(swatch_tpl.format(ref_hex))
            grid_layout.addWidget(ref_square, row_index, 1)

            # create hyp swatch
            #
            hyp_square = QLabel()
            hyp_square.setMinimumHeight(min_h)
            hyp_square.setMinimumWidth(min_w)
            hyp_square.setSizePolicy(
                QSizePolicy.Policy.Expanding, QSizePolicy.Policy.Expanding
            )
            hyp_square.setStyleSheet(swatch_tpl.format(hyp_hex))
            grid_layout.addWidget(hyp_square, row_index, 2)

        # set size policy
        #
        legend_widget.setSizePolicy(
            QSizePolicy.Policy.Expanding, QSizePolicy.Policy.Expanding
        )

        # return widget
        #
        return legend_widget
    #
    # end of method

    def _create_histogram_layout(self):
        """
        method: _create_histogram_layout

        arguments:
          none

        return:
          QVBoxLayout: the layout for the histogram panel

        description:
          This method creates the histogram widget and its layout.
        """

        # create layout
        #
        layout = QVBoxLayout()

        # create label
        #
        hist_label = QLabel(self.config.get(CFG_KEY_LBL_HIST))
        hist_label.setObjectName(self.config.get(CFG_KEY_OBJ_HIST_LABEL))

        # get dimensions
        #
        w = int(self.config.get(CFG_KEY_HIST_FIG_WIDTH))
        h = int(self.config.get(CFG_KEY_HIST_FIG_HEIGHT))
        dpi = int(self.config.get(CFG_KEY_HIST_FIG_DPI))

        # create histogram widget
        #
        self.hist_widget =\
            HistogramCanvas(self.parent, width=w, height=h, dpi=dpi)

        # attach to overlays
        #
        self.overlays.attach_hist_widget(self.hist_widget)

        # update data
        #
        self.update_histogram_data()

        # add to layout
        #
        layout.addWidget(hist_label)
        layout.addWidget(self.hist_widget)

        # return layout
        #
        return layout
    #
    # end of method

    def _create_navigator_layout(self):
        """
        method: _create_navigator_layout

        arguments:
          none

        return:
          QVBoxLayout: the layout for the map navigator panel

        description:
          This method creates the map navigator widget and its layout.
        """

        # create layout
        #
        layout = QVBoxLayout()

        # create label
        #
        navigator_label = QLabel(self.config.get(CFG_KEY_LBL_NAVIGATOR))
        navigator_label.setObjectName(self.config.get(CFG_KEY_OBJ_NAVIGATOR_LABEL))

        # create navigator widget
        #
        self.navigator = MapNavigator(self.parent)

        # attach to overlays
        #
        self.overlays.attach_navigator(self.navigator)

        # add to layout
        #
        layout.addWidget(navigator_label, stretch=0)
        layout.addWidget(self.navigator, stretch=1)
        layout.setStretch(1, 1)

        # return layout
        #
        return layout
    #
    # end of method

    def _create_heatmap_layout(self):
        """
        method: _create_heatmap_layout

        arguments:
          none

        return:
          QVBoxLayout: the layout for the heatmap panel

        description:
          This method creates the heatmap widget and its layout.
        """

        # create layout
        #
        layout = QVBoxLayout()

        # create label
        #
        heat_label = QLabel(self.config.get(CFG_KEY_LBL_HEAT))
        heat_label.setObjectName(self.config.get(CFG_KEY_OBJ_HEAT_LABEL))

        # create heatmap widget
        #
        self.heatmap = HeatMapCanvas(self.parent)

        # attach to overlays
        #
        self.overlays.attach_heatmap(self.heatmap)

        # add to layout
        #
        layout.addWidget(heat_label)
        layout.addWidget(self.heatmap)

        # return layout
        #
        return layout
    #
    # end of method

    def _create_logging_layout(self):
        """
        method: _create_logging_layout

        arguments:
          none

        return:
          QVBoxLayout: the layout for the logging panel

        description:
          This method creates the logger widget and its layout.
        """

        # create layout
        #
        layout = QVBoxLayout()

        # create label
        #
        log_label = QLabel(self.config.get(CFG_KEY_LBL_LOGGING))
        log_label.setObjectName(self.config.get(CFG_KEY_OBJ_LOG_LABEL))

        # create logger
        #
        self.logger = Logger(self.parent)

        # add to layout
        #
        layout.addWidget(log_label)
        layout.addWidget(self.logger)

        # return layout
        #
        return layout
    #
    # end of method

    def image_block(self):
        """
        method: image_block

        arguments:
          none

        return:
          QHBoxLayout: the layout containing the image viewer

        description:
          This method creates the image viewer block.
        """

        # create layout
        #
        image_layout = QHBoxLayout()
        image_layout.setSpacing(10)

        # create photo viewer
        #
        self.image_viewer = PhotoViewer(self.parent)
        self.image_viewer.resolutionChangeRequested.connect(
            self.on_resolution_change_requested
        )

        # create confidence bar
        #
        self.image_conf_bar = QProgressBar()
        self.image_conf_bar.setObjectName(self.config.get(CFG_KEY_OBJ_CONF_PBAR))
        self.image_conf_bar.setOrientation(Qt.Orientation.Vertical)
        self.image_conf_bar.setFixedWidth(int(self.config.get(CFG_KEY_PBAR_WIDTH)))

        # add widgets to layout
        #
        image_layout.addWidget(self.image_viewer)
        image_layout.addWidget(self.image_conf_bar)

        # return layout
        #
        return image_layout
    #
    # end of method

    def file_info_block(self):
        """
        method: file_info_block

        arguments:
          none

        return:
          QGroupBox: the file info group box

        description:
          This method creates the file information block.
        """

        # create group box
        #
        file_info_block = QGroupBox()

        # create layout and label
        #
        file_name_group = QHBoxLayout()
        file_name_group.setObjectName(self.config.get(CFG_KEY_OBJ_FILE_NAME_GROUP))
        file_name_label = QLabel(self.config.get(CFG_KEY_LBL_FILE_PREFIX))
        file_name_label.setObjectName(self.config.get(CFG_KEY_OBJ_FILE_NAME_LABEL))
        file_name_label.setAttribute(Qt.WidgetAttribute.WA_TranslucentBackground)
        file_name_label.setAlignment(Qt.AlignmentFlag.AlignLeft)
        file_name_label.setWordWrap(True)

        # assemble block
        #
        file_name_group.addWidget(file_name_label)
        file_info_layout = QVBoxLayout()
        file_info_layout.addLayout(file_name_group)
        file_info_block.setLayout(file_info_layout)

        # store reference
        #
        self.file_name_label = file_name_label

        # return block
        #
        return file_info_block
    #
    # end of method

    def _update_conf_bar(self, conf):
        """
        method: _update_conf_bar

        arguments:
          conf: confidence value (0.0 to 1.0)

        return:
          none

        description:
          This method updates the confidence bar with the current
          value and color.
        """

        # check existence
        #
        if self.image_conf_bar is None:
            return

        # clamp value
        #
        clamped = max(0.0, min(float(conf), 1.0))

        # set value
        #
        self.image_conf_bar.setValue(int(clamped * 100))

        # calculate color
        #
        hue_max = int(self.config.get(CFG_KEY_CONF_HUE_MAX))
        hue = int((1.0 - clamped) * hue_max)
        s = int(self.config.get(CFG_KEY_CONF_HSV_S))
        v = int(self.config.get(CFG_KEY_CONF_HSV_V))
        top_colour = QColor.fromHsv(hue, s, v).name()

        # apply style
        #
        tpl = self.config.get(CFG_KEY_STYLE_CONF_PBAR_TEMPLATE)
        token = self.config.get(CFG_KEY_STYLE_CONF_PBAR_TOKEN_TOP)
        style = tpl.replace(token, top_colour)
        self.image_conf_bar.setStyleSheet(style)
    #
    # end of method

    def update_file_name(self, file_name):
        """
        method: update_file_name

        arguments:
          file_name: the file name to display

        return:
          none

        description:
          This method updates the file name label text.
        """

        # update label
        #
        if self.file_name_label:
            prefix = self.config.get(CFG_KEY_LBL_FILE_PREFIX)
            self.file_name_label.setText(prefix + str(file_name))
    #
    # end of method

    def update_legend_colors(self):
        """
        method: update_legend_colors

        arguments:
          none

        return:
          none

        description:
          This method refreshes the legend colors based on
          configuration changes.
        """

        # check layout
        #
        if self.legend_layout is None or self.legend_layout.count() < 2:
            return

        # remove old widget
        #
        old_item = self.legend_layout.itemAt(1)
        old_widget = old_item.widget() if old_item else None
        if old_widget is not None:
            old_widget.setParent(None)
            old_widget.deleteLater()

        # create new widget
        #
        new_legend = self._build_legend_swatch_widget()
        self.legend_layout.addWidget(new_legend, 1)
        self.legend_layout.setStretch(1, 1)
    #
    # end of method

    @property
    def frame_csv_path(self):
        """
        property: frame_csv_path

        description:
          Getter for the frame-level CSV path from the session.
        """

        # return path
        #
        return self.session.frame_csv_path

    @frame_csv_path.setter
    def frame_csv_path(self, v):
        """
        property: frame_csv_path

        description:
          Setter for the frame-level CSV path in the session.
        """

        # set path
        #
        self.session.frame_csv_path = v

    def set_runtime_cache_limits(self, qt_pixmap_cache_mb=None,
                                 ready_queue_max=None, tile_cache_cap=None):
        """
        method: set_runtime_cache_limits

        arguments:
          qt_pixmap_cache_mb: limit for Qt Pixmanavigatorche
          ready_queue_max: limit for ready queue
          tile_cache_cap: limit for tile cache

        return:
          none

        description:
          This method passes cache limit settings to the stream controller.
        """

        # delegate to stream
        #
        self.stream.set_runtime_cache_limits(qt_pixmap_cache_mb,
                                             ready_queue_max, tile_cache_cap)
    #
    # end of method

    def install_stream_hooks(self):
        """
        method: install_stream_hooks

        arguments:
          none

        return:
          none

        description:
          This method connects view signals to stream and overlay slots.
        """

        # check if already installed
        #
        if self.stream_hook_installed:
            return

        # check viewer
        #
        if self.image_viewer is None:
            return

        # set flag
        #
        self.stream_hook_installed = True

        # connect signals
        #
        self.image_viewer.viewRectChanged.connect(
            self.stream.on_view_rect_changed
        )
        self.image_viewer.viewRectChanged.connect(
            self.overlays.sync_heatmap_viewport
        )
    #
    # end of method

    def on_resolution_change_requested(self, suggested_pyramid_level):
        """
        method: on_resolution_change_requested

        arguments:
          suggested_pyramid_level: the requested pyramid level

        return:
          none

        description:
          This method delegates resolution changes to the stream controller.
        """

        # delegate to stream
        #
        self.stream.on_resolution_change_requested(suggested_pyramid_level)
    #
    # end of method

    def display_image_at_level(self, pyramid_level):
        """
        method: display_image_at_level

        arguments:
          pyramid_level: the pyramid level to display

        return:
          none

        description:
          This method delegates image display to the stream controller.
        """

        # delegate to stream
        #
        self.stream.display_image_at_level(pyramid_level)
    #
    # end of method

    def set_slide_state(self, full_path, base_w, base_h, thumbnail_pm,
                        level_dims_list, level_downsamples_list, mil_full_res):
        """
        method: set_slide_state

        arguments:
          full_path: full path to the slide
          base_w: base width
          base_h: base height
          thumbnail_pm: thumbnail pixmap
          level_dims_list: list of level dimensions
          level_downsamples_list: list of downsample factors
          mil_full_res: full resolution mil handle

        return:
          none

        description:
          This method updates the session state with new slide information.
        """

        # update session
        #
        self.session.set_slide(
            path=full_path,
            base_w=base_w,
            base_h=base_h,
            thumbnail_pm=thumbnail_pm,
            level_dims=level_dims_list,
            downsamples=level_downsamples_list,
            mil_full_res=mil_full_res
        )
    #
    # end of method

    def set_heatmap_streaming(self, enabled):
        """
        method: set_heatmap_streaming

        arguments:
          enabled: boolean flag

        return:
          none

        description:
          This method enables or disables heatmap streaming via the overlay
          controller.
        """

        # delegate to overlays
        #
        self.overlays.set_heatmap_streaming(enabled)
    #
    # end of method

    def load_heatmap_overlay(self, frame_csv_path):
        """
        method: load_heatmap_overlay

        arguments:
          frame_csv_path: path to the frame-level CSV file

        return:
          none

        description:
          This method loads the heatmap overlay and updates the confidence bar.
        """

        # load overlay
        #
        max_conf = self.overlays.load_heatmap_overlay(
            frame_csv_path,
            self.stream.tile_items_by_level
        )

        # update confidence bar
        #
        self._update_conf_bar(max_conf)
    #
    # end of method

    def load_ref_annotations(self, ref_file):
        """
        method: load_ref_annotations

        arguments:
          ref_file: path to the reference annotation file

        return:
          none

        description:
          This method loads reference annotations via the overlay controller.
        """

        # delegate to overlays
        #
        self.overlays.load_ref_annotations(ref_file)
    #
    # end of method

    def draw_hyp_annotations(self, hyp_file):
        """
        method: draw_hyp_annotations

        arguments:
          hyp_file: path to the hypothesis annotation file

        return:
          none

        description:
          This method loads hypothesis annotations via the overlay controller.
        """

        # delegate to overlays
        #
        self.overlays.draw_hyp_annotations(hyp_file)
    #
    # end of method

    def update_histogram_data(self):
        """
        method: update_histogram_data

        arguments:
          none

        return:
          none

        description:
          This method triggers a histogram update via the overlay controller.
        """

        # delegate to overlays
        #
        self.overlays.update_histogram_data()
    #
    # end of method

    def reset_for_new_image(self):
        """
        method: reset_for_new_image

        arguments:
          none

        return:
          none

        description:
          This method resets the display and controllers for a new image.
        """

        # reset confidence bar
        #
        self._update_conf_bar(0.0)

        # reset overlays
        #
        self.overlays.reset_overlays()

        # reset stream
        #
        self.stream.reset_stream_state()

        # reset session
        #
        self.session.reset()
    #
    # end of method

    def apply_overlay_preferences(self):
        """
        method: apply_overlay_preferences

        arguments:
          none

        return:
          none

        description:
          This method applies overlay preference changes.
        """

        # delegate to overlays
        #
        self.overlays.restyle_all()
    #
    # end of method

    def apply_stream_preferences(self):
        """
        method: apply_stream_preferences

        arguments:
          none

        return:
          none

        description:
          This method applies stream preference changes.
        """

        # delegate to stream
        #
        self.stream.apply_stream_preferences()
    #
    # end of method
#
# end of class

class SlideSession:
    """
    class: SlideSession

    arguments:
      none

    description:
      This class manages the session state for the current slide, including
      metadata, file paths, and the full-resolution MIL handle.
    """

    def __init__(self):
        """
        method: constructor

        arguments:
          none

        return:
          none

        description:
          This method initializes the SlideSession class.
        """

        # reset the session
        #
        self.reset()
    #
    # end of method

    def reset(self):
        """
        method: reset

        arguments:
          none

        return:
          none

        description:
          This method resets the session variables to their default states.
        """

        # reset image location variables
        #
        self.image_location = None
        self.image_location_valid = False

        # reset the frame csv path
        #
        self.frame_csv_path = None

        # reset the thumbnail
        #
        self.thumbnail = QPixmap()

        # reset dimensions
        #
        self.base_width = 0
        self.base_height = 0

        # reset the full resolution handle
        #
        self.mil_full_res = None

        # reset level information
        #
        self.level_dimensions = []
        self.level_downsamples = []
        self.max_level_index = 0
    #
    # end of method

    def set_slide(self, path, base_w, base_h, thumbnail_pm,
                  level_dims, downsamples, mil_full_res):
        """
        method: set_slide

        arguments:
          path: the file path to the slide
          base_w: the base width of the slide
          base_h: the base height of the slide
          thumbnail_pm: the thumbnail pixmap
          level_dims: list of dimensions for each level
          downsamples: list of downsample factors for each level
          mil_full_res: the full resolution MIL handle

        return:
          none

        description:
          This method sets the slide information for the session.
        """

        # set the image location
        #
        self.image_location = str(path)
        self.image_location_valid = True

        # set the dimensions
        #
        self.base_width = int(base_w)
        self.base_height = int(base_h)

        # set the thumbnail
        #
        self.thumbnail =\
            thumbnail_pm if thumbnail_pm is not None else QPixmap()

        # set the level dimensions and downsamples
        #
        self.level_dimensions = list(level_dims or [])
        self.level_downsamples = list(downsamples or [])

        # set the max level index
        #
        self.max_level_index = max(0, len(self.level_dimensions) - 1)

        # set the full resolution handle
        #
        self.mil_full_res = mil_full_res
    #
    # end of method
#
# end of class

class _TileSignals(QObject):
    """
    class: _TileSignals

    arguments:
      none

    description:
      This class defines the signals used for tile streaming communication.
    """

    # define signals
    #
    tile_ready = pyqtSignal(tuple, object, int, int)

#
# end of class

class TileStreamController:
    """
    class: TileStreamController

    arguments:
      parent: the parent widget
      config: the configuration object
      session: the session object
      overlays: the overlay controller

    description:
      This class manages the streaming of image tiles, handling the fetching,
      caching, and displaying of tiles at different pyramid levels.
    """

    def __init__(self, parent, config, session, overlays):
        """
        method: constructor

        arguments:
          parent: the parent widget
          config: the configuration object
          session: the session object
          overlays: the overlay controller

        return:
          none

        description:
          This method initializes the TileStreamController class.
        """

        # set the parent
        #
        self.parent = parent

        # set the configuration
        #
        self.config = config

        # set the session
        #
        self.session = session

        # set the overlays
        #
        self.overlays = overlays

        # initialize viewer references
        #
        self.image_viewer = None
        self.heatmap = None
        self.navigator = None

        # initialize signals
        #
        self._tile_signals = _TileSignals()
        self._tile_signals.tile_ready.connect(
            self._on_tile_ready,
            QtCore.Qt.ConnectionType.QueuedConnection
        )

        # load configuration values
        #
        self.stream_tick_ms = int(self.config.get(CFG_KEY_STREAM_TICK_MS))
        self.submit_budget_default =\
            int(self.config.get(CFG_KEY_TILE_SUBMIT_BUDGET))
        self.ready_queue_max =\
            int(self.config.get(CFG_KEY_CACHE_READY_QUEUE_MAX))
        self.settle_ms = int(self.config.get(CFG_KEY_STREAM_SETTLE_MS))
        self.ui_tiles_per_tick =\
            int(self.config.get(CFG_KEY_STREAM_UI_TILES_PER_TICK))

        # initialize the stream timer
        #
        self.stream_timer = QTimer(self.parent)
        self.stream_timer.setInterval(self.stream_tick_ms)
        self.stream_timer.setSingleShot(True)
        self.stream_timer.setTimerType(QtCore.Qt.TimerType.PreciseTimer)
        self.stream_timer.timeout.connect(self._on_stream_tick)

        # initialize the settle timer
        #
        self.settle_timer = None

        # initialize cache variables
        #
        self.tile_cache_cap = 0
        self.cache_enforcer_timer = None
        self.tile_meta_by_level = {}
        self.loaded_tiles_by_level = {}
        self.tile_items_by_level = {}

        # initialize state variables
        #
        self.tile_clock_main = 0
        self.inflight_tiles = set()
        self.tile_ready_queue = deque()
        self.tile_fail_deadline = {}
        self.interacting = False
        self.more_tiles_pending = False
        self.catchup_deadline_ms = 0
        self.tile_generation = 0

        # initialize executor and mil storage
        #
        self.tile_executor = None
        self._mil_by_thread = {}
        self._mil_by_thread_lock = threading.Lock()

        # initialize tracking variables
        #
        self._last_needed_key = None
        self._last_needed_tiles = []
        self._last_main_prune_ms = 0
        self._last_cap_enforce_ms = 0
    #
    # end of method

    def attach_viewer(self, viewer):
        """
        method: attach_viewer

        arguments:
          viewer: the photo viewer object

        return:
          none

        description:
          This method attaches the photo viewer to the controller.
        """

        # set the viewer
        #
        self.image_viewer = viewer

        # attach viewer to overlays
        #
        self.overlays.attach_viewer(viewer)
    #
    # end of method

    def attach_heatmap(self, heatmap):
        """
        method: attach_heatmap

        arguments:
          heatmap: the heatmap canvas object

        return:
          none

        description:
          This method attaches the heatmap canvas to the controller.
        """

        # set the heatmap
        #
        self.heatmap = heatmap

        # attach heatmap to overlays
        #
        self.overlays.attach_heatmap(heatmap)
    #
    # end of method

    def attach_navigator(self, navigator):
        """
        method: attach_navigator

        arguments:
          navigator: the map navigator object

        return:
          none

        description:
          This method attaches the map navigator to the controller.
        """

        # set the navigator
        #
        self.navigator = navigator

        # attach navigator to overlays
        #
        self.overlays.attach_navigator(navigator)
    #
    # end of method

    def set_runtime_cache_limits(self, qt_pixmap_cache_mb=None,
                                 ready_queue_max=None, tile_cache_cap=None):
        """
        method: set_runtime_cache_limits

        arguments:
          qt_pixmap_cache_mb: limit for QPixmapCache in MB
          ready_queue_max: maximum size of the ready queue
          tile_cache_cap: maximum number of tiles in cache

        return:
          none

        description:
          This method updates the runtime cache limits.
        """

        # update pixmap cache limit
        #
        if qt_pixmap_cache_mb is not None:
            kb_limit = int(max(0, int(qt_pixmap_cache_mb)))\
                * DEF_QPIX_CACHE_KB_PER_MB
            QPixmapCache.setCacheLimit(kb_limit)

        # update ready queue max
        #
        if ready_queue_max is not None:
            self.ready_queue_max = max(1, int(ready_queue_max))
            while len(self.tile_ready_queue) > self.ready_queue_max:
                self.tile_ready_queue.pop()

        # update tile cache capacity
        #
        if tile_cache_cap is not None:
            self.tile_cache_cap = max(0, int(tile_cache_cap))
            if self.tile_cache_cap > 0:
                self._enforce_tile_cache_cap()
                self._start_cache_enforcer_timer()
            else:
                self._stop_cache_enforcer_timer()
    #
    # end of method

    def _start_cache_enforcer_timer(self):
        """
        method: _start_cache_enforcer_timer

        arguments:
          none

        return:
          none

        description:
          This method starts the timer responsible for enforcing cache limits.
        """

        # get the interval
        #
        interval_ms = self.config.get(CFG_KEY_CACHE_ENFORCER_INTERVAL_MS)
        interval_ms = max(50, interval_ms)

        # check if timer exists
        #
        if self.cache_enforcer_timer is None:
            self.cache_enforcer_timer = QTimer(self.parent)
            self.cache_enforcer_timer.setSingleShot(False)
            self.cache_enforcer_timer.setTimerType(
                QtCore.Qt.TimerType.VeryCoarseTimer
            )
            self.cache_enforcer_timer.timeout.connect(
                self._enforce_tile_cache_cap
            )
            self.cache_enforcer_timer.setInterval(interval_ms)
            self.cache_enforcer_timer.start()
            return

        # update interval if changed
        #
        if self.cache_enforcer_timer.interval() != interval_ms:
            self.cache_enforcer_timer.setInterval(interval_ms)

        # start timer if not active
        #
        if not self.cache_enforcer_timer.isActive():
            self.cache_enforcer_timer.start()
    #
    # end of method

    def _stop_cache_enforcer_timer(self):
        """
        method: _stop_cache_enforcer_timer

        arguments:
          none

        return:
          none

        description:
          This method stops the cache enforcer timer.
        """

        # stop the timer
        #
        if self.cache_enforcer_timer is not None:
            self.cache_enforcer_timer.stop()
    #
    # end of method

    def _enforce_tile_cache_cap(self):
        """
        method: _enforce_tile_cache_cap

        arguments:
          none

        return:
          none

        description:
          This method enforces the tile cache capacity
          limit by removing old tiles.
        """

        # get current time
        #
        now_ms = int(time.monotonic() * 1000)

        # check guard time
        #
        if self._last_main_prune_ms and\
           now_ms - self._last_main_prune_ms < DEF_CACHE_ENFORCE_GUARD_MS:
            return

        # get capacity
        #
        cap = int(self.tile_cache_cap)

        # check if enforcement is needed
        #
        if cap <= 0 or not self.tile_meta_by_level:
            return

        # update last enforcement time
        #
        self._last_cap_enforce_ms = now_ms

        # enforce limit per level
        #
        for level, meta in list(self.tile_meta_by_level.items()):
            if not meta or len(meta) <= cap:
                continue

            # identify victims based on LRU
            #
            victims = sorted(((int(v[TILEMETA_LAST_KEY]), k)
                              for k, v in meta.items()), key=lambda t: t[0])
            to_remove = max(0, len(meta) - cap)

            # remove victims
            #
            for idx in range(to_remove):
                self._remove_tile_if_exists(level, victims[idx][1])
    #
    # end of method

    def _ensure_settle_timer(self):
        """
        method: _ensure_settle_timer

        arguments:
          none

        return:
          none

        description:
          This method ensures the settle timer is initialized.
        """

        # check if timer exists
        #
        if self.settle_timer is not None:
            return

        # create the timer
        #
        self.settle_timer = QTimer(self.parent)
        self.settle_timer.setSingleShot(True)
        self.settle_timer.setTimerType(QtCore.Qt.TimerType.PreciseTimer)
        self.settle_timer.timeout.connect(self._on_settle_timeout)
    #
    # end of method

    def _ensure_tile_executor(self):
        """
        method: _ensure_tile_executor

        arguments:
          none

        return:
          none

        description:
          This method ensures the thread pool executor is initialized.
        """

        # check if executor exists
        #
        if self.tile_executor is not None:
            return

        # get max workers
        #
        max_workers = int(self.config.get(CFG_KEY_NUM_TILE_THREADS))

        # create the executor
        #
        self.tile_executor = ThreadPoolExecutor(max_workers=max_workers)
    #
    # end of method

    def _update_base_photo_visibility(self, pyramid_level, force_show=False):
        """
        method: _update_base_photo_visibility

        arguments:
          pyramid_level: current pyramid level
          force_show: flag to force visibility

        return:
          none

        description:
          This method updates the visibility of the base photo based on level.
        """

        # check for viewer
        #
        if self.image_viewer is None:
            return

        # get threshold
        #
        hide_at_or_below = int(self.config.get(CFG_KEY_HIDE_BG_AT_OR_BELOW))

        # check negative threshold
        #
        if hide_at_or_below < 0:
            self.image_viewer.set_base_photo_visible(True)
            return

        # determine visibility
        #
        visible = int(pyramid_level) > hide_at_or_below or bool(force_show)

        # set visibility
        #
        self.image_viewer.set_base_photo_visible(visible)
    #
    # end of method

    def _remove_tile_if_exists(self, level, tile_key):
        """
        method: _remove_tile_if_exists

        arguments:
          level: pyramid level
          tile_key: tile identifier tuple

        return:
          none

        description:
          This method removes a specific tile from all structures.
        """

        # convert level to int
        #
        level = int(level)

        # remove from scene
        #
        if self.image_viewer is not None:
            scene = self.image_viewer.scene()
            items_for_level = self.tile_items_by_level.get(level)
            if items_for_level and tile_key in items_for_level:
                item = items_for_level.pop(tile_key)
                if scene is not None:
                    scene.removeItem(item)

        # remove from loaded set
        #
        loaded_for_level = self.loaded_tiles_by_level.get(level)
        if loaded_for_level:
            loaded_for_level.discard(tile_key)

        # remove from heatmap
        #
        if self.heatmap is not None:
            self.heatmap.remove_tile(level, tile_key)

        # remove from metadata
        #
        meta_for_level = self.tile_meta_by_level.get(level)
        if meta_for_level:
            meta_for_level.pop(tile_key, None)
    #
    # end of method

    def _mark_interaction(self):
        """
        method: _mark_interaction

        arguments:
          none

        return:
          none

        description:
          This method handles user interaction state and timers.
        """

        # ensure timer exists
        #
        self._ensure_settle_timer()

        # check if already interacting
        #
        if self.interacting:
            self.settle_timer.start(int(self.settle_ms))
            return

        # set interacting state
        #
        self.interacting = True

        # increment generation
        #
        self.tile_generation += 1

        # clear inflight and queue
        #
        self.inflight_tiles.clear()
        self.tile_ready_queue.clear()

        # stop stream timer
        #
        if self.stream_timer.isActive():
            self.stream_timer.stop()

        # start settle timer
        #
        self.settle_timer.start(int(self.settle_ms))
    #
    # end of method

    def on_view_rect_changed(self, view_rect_scene):
        """
        method: on_view_rect_changed

        arguments:
          view_rect_scene: the new viewport rectangle in scene coordinates

        return:
          none

        description:
          This method is called when the viewport changes.
        """

        # check for null rect
        #
        if view_rect_scene is None or view_rect_scene.isNull():
            return

        # mark interaction
        #
        self._mark_interaction()
    #
    # end of method

    def _on_settle_timeout(self):
        """
        method: _on_settle_timeout

        arguments:
          none

        return:
          none

        description:
          This method is called when the settle timer expires.
        """

        # clear interacting state
        #
        self.interacting = False

        # set catchup deadline
        #
        ms = int(self.config.get(CFG_KEY_STREAM_FORCE_CATCHUP_MS))
        now_ms = int(time.monotonic() * 1000)
        self.catchup_deadline_ms = now_ms + max(0, int(ms))

        # restart stream timer
        #
        if not self.stream_timer.isActive():
            self.stream_timer.start()
    #
    # end of method

    def _plan_level_bounds(self, level, viewport_scene):
        """
        method: _plan_level_bounds

        arguments:
          level: pyramid level
          viewport_scene: viewport rectangle in scene coordinates

        return:
          tuple: calculated bounds and dimensions

        description:
          This method calculates the tile boundaries for the current view.
        """

        # get level dimensions and downsample
        #
        level_width, level_height = self.session.level_dimensions[level]
        level_downsample = float(self.session.level_downsamples[level])

        # calculate view bounds in level coordinates
        #
        view_x_level = int(math.floor(viewport_scene.x() / level_downsample))
        view_y_level = int(math.floor(viewport_scene.y() / level_downsample))
        view_w_level = int(math.ceil(viewport_scene.width() / level_downsample))
        view_h_level = int(math.ceil(viewport_scene.height() / level_downsample))

        # calculate margins
        #
        margin_scene_px = int(self.config.get(CFG_KEY_VIEWPORT_MARGIN))
        margin_level = int(round(margin_scene_px / level_downsample))
        retain_level = margin_level * 2

        # calculate request bounds
        #
        request_min_x = max(0, view_x_level - margin_level)
        request_min_y = max(0, view_y_level - margin_level)
        request_max_x =\
            min(level_width, view_x_level + view_w_level + margin_level)
        request_max_y =\
            min(level_height, view_y_level + view_h_level + margin_level)

        # calculate retain bounds
        #
        retain_min_x = max(0, view_x_level - retain_level)
        retain_min_y = max(0, view_y_level - retain_level)
        retain_max_x =\
            min(level_width, view_x_level + view_w_level + retain_level)
        retain_max_y =\
            min(level_height, view_y_level + view_h_level + retain_level)

        # calculate view center
        #
        view_center_x = view_x_level + view_w_level / 2.0
        view_center_y = view_y_level + view_h_level / 2.0

        # return results
        #
        return (
            int(level_width), int(level_height), float(level_downsample),
            view_x_level, view_y_level, view_w_level, view_h_level,
            request_min_x, request_min_y, request_max_x, request_max_y,
            retain_min_x, retain_min_y, retain_max_x, retain_max_y,
            view_center_x, view_center_y
        )
    #
    # end of method

    def _compute_needed_tiles_cached(self, level, level_width,
                                     level_height, request_min_x,
                                     request_min_y, request_max_x,
                                     request_max_y, tile_size_level,
                                     view_center_x, view_center_y):
        """
        method: _compute_needed_tiles_cached

        arguments:
          level: pyramid level
          level_width: width of level
          level_height: height of level
          request_min_x: request min x
          request_min_y: request min y
          request_max_x: request max x
          request_max_y: request max y
          tile_size_level: tile size at level
          view_center_x: view center x
          view_center_y: view center y

        return:
          list: list of needed tile rectangles

        description:
          This method computes the list of tiles needed for the current view,
          caching the result to avoid redundant calculations.
        """

        # create cache key
        #
        key = (int(level), int(tile_size_level),
               int(request_min_x), int(request_min_y),
               int(request_max_x), int(request_max_y),
               int(view_center_x), int(view_center_y))

        # check cache
        #
        if self._last_needed_key == key and self._last_needed_tiles:
            return list(self._last_needed_tiles)

        # align start coordinates
        #
        x_start = request_min_x // tile_size_level * tile_size_level
        y_start = request_min_y // tile_size_level * tile_size_level

        # collect needed tiles
        #
        needed_tiles = []
        y_cur = y_start
        while y_cur < request_max_y and y_cur < level_height:
            tile_h = min(tile_size_level, level_height - y_cur)
            x_cur = x_start
            while x_cur < request_max_x and x_cur < level_width:
                tile_w = min(tile_size_level, level_width - x_cur)
                needed_tiles.append((x_cur, y_cur, tile_w, tile_h))
                x_cur += tile_size_level
            y_cur += tile_size_level

        # sort tiles by distance to center
        #
        if len(needed_tiles) > 1:
            needed_tiles.sort(
                key=lambda r: (r[0] + r[2] / 2.0 - view_center_x) ** 2 +
                              (r[1] + r[3] / 2.0 - view_center_y) ** 2
            )

        # update cache
        #
        self._last_needed_key = key
        self._last_needed_tiles = list(needed_tiles)

        # return result
        #
        return needed_tiles
    #
    # end of method

    def _on_stream_tick(self):
        """
        method: _on_stream_tick

        arguments:
          none

        return:
          none

        description:
          This method is the main loop for the tile streaming logic.
        """

        # check if interacting
        #
        if self.interacting:
            return

        # drain ui queue
        #
        self._drain_tile_ui_queue()

        # check for viewer
        #
        if self.image_viewer is None:
            logging.warning('no image_viewer for stream tick')
            return

        # get view rect
        #
        rect = self.image_viewer.mapToScene(
            self.image_viewer.viewport().rect()
        ).boundingRect()

        # stream tiles if rect is valid
        #
        if not rect.isNull():
            pyramid_level = int(self.image_viewer.current_pyramid_level())
            if self.tile_generation == 0:
                self.tile_generation = 1
            self._stream_tiles_for_level(pyramid_level, self.tile_generation)

        # check status
        #
        any_inflight = bool(self.inflight_tiles)
        any_queued = bool(self.tile_ready_queue)
        any_pending = bool(self.more_tiles_pending)
        now_ms = int(time.monotonic() * 1000)
        in_catchup = now_ms < int(self.catchup_deadline_ms)

        # schedule next tick
        #
        if any_queued or any_inflight or any_pending or in_catchup:
            self.stream_timer.start(self.stream_tick_ms)
    #
    # end of method

    def _tile_in_backoff(self, level, tile_key, now_ms):
        """
        method: _tile_in_backoff

        arguments:
          level: pyramid level
          tile_key: tile identifier tuple
          now_ms: current time in milliseconds

        return:
          bool: true if tile is in backoff period

        description:
          This method checks if a tile is currently in a backoff
          period after a failure.
        """

        # create lookup key
        #
        lk = (int(level), tile_key)

        # check deadline
        #
        deadline = self.tile_fail_deadline.get(lk)

        # return result
        #
        return bool(deadline and now_ms < int(deadline))
    #
    # end of method

    def _stream_tiles_for_level(self, level, generation_id):
        """
        method: _stream_tiles_for_level

        arguments:
          level: pyramid level
          generation_id: current tile generation id

        return:
          none

        description:
          This method identifies missing tiles and submits fetch jobs.
        """

        # ensure inputs are int
        #
        level = int(level)

        # check session state
        #
        if not self.session.level_dimensions or\
           not self.session.level_downsamples:
            self.more_tiles_pending = False
            logging.warning('level dims/downs not initialized')
            return

        # check level range
        #
        if level < 0 or level >= len(self.session.level_dimensions):
            self.more_tiles_pending = False
            logging.warning('requested level %d out of range', level)
            return

        # check viewer
        #
        if self.image_viewer is None:
            self.more_tiles_pending = False
            logging.warning('no base photo to stream against')
            return

        # check image location
        #
        if not self.session.image_location_valid or\
           not self.session.image_location:
            self.more_tiles_pending = False
            return

        # get slide path
        #
        slide_path = str(self.session.image_location)

        # ensure executor
        #
        self._ensure_tile_executor()

        # get submit budget
        #
        submit_budget = max(1, int(self.submit_budget_default))

        # get viewport
        #
        viewport_scene = self.image_viewer.mapToScene(
            self.image_viewer.viewport().rect()
        ).boundingRect()
        if viewport_scene.isNull():
            self.more_tiles_pending = False
            return

        # plan bounds
        #
        (level_width, level_height, level_downsample,
         view_x_level, view_y_level, view_w_level, view_h_level,
         request_min_x, request_min_y, request_max_x, request_max_y,
         retain_min_x, retain_min_y, retain_max_x, retain_max_y,
         view_center_x, view_center_y) =\
            self._plan_level_bounds(level, viewport_scene)

        # initialize dictionaries
        #
        self.loaded_tiles_by_level.setdefault(level, set())
        self.tile_items_by_level.setdefault(level, {})
        self.tile_meta_by_level.setdefault(level, {})
        loaded_keys = self.loaded_tiles_by_level[level]
        scene_items = self.tile_items_by_level[level]

        # prune distant tiles
        #
        if loaded_keys:
            scene = self.image_viewer.scene()
            drop_keys = []
            for key in list(loaded_keys):
                tile_x, tile_y, tile_w, tile_h = key
                out_x = tile_x + tile_w < retain_min_x or tile_x > retain_max_x
                out_y = tile_y + tile_h < retain_min_y or tile_y > retain_max_y
                if out_x or out_y:
                    drop_keys.append(key)

            for key in drop_keys:
                if key in scene_items and scene is not None:
                    scene.removeItem(scene_items[key])
                    scene_items.pop(key, None)
                loaded_keys.discard(key)
                if self.heatmap is not None:
                    self.heatmap.remove_tile(level, key)
                self.tile_meta_by_level[level].pop(key, None)

        # calculate tile size
        #
        base_tile_size = int(self.config.get(CFG_KEY_TILE_SIZE))
        tile_size_level =\
            int(np.clip(base_tile_size / max(1.0, level_downsample),
                        DEF_TILE_SIZE_MIN_PX, DEF_TILE_SIZE_MAX_PX))

        # compute needed tiles
        #
        needed_tiles = self._compute_needed_tiles_cached(
            level=level,
            level_width=level_width,
            level_height=level_height,
            request_min_x=request_min_x,
            request_min_y=request_min_y,
            request_max_x=request_max_x,
            request_max_y=request_max_y,
            tile_size_level=tile_size_level,
            view_center_x=view_center_x,
            view_center_y=view_center_y
        )

        # filter candidates
        #
        now_ms = int(time.monotonic() * 1000)
        candidate_tiles = []
        for rect in needed_tiles:
            key = (int(rect[0]), int(rect[1]), int(rect[2]), int(rect[3]))
            if key in loaded_keys:
                continue
            inflight_key = (int(level), *key)
            if inflight_key in self.inflight_tiles:
                continue
            if self._tile_in_backoff(level, key, now_ms):
                continue
            candidate_tiles.append(rect)

        # submit jobs
        #
        submitted = 0
        for tile_rect in candidate_tiles:
            if submit_budget <= 0:
                break
            tile_key = (int(tile_rect[0]), int(tile_rect[1]),
                        int(tile_rect[2]), int(tile_rect[3]))
            self.inflight_tiles.add((int(level), *tile_key))
            fut = self.tile_executor.submit(
                self._worker_read_tile, tile_rect,
                level, generation_id, slide_path
            )
            fut.add_done_callback(self._emit_done)
            submit_budget -= 1
            submitted += 1

        # update pending status
        #
        self.more_tiles_pending = len(candidate_tiles) > submitted

        # sync overlays
        #
        self.overlays.sync_heatmap_viewport()
        if self.navigator is not None:
            self.navigator.update_viewport_from_viewer(self.image_viewer)
    #
    # end of method

    def _emit_done(self, fut):
        """
        method: _emit_done

        arguments:
          fut: the future object

        return:
          none

        description:
          This method processes the completed future and
          emits the result signal.
        """

        # check cancellation
        #
        if fut.cancelled():
            logging.warning('tile future cancelled')
            return

        # get result
        #
        result = fut.result()

        # emit signal
        #
        self._tile_signals.tile_ready.emit(*result)
    #
    # end of method

    def _worker_read_tile(self, tile_rect, level, generation_id, slide_path):
        """
        method: _worker_read_tile

        arguments:
          tile_rect: tile rectangle
          level: pyramid level
          generation_id: generation id
          slide_path: path to slide file

        return:
          tuple: result data

        description:
          This method runs in a worker thread to read and process a tile.
        """

        # unpack tile rect
        #
        tile_x, tile_y, tile_w, tile_h = tile_rect

        # check generation
        #
        if generation_id != int(self.tile_generation):
            return (tile_rect, None, int(level), int(generation_id))

        # get thread id
        #
        tid = threading.get_ident()

        # get or open mil handle
        #
        with self._mil_by_thread_lock:
            per_thread = self._mil_by_thread.get(tid)
            if per_thread is None:
                per_thread = {}
                self._mil_by_thread[tid] = per_thread
            mil = per_thread.get(int(level))
            if mil is None:
                mil = nit.Mil()
                if not mil.open(slide_path):
                    logging.error('Mil.open failed')
                    return (tile_rect, None, int(level), int(generation_id))
                mil.set_level(int(level))
                per_thread[int(level)] = mil

        # read data
        #
        patch = mil.read_data(
            (tile_x, tile_y), npixx=int(tile_w), npixy=int(tile_h)
        )
        if patch is None:
            logging.warning('read_data returned None for tile %s L=%d',
                            str(tile_rect), level)
            return (tile_rect, None, int(level), int(generation_id))

        # convert to qimage
        #
        qimg = _patch_to_qimage(patch)
        if qimg.isNull():
            logging.warning('QImage build failed for tile %s', str(tile_rect))
            return (tile_rect, None, int(level), int(generation_id))

        # draw overlay if needed
        #
        qimg_drawn = qimg
        overlay_img = self.overlays.render_heat_overlay_tile(
            level, (tile_x, tile_y, tile_w, tile_h)
        )
        if overlay_img is not None:
            qimg_drawn = qimg.copy()
            painter = QPainter(qimg_drawn)
            painter.drawImage(0, 0, overlay_img)
            painter.end()

        # return result
        #
        return (tile_rect, qimg_drawn, int(level), int(generation_id))
    #
    # end of method

    def _commit_tile_to_scene(self, tile_rect, qimg_payload,
                              level, generation_id):
        """
        method: _commit_tile_to_scene

        arguments:
          tile_rect: tile rectangle
          qimg_payload: QImage data
          level: pyramid level
          generation_id: generation id

        return:
          none

        description:
          This method adds the processed tile to the scene.
        """

        # check generation
        #
        if generation_id != int(self.tile_generation):
            return

        # unpack tile rect
        #
        tile_x, tile_y, tile_w, tile_h = tile_rect
        tile_key = (int(tile_x), int(tile_y), int(tile_w), int(tile_h))

        # remove from inflight
        #
        self.inflight_tiles.discard((int(level), *tile_key))

        # check payload
        #
        if qimg_payload is None:
            lk = (int(level), tile_key)
            self.tile_fail_deadline[lk] =\
                int(time.monotonic() * 1000) + DEF_TILE_BACKOFF_MAX_MS
            self.more_tiles_pending = True
            logging.warning('tile payload None; backoff %d ms for %s',
                            DEF_TILE_BACKOFF_MAX_MS, str(lk))
            return

        # initialize dictionaries
        #
        self.loaded_tiles_by_level.setdefault(int(level), set())
        self.tile_items_by_level.setdefault(int(level), {})
        self.tile_meta_by_level.setdefault(int(level), {})
        loaded_set = self.loaded_tiles_by_level[int(level)]
        scene_items = self.tile_items_by_level[int(level)]

        # convert to pixmap
        #
        pixmap_main = QPixmap.fromImage(
            qimg_payload, Qt.ImageConversionFlag.NoFormatConversion
        )
        if pixmap_main.isNull():
            logging.error('tile pixmap isNull after conversion')
            return

        # get scene
        #
        scene = self.image_viewer.scene()

        # remove old item
        #
        if tile_key in scene_items:
            old_item = scene_items.pop(tile_key)
            if scene is not None:
                scene.removeItem(old_item)

        # create new item
        #
        tile_item = QGraphicsPixmapItem(pixmap_main)
        tile_item.setTransformationMode(Qt.TransformationMode.FastTransformation)
        tile_item.setCacheMode(QGraphicsItem.CacheMode.DeviceCoordinateCache)

        # set position and scale
        #
        level_downsample = float(self.session.level_downsamples[int(level)])
        tile_item.setPos(
            float(tile_x) * level_downsample, float(tile_y) * level_downsample
        )
        tile_item.setScale(level_downsample)

        # set z value
        #
        z_photo = self.image_viewer.base_photo_z()
        z_base = z_photo + float(self.config.get(CFG_KEY_Z_TILES_OFFSET))
        z_delta = (int(self.session.max_level_index) - int(level))\
            * DEF_Z_TILE_LEVEL_SPACING
        tile_item.setZValue(z_base + z_delta)

        # add to scene
        #
        if scene is not None:
            scene.addItem(tile_item)

        # update tracking
        #
        scene_items[tile_key] = tile_item
        loaded_set.add(tile_key)
        self.tile_clock_main += 1
        self.tile_meta_by_level[int(level)][tile_key] = {
            TILEMETA_ITEM_KEY: tile_item,
            TILEMETA_LAST_KEY: int(self.tile_clock_main)
        }

        # mirror to heatmap
        #
        if self.heatmap is not None:
            self.heatmap.mirror_tile(
                int(level), tile_key, pixmap_main, level_downsample
            )
    #
    # end of method

    def _on_tile_ready(self, tile_rect, qimg_payload, level, generation_id):
        """
        method: _on_tile_ready

        arguments:
          tile_rect: tile rectangle
          qimg_payload: QImage payload
          level: pyramid level
          generation_id: generation id

        return:
          none

        description:
          This method handles the tile ready signal and queues the
          tile for display.
        """

        # check generation
        #
        if generation_id != int(self.tile_generation):
            return

        # check interaction
        #
        if self.interacting:
            self.inflight_tiles.discard(
                (int(level),
                 int(tile_rect[0]), int(tile_rect[1]),
                 int(tile_rect[2]), int(tile_rect[3]))
            )
            return

        # manage queue
        #
        max_q = max(1, int(self.ready_queue_max))
        entry = (tile_rect, qimg_payload, int(level), int(generation_id))
        if len(self.tile_ready_queue) >= max_q:
            self.tile_ready_queue.popleft()
        self.tile_ready_queue.append(entry)

        # ensure timer is running
        #
        if not self.stream_timer.isActive():
            self.stream_timer.start()
    #
    # end of method

    def _drain_tile_ui_queue(self):
        """
        method: _drain_tile_ui_queue

        arguments:
          none

        return:
          none

        description:
          This method processes items in the ready queue
          and commits them to the scene.
        """

        # check status
        #
        if self.interacting or not self.tile_ready_queue:
            return

        # check viewer
        #
        if self.image_viewer is None:
            return

        # disable updates
        #
        self.image_viewer.setUpdatesEnabled(False)

        # process batch
        #
        n = min(int(self.ui_tiles_per_tick), len(self.tile_ready_queue))
        for _ in range(n):
            t_rect, payload, level, gid = self.tile_ready_queue.popleft()
            self._commit_tile_to_scene(t_rect, payload, level, gid)

        # enable updates
        #
        self.image_viewer.setUpdatesEnabled(True)
        self.image_viewer.viewport().update()
    #
    # end of method

    def display_image_at_level(self, pyramid_level):
        """
        method: display_image_at_level

        arguments:
          pyramid_level: the desired pyramid level

        return:
          none

        description:
          This method initiates the display of the image at
          the specified level.
        """

        # check downsamples
        #
        if not self.session.level_downsamples:
            logging.warning('no downsamples to display at level')
            return

        # clamp level
        #
        target_level =\
            max(0, min(int(pyramid_level), int(self.session.max_level_index)))
        # update generation
        #
        self.tile_generation += 1
        generation_id = int(self.tile_generation)

        # initialize dictionaries
        #
        self.loaded_tiles_by_level.setdefault(target_level, set())
        self.tile_items_by_level.setdefault(target_level, {})
        self.inflight_tiles.clear()

        # setup heatmap
        #
        if self.heatmap is not None:
            self.heatmap.enable_stream_tiles(
                self.overlays.heatmap_stream_enabled
            )
            self.heatmap.prune_to_level(target_level)

        # start streaming
        #
        self._stream_tiles_for_level(target_level, generation_id)

        # schedule post display tick
        #
        QTimer.singleShot(
            DEF_POST_DISPLAY_IMAGE_DELAY_MS, self._post_display_image_tick
        )
    #
    # end of method

    def _post_display_image_tick(self):
        """
        method: _post_display_image_tick

        arguments:
          none

        return:
          none

        description:
          This method performs post-display updates for
          NAVIGATOR and heatmap viewports.
        """

        # update navigator viewport
        #
        if self.navigator is not None and self.image_viewer is not None:
            self.navigator.update_viewport_from_viewer(self.image_viewer)

        # sync heatmap viewport
        #
        self.overlays.sync_heatmap_viewport()
    #
    # end of method

    def on_resolution_change_requested(self, suggested_pyramid_level):
        """
        method: on_resolution_change_requested

        arguments:
          suggested_pyramid_level: the requested pyramid level

        return:
          none

        description:
          This method handles resolution change requests from the viewer.
        """

        # check viewer
        #
        if self.image_viewer is None:
            logging.warning('resolution change without viewer')
            return

        # get levels
        #
        prev_level = int(self.image_viewer.current_pyramid_level())
        new_level = int(suggested_pyramid_level)

        # update level
        #
        if prev_level != new_level:
            self.image_viewer.set_current_pyramid_level(new_level)

        # update base photo visibility
        #
        self._update_base_photo_visibility(new_level, force_show=False)

        # reset state
        #
        self.interacting = False
        self.tile_generation += 1
        self.inflight_tiles.clear()

        # purge unused levels
        #
        keep = {int(new_level), int(self.session.max_level_index)}
        self._purge_levels_except(keep_levels=keep)

        # initialize dictionaries
        #
        self.loaded_tiles_by_level.setdefault(int(new_level), set())
        self.tile_items_by_level.setdefault(int(new_level), {})

        # setup heatmap
        #
        if self.heatmap is not None:
            self.heatmap.enable_stream_tiles(
                self.overlays.heatmap_stream_enabled
            )

        # start streaming
        #
        self._stream_tiles_for_level(new_level, self.tile_generation)
        self._resync_heatmap_tiles()
    #
    # end of method

    def _purge_levels_except(self, keep_levels):
        """
        method: _purge_levels_except

        arguments:
          keep_levels: set of levels to keep

        return:
          none

        description:
          This method purges data for levels not in the keep set.
        """

        # get keep set
        #
        keep_set = {int(k) for k in keep_levels}

        # check viewer
        #
        if self.image_viewer is None:
            return

        # get scene
        #
        scene = self.image_viewer.scene()

        # iterate levels
        #
        for level, items in list(self.tile_items_by_level.items()):
            if int(level) in keep_set:
                continue

            # remove items
            #
            for item in list(items.values()):
                if scene is not None:
                    scene.removeItem(item)
            items.clear()

            # clear loaded set
            #
            loaded = self.loaded_tiles_by_level.get(level)
            if loaded:
                loaded.clear()

            # clear metadata
            #
            meta = self.tile_meta_by_level.get(level)
            if meta:
                meta.clear()

        # clear heatmap tiles
        #
        if self.heatmap is not None:
            for level in list(self.tile_items_by_level.keys()):
                if int(level) not in keep_set:
                    self.heatmap.clear_tiles(level=int(level))

        # clear inflight tiles
        #
        if self.inflight_tiles:
            self.inflight_tiles =\
                {t for t in self.inflight_tiles if int(t[0]) in keep_set}
    #
    # end of method

    def _resync_heatmap_tiles(self):
        """
        method: _resync_heatmap_tiles

        arguments:
          none

        return:
          none

        description:
          This method calls the overlay controller to resync heatmap tiles.
        """

        # call overlay method
        #
        self.overlays._resync_heatmap_tiles(self.tile_items_by_level)
    #
    # end of method

    def prune_main_tiles(self):
        """
        method: prune_main_tiles

        arguments:
          none

        return:
          none

        description:
          This method prunes tiles that are out of the viewport.
        """

        # check preconditions
        #
        if self.image_viewer is None or not self.session.level_downsamples or\
           (not self.session.level_dimensions):
            return

        # get current level
        #
        level = int(self.image_viewer.current_pyramid_level())
        if level < 0 or level >= len(self.session.level_downsamples):
            return

        # get scene rect
        #
        rect_scene = self.image_viewer.mapToScene(
            self.image_viewer.viewport().rect()
        ).boundingRect()
        if rect_scene.isNull():
            return

        # get plan bounds
        #
        #
        (_, _, _, _, _, _, _, _, _, _, _, retain_min_x, retain_min_y,
         retain_max_x, retain_max_y, _, _) =\
             self._plan_level_bounds(level, rect_scene)

        # identify tiles to remove
        #
        loaded_set = self.loaded_tiles_by_level.get(level, set())
        max_remove = int(self.config.get(CFG_KEY_TILE_UNLOAD_BUDGET))
        to_remove = []
        for key in list(loaded_set):
            tx, ty, tw, th = key
            out_x = tx + tw < retain_min_x or tx > retain_max_x
            out_y = ty + th < retain_min_y or ty > retain_max_y
            if out_x or out_y:
                to_remove.append(key)

        # remove tiles
        #
        removed = 0
        for key in to_remove:
            if removed >= max_remove:
                break
            self._remove_tile_if_exists(level, key)
            removed += 1

        # update prune time
        #
        now_ms = int(time.monotonic() * 1000)
        self._last_main_prune_ms = now_ms

        # enforce cache cap
        #
        if self.tile_cache_cap > 0:
            if not self._last_cap_enforce_ms or\
               now_ms - self._last_cap_enforce_ms >= DEF_CACHE_ENFORCE_GUARD_MS:
                self._enforce_tile_cache_cap()
    #
    # end of method

    def reset_stream_state(self):
        """
        method: reset_stream_state

        arguments:
          none

        return:
          none

        description:
          This method resets the state of the stream controller.
        """

        # increment generation
        #
        self.tile_generation += 1

        # check viewer
        #
        if self.image_viewer is None:
            logging.warning('no scene during reset')
            self.loaded_tiles_by_level.clear()
            self.tile_items_by_level.clear()
            self.tile_meta_by_level.clear()
            self.tile_clock_main = 0
            self.inflight_tiles.clear()
            self.tile_ready_queue.clear()
            self._mil_by_thread = {}
            self.session.image_location = None
            self.session.image_location_valid = False
            self.session.level_dimensions = []
            self.session.level_downsamples = []
            self.session.max_level_index = 0
            self.interacting = False
            self.more_tiles_pending = False
            self.catchup_deadline_ms = 0
            self.tile_fail_deadline = {}
            self._last_needed_key = None
            self._last_needed_tiles = []
            return

        # get scene
        #
        scene = self.image_viewer.scene()
        if scene is None:
            logging.warning('no scene during reset')
            return

        # clear base photo
        #
        self.image_viewer.clear_base_photo()

        # clear items
        #
        for _level, items in list(self.tile_items_by_level.items()):
            for item in list(items.values()):
                scene.removeItem(item)
            items.clear()
        self.tile_items_by_level.clear()
        self.loaded_tiles_by_level.clear()
        self.tile_meta_by_level.clear()

        # reset state variables
        #
        self.tile_clock_main = 0
        self.inflight_tiles.clear()
        self.tile_ready_queue.clear()

        # clear heatmap
        #
        if self.heatmap is not None:
            self.heatmap.clear_tiles()

        # clear mil threads
        #
        self._mil_by_thread = {}

        # shutdown executor
        #
        if self.tile_executor is not None:
            self.tile_executor.shutdown(wait=False, cancel_futures=True)
            self.tile_executor = None

        # reset session
        #
        self.session.image_location = None
        self.session.image_location_valid = False
        self.session.level_dimensions = []
        self.session.level_downsamples = []
        self.session.max_level_index = 0

        # reset status
        #
        self.interacting = False
        self.more_tiles_pending = False
        self.catchup_deadline_ms = 0
        self.tile_fail_deadline = {}
        self._last_needed_key = None
        self._last_needed_tiles = []

        # update viewport
        #
        self.image_viewer.viewport().update()
    #
    # end of method

    def apply_stream_preferences(self):
        """
        method: apply_stream_preferences

        arguments:
          none

        return:
          none

        description:
          This method applies the current stream configuration settings.
        """

        # load preferences
        #
        self.ui_tiles_per_tick =\
            int(self.config.get(CFG_KEY_STREAM_UI_TILES_PER_TICK))
        self.settle_ms =\
            int(self.config.get(CFG_KEY_STREAM_SETTLE_MS))
        self.stream_tick_ms =\
            int(self.config.get(CFG_KEY_STREAM_TICK_MS))
        self.submit_budget_default =\
            int(self.config.get(CFG_KEY_TILE_SUBMIT_BUDGET))

        # update timer
        #
        if self.stream_timer is not None:
            self.stream_timer.setInterval(int(self.stream_tick_ms))

        # reset executor
        #
        if self.tile_executor is not None:
            self.tile_executor.shutdown(wait=False, cancel_futures=True)
            self.tile_executor = None

        # update generation
        #
        self.tile_generation = int(self.tile_generation) + 1

        # clear queues
        #
        self.inflight_tiles.clear()
        self.tile_ready_queue.clear()
        self.more_tiles_pending = False

        # apply changes
        #
        self.prune_main_tiles()
        self._enforce_tile_cache_cap()

        # tick heatmap
        #
        if self.image_viewer is not None and self.heatmap is not None:
            vis = self.image_viewer.mapToScene(
                self.image_viewer.viewport().rect()
            ).boundingRect()
            self.heatmap.tick_prune_stream_tiles(vis)
    #
    # end of method
#
# end of class

#
# end of file
