#!/usr/bin/env python
#
# file: $NEDC_NFC/class/python/nedc_dpath_demo_tools/nedc_dpath_demo_config.py
#
# revision history:
#
# 20251217 (DH): initial version
#
# This file contains the configuration management for the DPATH demo tool,
# utilizing a Singleton pattern to manage QSettings and application-wide
# constants.
#------------------------------------------------------------------------------

# import system modules
#
import os

# import PyQt modules
#
from PyQt6.QtCore import QSettings

# import NEDC modules
#
import nedc_file_tools as nft

#------------------------------------------------------------------------------
#
# global variables are listed here
#
#------------------------------------------------------------------------------

# set the filename using basename
#
__FILE__ = os.path.basename(__file__)

# define the organization name for QSettings
#
ORG_NAME = 'NEDC'

# define the application key for QSettings
#
APP_KEY = 'PABCC_DEMO'

# define standard annotation labels
#
ANNOTATION_LABELS = (
    'bckg', 'norm', 'nneo', 'indc', 'dcis', 'artf', 'null', 'infl', 'susp'
)

# define default colors for reference annotations
#
DEFAULT_REF_COLORS = {
    'bckg': '#5E5E5E', 'norm': '#00ffff', 'nneo': '#019161', 'indc': '#ff0000',
    'dcis': '#ffff00', 'artf': '#673891', 'null': '#000000', 'infl': '#689111',
    'susp': '#689111'
}

# define default colors for hypothesis annotations
#
DEFAULT_HYP_COLORS = {
    'bckg': '#DEDEDE', 'norm': '#b9fff9', 'nneo': '#02EFA0', 'indc': '#ff8080',
    'dcis': '#ffff7f', 'artf': '#d294ff', 'null': '#A2A2A2', 'infl': '#e1faac',
    'susp': '#e1faac'
}

# define the application stylesheet
#
DEF_STYLE_SHEET = """
#MainWindow {
    background-color: #ececec;
}

#legend_container {
    background-color: white;
    border: 1px solid #8f8f91;
}

QGroupBox {
    background-color: white;
    border: 1px solid #8f8f91;
    min-width: 600px;
}

QHBoxLayout {
    min-height: 32px;
}

QPlainTextEdit {
    border: 1px solid #8f8f91;
}

QMenuBar {
    background-color: #B884A2;
    color: white;
    font-weight: bold;
    spacing: 3px;
}

QMenuBar::item {
    padding: 10px 10px;
}

QMenuBar::item:selected {
    background: #f4f4f4;
    color: black;
    font-weight: bold;
}

QMenu::item {
    min-height: 20px;
    padding: 10px 15px 10px 20px;
    font-size: 16px;
    font-weight: normal;
}

QMenu::item:selected {
    background-color: #f4f4f4;
    color: black;
    border: 1px solid #a8a8a8;
}

QPushButton {
    border: 1px solid #8f8f91;
    background-color: white;
    padding: 10px;
    border-radius: 6px;
}

QPushButton:hover {
    background-color: #f4f4f4;
}

QPushButton:pressed {
    background-color: qlineargradient(
        x1:0, y1:0, x2:0, y2:1,
        stop:0 #dadbde, stop:1 #f6f7fa
    );
}

QLabel {
    background-color: transparent;
}

#file_name_label,
#logging_label,
#legend_label,
#heatmap_label,
#rgb_channels_histogram_label,
#navigator_overlay_label {
    font-size: 14px;
    font-weight: bold;
}

#image_confidence_progress_bar {
    border: 1px solid #888;
    background-color: #eee;
    text-align: center;
}

#image_confidence_progress_bar::chunk {
    background: qlineargradient(
        x1:0, y1:1, x2:0, y2:0,
        stop:0.00 #0000FF,   /* blue    */
        stop:0.33 #00FFFF,   /* cyan    */
        stop:0.66 #FF00FF,   /* magenta */
        stop:1.00 #FF0000    /* red     */
    );
}
""".strip()

# define supported models
#
SUPPORTED_MODELS = ('eb0',)

#------------------------------------------------------------------------------
#
# classes are listed here
#
#------------------------------------------------------------------------------

class Singleton(type):
    """
    class: Singleton

    arguments:
      none

    description:
      A metaclass that implements the Singleton pattern, ensuring that
      only one instance of a class exists.
    """

    # define internal instance dictionary
    #
    _instances = {}

    def __call__(cls, *args, **kwargs):
        """
        method: __call__

        arguments:
          args: positional arguments
          kwargs: keyword arguments

        return:
          instance: the single instance of the class

        description:
          Returns the existing instance or creates a new one.
        """

        # check if instance exists
        #
        if cls not in cls._instances:

            # create and store instance
            #
            cls._instances[cls] = super().__call__(*args, **kwargs)

        # exit gracefully
        # return instance
        #
        return cls._instances[cls]
    #
    # end of method
#
# end of class

class AppConfig(metaclass=Singleton):
    """
    class: AppConfig

    arguments:
      first_run: a boolean flag to determine if default settings are initialized

    description:
      This class manages the application configuration using QSettings.
    """

    def __init__(self, first_run = False):
        """
        method: constructor

        arguments:
          first_run: determine if settings should be initialized

        return: none

        description:
          Initializes the QSettings object and color maps.
        """

        # initialize the settings object
        #
        self.setting = QSettings(ORG_NAME, APP_KEY)

        # initialize reference color map
        #
        self.ref_color_map = None

        # initialize hypothesis color map
        #
        self.hyp_color_map = None

        # check if initialization is required
        #
        if first_run:

            # initialize default values
            #
            self._init_setting_value()

        # create color mapping dictionaries
        #
        self.create_color_map_dict()

        # exit gracefully
        #
        return None
    #
    # end of method

    def set(self, key, value):
        """
        method: set

        arguments:
          key: the setting key
          value: the value to store

        return: none

        description:
          Sets a specific configuration value.
        """

        # set setting value
        #
        self.setting.setValue(key, value)

        # exit gracefully
        #
        return None
    #
    # end of method

    def get(self, key, default=None):
        """
        method: get

        arguments:
          key: the setting key
          default: the default value if key is not found

        return: retrieved value (with basic type coercion)

        description:
          Retrieves a specific configuration value and performs conservative
          type coercion to prevent common QSettings string-return issues
          (e.g., "true"/"false" coming back as str).
        """

        # retrieve the raw value from the settings store (Qt may return str/bool/int/etc.)
        #
        val = self.setting.value(key, defaultValue=default)

        # check for byte-like values (some backends can return bytes)
        #
        if isinstance(val, (bytes, bytearray)):

            # attempt to decode bytes into a UTF-8 string
            #
            try:

                # decode the byte-like value to a string
                #
                val = val.decode('utf-8')

            # handle any decoding failures by returning the raw value unchanged
            #
            except Exception:

                # exit gracefully
                # return the raw value
                #
                return val

        # if the value is a string, attempt conservative type coercion
        #
        if isinstance(val, str):

            # normalize the string for comparisons (strip whitespace and lowercase)
            #
            s = val.strip().lower()

            # convert common truthy strings to boolean True
            #
            if s in ('true', 't', 'yes', 'y', 'on'):

                # exit gracefully
                # return boolean True
                #
                return True

            # convert common falsy strings to boolean False
            #
            if s in ('false', 'f', 'no', 'n', 'off'):

                # exit gracefully
                # return boolean False
                #
                return False

            # if the provided default is a non-bool int, try to coerce numeric strings to int
            #
            if isinstance(default, int) and not isinstance(default, bool):

                # attempt integer conversion of the string value
                #
                try:

                    # convert and return an integer
                    #
                    return int(val)

                # ignore conversion failures and keep the original value
                #
                except Exception:

                    # do nothing (preserve val as-is)
                    #
                    pass

            # if the provided default is a float, try to coerce numeric strings to float
            #
            if isinstance(default, float):

                # attempt float conversion of the string value
                #
                try:

                    # convert and return a float
                    #
                    return float(val)

                # ignore conversion failures and keep the original value
                #
                except Exception:

                    # do nothing (preserve val as-is)
                    #
                    pass

        # exit gracefully
        # return the (possibly coerced) value
        #
        return val
    #
    # end of method

    def sync(self):
        """
        method: sync

        arguments: none

        return: none

        description:
          Syncs current settings to persistent storage.
        """

        # perform the sync
        #
        self.setting.sync()

        # exit gracefully
        #
        return None
    #
    # end of method

    def reset(self):
        """
        method: reset

        arguments: none

        return: none

        description:
          Resets application configuration to factory defaults.
        """

        # initialize default values
        #
        self._init_setting_value()

        # update color maps
        #
        self.create_color_map_dict()

        # exit gracefully
        #
        return None
    #
    # end of method

    def create_color_map_dict(self):
        """
        method: create_color_map_dict

        arguments: none

        return:
          color_maps: a tuple containing ref and hyp color maps

        description:
          Generates dictionaries for label-to-color mapping.
        """

        # generate reference color map
        #
        self.ref_color_map = {
            lbl: self.setting.value(f'AnnotationsColor/{lbl}_ref') \
            for lbl in ANNOTATION_LABELS}

        # generate hypothesis color map
        #
        self.hyp_color_map = {
            lbl: self.setting.value(f'AnnotationsColor/{lbl}_hyp') \
            for lbl in ANNOTATION_LABELS}

        # exit gracefully
        # return mappings
        #
        return (self.ref_color_map, self.hyp_color_map)
    #
    # end of method

    def _init_setting_value(self):
        """
        method: _init_setting_value

        arguments: none

        return: none

        description:
          Initializes default configuration values in the registry.
        """

        # set the app name
        #
        self.setting.setValue('APP_NAME', 'PABCC Demo Tool')

        # set the search directory
        #
        self.setting.setValue('DEFAULT_SEARCH_DIR',
                              nft.get_fullpath('$NEDC_NFC'))

        # set the default width
        #
        self.setting.setValue('WIDTH', 1000)

        # set the default height
        #
        self.setting.setValue('HEIGHT', 1100)

        # set the active model
        #
        self.setting.setValue('ACTIVE_MODEL', 'eb0')

        # begin eb0 model group
        #
        self.setting.beginGroup('Model/eb0')

        # set decoder path
        #
        self.setting.setValue(
            'decoder_executable',
            nft.get_fullpath('$NEDC_NFC/src/decoder/nedc_dpath_eb0_decode.py')
        )

        # set parameter file path
        #
        self.setting.setValue(
            'decoder_params',
            nft.get_fullpath(
                '$NEDC_NFC/docs/params/nedc_dpath_eb0_decode_params_v00.toml'
            )
        )

        # end model group
        #
        self.setting.endGroup()

        # set post proc parameter file
        #
        self.setting.setValue(
            'POSTPROC_PARAM_FILE',
            nft.get_fullpath(
                '$NEDC_NFC/docs/params/nedc_dpath_post_proc_params_v00.toml'
            )
        )

        # set hypothesis opacity
        #
        self.setting.setValue('HYP_OPACITY', 0.275)

        # set reference edge width
        #
        self.setting.setValue('REF_EDGE_WIDTH', 5)

        # set hypothesis edge width
        #
        self.setting.setValue('HYP_EDGE_WIDTH', 1)

        # set confidence threshold
        #
        self.setting.setValue('CONFIDENCE_THRESHOLD', 0.05)

        # set scaling epsilon
        #
        self.setting.setValue('SCALE_EPSILON', 0.01)

        # set heat-on-photo toggle
        #
        self.setting.setValue('SHOW_HEAT_ON_PHOTO', False)

        # set GL viewport toggle
        #
        self.setting.setValue('USE_GL_VIEWPORT', False)

        # set viewport margin
        #
        self.setting.setValue('VIEWPORT_MARGIN', 512)

        # set default tile size
        #
        self.setting.setValue('TILE_SIZE', 1024)

        # set background hide threshold
        #
        self.setting.setValue('HIDE_BG_AT_OR_BELOW', -1)

        # set frame opacity
        #
        self.setting.setValue('FRAME_OPACITY', 0.6)

        # set thumbnail max edge
        #
        self.setting.setValue('THUMB_MAX_EDGE', 1024)

        # begin stream group
        #
        self.setting.beginGroup('Stream')

        # set tiles per tick
        #
        self.setting.setValue('UI_TILES_PER_TICK', 64)

        # set tick ms
        #
        self.setting.setValue('STREAM_TICK_MS', 16)

        # set settle ms
        #
        self.setting.setValue('STREAM_SETTLE_MS', 200)

        # set force catchup ms
        #
        self.setting.setValue('STREAM_FORCE_CATCHUP_MS', 1500)

        # set submit budget
        #
        self.setting.setValue('TILE_SUBMIT_BUDGET', 128)

        # set unload budget
        #
        self.setting.setValue('TILE_UNLOAD_BUDGET', 64)

        # set heat index cell size
        #
        self.setting.setValue('HEAT_INDEX_CELL', 64)

        # set worker threads
        #
        self.setting.setValue('NUM_TILE_THREADS', 4)

        # end stream group
        #
        self.setting.endGroup()

        # begin cache group
        #
        self.setting.beginGroup('Cache')

        # set limits enabled toggle
        #
        self.setting.setValue('LIMITS_ENABLED', False)

        # set pixmap mb
        #
        self.setting.setValue('QT_PIXMAP_MB', 256)

        # set ready queue max
        #
        self.setting.setValue('READY_QUEUE_MAX', 128)

        # set tile cache cap
        #
        self.setting.setValue('TILE_CACHE_CAP', 400)

        # set cache enforcer interval
        #
        self.setting.setValue('ENFORCER_INTERVAL_MS', 250)

        # end cache group
        #
        self.setting.endGroup()

        # initialize labels on list
        #
        labels_on = {'infl', 'dcis', 'nneo', 'indc'}

        # begin visibility ref group
        #
        self.setting.beginGroup('AnnotationsVisibleRef')

        # set initial visibilities for ref
        #
        for lbl in ANNOTATION_LABELS:
            self.setting.setValue(lbl, lbl in labels_on)

        # end visibility ref group
        #
        self.setting.endGroup()

        # begin visibility hyp group
        #
        self.setting.beginGroup('AnnotationsVisibleHyp')

        # set initial visibilities for hyp
        #
        for lbl in ANNOTATION_LABELS:
            self.setting.setValue(lbl, lbl in labels_on)

        # end visibility hyp group
        #
        self.setting.endGroup()

        # begin annotation color group
        #
        self.setting.beginGroup('AnnotationsColor')

        # set ref colors
        #
        for lbl, col in DEFAULT_REF_COLORS.items():
            self.setting.setValue(f'{lbl}_ref', col)

        # set hyp colors
        #
        for lbl, col in DEFAULT_HYP_COLORS.items():
            self.setting.setValue(f'{lbl}_hyp', col)

        # end annotation color group
        #
        self.setting.endGroup()

        # set application style sheet
        #
        self.setting.setValue('STYLE_SHEET', DEF_STYLE_SHEET)

        # begin paths group
        #
        self.setting.beginGroup('Paths')

        # set last open directory
        #
        self.setting.setValue('LAST_OPEN_DIR', self.get('DEFAULT_SEARCH_DIR'))

        # set last save directory
        #
        self.setting.setValue('LAST_SAVE_DIR', nft.get_fullpath('~/'))

        # set export directory
        #
        self.setting.setValue('EXPORT_DIR', nft.get_fullpath('~/exports'))

        # set logging directory
        #
        self.setting.setValue('LOG_DIR', nft.get_fullpath('~/logs'))

        # end paths group
        #
        self.setting.endGroup()

        # begin panel group
        #
        self.setting.beginGroup('UIPanels')

        # set legend visibility
        #
        self.setting.setValue('SHOW_LEGEND', True)

        # set logging visibility
        #
        self.setting.setValue('SHOW_LOGGING', True)

        # set heatmap visibility
        #
        self.setting.setValue('SHOW_HEATMAP', True)

        # set histogram visibility
        #
        self.setting.setValue('SHOW_HISTOGRAM', True)
        self.setting.setValue('SHOW_RGB_HISTOGRAM', True)

        # set NAVIGATOR overlay visibility
        #
        self.setting.setValue('SHOW_NAVIGATOR_OVERLAY', True)

        # end panel group
        #
        self.setting.endGroup()

        # begin general UI group
        #
        self.setting.beginGroup('UI')

        # set splitter handle width
        #
        self.setting.setValue('SPLITTER_HANDLE_W', 6)

        # set init log height
        #
        self.setting.setValue('SPLITTER_INIT_LOG_H', 100)

        # set init side height
        #
        self.setting.setValue('SPLITTER_INIT_SIDE_H', 300)

        # set singleshot delay
        #
        self.setting.setValue('SINGLESHOT_INIT_DELAY_MS', 0)

        # set qtest wait
        #
        self.setting.setValue('QTEST_WAIT0_MS', 0)

        # set pbar width
        #
        self.setting.setValue('PBAR_WIDTH', 30)

        # set swatch height
        #
        self.setting.setValue('SWATCH_MIN_H', 18)

        # set swatch width
        #
        self.setting.setValue('SWATCH_MIN_W', 10)

        # set panel width
        #
        self.setting.setValue('PANEL_MIN_W', 120)

        # end general UI group
        #
        self.setting.endGroup()

        # begin scene group
        #
        self.setting.beginGroup('Scene')

        # set tiles z-offset
        #
        self.setting.setValue('Z_TILES_OFFSET', 1.0)

        # set annotation z-offset
        #
        self.setting.setValue('Z_ANNOT_OFFSET', 10000.0)

        # end scene group
        #
        self.setting.endGroup()

        # begin process group
        #
        self.setting.beginGroup('Process')

        # set process wait time
        #
        self.setting.setValue('WAIT_FOR_STARTED_MS', 5000)

        # end process group
        #
        self.setting.endGroup()

        # begin heatmap group
        #
        self.setting.beginGroup('Heatmap')

        # set color map name
        #
        self.setting.setValue('COLORMAP', 'viridis')

        # set alpha value
        #
        self.setting.setValue('ALPHA', 0.45)

        # set smoothing sigma
        #
        self.setting.setValue('SMOOTHING_SIGMA', 0.0)

        # set stroke width
        #
        self.setting.setValue('HEAT_STROKE_MIN_W', 1)

        # end heatmap group
        #
        self.setting.endGroup()

        # begin histogram group
        #
        self.setting.beginGroup('RGBHistogram')

        # set bin count
        #
        self.setting.setValue('BINS', 256)

        # set normalization toggle
        #
        self.setting.setValue('NORMALIZE', True)

        # end histogram group
        #
        self.setting.endGroup()

        # begin NAVIGATOR group
        #
        self.setting.beginGroup('NAVIGATOROverlay')

        # set alpha value
        #
        self.setting.setValue('ALPHA', 0.35)

        # set display mode
        #
        self.setting.setValue('MODE', 'contours')

        # end NAVIGATOR group
        #
        self.setting.endGroup()

        # begin legend group
        #
        self.setting.beginGroup('Legend')

        # set legend position
        #
        self.setting.setValue('POSITION', 'right')

        # set compact toggle
        #
        self.setting.setValue('COMPACT', False)

        # end legend group
        #
        self.setting.endGroup()

        # begin logging group
        #
        self.setting.beginGroup('Logging')

        # set enablement toggle
        #
        self.setting.setValue('ENABLED', False)

        # set logging level
        #
        self.setting.setValue('LEVEL', 'INFO')

        # set log file path
        #
        self.setting.setValue('FILE', nft.get_fullpath('~/logs/pabcc_demo.log'))

        # end logging group
        #
        self.setting.endGroup()

        # begin window config group
        #
        self.setting.beginGroup('MainWConf')

        # set x position
        #
        self.setting.setValue('MAIN_WIN_POS_X', 100)

        # set y position
        #
        self.setting.setValue('MAIN_WIN_POS_Y', 100)

        # end window config group
        #
        self.setting.endGroup()

        # begin messages group
        #
        self.setting.beginGroup('Messages')

        # set error title
        #
        self.setting.setValue('CONFIG_ERROR_TITLE', 'Configuration Error')

        # set busy title
        #
        self.setting.setValue('BUSY_TITLE', 'Busy')

        # set missing title
        #
        self.setting.setValue('MISSING_INPUT_TITLE', 'Missing input')

        # set body prefix
        #
        self.setting.setValue('MISSING_INPUT_BODY_PREFIX',
                              'Cannot find decoded file:\n\n')

        # set decode fail title
        #
        self.setting.setValue('DECODE_FAILED_TITLE', 'Decode failed')

        # set post proc fail title
        #
        self.setting.setValue('POSTPROC_FAILED_TITLE', 'Post-processing failed')

        # set decode label
        #
        self.setting.setValue('DECODE_LABEL', 'decode')

        # set post proc label
        #
        self.setting.setValue('POSTPROC_LABEL', 'post-processing')

        # set start failure message
        #
        self.setting.setValue('FAILED_COULD_NOT_START', 'Could not start {}.')

        # set missing output message
        #
        self.setting.setValue(
            'FAILED_NO_OUTPUT_FMT',
            '{} reported success but the expected file was not found:\n\n{}'
        )

        # set exit failure message
        #
        self.setting.setValue('FAILED_EXIT_FMT', '{} exited with code {}.')

        # set generic failure message
        #
        self.setting.setValue('GENERIC_FAILED_TITLE_FMT', '{} failed')

        # set about title
        #
        self.setting.setValue('ABOUT_TITLE', 'About')

        # set about body
        #
        self.setting.setValue('ABOUT_BODY', 'PABCC Demo Tool')

        # set help title
        #
        self.setting.setValue('HELP_TITLE', 'Help')

        # set help body
        #
        self.setting.setValue('HELP_BODY',
                              'Help content is not yet available.')

        # set missing model config message
        #
        self.setting.setValue(
            'MISSING_MODEL_CONFIG_FMT',
            "Missing model config for '{}'.\n\n"
            + "Check 'Model/{}/decoder_executable' and\n"
            + "'Model/{}/decoder_params'."
        )

        # set null thumb log
        #
        self.setting.setValue('NULL_THUMB',
                              'HeatManavigatornvas: null thumbnail')

        # end messages group
        #
        self.setting.endGroup()

        # begin labels group
        #
        self.setting.beginGroup('Labels')

        # set file dialog title
        #
        self.setting.setValue('TITLE_OPEN_SVS', 'Open SVS File')

        # set file extension filter
        #
        self.setting.setValue('FILTER_SVS', 'SVS Files (*.svs);;All files (*.*)')

        # set legend title
        #
        self.setting.setValue('LEGEND', 'Legend')

        # set histogram title
        #
        self.setting.setValue('HIST', 'RGB Channels Histogram')

        # set navigator title
        #
        self.setting.setValue('NAVIGATOR', 'Map Navigator')

        # set heatmap title
        #
        self.setting.setValue('HEAT', 'Heatmap')

        # set logging title
        #
        self.setting.setValue('LOGGING', 'Logging')

        # set file prefix
        #
        self.setting.setValue('FILE_PREFIX', 'File: ')

        # set x-axis label
        #
        self.setting.setValue('AXIS_X', 'Pixel Intensity Value')

        # set y-axis label
        #
        self.setting.setValue('AXIS_Y', 'Normalized Freq')

        # set color channel labels
        #
        self.setting.setValue('LINE_R', 'R')
        self.setting.setValue('LINE_G', 'G')
        self.setting.setValue('LINE_B', 'B')

        # set legend headers
        #
        self.setting.setValue('LEGEND_HEADER_REF', 'ref')
        self.setting.setValue('LEGEND_HEADER_HYP', 'hyp')

        # end labels group
        #
        self.setting.endGroup()

        # begin ui objects group
        #
        self.setting.beginGroup('UIObjects')

        # set object names for styling
        #
        self.setting.setValue('OBJ_MAIN_WIN', 'MainWindow')
        self.setting.setValue('OBJ_FILE_NAME_GROUP', 'file_name_group')
        self.setting.setValue('OBJ_FILE_NAME_LABEL', 'file_name_label')
        self.setting.setValue('OBJ_LEGEND_CONTAINER', 'legend_container')
        self.setting.setValue('OBJ_HIST_LABEL', 'rgb_channels_histogram_label')
        self.setting.setValue('OBJ_NAVIGATOR_LABEL', 'navigator_overlay_label')
        self.setting.setValue('OBJ_HEAT_LABEL', 'heatmap_label')
        self.setting.setValue('OBJ_LOG_LABEL', 'logging_label')
        self.setting.setValue('OBJ_CONF_PBAR', 'image_confidence_progress_bar')

        # end ui objects group
        #
        self.setting.endGroup()

        # begin styles group
        #
        self.setting.beginGroup('Styles')

        # set grid style
        #
        self.setting.setValue('LEGEND_GRID', 'QLabel { font-size: 12pt; }')

        # set swatch template
        #
        self.setting.setValue('SWATCH_TEMPLATE', 'background:{};')

        # set progress bar template
        #
        self.setting.setValue(
            'CONF_PBAR_TEMPLATE',
            '#image_confidence_progress_bar {\n'
            + '  border: 1px solid #888;\n'
            + '  background-color: #eee;\n'
            + '  text-align: center;\n'
            + '}\n'
            + '#image_confidence_progress_bar::chunk {\n'
            + '  background: qlineargradient(\n'
            + '    x1:0, y1:1, x2:0, y2:0,\n'
            + '    stop:0 #0000FF,\n'
            + '    stop:1 {TOP_COLOUR}\n'
            + '  );\n'
            + '}\n'
        )

        # set color token
        #
        self.setting.setValue('CONF_PBAR_TOKEN_TOP', '{TOP_COLOUR}')

        # set border css
        #
        self.setting.setValue('LOGGER_BORDER_CSS', 'border: 1px solid #8f8f91;')

        # end styles group
        #
        self.setting.endGroup()

        # begin file extensions group
        #
        self.setting.beginGroup('FileExt')

        # set extension strings
        #
        self.setting.setValue('EXT_CSVF', 'csvf')
        self.setting.setValue('EXT_CSVH', 'csvh')
        self.setting.setValue('EXT_CSV', nft.DEF_EXT_CSV)

        # set post proc program name
        #
        self.setting.setValue('POSTPROC_PROG', nft.get_fullpath('$NEDC_NFC/src/nedc_dpath_post_proc.py'))

        # end file extensions group
        #
        self.setting.endGroup()

        # begin formatting group
        #
        self.setting.beginGroup('Formatting')

        # set logger format
        #
        self.setting.setValue('LOGGER_FORMAT',
                              '%(asctime)s - %(levelname)s - %(message)s')

        # set progress bar format
        #
        self.setting.setValue('PBAR_FORMAT_PERCENT', '%p%')

        # end formatting group
        #
        self.setting.endGroup()

        # begin color components group
        #
        self.setting.beginGroup('Colors')

        # set pen colors
        #
        self.setting.setValue('NAVIGATOR_RECT_PEN_R', 220)
        self.setting.setValue('NAVIGATOR_RECT_PEN_G', 40)
        self.setting.setValue('NAVIGATOR_RECT_PEN_B', 40)

        # set brush colors
        #
        self.setting.setValue('NAVIGATOR_RECT_BRUSH_R', 220)
        self.setting.setValue('NAVIGATOR_RECT_BRUSH_G', 40)
        self.setting.setValue('NAVIGATOR_RECT_BRUSH_B', 40)
        self.setting.setValue('NAVIGATOR_RECT_BRUSH_A', 80)

        # set background colors
        #
        self.setting.setValue('BG_WHITE_R', 255)
        self.setting.setValue('BG_WHITE_G', 255)
        self.setting.setValue('BG_WHITE_B', 255)

        # set plot line colors
        #
        self.setting.setValue('HIST_COLOR_RED', 'red')
        self.setting.setValue('HIST_COLOR_GREEN', 'green')
        self.setting.setValue('HIST_COLOR_BLUE', 'blue')

        # end colors group
        #
        self.setting.endGroup()

        # begin dimensions group
        #
        self.setting.beginGroup('Dimensions')

        # set initial dimensions
        #
        self.setting.setValue('INIT_VIEWPORT_W', 800)
        self.setting.setValue('INIT_VIEWPORT_H', 1000)

        # set figure dimensions
        #
        self.setting.setValue('HIST_FIG_WIDTH', 4)
        self.setting.setValue('HIST_FIG_HEIGHT', 6)
        self.setting.setValue('HIST_FIG_DPI', 100)

        # set legend margins
        #
        self.setting.setValue('LEGEND_MARGIN_LR', 6)
        self.setting.setValue('LEGEND_MARGIN_TB', 2)
        self.setting.setValue('LEGEND_GRID_SPACING', 2)

        # set logging min height
        #
        self.setting.setValue('LOGGER_MIN_HEIGHT_PX', 50)

        # set pen width
        #
        self.setting.setValue('NAVIGATOR_RECT_PEN_WIDTH', 3)

        # end dimensions group
        #
        self.setting.endGroup()

        # begin behavior group
        #
        self.setting.beginGroup('Behavior')

        # set zoom factors
        #
        self.setting.setValue('DEF_SCALE_FACTOR', 1.25)
        self.setting.setValue('SCALE_EPS_DEFAULT', 1e-06)
        self.setting.setValue('SCALE_REL_EPSILON', 1e-09)

        # set confidence offsets
        #
        self.setting.setValue('CONF_OFFSET_INDC', 0.75)
        self.setting.setValue('CONF_OFFSET_DCIS', 0.5)

        # set confidence factors
        #
        self.setting.setValue('CONF_DIVISOR', 4.0)
        self.setting.setValue('CONF_SCALE_LOW', 0.5)

        # set timer intervals
        #
        self.setting.setValue('FOLLOW_TIMER_MS', 16)
        self.setting.setValue('PRUNE_SCHEDULE_DELAY_MS', 16)
        self.setting.setValue('MOUSE_DRAG_PRUNE_DELAY_MS', 32)

        # set cache timings
        #
        self.setting.setValue('CACHE_ENFORCER_DEFAULT_INTERVAL_MS', 250)
        self.setting.setValue('CACHE_ENFORCE_GUARD_MS', 100)

        # set progress bar timings
        #
        self.setting.setValue('JOB_PBAR_STEP_INTERVAL_MS', 150)
        self.setting.setValue('JOB_PBAR_COMPLETE_HOLD_MS', 1000)

        # set backoff and spacing
        #
        self.setting.setValue('TILE_BACKOFF_MAX_MS', 2000)
        self.setting.setValue('Z_TILE_LEVEL_SPACING', 0.01)

        # set delay for display
        #
        self.setting.setValue('POST_DISPLAY_IMAGE_DELAY_MS', 300)

        # set wheel behavior
        #
        self.setting.setValue('WHEEL_DEGREES_PER_STEP', 120)
        self.setting.setValue('DIM_DENOM_FLOOR', 1.0)

        # set histogram logic
        #
        self.setting.setValue('HIST_NORMALIZE_EPS', 1e-12)
        self.setting.setValue('HIST_LEGEND_LOC', 'upper left')

        # end behavior group
        #
        self.setting.endGroup()

        # begin rendering group
        #
        self.setting.beginGroup('Rendering')

        # set tile limits
        #
        self.setting.setValue('TILE_SIZE_MIN_PX', 512)
        self.setting.setValue('TILE_SIZE_MAX_PX', 1536)

        # set LRU limit
        #
        self.setting.setValue('LRU_LIMIT_DEFAULT', 400)

        # set band and edge limits
        #
        self.setting.setValue('GUARD_BAND_SCENE_PX_DEFAULT', 256.0)
        self.setting.setValue('MIN_VIEWPORT_EDGE_PX', 6.0)

        # set tolerance and z-values
        #
        self.setting.setValue('FOLLOW_TOLERANCE_FRACTION', 0.002)
        self.setting.setValue('OVERLAY_Z_VALUE', 10.0)
        self.setting.setValue('TILE_Z_VALUE', 0.1)
        self.setting.setValue('NAVIGATOR_RECT_Z', 10.0)

        # set HSV limits
        #
        self.setting.setValue('HSV_SAT_MAX', 255)
        self.setting.setValue('HSV_VAL_MAX', 255)
        self.setting.setValue('HUE_RANGE_BLUE_TO_RED_MAX', 240)
        self.setting.setValue('ALPHA_SCALE_255', 255)

        # set histogram rendering params
        #
        self.setting.setValue('HIST_NUM_BINS', 51)
        self.setting.setValue('HIST_X_MIN', 0)
        self.setting.setValue('HIST_X_MAX', 255)
        self.setting.setValue('HIST_SMOOTH_SIGMA', 2)

        # set stroke dimensions
        #
        self.setting.setValue('OVERLAY_DEFAULT_STROKE_PX', 20)
        self.setting.setValue('OVERLAY_MIN_STROKE_PX', 1)

        # set hsv colors for confidence
        #
        self.setting.setValue('CONF_HSV_S', 255)
        self.setting.setValue('CONF_HSV_V', 255)

        # set hue limits
        #
        self.setting.setValue('CONF_HUE_MAX', 240)
        self.setting.setValue('HEAT_HUE_MAX', 240)

        # set stages for progress bar
        #
        self.setting.setValue('OPEN_PBAR_STAGE1', 25)
        self.setting.setValue('OPEN_PBAR_STAGE2', 50)
        self.setting.setValue('OPEN_PBAR_STAGE3', 75)
        self.setting.setValue('OPEN_PBAR_STAGE4', 100)

        # end rendering group
        #
        self.setting.endGroup()

        # begin menu string group
        #
        self.setting.beginGroup('MenuStrings')

        # set titles
        #
        self.setting.setValue('TITLE_PREFERENCES', 'Preferences')
        self.setting.setValue('TITLE_EDIT_MENU', 'Edit')
        self.setting.setValue('TITLE_FILE_MENU', 'File')
        self.setting.setValue('TITLE_VIEW_MENU', 'View')
        self.setting.setValue('TITLE_PROCESS_MENU', 'Process')
        self.setting.setValue('TITLE_PABCC_MENU', 'PABCC')
        self.setting.setValue('TITLE_HELP_MENU', 'Help')

        # end menu string group
        #
        self.setting.endGroup()

        # begin menu action group
        #
        self.setting.beginGroup('MenuActions')

        # set action texts
        #
        self.setting.setValue('ACTION_OPEN', 'Open...')
        self.setting.setValue('ACTION_EXIT', 'Exit')
        self.setting.setValue('ACTION_PREFERENCES', 'Preferences')
        self.setting.setValue('ACTION_SAVE_SETTINGS', 'Save Settings')
        self.setting.setValue('ACTION_RESET_SETTINGS', 'Reset Settings')
        self.setting.setValue('ACTION_TOGGLE_LEGEND', 'Toggle Legend')
        self.setting.setValue('ACTION_TOGGLE_HISTOGRAM', 'Toggle Histogram')
        self.setting.setValue('ACTION_TOGGLE_NAVIGATOR',
                              'Toggle Navigator Overlay')
        self.setting.setValue('ACTION_TOGGLE_HEATMAP', 'Toggle Heatmap')
        self.setting.setValue('ACTION_TOGGLE_LOGGING', 'Toggle Logging')
        self.setting.setValue('ACTION_DECODE', 'Decode')
        self.setting.setValue('ACTION_POSTPROCESS', 'Post-process')
        self.setting.setValue('ACTION_ABOUT', 'About')
        self.setting.setValue('ACTION_HELP', 'View Help')

        # end menu action group
        #
        self.setting.endGroup()

        # begin menu tab group
        #
        self.setting.beginGroup('MenuTabs')

        # set tab labels
        #
        self.setting.setValue('TAB_GENERAL', 'General')
        self.setting.setValue('TAB_PROCESS', 'Process')
        self.setting.setValue('TAB_COLORS', 'Colors')
        self.setting.setValue('TAB_CACHE', 'Cache')

        # end menu tab group
        #
        self.setting.endGroup()

        # begin menu button group
        #
        self.setting.beginGroup('MenuButtons')

        # set button texts
        #
        self.setting.setValue('BTN_RESET', 'Reset to Defaults')
        self.setting.setValue('BTN_OK', 'OK')
        self.setting.setValue('BTN_CANCEL', 'Cancel')
        self.setting.setValue('BTN_BROWSE', '...')

        # end menu button group
        #
        self.setting.endGroup()

        # begin menu label group
        #
        self.setting.beginGroup('MenuLabels')

        # set detailed labels
        #
        self.setting.setValue('LBL_DEFAULT_FOLDER', 'Default folder:')
        self.setting.setValue('LBL_REF_EDGE', 'Ref edge width (px):')
        self.setting.setValue('LBL_HYP_EDGE', 'Hyp edge width (px):')
        self.setting.setValue('LBL_HYP_OPACITY', 'Hyp/heatmap opacity:')
        self.setting.setValue('LBL_TILES_PER_TICK', 'Tiles per tick:')
        self.setting.setValue('LBL_STREAM_SETTLE_MS', 'Stream settle (ms):')
        self.setting.setValue('LBL_TILE_LOAD_BUDGET', 'Tile load budget:')
        self.setting.setValue('LBL_TILE_UNLOAD_BUDGET', 'Tile unload budget:')
        self.setting.setValue('LBL_HEAT_INDEX_CELL', 'Heat index cell:')
        self.setting.setValue('LBL_TILE_WORKERS', 'Tile worker threads:')
        self.setting.setValue('LBL_ACTIVE_MODEL', 'Active Model:')
        self.setting.setValue('LBL_DECODER_EXEC', 'Decoder executable:')
        self.setting.setValue('LBL_DECODER_PFILE', 'Decoder parameter file:')
        self.setting.setValue('LBL_POSTPROC_PFILE',
                              'Post-processor parameter file:')
        self.setting.setValue('LBL_CONF_THRESHOLD',
                              'Hyp confidence threshold:')
        self.setting.setValue('LBL_CACHE_ENABLE', 'Enable cache limits')
        self.setting.setValue('LBL_QT_PIXMAP_CACHE', 'Qt pixmap cache:')
        self.setting.setValue('LBL_READY_QUEUE_MAX', 'Tile ready-queue max:')
        self.setting.setValue('LBL_TILE_CACHE_CAP',
                              'Tile cache cap (per level):')
        self.setting.setValue('LBL_COLOR_VIS_SHOW', 'show')

        # set suffixes
        #
        self.setting.setValue('SUFFIX_MB', ' MB')
        self.setting.setValue('SUFFIX_TILES', ' tiles')

        # end menu label group
        #
        self.setting.endGroup()

        # begin menu dialog group
        #
        self.setting.beginGroup('MenuDialogs')

        # set titles
        #
        self.setting.setValue('TITLE_DECODER_PFILE',
                              'Choose decoder parameter file')
        self.setting.setValue('TITLE_POSTPROC_PFILE',
                              'Choose post-processor parameter file')

        # set errors
        #
        self.setting.setValue('ERR_INVALID_PATH_TITLE', 'Invalid path')
        self.setting.setValue('ERR_INVALID_FILE_TITLE', 'Invalid file')
        self.setting.setValue('ERR_DIR_DOES_NOT_EXIST',
                              'The default folder does not exist.')
        self.setting.setValue('ERR_POSTPROC_PFILE_MISSING',
                              'Post-processor parameter file does not exist.')

        # end menu dialog group
        #
        self.setting.endGroup()

        # set filter path
        #
        self.setting.setValue('FileExt/FILTER_TOML',
                              'TOML files (*.toml);;All files (*.*)')

        # begin constraint group
        #
        self.setting.beginGroup('WidgetConstraints')

        # set edge constraints
        #
        self.setting.setValue('EDGE_WIDTH_MIN', 1)
        self.setting.setValue('EDGE_WIDTH_MAX', 200)

        # set opacity constraints
        #
        self.setting.setValue('OPACITY_MIN', 0.05)
        self.setting.setValue('OPACITY_MAX', 1.0)
        self.setting.setValue('OPACITY_STEP', 0.05)
        self.setting.setValue('OPACITY_DECIMALS', 2)

        # set spin constraints
        #
        self.setting.setValue('INT_SPIN_MIN', 1)
        self.setting.setValue('INT_SPIN_MAX_LARGE', 100000)

        # set settle constraints
        #
        self.setting.setValue('SETTLE_MS_MIN', 0)
        self.setting.setValue('SETTLE_MS_MAX', 100000)
        self.setting.setValue('SETTLE_MS_STEP', 50)

        # set cache mb constraints
        #
        self.setting.setValue('CACHE_PIXMAP_MB_MIN', 0)
        self.setting.setValue('CACHE_PIXMAP_MB_MAX', 4096)
        self.setting.setValue('CACHE_PIXMAP_MB_STEP', 16)

        # set ready queue constraints
        #
        self.setting.setValue('READY_QUEUE_MIN', 1)
        self.setting.setValue('READY_QUEUE_MAX', 4096)
        self.setting.setValue('READY_QUEUE_STEP', 8)

        # set tile cache constraints
        #
        self.setting.setValue('TILE_CACHE_CAP_MIN', 0)
        self.setting.setValue('TILE_CACHE_CAP_MAX', 10000)
        self.setting.setValue('TILE_CACHE_CAP_STEP', 10)

        # end constraint group
        #
        self.setting.endGroup()

        # set default model name
        #
        self.setting.setValue('Models/DEFAULT_MODEL_NAME', 'eb0')

        # exit gracefully
        #
        return None

#
# end of file
