#!/usr/bin/env python
#
# file: $NEDC_NFC/util/python/nedc_imld/v5.0.1/imld.py
#
# revision history:
#
# 20250925 (SA): refactored code to match IMLD standards
#
# This file sets up and runs the IMLD app.
#-------------------------------------------------------------------------------

# import system modules
#
import os
import sys
import threading
import webbrowser
import warnings

# add the backend directory to the Python path
#
sys.path.append(os.path.join(os.path.dirname(__file__), 'app', 'backend'))

# import nedc modules
#
import nedc_debug_tools as ndt

# import IMLD class and app
#
from app import IMLD
from app.extensions.base import app

# suppress only the PytzUsageWarning from APScheduler
#
warnings.filterwarnings("ignore", module="apscheduler.util")

#-------------------------------------------------------------------------------
#
# global variables are listed here
#
#-------------------------------------------------------------------------------

# set the filename using basename
#
__FILE__ = os.path.basename(__file__)

# set default port
#
DEFAULT_PORT = 5000

# read port once for browser launch
#
PORT = int(os.getenv('IMLD_PORT', DEFAULT_PORT))

# set url for browser launch
#
URL = f"http://127.0.0.1:{PORT}"

#-------------------------------------------------------------------------------
#
# global instances are listed here
#
#-------------------------------------------------------------------------------

# declare a global debug object so we can use it in functions
#
dbgl = ndt.Dbgl()

# instance of IMLD class
#
imld = IMLD()

#-------------------------------------------------------------------------------
#
# runtime execution status and flags are listed here
#
#-------------------------------------------------------------------------------

# truth value, whether default web browser was successfully opened
#
browser_opened = webbrowser.open(f"http://127.0.0.1:{PORT}")

#-------------------------------------------------------------------------------
#
# functions are listed here
#
#-------------------------------------------------------------------------------

# function: run_flask
# 
def run_flask(imld_instance):
    """
    function: run_flask

    arguments:
     imld_instance: instance of IMLD app

    returns: 
     a boolean value indicating status
    
    description:
     This function starts the Flask server in a background thread. 
    """

    # run the IMLD application
    #
    imld_instance.run()

    # exit gracefully
    #
    return True

def main(argv):

    # check if IMLD instance was created successfully
    #
    if imld is None:
        print("Error: %s (line: %s) %s: imld object creation failed" %
            (__FILE__, ndt.__LINE__, ndt.__NAME__))
        sys.exit(os.EX_SOFTWARE)
    
    # start Flask app in a background thread
    #
    flask_thread = threading.Thread(target=run_flask, args = (imld,),
                                    daemon=True)
    flask_thread.start()

    # verify the thread started successfully
    #
    if flask_thread.is_alive() is False:
        print("Error: %s (line: %s) %s: failed to start Flask thread" %
              (__FILE__, ndt.__LINE__, ndt.__NAME__))
        sys.exit(os.EX_SOFTWARE)

    # check if browser was opened
    if browser_opened is False:
        print("Error: %s (line: %s) %s: failed to open browser" %
              (__FILE__, ndt.__LINE__, ndt.__NAME__))
        sys.exit(os.EX_SOFTWARE)
        
    # keep the main thread alive while Flask runs
    #
    flask_thread.join()
    
    # check final status of thread after joining
    #
    if flask_thread.is_alive() is False:
        print("Error: %s (line: %s) %s: failed to start Flask thread" %
              (__FILE__, ndt.__LINE__, ndt.__NAME__))
        sys.exit(os.EX_SOFTWARE)

    # exit gracefully
    #
    return True

# begin gracefully
#
if __name__ == '__main__':
    main(sys.argv[0:])
    
#
# end of file
