// file: $(NEDC_NFC)/util/python/nedc_imld/v5.0.1/app/static/components/ToolbarC
//       omponents.js
//
// This component defines a set of custom HTML elements that make up the
// interactive toolbar for a plotting/data visualization application. The
// toolbar includes buttons, checkboxes, dropdowns, popups, and forms to
// control plot parameters, ranges, normalization, data swapping, and file
// operations. All components are implemented using Shadow DOM for
// encapsulation and include styling, event handling, and state management.
// Custom events are dispatched to an EventBus for inter-component
// communication. This component implements the AlgoTool toolbar for selecting
// algorithms, configuring parameters, and training or evaluating them while
// communicating with the server.
//

// Revision History:
//
// 20251010 (SA): refactored code to meet ISIP standards
//

// import Event Bus to handle events
//
import { EventBus } from "./Events.js";

//******************************************************************************
//
// Section 1: General Toolbar/Dropdown Components
//
//******************************************************************************

//------------------------------------------------------------------------------
//
// IMLD Tools Class: Toolbar_Button
//
//------------------------------------------------------------------------------

class Toolbar_Button extends HTMLElement {
  /*
  class: Toolbar_Button 

  description:
   This class represents a custom toolbar button element. It allows for
   creating a button with a label and some basic styling. It also provides
   the functionality of adding an event listener to the button so that
   when clicked, it can dispatch a custom event to the window, typically
   for interacting with other components like a plot. This class extends
   HTMLElement and uses Shadow DOM to encapsulate the styles and structure
   of the button.
  */

  //****************************************************************************
  //
  // public methods: required methods such as constructors
  //
  //****************************************************************************
    
  constructor() {
    /*
    method: Toolbar_Button::constructor

    args: None

    return:
     Toolbar_Button instance

    description:
     This is the constructor for the Toolbar_Button class. It is called
     when a new instance of the class is created. The constructor attaches
     a shadow DOM to the element with the "open" mode, which allows
     styling and structure to be encapsulated within the component.
    */

    // call the parent constructor
    //
    super();

    // create a shadow root for the component
    //
    this.attachShadow({ mode: "open" });
  }
  //
  // end of method

  //****************************************************************************
  //
  // lifecycle methods
  //
  //****************************************************************************
    
  connectedCallback() {
    /*
    method: Toolbar_Button::connectedCallback

    args: None

    return: None

    description:
     This method is called when the Toolbar_Button element is added to the
     DOM. It triggers the rendering of the button and adds a click event
     listener to it. This lifecycle method ensures that the button is
     properly initialized when the component is inserted into the DOM.
    */

    // render the component
    //
    this.render();

    // add click listener
    //  
    this.addClickListener();
  }
  //
  // end of method

  //****************************************************************************
  //
  // rendering methods
  //
  //****************************************************************************
    
  render() {
    /*
    method: Toolbar_Button::render

    args: None

    return: None

    description:
     This method is responsible for rendering the button element in the
     shadow DOM. It defines the button's HTML structure, style, and the
     label that appears on the button. The label is fetched from the
     "label" attribute of the element, with a fallback value of "Button"
     if the attribute is not provided.
    */

    // get label from attribute
    //  
    const label = this.getAttribute("label") || "Button"; 

    // define the HTML structure and CSS styles
    //
    this.shadowRoot.innerHTML = `
      <style>
        .toolbar-button {
          background-color: white;
          color: black;
          font-family: 'Inter', sans-serif;
          font-weight: 100;
          font-size: 1em;
          padding: 5px 30px;
          border: none;
          cursor: pointer;
          min-width: 220px;
          white-space: nowrap;
          text-align: left;
        }

        .toolbar-button:hover {
          background-color: #c9c9c9;
        }

      </style>

      <button class="toolbar-button">${label}</button>
    `;
  }

  //****************************************************************************
  //
  // event handling methods
  //
  //****************************************************************************
    
  addClickListener() {
    /*
    method: Toolbar_Button::addClickListener

    args: None

    return: None

    description:
     This method adds a click event listener to the button. When the
     button is clicked, it dispatches a custom "clearPlot" event with the
     `clear` and `plotId` attributes as event details. This event can be
     used to communicate with other components, such as clearing a plot
     based on the values of these attributes.
    */

    // get the button element from the shadow DOM
    //
    const button = this.shadowRoot.querySelector(".toolbar-button");

    // get the label attribute value for conditional logic
    //
    const clear = this.getAttribute("clear");
    const plotID = this.getAttribute("plotId");

    // add an event listener to handle the button click event
    //
    button.addEventListener("click", () => {

      // send a custom event to the window which the plot component
      // is listening for. the plot component will clear the plot
      // based on the clear attribute.
      //
      EventBus.dispatchEvent(
        new CustomEvent("clearPlot", {
          detail: {
            type: clear,
            plotID: plotID,
          },
        })
      );
    });
  }
  //
  // end of method
}
//
// end of class

//------------------------------------------------------------------------------
//
// IMLD Tools Class: Toolbar_CheckboxButton
//
//------------------------------------------------------------------------------

class Toolbar_CheckboxButton extends HTMLElement {
  /*
  class: Toolbar_CheckboxButton

  description:
   This class represents a checkbox button in a toolbar component. It
   manages the checkbox's checked state and toggles its appearance when
   clicked. The class uses a shadow DOM to encapsulate its styles and
   structure, which includes a button with a checkbox input and
   associated styles for hover and layout. It also listens for clicks
   outside of the button to close the button's state when clicked
   elsewhere on the document.
  */

  //****************************************************************************
  //
  // public methods: required methods such as constructors
  //
  //****************************************************************************
    
  constructor() {
    /*
    method: Toolbar_CheckboxButton::constructor

    args: None

    return:
     Toolbar_CheckboxButton instance

    description:
     This is the constructor for the Toolbar_CheckboxButton class. It is
     called when a new instance of the class is created. The constructor
     attaches a shadow DOM to the element with the "open" mode, which
     allows styling and structure to be encapsulated within the component.
    */

    // call the parent constructor
    //
    super();

    // create a shadow root for the component
    //
    this.attachShadow({ mode: "open" });

    // create a variable to hold initial state of checkbox and if it's open
    //
    this.checked = false;
    this.isOpen = false;
  }
  //
  // end of method

  //****************************************************************************
  //
  // lifecycle methods
  //
  //****************************************************************************
    
  connectedCallback() {
    /*
    method: Toolbar_CheckboxButton::connectedCallback

    args: None

    return: None

    description:
     This lifecycle method is called when the component is inserted into
     the DOM. It triggers the rendering of the component and sets up a
     global click event listener to detect clicks outside of the
     component in order to close the button if needed.
    */

    // render the state
    //  
    this.render();

    // add global click listener      
    //
    document.addEventListener("click", this.handleDocumentClick.bind(this));
  }
  //
  // end of method

  disconnectedCallback() {
    /*
    method: Toolbar_CheckboxButton::disconnectedCallback

    args: None

    return: None

    description:
     This lifecycle method is called when the component is removed from
     the DOM. It cleans up by removing the global click event listener to
     prevent memory leaks and unnecessary event handling after the
     component is removed.
    */

    // remove event listener
    //  
    document.removeEventListener("click", this.handleDocumentClick.bind(this)); 
  }  
  //
  // end of method

  //****************************************************************************
  //
  // rendering methods
  //
  //****************************************************************************
    
  render() {
    /*
    method: Toolbar_CheckboxButton::render

    args: None

    return: None

    description:
     This method is responsible for rendering the component's structure
     inside the shadow DOM. It creates the button with a checkbox and
     applies the relevant styles. It also adds a click event listener to
     toggle the checkbox state and the button's open state when clicked.
    */

    // get label from attribute
    //      
    const label = this.getAttribute("label") || "Button"; 

    // define the HTML structure and CSS styles for the component
    //
    this.shadowRoot.innerHTML = `
      <style>
        .toolbar-checkbox-button {
          background-color: white;
          color: black;
          font-family: 'Inter', sans-serif;
          font-weight: 100;
          font-size: 1em;
          padding: 5px 0; /* Remove left padding, keep top/bottom padding */
          border: none;
          cursor: pointer;
          min-width: 220px;
          white-space: nowrap;
          text-align: left;
          display: flex; /* Use flexbox for alignment */
          align-items: center; /* Center align items vertically */
        }

        .toolbar-checkbox-button:hover {
          background-color: #c9c9c9;
        }

        input[type="checkbox"] {
          margin-right: 7px; /* Space between checkbox and label */
          margin-left: 10px;
        }
      </style>

      <button class="toolbar-checkbox-button" id="checkboxButton">
        <input type="checkbox" id="checkbox" ?checked="${this.checked}" />
        ${label}
      </button>
    `;

    // get button and checkbox elements from shadow DOM
    //
    const button = this.shadowRoot.querySelector("#checkboxButton");
    const checkbox = this.shadowRoot.querySelector("#checkbox");

    // add click event listener to toggle checkbox and update component status
    //  
    button.addEventListener("click", (event) => {
      event.stopPropagation(); // prevent event from bubbling up
      this.checked = !this.checked; // toggle the checked state
      checkbox.checked = this.checked; // update the checkbox state
      this.isOpen = true; // mark the button as open
    });
  }
  //
  // end of method

  //****************************************************************************
  //
  // event handling methods
  //
  //****************************************************************************
    
  handleDocumentClick(event) {
    /*
    method: Toolbar_CheckboxButton::handleDocumentClick

    args:
    event (Event): The click event triggered on the document.

    return: None

    description:
     This method is called whenever a click event occurs on the document.
     It checks if the click happened outside the button, and if so, it
     closes the button's state by setting `isOpen` to false. This ensures
     the button behaves like a dropdown, closing when clicked outside.
    */

    // get button element from shadow DOM
    //  
    const button = this.shadowRoot.querySelector("#checkboxButton");

    // check if the clicked target is outside of the button
    //
    if (this.isOpen && !button.contains(event.target)) {
	  
      this.isOpen = false; // close the button
	
      // optionally, reset checkbox state if needed
      // this.checked = false;
      // this.shadowRoot.querySelector('#checkbox').checked = this.checked;
      // update checkbox state
    }
  }
  //
  // end of method
}
//
// end of class

//------------------------------------------------------------------------------
//
// IMLD Tools Class: Toolbar_DropdownClear
//
//------------------------------------------------------------------------------

class Toolbar_DropdownClear extends HTMLElement {
  /*
  class: Toolbar_DropdownClear

  description:
   This class represents a custom dropdown toolbar button with
   functionality to clear data, results, or everything related to a plot.
   The dropdown menu is displayed when the user hovers over the button.
   The options include clearing data, results, or everything associated
   with the plot. The class utilizes a shadow DOM for encapsulation and
   contains styling for the button and the dropdown menu.
  */

  //****************************************************************************
  //
  // public methods: required methods such as constructors
  //
  //****************************************************************************
    
  constructor() {
    /*
    method: Toolbar_DropdownClear::constructor

    args: None

    return:
      Toolbar_DropdownClear instance

    description:
     The constructor for the Toolbar_DropdownClear class. It initializes
     the custom element and attaches a shadow root to it in "open" mode
     for encapsulation. The constructor is automatically called when a
     new instance of the class is created.
    */

    // call the parent constructor
    //
    super();

    // create a shadow root for the component
    //
    this.attachShadow({ mode: "open" });
  }
  //
  // end of method

  //****************************************************************************
  //
  // lifecycle methods
  //
  //****************************************************************************
    
  connectedCallback() {
    /*
    method: Toolbar_DropdownClear::connectedCallback

    args: None

    return: None

    description:
     This method is invoked when the custom element is added to the
     document's DOM. It triggers the rendering of the toolbar button and
     dropdown menu, and adds hover event listeners for showing and hiding
     the dropdown. This ensures the element is functional when it becomes
     part of the DOM.
    */

    // render the initial state
    //  
    this.render();

    // add event listeners for hover functionality
    //  
    this.addHoverListeners();
  }
  //
  // end of method

  //****************************************************************************
  //
  // rendering methods
  //
  //****************************************************************************
    
  render() {
    /*
    method: Toolbar_DropdownClear::render

    args: None

    return: None

    description:
     This method renders the HTML structure and styles for the dropdown
     button and the associated dropdown menu. The button's label and
     plotId are fetched from the attributes, and the shadow DOM is
     populated with the appropriate styles and elements. The dropdown
     menu contains three options: Clear Data, Clear Results, and Clear
     All.
    */

    // get label and plotId from attribute
    //  
    const label = this.getAttribute("label") || "Button"; 
    const plotId = this.getAttribute("plotId");

    // define the HTML structure and CSS styles
    //
    this.shadowRoot.innerHTML = `
      <style>

        .toolbar-item {
          position: relative;
          display: inline-block;
        }

        .toolbar-button {
          background-color: white;
          color: black;
          font-family: 'Inter', sans-serif;
          font-weight: 100;
          font-size: 1em;
          padding: 5px 30px;
          border: none;
          cursor: pointer;
          min-width: 220px;
          white-space: nowrap;
          text-align: left;
          position: relative; /* Needed for absolute positioning of dropdown */
        }

        /* Add the triangle using ::after pseudo-element */
        .toolbar-button::after {
          content: ''; /* Empty content for triangle */
          position: absolute;
          right: 10px; /* Distance from the right edge */
          top: 50%;
          transform: translateY(-50%); /* Vertically center the triangle */
          border-width: 5px;
          border-style: solid;
          border-color: transparent transparent transparent black; /* Creates a
                        right-pointing triangle */
        }

        .toolbar-button:hover,
        .toolbar-button.active {
          background-color: #c9c9c9; /* Highlight color */
        }

        /* Dropdown menu styling */
        .dropdown-menu {
          display: none; /* Initially hidden */
          position: absolute;
          top: 0; /* Aligns with the top of the button */
          left: calc(100% + 0.7px); /* Positions to the right of the button */
          background-color: white;
          z-index: 1000; /* Ensure it's on top */
          min-width: 150px; /* Match button width */
          border: 1px solid #ccc;
        }

        .dropdown-menu.show {
          display: block; /* Show when needed */
        }

        .dropdown-item {
          background-color: white;
          color: black;
          font-family: 'Inter', sans-serif;
          font-weight: 100;
          font-size: 1em;
          padding: 5px 20px;
          border: none;
          cursor: pointer;
          min-width: 180px;
          white-space: nowrap;
          text-align: left;
        }

        .dropdown-item:hover {
          background-color: #c9c9c9; /* Hover effect for dropdown items */
        }
      </style>
        
      <div class="toolbar-item">
        <button class="toolbar-button">${label}</button>
        <div class="dropdown-menu" id="dropdown-menu">
          <toolbar-button label="Clear Data" clear="data"
          plotId=${plotId}></toolbar-button>
          <toolbar-button label="Clear Results" clear="results"
          plotId=${plotId}></toolbar-button>
          <toolbar-button label="Clear All" clear="all"
          plotId=${plotId}></toolbar-button>
        </div>
      </div>
    `;
  }
  //
  // end of method

  //****************************************************************************
  //
  // event handling methods
  //
  //****************************************************************************
    
  addHoverListeners() {
    /*
    method: Toolbar_DropdownClear::addHoverListeners

    args: None

    return: None

    description:
     This method adds event listeners for mouse hover interactions. When
     the user hovers over the toolbar button, the dropdown menu is
     displayed, and the button is highlighted. Conversely, when the user
     stops hovering over the button or dropdown menu, the dropdown menu
     is hidden, and the button's highlight is removed. This creates an
     interactive hover effect for the dropdown.
    */

    // get references to toolbar button and its dropdown menu
    //  
    const button = this.shadowRoot.querySelector(".toolbar-button");
    const dropdownMenu = this.shadowRoot.getElementById("dropdown-menu");

    // show the dropdown on hover
    //
    button.addEventListener("mouseenter", () => {
      dropdownMenu.classList.add("show");
      button.classList.add("active"); // add active class to highlight button
    });

    // hide the dropdown when not hovering over both the button and dropdown
    //
    button.addEventListener("mouseleave", () => {
      if (!dropdownMenu.matches(":hover")) {
        dropdownMenu.classList.remove("show");
        button.classList.remove("active"); // remove active class when hiding
      }
    });

    // show dropdown and highlight button when mouse enters menu
    //  
    dropdownMenu.addEventListener("mouseenter", () => {
      dropdownMenu.classList.add("show"); // keep dropdown open
      button.classList.add("active"); // keep button highlighted
    });

    dropdownMenu.addEventListener("mouseleave", () => {

      // hide when not hovering over dropdown
      //	
      dropdownMenu.classList.remove("show"); 

      // remove highlight when leaving dropdown
      //	
      button.classList.remove("active"); 
    });
  }
  //
  // end of method
}
//
// end of class

//------------------------------------------------------------------------------
//
// IMLD Tools Class: Toolbar_DropdownSettings
//
//------------------------------------------------------------------------------

class Toolbar_DropdownSettings extends HTMLElement {
  /*
  class: Toolbar_DropdownSettings

  description:
   This class defines a custom toolbar dropdown element. It contains a
   button that, when hovered over, reveals a dropdown menu with different
   toolbar items (e.g., "Set Ranges" and "Set Gaussian"). The dropdown
   menu is shown or hidden based on the user's interaction with the
   button or the dropdown. It is intended to be used in a toolbar
   interface, providing users with quick access to various settings.
  */

  //***************************************************************************
  //
  // public methods: required methods such as constructors
  //
  //****************************************************************************
    
  constructor() {
    /*
    method: Toolbar_DropdownSettings::constructor

    args: None

    return:
      Toolbar_DropdownSettings instance

    description:
     This is the constructor for the Toolbar_DropdownSettings class. It
     initializes the component by attaching a shadow DOM with the "open"
     mode, which encapsulates the styles and structure of the
     component.
    */

    // call parent constructor
    //  
    super();

    // create shadow root for constructor
    //  
    this.attachShadow({ mode: "open" });
  }
  //
  // end of method

  //****************************************************************************
  //
  // lifecycle methods
  //
  //****************************************************************************
    
  connectedCallback() {
    /*
    method: Toolbar_DropdownSettings::connectedCallback

    args: None

    return: None

    description:
     This method is invoked when the element is connected to the DOM. It
     triggers the rendering of the toolbar dropdown and adds hover event
     listeners to manage the display of the dropdown menu.
    */

    // render the components
    //  
    this.render();

    // add event listeners for hover functionality
    //  
    this.addHoverListeners();
  }
  //
  // end of method

  //****************************************************************************
  //
  // rendering and state update methods
  //
  //****************************************************************************
   
  render() {
    /*
    method: Toolbar_DropdownSettings::render

    args: None

    return: None

    description:
     This method is responsible for rendering the HTML structure and
     styles of the toolbar dropdown component. It creates the button for
     the toolbar and the dropdown menu, applying the appropriate styles
     and structure to the shadow DOM. The label of the button is set from
     the element's `label` attribute or defaults to "Button".
    */

    // get label from attribute
    //       
    const label = this.getAttribute("label") || "Button"; 

    // define the HTML structure and CSS styles
    //
    this.shadowRoot.innerHTML = `
      <style>

        .toolbar-item {
          position: relative;
          display: inline-block;
        }

        .toolbar-button {
          background-color: white;
          color: black;
          font-family: 'Inter', sans-serif;
          font-weight: 100;
          font-size: 1em;
          padding: 5px 30px;
          border: none;
          cursor: pointer;
          min-width: 220px;
          white-space: nowrap;
          text-align: left;
          position: relative; /* Needed for absolute positioning of dropdown */
        }

        /* Add the triangle using ::after pseudo-element */
        .toolbar-button::after {
          content: ''; /* Empty content for triangle */
          position: absolute;
          right: 10px; /* Distance from the right edge */
          top: 50%;
          transform: translateY(-50%); /* Vertically center the triangle */
          border-width: 5px;
          border-style: solid;
          border-color: transparent transparent transparent black; /* Creates a
                        right-pointing triangle */
        }

        .toolbar-button:hover,
        .toolbar-button.active {
          background-color: #c9c9c9; /* Highlight color */
        }

        /* Dropdown menu styling */
        .dropdown-menu {
          display: none; /* Initially hidden */
          position: absolute;
          top: 0px; /* Aligns with the top of the button */
          left: calc(100% + 0.7px); /* Positions to the right of the button */
          background-color: white;
          z-index: 1000; /* Ensure it's on top */
          min-width: 150px; /* Match button width */
          border: 1px solid #ccc;
        }

        .dropdown-menu.show {
          display: block; /* Show when needed */
        }

        .dropdown-item {
          background-color: white;
          color: black;
          font-family: 'Inter', sans-serif;
          font-weight: 100;
          font-size: 1em;
          padding: 5px 30px;
          border: none;
          cursor: pointer;
          min-width: 180px;
          white-space: nowrap;
          text-align: left;
        }

        .dropdown-item:hover {
          background-color: #c9c9c9; /* Hover effect for dropdown items */
        }
      </style>

      <div class="toolbar-item">
        <button class="toolbar-button">${label}</button>
        <div class="dropdown-menu" id="dropdown-menu">
          <toolbar-set-ranges label="Set Ranges"></toolbar-set-ranges>
          <toolbar-set-gaussian label="Set Gaussian"></toolbar-set-gaussian>
          <toolbar-normalize label="Normalize Data"></toolbar-normalize>
        </div>
      </div>
    `;
  }
  //
  // end of method

  setNormalize(status) {
    /*
    method: Toolbar_DropdownSettings::setNormalize

    args:
      status (boolean): The status of the normalize button.

    return: None

    description:
     This method sets the normalize button's state based on the provided
     status. It updates the button's appearance and functionality
     accordingly.
    */

    // set state of normalize button, updates it accordingly
    //  
    this.shadowRoot.querySelector("toolbar-normalize").setNormalize(status);
      
  }

  //***************************************************************************
  //
  // event handling methods
  //
  //****************************************************************************
    
  addHoverListeners() {
    /*
    method: Toolbar_DropdownSettings::addHoverListeners

    args: None

    return: None

    description:
     This method adds event listeners for mouse hover interactions. It
     listens for mouseenter and mouseleave events on both the toolbar
     button and the dropdown menu to manage the visibility of the
     dropdown menu and the active state of the toolbar button. When
     hovering over the button, the dropdown menu is shown, and when
     the mouse leaves the button or the dropdown (if no popups are open),
     the dropdown menu is hidden.
    */

    // gets button and dropdown menu from shadowDOM
    //  
    const button = this.shadowRoot.querySelector(".toolbar-button");
    const dropdownMenu = this.shadowRoot.getElementById("dropdown-menu");

    // show the dropdown on hover
    //
    button.addEventListener("mouseenter", () => {
      dropdownMenu.classList.add("show");
      button.classList.add("active"); // add active class to highlight button
    });

    // hide the dropdown when not hovering over both the button and
    // dropdown
    //
    button.addEventListener("mouseleave", () => {

      // check if any popup inside the dropdown is open
      //
      const openPopups = dropdownMenu.querySelectorAll("toolbar-popup-button");

      // check if any of the popups is open
      //
      const isAnyPopupOpen = Array.from(openPopups).some(
        (popup) => popup.isPopupOpen
      );

      // hides dropdown and removes highlight if mouse not over & no popup open
      //	
      if (!dropdownMenu.matches(":hover") && !isAnyPopupOpen) {
        dropdownMenu.classList.remove("show");
        button.classList.remove("active"); // remove active class when hiding
      }
    });

    // shows dropdown and keeps highlight if mouse is over it
    //  
    dropdownMenu.addEventListener("mouseenter", () => {
      dropdownMenu.classList.add("show"); // keep dropdown open
      button.classList.add("active"); // keep button highlighted
    });

    // hide dropdown, remove button higlight when mouse leaves menu
    //      
    dropdownMenu.addEventListener("mouseleave", () => {
	  
      // check if any popup inside the dropdown is open
      //
      const openPopups = dropdownMenu.querySelectorAll(
        "toolbar-set-ranges, toolbar-set-gaussian"
      );

      // check if any of the popups is openc
      //
      const isAnyPopupOpen = Array.from(openPopups).some(
        (popup) => popup.isPopupOpen
      );

	if (!isAnyPopupOpen) {

	// hide when not hovering over dropdown
	//    
        dropdownMenu.classList.remove("show"); 

	// remove highlight when leaving dropdown
	//    
        button.classList.remove("active"); 
      }
    });
  }
  //
  // end of method
}
//
// end of class

//-----------------------------------------------------------------------------
//
// IMLD Tools Class: Toolbar_OpenFileButton
//
//-----------------------------------------------------------------------------

class Toolbar_OpenFileButton extends HTMLElement {
  /*
  class: Toolbar_OpenFileButton

  description:
   This class defines a custom web component that represents a button in
   a toolbar for opening a file. The button triggers a hidden file input
   field when clicked, allowing the user to select a file. Based on the
   label of the button, it dispatches custom events to load different
   types of data (e.g., training data, evaluation data, parameters, or
   models) through an event bus.
  */

  //***************************************************************************
  //
  // public methods: required methods such as constructors
  //
  //****************************************************************************
    
  constructor() {
    /*
    method: Toolbar_OpenFileButton::constructor

    args: None

    return:
      Toolbar_OpenFileButton instance

    description:
     This is the constructor for the Toolbar_OpenFileButton class. It
     initializes the shadow DOM for the component, creates a hidden file
     input element, and sets up its attributes, including the input type
     and display properties.
    */

    // call parent constructor
    //  
    super();
      
    this.attachShadow({ mode: "open" }); // creates shadow DOM
    this.fileInput = document.createElement("input"); // input element 
    this.fileInput.type = "file"; // set the input type to file
    this.fileInput.style.display = "none"; // hide the input
  }
  //
  // end of method

  //***************************************************************************
  //
  // lifecycle methods
  //
  //****************************************************************************
    
  connectedCallback() {
    /*
    method: Toolbar_OpenFileButton::connectedCallback

    args: None

    return: None

    description:
     This method is called when the element is inserted into the DOM. It
     renders the button in the shadow DOM, appends the hidden file input,
     and attaches a click listener to the button. It triggers the file
     input when the button is clicked.
    */

    // render the components
    //  
    this.render();

    // append hidden file input to shadow DOM
    //  
    this.shadowRoot.appendChild(this.fileInput);
      
    // add click listener
    //  
    this.addClickListener();
  }
  //
  // end of method

  //****************************************************************************
  //
  // rendering methods
  //
  //****************************************************************************
    
  render() {
    /*
    method: Toolbar_OpenFileButton::render

    args: None

    return: None

    description:
     This method renders the button's HTML structure and styles inside
     the shadow DOM. It sets the button's label based on the value of the
     "label" attribute, or defaults to "Button". The button is styled
     with a simple white background, black text, and hover effects.
    */

    // get label from attribute
    //
    const label = this.getAttribute("label") || "Button"; 

    // define the HTML structure and CSS styles
    //
    this.shadowRoot.innerHTML = `
      <style>

        .toolbar-openfile-button {
          background-color: white;
          color: black;
          font-family: 'Inter', sans-serif;
          font-weight: 100;
          font-size: 1em;
          padding: 5px 30px;
          border: none;
          cursor: pointer;
          min-width: 232px;
          white-space: nowrap;
          text-align: left;
        }

        .toolbar-openfile-button:hover {
          background-color: #c9c9c9;
        }

      </style>

      <button class="toolbar-openfile-button">${label}</button>
    `;
  }
  //
  // end of method

  //****************************************************************************
  //
  // event handling methods
  //
  //****************************************************************************
   
  addClickListener() {
    /*
    method: Toolbar_OpenFileButton::addClickListener

    args: None

    return: None

    description:
     This method adds event listeners to the button and the file input.
     It listens for a click on the button to trigger the file input
     click, and listens for a change event on the file input to handle
     the selection of a file. Based on the button's label, it dispatches
     a corresponding custom event (e.g., "loadData", "loadAlgParams", or
     "loadModel") to the EventBus with the selected file as the event
     detail.
    */

    // get the buttom element from the shadow DOM
    //
    const button = this.shadowRoot.querySelector(".toolbar-openfile-button");

    // get the label attribute value for conditional logic
    //
    const label = this.getAttribute("label");

    // add an event listener to handle the button click event
    //
    button.addEventListener("click", () => {
      this.fileInput.click(); // trigger the file input click
    });

    // add the file input change listener and pass the label explicitly
    //
    this.fileInput.addEventListener("change", (event) => {

      // handle loading training data
      //	
      if (label == "Load Train Data") {
        const formData = new FormData();
        formData.append("file", event.target.files[0]);
        formData.append("plotID", "train");

	// dispath custom 'loadData' event with form data
	//  
        EventBus.dispatchEvent(
          new CustomEvent("loadData", {
            detail: formData,
          })
        );
      }

      // handle loading eval data
      //	
      else if (label == "Load Eval Data") {
        const formData = new FormData();
        formData.append("file", event.target.files[0]);
        formData.append("plotID", "eval");

	// dispath custom "loadData" event with form data
	//  
        EventBus.dispatchEvent(
          new CustomEvent("loadData", {
            detail: formData,
          })
        );
      }

      // handle loading model params
      //	
      else if (label == "Load Model Parameters") {

	// dispatch the loadParameters event to the EventBus
        // the event listener is in Events.js
        //
        EventBus.dispatchEvent(
          new CustomEvent("loadAlgParams", {
            detail: {
              file: event.target.files[0],
            },
          })
        );

        // reset the file input
        //
        event.target.value = "";
      }

      // handle loading model
      //	
      else if (label == "Load Model") {

	// dispatch the loadModel event to the EventBus
        // the event listener is in Events.js
        //
        EventBus.dispatchEvent(
          new CustomEvent("loadModel", {
            detail: {
              file: event.target.files[0],
            },
          })
        );

        // reset the file input
        //
        event.target.value = "";
      }
    });
  }
  //
  // end of method
}
//
// end of class

//-----------------------------------------------------------------------------
//
// IMLD Tools Class: Toolbar_SaveFileButton
//
//-----------------------------------------------------------------------------

class Toolbar_SaveFileButton extends HTMLElement {
  /*
  class: Toolbar_SaveFileButton

  description:
   This class defines a custom toolbar button for saving files. The
   button's label is customizable through the `label` attribute. When
   clicked, the button triggers different events based on its label.
   The class is structured to handle various save actions, such as
   saving training data, evaluation data, algorithm parameters, or
   models. It encapsulates the logic for adding a click event listener
   and dispatching the appropriate event when clicked.
  */

  //****************************************************************************
  //
  // public methods: required methods such as constructors
  //
  //****************************************************************************
    
  constructor() {
    /*
    method: Toolbar_SaveFileButton::constructor

    args: None

    return:
     Toolbar_SaveFileButton instance

    description:
     This is the constructor for the Toolbar_SaveFileButton class. It
     initializes the custom element, attaches a shadow root, and sets up
     the component. The constructor doesn't take any parameters.
    */

    // call parent constructor
    //  
    super();

    // attach shadow DOM
    //  
    this.attachShadow({ mode: "open" });
  }
  //
  // end of method

  //****************************************************************************
  //
  // lifecycle methods
  //
  //****************************************************************************
    
  connectedCallback() {
    /*
    method: Toolbar_SaveFileButton::connectedCallback

    args: None

    return: None

    description:
     This method is invoked when the custom element is added to the DOM.
     It is responsible for rendering the button and adding the click
     event listener to the button. It is automatically called by the
     browser when the element is inserted into the document.
    */

    // render initial state
    //
    this.render();

    // add global click listener
    //
    this.addClickListener();
  }
  //
  // end of method

  //****************************************************************************
  //
  // rendering methods
  //
  //****************************************************************************
    
  render() {
    /*
    method: Toolbar_SaveFileButton::render

    args: None

    return: None

    description:
     This method renders the button element inside the shadow DOM. It
     applies basic styling and inserts the button's label, which can be
     customized via the `label` attribute. If the `label` attribute is
     not provided, it defaults to "Save File".
    */

    // get label from attribute
    //      
    const label = this.getAttribute("label") || "Save File"; 

    // define the HTML structure and CSS styles
    //
    this.shadowRoot.innerHTML = `
      <style>
        .toolbar-openfile-button {
          background-color: white;
          color: black;
          font-family: 'Inter', sans-serif;
          font-weight: 100;
          font-size: 1em;
          padding: 5px 30px;
          border: none;
          cursor: pointer;
          min-width: 232px;
          white-space: nowrap;
          text-align: left;
        }

        .toolbar-openfile-button:hover {
          background-color: #c9c9c9;
        }
      </style>

      <button class="toolbar-openfile-button">${label}</button>
    `;
  }
  //
  // end of method

  //****************************************************************************
  //
  // event handling methods
  //
  //****************************************************************************
    
  addClickListener() {
    /*
    method: Toolbar_SaveFileButton::addClickListener

    args: None

    return: None

    description:
     This method adds a click event listener to the button element. When
     the button is clicked, the method checks the value of the button's
     `label` attribute and dispatches a corresponding event to the
     EventBus for different save actions: "Save Train As...", "Save Eval
     As...", "Save Parameters As...", and "Save Model As...". If the
     label doesn't match any of these cases, no action is taken.
    */

    // get the button element from the shadow DOM
    //
    const button = this.shadowRoot.querySelector(".toolbar-openfile-button");

    // get the label attribute value for conditional logic
    //
    const label = this.getAttribute("label");

    // add an event listener to handle the button click event
    //
    button.addEventListener("click", () => {

      // check the label to determine the action
      //
      switch (label) {

	// save training data
	//  
        case "Save Train Data":
          EventBus.dispatchEvent(
            new CustomEvent("saveData", {
              detail: {
                plotID: "train",
              },
            })
          );
          break;

	// save eval data
	//  
        case "Save Eval Data":
          EventBus.dispatchEvent(
            new CustomEvent("saveData", {
              detail: {
                plotID: "eval",
              },
            })
          );
          break;

	// save algorithm params
	//  
        case "Save Model Parameters":
          EventBus.dispatchEvent(new CustomEvent("saveAlgParams"));
          break;

	// save trained model
	//  
        case "Save Model":
          EventBus.dispatchEvent(new CustomEvent("saveModel"));
          break;

	// do nothing for unrecognized labels
	//  
        default:
          break;
      }
    });
  }
  //
  // end of method
}
//
// end of class

//-----------------------------------------------------------------------------
//
// IMLD Tools Class: Toolbar_Popup Button
//
//-----------------------------------------------------------------------------

class Toolbar_PopupButton extends HTMLElement {
  /*
  class: Toolbar_PopupButton

  description:
   This class defines a custom button element with an attached popup.
   When the button is clicked, a popup appears with content that can be
   closed by clicking the close button. The popup also includes an
   overlay background. The button and popup behavior is controlled using
   JavaScript to toggle visibility and animations.
  */

  //****************************************************************************
  //
  // public methods: required methods such as constructors
  //
  //****************************************************************************
    
  constructor() {
    /*
    method: Toolbar_PopupButton::constructor

    args: None

    return:
     Toolbar_PopupButton instance

    description:
     This is the constructor for the Toolbar_PopupButton class. It is
     called when a new instance of the class is created. The constructor
     initializes the shadow DOM and sets an initial state for the popup
     (closed by default).
    */

    // call parent constructor
    //  
    super();

    // attach shadow DOM
    //  
    this.attachShadow({ mode: "open" });

    // track popup state
    //  
    this.isPopupOpen = false; 
  }
  //
  // end of method

  //****************************************************************************
  //
  // lifecycle methods
  //
  //****************************************************************************
    
  connectedCallback() {
    /*
    method: Toolbar_PopupButton::connectedCallback

    args: None 

    return: None

    description:
     This method is called when the component is added to the DOM. It
     triggers the `render` method to display the button, popup, and
     overlay within the shadow DOM. It also sets up event listeners 
     to manage button and popup interactions (open/close).
    */

    // render the components
    //  
    this.render();
  }
  //
  // end of method

  //****************************************************************************
  //
  // rendering methods
  //
  //****************************************************************************
    
  render() {
    /*
    method: Toolbar_PopupButton::render

    args: None

    return: None

    description:
     This method generates the HTML structure for the button, popup,
     and overlay within the shadow DOM. It includes styling for the
     popup and button, as well as the functionality to open and close
     the popup when the button or close button is clicked.
    */

    // get label from attribute
    //      
    const label = this.getAttribute("label") || "Button"; 

    // define the HTML structure and CSS styles
    //
    this.shadowRoot.innerHTML = `
      <style>
        .toolbar-popup-button {
          background-color: white;
          color: black;
          font-family: 'Inter', sans-serif;
          font-weight: 100;
          font-size: 1em;
          padding: 5px 30px;
          border: none;
          cursor: pointer;
          min-width: 220px;
          white-space: nowrap;
          text-align: left;
        }

        .toolbar-popup-button:hover {
          background-color: #c9c9c9;
        }

        /* Popup styling */
        .popup {
          display: none; /* Initially hidden */
          position: fixed;
          top: 50%;
          left: 50%;
          transform: translate(-50%, -50%) scale(0); /* Start scaled down */
          width: 300px;
          height: 200px; /* Increased height */
          padding: 20px;
          background-color: white;
          border-radius: 15px; /* Rounded corners */
          box-shadow: 0 4px 20px rgba(0, 0, 0, 0.3);
          z-index: 1000; /* Ensure it's on top */
          opacity: 0; /* Start fully transparent */
          transition: opacity 0.1s ease, transform 0.s ease; /* Transition for
                      opening/closing */
        }

        .popup.show {
          display: block; /* Show when needed */
          opacity: 1; /* Fully opaque when shown */
          transform: translate(-50%, -50%) scale(1); /* Scale to original size
                     */
        }

        .popup h2 {
          margin: 0 0 20px 0;
        }

        /* Close button styling */
        .close-btn {
          position: absolute;
          top: 10px;
          right: 10px;
          background: transparent;
          border: none;
          font-size: 16px;
          cursor: pointer;
          color: #333;
        }

        /* Overlay styling */
        .overlay {
          display: none; /* Initially hidden */
          position: fixed;
          top: 0;
          left: 0;
          width: 100%;
          height: 100%;
          background: rgba(0, 0, 0, 0.5); /* Semi-transparent background */
          z-index: 999; /* Ensure it's below the popup */
        }

        .overlay.show {
          display: block; /* Show overlay when needed */
        }
      </style>

      <button class="toolbar-popup-button">${label}</button>
      <div class="overlay" id="overlay"></div>
      <div class="popup" id="popup">
        <button class="close-btn" id="close-btn">X</button>
        <h2>Popup Title</h2>
        <p>This is the popup content!</p>
      </div>
    `;

    // get elements
    //
    const button = this.shadowRoot.querySelector(".toolbar-popup-button");
    const popup = this.shadowRoot.getElementById("popup");
    const overlay = this.shadowRoot.getElementById("overlay");
    const closeBtn = this.shadowRoot.getElementById("close-btn");

    // show the popup when clicking the button
    //
    button.addEventListener("click", (event) => {
      event.stopPropagation();
      this.togglePopup();
    });

    // close the popup when clicking the close button
    //
    closeBtn.addEventListener("click", (event) => {
      event.stopPropagation();
      this.closePopup();
    });

    // stop event propagation on popup to avoid closing when clicking
    // inside it
    //
    popup.addEventListener("click", (event) => {
      event.stopPropagation();
    });
  }
  //
  // end of method

  //****************************************************************************
  //
  // popup management methods
  //
  //****************************************************************************
    
  togglePopup() {
    /*
    method: Toolbar_PopupButton::togglePopup

    args: None

    return: None

    description:
     Toggles the visibility of the Toolbar_PopupButton modal and its
     overlay. If the popup is currently hidden, this method makes it
     visible; otherwise, it closes the popup by calling `closePopup()`.
     It also updates the internal `isPopupOpen` state to reflect the
     current visibility.
    */

    // get references to popup and overlay elements in shadow DOM
    //  
    const popup = this.shadowRoot.getElementById("popup");
    const overlay = this.shadowRoot.getElementById("overlay");

    // toggle popup state
    //  
    this.isPopupOpen = !this.isPopupOpen;

    // if popup open, add "show" class
    //  
    if (this.isPopupOpen) {
      popup.classList.add("show");
      overlay.classList.add("show");
      popup.style.display = "block";
      overlay.style.display = "block";
    }
      
    // else, call closePopup method
    //  
    else {
      this.closePopup();
    }
  }
  //
  // end of method

  closePopup() {
    /*
    method: Toolbar_PopupButton::closePopup

    args: None
    
    return: None

    description:
     Closes the Toolbar_PopupButton modal and overlay by removing the
     visible classes and setting their display to "none" after a short
     delay to allow CSS transitions to complete. Also updates the
     internal `isPopupOpen` flag to indicate that the popup is closed.
    */
  
    // get references to popup and overlay elements
    //      
    const popup = this.shadowRoot.getElementById("popup");
    const overlay = this.shadowRoot.getElementById("overlay");

    // hide popup and overlay visually
    //
    popup.classList.remove("show");
    overlay.classList.remove("show");

    // fully remove elements from layout after short delay
    //      
    setTimeout(() => {
      popup.style.display = "none";
      overlay.style.display = "none";
    }, 100);

    // update internal state to reflect the popup is closed
    //      
    this.isPopupOpen = false;
  }
  //
  // end of method
}
//
// end of class

//******************************************************************************
//
// Section 2: Popup and Form Components
//
//******************************************************************************

//------------------------------------------------------------------------------
//
// IMLD Tools Class: Toolbar_SetGaussian
//
//------------------------------------------------------------------------------

class Toolbar_SetGaussian extends HTMLElement {
  /*
  class: Toolbar_SetGaussian 

  description:
   This class manages the toolbar for setting Gaussian draw parameters.
   It contains functionality for rendering the toolbar, showing and
   hiding a popup form, and handling user input for the Gaussian
   parameters such as covariance and number of points. The class also
   includes styling for the popup, buttons, and form inputs, and
   provides interactivity for preset and submit actions.
  */

  //****************************************************************************
  //
  // public methods: required methods such as constructors
  //
  //****************************************************************************
    
  constructor() {
    /*
    method: Toolbar_SetGaussian::constructor

    args: None

    return:
     Toolbar_SetGaussian instance

    description:
     This is the constructor for the Toolbar_SetGaussian class. It sets
     up the shadow DOM, initializes the `isPopupOpen` flag to track the
     popup state, and attaches the necessary styles and elements for the
     toolbar. It also sets up the rendering of the toolbar and the event
     listeners for interactivity.
    */

    // call parent constructor
    //  
    super();

    // attach shadow DOM
    //  
    this.attachShadow({ mode: "open" });

    // track popup state  
    this.isPopupOpen = false;
  }
  //
  // end of method

  //****************************************************************************
  //
  // lifecycle methods
  //
  //****************************************************************************
    
  connectedCallback() {
    /*
    method: Toolbar_SetGaussian::connectedCallback

    args: None

    return: None

    description:
     This method is invoked when the custom element is attached to the
     DOM. It calls the render method to display the toolbar and adds the
     event listeners to enable interactivity. The method ensures the
     toolbar is ready to interact with the user.
    */

    // render the components
    //  
    this.render();

    // add event listeners for interactivity
    //
    this.addEventListeners();
  }
  //
  // end of method

  //****************************************************************************
  //
  // rendering methods
  //
  //****************************************************************************
    
  render() {
    /*
    method: Toolbar_SetGaussian::render

    args: None

    return: None

    description:
     This method handles the rendering of the toolbar and its associated
     styles and HTML elements. It creates a shadow DOM structure with a
     button to trigger the popup and the corresponding popup with
     buttons for preset and submit actions. It also sets up the dynamic
     form container for setting  Gaussian parameters such as the number
     of points and covariance matrix.
    */

    // define the HTML structure and CSS styles
    //
    this.shadowRoot.innerHTML = `
      <style>

        /* Button styles */
        .toolbar-popup-button {
          background-color: white;
          color: black;
          font-family: 'Inter', sans-serif;
          font-weight: 100;
          font-size: 1em;
          padding: 5px 30px;
          border: none;
          cursor: pointer;
          min-width: 220px;
          white-space: nowrap;
          text-align: left;
        }

        .toolbar-popup-button:hover {
          background-color: #c9c9c9;
        }

        /* Popup styling */
        .popup {
          display: none; /* Initially hidden */
          position: fixed;
          top: 50%;
          left: 50%;
          transform: translate(-50%, -50%) scale(0); /* Start scaled down */
          width: 45vw; /* Set a fixed width */
          max-width: 90%; /* Allow the width to shrink if needed */
          max-height: 80vh; /* Limit the height to 80% of the viewport height */
          padding: 15px;
          padding-top: 10px;
          padding-bottom: 10px;
          background-color: white;
          border-radius: 15px; /* Rounded corners */
          box-shadow: 0 4px 20px rgba(0, 0, 0, 0.3);
          z-index: 1000; /* Ensure it's on top */
          opacity: 0; /* Start fully transparent */
          transition: opacity 0.1s ease, transform 0.2s ease; /* Transition for
                      opening/closing */
          overflow: auto; /* Allow scrolling inside the popup if the content
                    overflows */
        }

        .popup.show {
          display: block; /* Show when needed */
          opacity: 1; /* Fully opaque when shown */
          transform: translate(-50%, -50%) scale(1); /* Scale to original size
                     */
        }

        .popup h2 {
          font-family: 'Inter', sans-serif;
          font-size: 1.2em;
          margin: 0 0 8px 0;
        }

        /* Close button styling */
        .close-btn {
          position: absolute;
          top: 10px;
          right: 10px;
          background: transparent;
          border: none;
          font-size: 16px;
          cursor: pointer;
          color: #333;
        }

        /* Overlay styling */
        .overlay {
          display: none; /* Initially hidden */
          position: fixed;
          top: 0;
          left: 0;
          width: 100%;
          height: 100%;
          background: rgba(0, 0, 0, 0.5); /* Semi-transparent background */
          z-index: 999; /* Ensure it's below the popup */
        }

        .overlay.show {
          display: block; /* Show overlay when needed */
        }

        .button-container {
          display: flex;
          justify-content: space-between;
          gap: 0.5vw;
          width: 100%;
          margin: 1vh 0 0.1vw;
        }

        .button, .reset {
          flex: 1; /* Makes each button take up equal width */
          padding: 0.2vh 0.4vw;
          border-radius: 1vw; /* Makes buttons rounded */
          background-color: #4CAF50; /* Sets button background color */
          color: white;
          border: none;
          cursor: pointer;
          font-family: 'Inter', sans-serif;
          font-size: 1em;
        }

        .button:hover, .reset:hover {
          background-color: #2a732e;
        }

      </style>

      <!-- Button to trigger the popup -->
      <button class="toolbar-popup-button">Set Gaussian</button>
      
      <!-- Background overlay -->
      <div class="overlay" id="overlay"></div>

      <!-- Popup container -->
      <div class="popup" id="popup">
        <button class="close-btn" id="close-btn">X</button>
        <h2>Set Gaussian Draw Parameters</h2>
        <div id="form-div">
          <div class="button-container">
            <button type="button" class="button"
            id="presetButton">Presets</button>
            <button type="submit" class="button"
            id="submitButton">Submit</button>
          </div>
        </div>      
      </div>
    `;

    // get elements within the shadow DOM
    //
    const button = this.shadowRoot.querySelector(".toolbar-popup-button");
    const popup = this.shadowRoot.getElementById("popup");
    const closeBtn = this.shadowRoot.getElementById("close-btn");

    // create a style element
    //
    const style = `
      /* Styling the main container for form inputs */
      .form-container {
        display: flex;
        flex-direction: column;
      }

      /* Styling for individual input containers */
      .num-container {
        border: 2px solid #ccc;
        padding: 0.4vw;
        border-radius: 0.4vw;
        width: 100%;
        margin: 0.4vh 0.15vw 0.1vw;
        box-sizing: border-box;
      }

      /* Label styling for input fields */
      .num-container label {
        padding-left: 0.5vw;
        font-family: 'Inter', sans-serif;
        font-size: 0.9em;
        font-weight: bold;
        margin-bottom: 0.3vw;
        display: block;
      }

      /* Grid layout for input fields */
      .num-input {
        display: grid;
        gap: 0.5vw;
      }

      /* Input field styling */
      input {
        padding: 0.4vw;
        border: 1px solid #ccc;
        border-radius: 0.4vw;
        font-size: 0.75em;
        box-sizing: border-box;
        width: 100%;
      }

      /* Input field focus state */
      input:focus {
        border-color: #7441BA;
        border-width: 2px;
        outline: none;
      }
    `;

    // create a dynamic form container for the distribution key
    //
    this.form = new FormContainer(
      {
        name: "Gaussian Draw Parameters",
        params: {
          numPoints: {
            name: "Size of Gaussian Mass",
            type: "int",
            range: [0, 100],
            default: 15,
          },
          cov: {
            name: "Covariance Matrix",
            type: "matrix",
            dimensions: [2, 2],
            default: [
              [0.025, 0],
              [0, 0.025],
            ],
          },
        },
      },
      style
    );

    // append the form to the popup before the button container
    //
    const formDiv = this.shadowRoot.getElementById("form-div");
    formDiv.insertBefore(this.form, formDiv.firstChild);

    // show the popup when the button is clicked
    //
    button.onclick = (event) => {

      // prevent event propagation to avoid unintended behavior
      //
      event.stopPropagation();

      // call togglePopup method to show/hide popup
      //
      this.togglePopup();
    };

    // close the popup when clicking the close button
    //
    closeBtn.onclick = (event) => {

      // prevent event propagation to avoid conflicts
      //
      event.stopPropagation();

      // call closePopup method to hide popup
      //
      this.closePopup();
    };

    // stop event propagation on popup to avoid closing when clicking
    // inside it
    //
    popup.onclick = (event) => {

      // stop event from bubbling up to parent listeners
      //
      event.stopPropagation(); 
    };
  }
  //
  // end of method

  //****************************************************************************
  //
  // event handling methods
  //
  //****************************************************************************
    
  addEventListeners() {
    /*
    method: Toolbar_SetGaussian::addEventListeners

    args: None

    return: None

    description:
     This method sets up event listeners for the preset and submit
     buttons in the popup. The preset button applies default values to
     the form, while the submit button dispatches the selected Gaussian 
     parameters as a custom event and closes the popup.
    */

    // set up button to clear inputs and apply preset values
    //
    const presetButton = this.shadowRoot.querySelector("#presetButton");
    const submitButton = this.shadowRoot.querySelector("#submitButton");

    // fetch and apply preset values when preset button is clicked
    //
    presetButton.onclick = () => {

      // set the defaults through the form object
      //
      this.form.setDefaults();
    };

    // fetch and apply preset values when preset button is clicked
    //
    submitButton.onclick = () => {

      // set the defaults through the form object
      //
      const [paramsDict, _] = this.form.submitForm();

      // dispath custom 'setGaussianParams' event
      //	
      EventBus.dispatchEvent(
        new CustomEvent("setGaussianParams", {
          detail: paramsDict,
        })
      );

      // close the popup
      //
      this.closePopup();
    };
  }
  //
  // end of method

  //****************************************************************************
  //
  // state update methods
  //
  //****************************************************************************
    
  togglePopup() {
    /*
    method: Toolbar_SetGaussian::togglePopup

    args: None

    return: None

    description:
     Toggles the visibility of the Toolbar_SetGaussian modal and its
     overlay. If the popup is currently hidden, this method makes it
     visible; otherwise, it closes the popup by calling `closePopup()`.
     It also updates the internal `isPopupOpen` state to reflect the
     current visibility.
    */

    // create popup and overlay element
    //
    const popup = this.shadowRoot.getElementById("popup");
    const overlay = this.shadowRoot.getElementById("overlay");

    // toggle popup state
    //
    this.isPopupOpen = !this.isPopupOpen;

    // show popup and overlap and ensure they are both visible
    //  
    if (this.isPopupOpen) {
      popup.classList.add("show");
      overlay.classList.add("show");
      popup.style.display = "block";
      overlay.style.display = "block";
    }

    // close popup if already open
    //  
    else {
      this.closePopup();
    }
  }
  //
  // end of method

  closePopup() {
    /*
    method: Toolbar_SetGaussian::closePopup

    args: None

    return: None

    description:
     Closes the Toolbar_SetGaussian modal and overlay by removing the
     visible classes and setting their display to "none" after a short
     delay to allow CSS transitions to complete. Also updates the
     internal `isPopupOpen` flag to indicate that the popup is closed.
    */

    // create popup and overlay element
    //
    const popup = this.shadowRoot.getElementById("popup");
    const overlay = this.shadowRoot.getElementById("overlay");

    // remove show class from popup and overlay
    //
    popup.classList.remove("show");
    overlay.classList.remove("show");

    // hide popup and overlay after transition ends
    //
    setTimeout(() => {
      popup.style.display = "none";
      overlay.style.display = "none";
    }, 100);

    // set popup state to closed
    //
    this.isPopupOpen = false;
  }
  //
  // end of method
}
//
// end of class

//-----------------------------------------------------------------------------
//
// IMLD Tools Class: Toolbar_SetRanges
//
//-----------------------------------------------------------------------------

class Toolbar_SetRanges extends HTMLElement {
  /*
  class: Toolbar_SetRanges

  description:
   This class is responsible for creating a toolbar with a popup for
   setting the plot ranges. It includes functionality for displaying the
   popup, setting ranges for the X and Y axes, providing preset values,
   and submitting the values. The class interacts with the shadow DOM 
   to provide encapsulated styling and functionality.
  */

  //****************************************************************************
  //
  // public methods: required methods such as constructors
  //
  //****************************************************************************
    
  constructor() {
    /*
    method: Toolbar_SetRanges::constructor

    args: None

    return:
     Toolbar_SetRanges instance

    description:
     This is the constructor for the Toolbar_SetRanges class. It
     initializes the component by attaching a shadow root to the element
     and setting the initial state of the popup (closed). It also
     prepares the state of the popup by setting up the flag to track
     whether the popup is open or not.
    */

    // call parent constructor
    //  
    super();

    // attach shadow DOM
    //  
    this.attachShadow({ mode: "open" });

    // track popup state
    //  
    this.isPopupOpen = false; 
  }
  //
  // end of method

  //****************************************************************************
  //
  // lifecycle methods
  //
  //****************************************************************************
    
  connectedCallback() {
    /*
    method: Toolbar_SetRanges::connectedCallback

    args: None

    return: None

    description:
     This method is called when the component is added to the DOM. It
     renders the toolbar and attaches event listeners for the toolbar
     buttons, such as the button to open the popup and the close button.
     It also appends the form for setting ranges to the popup and adds
     interactivity to handle the preset and submit actions.
    */

    // render the components
    //  
    this.render();

    // add event listeners for interactivity
    //
    this.addEventListeners();
  }
  //
  // end of method

  //****************************************************************************
  //
  // rendering methods
  //
  //****************************************************************************

  render() {
    /*
    method: Toolbar_SetRanges::render

    args: None

    return: None

    description:
     This method generates and inserts the HTML structure for the
     toolbar and popup into the shadow DOM. It includes styles for the
     popup, buttons, and overlay, as well as creating a form container 
     for setting the X and Y axis ranges. This method also sets up the
     visibility of the popup and its interactions, including closing the
     popup when the close button is clicked.
    */

    // define the HTML structure and CSS styles
    //
    this.shadowRoot.innerHTML = `
      <style>

        /* Button styles */
        .toolbar-popup-button {
          background-color: white;
          color: black;
          font-family: 'Inter', sans-serif;
          font-weight: 100;
          font-size: 1em;
          padding: 5px 30px;
          border: none;
          cursor: pointer;
          min-width: 220px;
          white-space: nowrap;
          text-align: left;
        }

        .toolbar-popup-button:hover {
          background-color: #c9c9c9;
        }

        /* Popup styling */
        .popup {
          display: none; /* Initially hidden */
          position: fixed;
          top: 50%;
          left: 50%;
          transform: translate(-50%, -50%) scale(0); /* Start scaled down */
          width: 45vw; /* Set a fixed width */
          max-width: 90%; /* Allow the width to shrink if needed */
          max-height: 80vh; /* Limit the height to 80% of the viewport height */
          padding: 15px;
          padding-top: 10px;
          padding-bottom: 10px;
          background-color: white;
          border-radius: 15px; /* Rounded corners */
          box-shadow: 0 4px 20px rgba(0, 0, 0, 0.3);
          z-index: 1000; /* Ensure it's on top */
          opacity: 0; /* Start fully transparent */
          transition: opacity 0.1s ease, transform 0.2s ease; /* Transition for
          opening/closing */
          overflow: auto; /* Allow scrolling inside the popup if the content
          overflows */
        }

        .popup.show {
          display: block; /* Show when needed */
          opacity: 1; /* Fully opaque when shown */
          transform: translate(-50%, -50%) scale(1); /* Scale to original size
                     */
        }

        .popup h2 {
          font-family: 'Inter', sans-serif;
          font-size: 1.2em;
          margin: 0 0 8px 0;
        }

        /* Close button styling */
        .close-btn {
          position: absolute;
          top: 10px;
          right: 10px;
          background: transparent;
          border: none;
          font-size: 16px;
          cursor: pointer;
          color: #333;
        }

        /* Overlay styling */
        .overlay {
          display: none; /* Initially hidden */
          position: fixed;
          top: 0;
          left: 0;
          width: 100%;
          height: 100%;
          background: rgba(0, 0, 0, 0.5); /* Semi-transparent background */
          z-index: 999; /* Ensure it's below the popup */
        }

        .overlay.show {
          display: block; /* Show overlay when needed */
        }

        .button-container {
          display: flex;
          justify-content: space-between;
          gap: 0.5vw;
          width: 100%;
          margin: 1vh 0 0.1vw;
        }

        .button, .reset {
          flex: 1; /* Makes each button take up equal width */
          padding: 0.2vh 0.4vw;
          border-radius: 1vw; /* Makes buttons rounded */
          background-color: #4CAF50; /* Sets button background color */
          color: white;
          border: none;
          cursor: pointer;
          font-family: 'Inter', sans-serif;
          font-size: 1em;
        }

        .button:hover, .reset:hover {
          background-color: #2a732e;
        }

      </style>

      <!-- Button to trigger the popup -->
      <button class="toolbar-popup-button">Set Ranges</button>
      
      <!-- Background overlay -->
      <div class="overlay" id="overlay"></div>

      <!-- Popup container -->
      <div class="popup" id="popup">
        <button class="close-btn" id="close-btn">X</button>
        <h2>Set Plot Ranges</h2>
        <div id="form-div">
          <div class="button-container">
            <button type="button" class="button"
            id="presetButton">Presets</button>
            <button type="submit" class="button"
            id="submitButton">Submit</button>
          </div>
        </div>
      </div>
    `;

    // get elements within the shadow DOM
    //
    const button = this.shadowRoot.querySelector(".toolbar-popup-button");
    const popup = this.shadowRoot.getElementById("popup");
    const closeBtn = this.shadowRoot.getElementById("close-btn");

    // create a style element
    //
    const style = `
      /* Styling the main container for form inputs */
      .form-container {
        display: flex;
        flex-direction: column;
      }

      /* Styling for individual input containers */
      .num-container {
        border: 2px solid #ccc;
        padding: 0.4vw;
        border-radius: 0.4vw;
        width: 100%;
        margin: 0.4vh 0.15vw 0.1vw;
        box-sizing: border-box;
      }

      /* Label styling for input fields */
      .num-container label {
        padding-left: 0.5vw;
        font-family: 'Inter', sans-serif;
        font-size: 0.9em;
        font-weight: bold;
        margin-bottom: 0.3vw;
        display: block;
      }

      /* Grid layout for input fields */
      .num-input {
        display: grid;
        gap: 0.5vw;
      }

      /* Input field styling */
      input {
        padding: 0.4vw;
        border: 1px solid #ccc;
        border-radius: 0.4vw;
        font-size: 0.75em;
        box-sizing: border-box;
        width: 100%;
      }

      /* Input field focus state */
      input:focus {
        border-color: #7441BA;
        border-width: 2px;
        outline: none;
      }
    `;

    // create a dynamic form container for the distribution key
    //
    this.form = new FormContainer(
      {
        name: "set_ranges",
        params: {
          x: {
            name: "X-axis bounds",
            type: "matrix",
            dimensions: [1, 2],
            default: [[-1, 1]],
          },
          y: {
            name: "Y-axis bounds",
            type: "matrix",
            dimensions: [1, 2],
            default: [[-1, 1]],
          },
        },
      },
      style
    );

    // append the form to the popup before the button container
    //
    const formDiv = this.shadowRoot.getElementById("form-div");
    formDiv.insertBefore(this.form, formDiv.firstChild);

    // show the popup when the button is clicked
    //
    button.onclick = (event) => {

      // prevent event propagation to avoid unintended behavior
      //
      event.stopPropagation();

      // call togglePopup method to show/hide popup
      //
      this.togglePopup();
    };

    // close the popup when clicking the close button
    //
    closeBtn.onclick = (event) => {

      // prevent event propagation to avoid conflicts
      //
      event.stopPropagation();

      // call closePopup method to hide popup
      //
      this.closePopup();
    };

    // stop event propagation on popup to avoid closing when clicking inside it
    //
    popup.onclick = (event) => {

      // stop event from bubbling up to parent listeners
      //	
      event.stopPropagation(); 
    };
  }
  //
  // end of method

  //****************************************************************************
  //
  // event handling methods
  //
  //****************************************************************************
    
  addEventListeners() {
    /*
    method: Toolbar_SetRanges::addEventListeners

    args: None

    return: None

    description:
     This method sets up event listeners for the preset and submit
     buttons. It defines the logic for applying preset values to the
     form and submitting the form data. Upon submission, it dispatches 
     a custom event with the form data and closes the popup.
    */

    // set up button to clear inputs and apply preset values
    //
    const presetButton = this.shadowRoot.querySelector("#presetButton");
    const submitButton = this.shadowRoot.querySelector("#submitButton");

    // fetch and apply preset values when preset button is clicked
    //
    presetButton.onclick = () => {

      // set the defaults through the form object
      //
      this.form.setDefaults();
    };

    // fetch and apply preset values when preset button is clicked
    //
    submitButton.onclick = () => {

      // set the defaults through the form object
      //
      const [paramsDict, _] = this.form.submitForm();

      // dispath custom "setRanges" event
      //	
      EventBus.dispatchEvent(
        new CustomEvent("setRanges", {
          detail: paramsDict,
        })
      );

      // close the popup
      //
      this.closePopup();
    };
  }
  //
  // end of method

  //****************************************************************************
  //
  // state update methods
  //
  //****************************************************************************
   
  togglePopup() {
    /*
    method: Toolbar_SetRages::togglePopup

    args: None

    return: None

    description:
     Toggles the visibility of the Toolbar_SetRages modal and its
     overlay. If the popup is currently hidden, this method makes it
     visible; otherwise, it closes the popup by calling `closePopup()`.
     It also updates the internal `isPopupOpen` state to reflect the
     current visibility.
    */

    // create popup and overlay element
    //
    const popup = this.shadowRoot.getElementById("popup");
    const overlay = this.shadowRoot.getElementById("overlay");

    // toggle popup state
    //
    this.isPopupOpen = !this.isPopupOpen;

    // show popup and overlap and ensure they are both visible
    //
    if (this.isPopupOpen) {
      popup.classList.add("show");
      overlay.classList.add("show");
      popup.style.display = "block";
      overlay.style.display = "block";
    }

    // close popup if already open
    //  
    else {
      this.closePopup();
    }
  }
  //
  // end of method

  closePopup() {
    /*
    method: Toolbar_SetRages::closePopup

    args: None

    return: None

    description:
     Closes the Toolbar_SetRages modal and overlay by removing the
     visible classes and setting their display to "none" after a short
     delay to allow CSS transitions to complete. Also updates the
     internal `isPopupOpen` flag to indicate that the popup is closed.
    */

    // create popup and overlay element
    //
    const popup = this.shadowRoot.getElementById("popup");
    const overlay = this.shadowRoot.getElementById("overlay");

    // remove show class from popup and overlay
    //
    popup.classList.remove("show");
    overlay.classList.remove("show");

    // hide popup and overlay after transition ends
    //
    setTimeout(() => {
      popup.style.display = "none";
      overlay.style.display = "none";
    }, 100);

    // set popup state to closed
    //
    this.isPopupOpen = false;
  }
  //
  // end of method
}
//
// end of class

//-----------------------------------------------------------------------------
//
// IMLD Tools Class: Toolbar_SwapButton
//
//-----------------------------------------------------------------------------

class Toolbar_SwapButton extends HTMLElement {
    /*
    class: Toolbar_SwapButton
  
    description:
     This class defines a custom element that renders a "Swap Data"
     button within a shadow DOM and handles swapping of train and eval
     plots. It attaches styling, binds click events to swap data,
     layout, config, and legends of Plotly charts, logs the swap, and
     notifies other components via a custom stateChange event.
    */

  //****************************************************************************
  //
  // public methods: required methods such as constructors
  //
  //****************************************************************************
    
  constructor() {
    /*
    method: Toolbar_SwapButton::constructor

    args: None

    return: None

    description:
     Initializes the Toolbar_SwapButton component by attaching a shadow
     DOM and setting default states for `checked` and `isOpen`, which
     track the checkbox status and open/closed state of the button.
    */

    // call parent constructor
    //  
    super();

    // attach shadow DOM
    //  
    this.attachShadow({ mode: "open" });
  }

  //****************************************************************************
  //
  // lifecycle methods
  //
  //****************************************************************************
   
  connectedCallback() {
    /*
    method: Toolbar_SwapButton::connectedCallback

    args: None

    return: None

    description:
     This method is called when the Toolbar_DownloadButton element is
     added to the DOM. It triggers the rendering of the button and adds
     a click event listener to it. This lifecycle method ensures that
     the button is properly initialized when the component is inserted
     into the DOM.
    */

    // render the component
    //
    this.render();

    // add click listener
    //  
    this.addClickListener();
  }

  //****************************************************************************
  //
  // rendering methods
  //
  //****************************************************************************
    
  render() {
      /*
    method: Toolbar_SwapButton::render

    args: None

    return: None

    description:
     This method generates and inserts the HTML structure for the
     toolbar and popup into the shadow DOM. It includes styles for the
     popup, buttons, and overlay, as well as creating a form container 
     for setting the X and Y axis ranges. This method also sets up the
     visibility of the popup and its interactions, including closing the
     popup when the close button is clicked.
      */

    // retrieve button label from attributes
    //  
    const label = this.getAttribute("label") || "Swap Data";

    // define the HTML structure and CSS styles
    //
    this.shadowRoot.innerHTML = `
      <style>
        .toolbar-swap-button {
          background-color: white;
          color: black;
          font-family: 'Inter', sans-serif;
          font-weight: 100;
          font-size: 1em;
          padding: 5px 30px;
          border: none;
          cursor: pointer;
          min-width: 220px;
          white-space: nowrap;
          text-align: left;
        }

        .toolbar-swap-button:hover {
          background-color: #c9c9c9;
        }

        .toolbar-swap-button:disabled {
          cursor: not-allowed;
          background-color: #e0e0e0;
          color:rgb(0, 0, 0);
        }
      </style>

      <button class="toolbar-swap-button">${label}</button>
    `;
  }

  //****************************************************************************
  //
  // event handling methods
  //
  //****************************************************************************
    
  addClickListener() {
    /*
    method: Toolbar_SwapButton::addClickListener

    args: None

    return: None

    description:
     This method adds a click event listener to the button element. When
     the button is clicked, the method checks the value of the button's
     `label` attribute and dispatches a corresponding event to the
     EventBus to swap Train and Eval data. 
    */

    // get reference to button element in shadow DOM
    //  
    const button = this.shadowRoot.querySelector(".toolbar-swap-button");

    // function that updates the button state based on the presence of
    // train and eval data
    //
    const updateButtonState = () => {

      // get train and eval plot elements
      //	
      const trainPlot = document.querySelector('[plotid="train"]');
      const evalPlot  = document.querySelector('[plotid="eval"]');

      // check if train data exists and is non-empty
      //	
      const hasTrainData = Array.isArray(trainPlot?.plotData) &&
	                   trainPlot.plotData.length > 0;

      // check if eval data exists and is non-empty
      //
      const hasEvalData  = Array.isArray(evalPlot?.plotData) &&
                  	   evalPlot.plotData.length > 0;

      // enable button only if both train and eval data exist
      //	
      button.disabled = !(hasTrainData && hasEvalData);
    };

    // initial state update
    //
    updateButtonState();

    // re-check when other components signal data/state changes
    //
    EventBus.addEventListener("stateChange", updateButtonState);

    // only perform swap if button is enabled
    //
    button.addEventListener("click", () => {
      if (!button.disabled) {
        this.swapPlots();
      }
    });
  }
  swapPlots() {
    /*
    method: Toolbar_SwapButton::swapPlots

    args: None

    return: None

    description:
     Swaps the train and eval plot data, layout, config, and legends,
     redraws both graphs, and logs the swap event in the process log.
    */

    // get references to train plot, eval plot, and process log
    //  
    const trainPlot = document.querySelector('[plotid="train"]');
    const evalPlot  = document.querySelector('[plotid="eval"]');
    const log       = document.querySelector('process-log');

    // log error and exit if either plot is missing
    //  
    if (!trainPlot || !evalPlot) {
      console.error("Can't find train or eval plot to swap.");
      return;
    }

    // swap underlying data containers
    //  
    const tmpData = trainPlot.data;
    trainPlot.data = evalPlot.data;
    evalPlot.data = tmpData;

    // swap all plot traces
    //  
    const tmpTraces = trainPlot.plotData;
    trainPlot.plotData = evalPlot.plotData;
    evalPlot.plotData = tmpTraces;

    // clear decision boundaries
    // remove any layout shapes (colored regions)
    //  
    trainPlot.layout.shapes = [];
    evalPlot.layout.shapes = [];
      
    // remove non-scatter traces (e.g., contour/surface)
    //  
    const isDataTrace = t => t.mode && t.mode.includes('markers');
    trainPlot.plotData = trainPlot.plotData.filter(isDataTrace);
    evalPlot.plotData = evalPlot.plotData.filter(isDataTrace);

    // swap layout (axes, titles, etc.)
    //
    const tmpLayout = trainPlot.layout;
    trainPlot.layout = evalPlot.layout;
    evalPlot.layout = tmpLayout;

    // swap config
    //  
    const tmpConfig = trainPlot.config;
    trainPlot.config = evalPlot.config;
    evalPlot.config = tmpConfig;

    // re-render both plots
    //  
    const trainDiv = trainPlot.querySelector('#plot');
    const evalDiv = evalPlot.querySelector('#plot');
    Plotly.react(trainDiv, trainPlot.plotData, trainPlot.layout,
		trainPlot.config);
    Plotly.react(evalDiv, evalPlot.plotData, evalPlot.layout, evalPlot.config);

    // update legends
    //  
    trainPlot.clearLegend();
    trainPlot.createLegend();
    evalPlot.clearLegend();
    evalPlot.createLegend();

    // log event
    //  
    if (log?.writePlain) {
      log.writePlain("Swapped data; cleared decision boundaries.");
    }
    EventBus.dispatchEvent(new CustomEvent("stateChange"));
  }
}
//
// end of class

//-----------------------------------------------------------------------------
//
// IMLD Tools Class: Toolbar_Normalize
//
//-----------------------------------------------------------------------------

class Toolbar_Normalize extends HTMLElement {
  /*
  method: AlgoTool::constructor

  args: None

  return:
   Toolbar_Normalize instance

  description:
   This class creates a toolbar checkbox button for enabling or
   disabling normalization in a plot. It manages its own state (checked
   and isOpen) and renders the checkbox inside a shadow DOM. The
   component listens for clicks on the button to toggle its state and
   dispatches a custom setNormalize event with the current status. It
   also listens for clicks outside the component to update the isOpen
   state, ensuring proper interaction behavior.
  */

  //****************************************************************************
  //
  // public methods: required methods such as constructors
  //
  //****************************************************************************
    
  constructor() {
    /*
    method: DrawCheckBox::constructor

    args: None

    return:
     DrawCheckBox instance

    description:
     Initializes the DrawCheckBox component by attaching a shadow DOM
     and setting default states for `checked` and `isOpen`, which track
     the checkbox status and open/closed state of the button.
    */

    // call the parent constructor
    //
    super();

    // create a shadow root for the component
    //
    this.attachShadow({ mode: "open" });

    // create variable to hold state of checkbox and button
    //
    this.checked = false;
    this.isOpen = false;
  }
  //
  // end of method

  //****************************************************************************
  //
  // lifecycle methods
  //
  //****************************************************************************
    
  connectedCallback() {
    /*
    method: DrawCheckBox::connectedCallback

    args: None

    return: None

    description:
     Called automatically when the custom element is added to the DOM.
     It initializes the checkbox by rendering its initial state and
     registers a global click listener to handle outside interactions.
    */

    // render the initial state
    //
    this.render();

    // add global click listener
    //
    document.addEventListener("click", this.handleDocumentClick.bind(this));
  }
  //
  // end of method

  disconnectedCallback() {
    /*
    method: DrawCheckBox::disconnectedCallback

    args: None

    return: None

    description:
     Lifecycle method called when the component is removed from the DOM.
     Cleans up the global event listener to prevent memory leaks.
    */

    // clean up and remove the global event listener
    //
    document.removeEventListener("click", this.handleDocumentClick.bind(this)); 
  }
  //
  // end of method

  setNormalize(status) {
    /*
    method: DrawCheckBox::setNormalize

    args: None

    return: None

    description:
     Updates the component's checked state and open state based on the
     provided status. Dispathes a "setNormalize" event with the current
     status for other parts of the application to react to. 
    */

    // update internal checked state, actual checkbox element, and open state
    //  
    this.checked = status;
    this.shadowRoot.querySelector("#checkbox").checked = status;
    this.isOpen = status;

    // notify listeners that normalization status has changed
    //  
    EventBus.dispatchEvent(
      new CustomEvent("setNormalize", {
        detail: {
          status: status,
        },
      })
    );
  }
  
  render() {
    /*
    method: DrawCheckBox::render

    args: None

    return: None

    description:
     Generates the HTMl and CSS for the component inside its shadow DOM.
     Sets up the checkbox button and attaches a click handler that toggles
     the component's state using setNormalize
     */

    // retrieve button label froma attributes
    //  
    const label = this.getAttribute("label") || "Button";

    // define the HTML structure and CSS styles
    //
    this.shadowRoot.innerHTML = `
      <style>
        .toolbar-checkbox-button {
          background-color: white;
          color: black;
          font-family: 'Inter', sans-serif;
          font-weight: 100;
          font-size: 1em;
          padding: 5px 0;
          border: none;
          cursor: pointer;
          min-width: 220px;
          white-space: nowrap;
          text-align: left;
          display: flex;
          align-items: center;
        }

        .toolbar-checkbox-button:hover {
          background-color: #c9c9c9;
        }

        input[type="checkbox"] {
          margin-right: 7px;
          margin-left: 10px;
        }
      </style>

      <button class="toolbar-checkbox-button" id="checkboxButton">
        <input type="checkbox" id="checkbox" />
        ${label}
      </button>
    `;

    // get references to the button element in the shadow DOM
    //  
    const button = this.shadowRoot.querySelector("#checkboxButton");

    button.onclick = (event) => {

      // prevent event from bubbling up
      //
      event.stopPropagation();

      // toggle normalization state, set it to opposite of checked value
      //	
      this.setNormalize(!this.checked);
    };
  }

  //****************************************************************************
  //
  // event handling methods
  //
  //****************************************************************************
    
  handleDocumentClick(event) {
    /*
    method: DrawCheckBox::handleDocumentClick

    args:
     event (Event): The global click event to evaluate

    return: None

    description:
     Handles global document clicks to determine if the user clicked
     outside the component. If so, and the component is open, it updates
     internal state to reflect the closed state.
    */

    // get reference to button element in shadow DOM
    //  
    const button = this.shadowRoot.querySelector("#checkboxButton");

    // check if the clicked target is outside of the button
    //
    if (this.isOpen && !button.contains(event.target)) {
	  
      this.isOpen = false; // close the button
	
      // optionally, reset checkbox state if needed
      // this.checked = false;
      // this.shadowRoot.querySelector('#checkbox').checked = this.checked;
      // Update checkbox state
    }
  }
  //
  // end of method
}
//
// end of class

//*****************************************************************************
//
// Section 3: Action and Download Button Components
//
//*****************************************************************************

//-----------------------------------------------------------------------------
//
// IMLD Tools Class: Toolbar_DownloadButton
//
//-----------------------------------------------------------------------------

class Toolbar_DownloadButton extends HTMLElement {
  /*
  class: Toolbar_DownloadButton 

  description:
   This class represents a custom toolbar download button element. It
   allows for creating a button with a label and some basic styling. It
   also provides the functionality of adding an event listener to the
   button so that when clicked, it can dispatch a custom event to the
   window, to download a file.
  */

  //****************************************************************************
  //
  // public methods: required methods such as constructors
  //
  //****************************************************************************
    
  constructor() {
    /*
    method: Toolbar_DownloadButton::constructor

    args: None

    return:
     Toolbar_DownloadButton instance

    description:
     This is the constructor for the Toolbar_DownloadButton class. It is
     called when a new instance of the class is created. The constructor
     attaches a shadow DOM to the element with the "open" mode, which
     allows styling and structure to be encapsulated within the
     component.
    */

    // call the parent constructor
    //
    super();

    // create a shadow root for the component
    //
    this.attachShadow({ mode: "open" });
  }
  //
  // end of method

  //****************************************************************************
  //
  // lifecycle methods
  //
  //****************************************************************************
    
  connectedCallback() {
    /*
    method: Toolbar_DownloadButton::connectedCallback

    args: None

    return: None

    description:
     This method is called when the Toolbar_DownloadButton element is
     added to the DOM. It triggers the rendering of the button and adds
     a click event listener to it. This lifecycle method ensures that
     the button is properly initialized when the component is inserted
     into the DOM.
    */

    // render the component
    //
    this.render();

    // add click listener
    //  
    this.addClickListener();
  }
  //
  // end of method

  //****************************************************************************
  //
  // rendering methods
  //
  //****************************************************************************
    
  render() {
    /*
    method: Toolbar_DownloadButton::render

    args: None

    return: None

    description:
     This method is responsible for rendering the button element in the
     shadow DOM. It defines the button's HTML structure, style, and the
     label that appears on the button. The label is fetched from the
     "label" attribute of the element, with a fallback value of "Button"
     if the attribute is not provided.
    */


    // get label from attribute
    //  
    const label = this.getAttribute("label") || "Button"; 

    // define the HTML structure and CSS styles
    //
    this.shadowRoot.innerHTML = `
      <style>
        .toolbar-button {
          background-color: white;
          color: black;
          font-family: 'Inter', sans-serif;
          font-weight: 100;
          font-size: 1em;
          padding: 5px 30px;
          border: none;
          cursor: pointer;
          min-width: 220px;
          white-space: nowrap;
          text-align: left;
        }

        .toolbar-button:hover {
          background-color: #c9c9c9;
        }

      </style>

      <button class="toolbar-button">${label}</button>
    `;
  }

  //****************************************************************************
  //
  // event handling methods
  //
  //****************************************************************************
    
  addClickListener() {
    /*
    method: Toolbar_DownloadButton::addClickListener

    args: None

    return: None

    description:
     This method adds a click event listener to the button. When the
     button is clicked, it dispatches a custom event to download a file
     to the user's downloads.
    */

    // get the button element from the shadow DOM
    //
    const button = this.shadowRoot.querySelector(".toolbar-button");

    // get the download type attribute value for conditional logic
    //
    const download_type = this.getAttribute("download");

    // add an event listener to handle the button click event
    //
    button.addEventListener("click", () => {
	  
      // send a custom event to the window which will 
      // download a file based on the download type atrribute
      //
      EventBus.dispatchEvent(
        new CustomEvent("download", {
          detail: {
            type: download_type
          },
        })
      );
    });
  }
  //
  // end of method
}
//
// end of class

//*****************************************************************************
//
// Section 4: Custom Element Registration
//
//*****************************************************************************

// register the custom elements
//
customElements.define("toolbar-button", Toolbar_Button);
customElements.define("toolbar-checkbox-button", Toolbar_CheckboxButton);
customElements.define("toolbar-dropdown-clear", Toolbar_DropdownClear);
customElements.define("toolbar-dropdown-settings", Toolbar_DropdownSettings);
customElements.define("toolbar-openfile-button", Toolbar_OpenFileButton);
customElements.define("toolbar-savefile-button", Toolbar_SaveFileButton);
customElements.define("toolbar-popup-button", Toolbar_PopupButton);
customElements.define("toolbar-set-gaussian", Toolbar_SetGaussian);
customElements.define("toolbar-set-ranges", Toolbar_SetRanges);
customElements.define("toolbar-swap-button", Toolbar_SwapButton);
customElements.define("toolbar-normalize", Toolbar_Normalize);
customElements.define("toolbar-download-button", Toolbar_DownloadButton);

//
// end of file
