// file: $(NEDC_NFC)/util/python/nedc_imld/v5.0.1/app/static/components/ProcessL
//       og.js
//
// Defines the ProcessLog custom web component, which provides a
// scrollable, styled logging interface for displaying process outputs,
// parameters, errors, and metrics in real time. The component
// encapsulates its own structure and styling using Shadow DOM, offering
// modularity and reusability in web apps. Utility methods are included
// for writing plain text, headers, errors, parameters, matrices, and
// metrics to the log container.
//

// Revision History:
//
// 20251010 (SA): refactored code to fit ISIP standards
//

//******************************************************************************
//
// Section 1: Logging Components
//
//******************************************************************************

//------------------------------------------------------------------------------
//
// IMLD Tools Class: ProcessLog
//
//------------------------------------------------------------------------------

class ProcessLog extends HTMLElement {
  /*
  class: ProcessLog

  description:
   This class is designed to provide a customizable logging interface for
   displaying process and analysis results in a scrollable container. It
   extends the HTMLElement class and creates a shadow root to encapsulate
   its styles and structure. To create a new component, copy this template
   and adjust the class and file names to match the desired component.
   Also, change the custom element name at the end of this file.
   Additional methods and properties may be added as needed.
  */

  //****************************************************************************
  //
  // public methods: required methods such as constructors
  //
  //****************************************************************************
    
  constructor() {
    /*
    method: ProcessLog::constructor

    args: None

    return:
     ProcessLog instance

    description:
     This is the constructor for the ProcessLog class. It initializes the
     component and creates a shadow root. It gets the HTML and CSS for the
     component that should be in the same directory as this file.
    */

    // call the parent constructor (HTMLElement)
    //
    super();

    // create a shadow root for the component
    //
    this.attachShadow({ mode: "open" });

    // get the name of the class
    //
    this.name = this.constructor.name;
  }
  //
  // end of method

  //****************************************************************************
  //
  // lifecycle methods
  //
  //****************************************************************************
  
  async connectedCallback() {
    /*
    method: ProcessLog::connectedCallback

    args: None

    return: None

    description:
     This method is called when the component is added to the DOM.
    */

    // render the component to the webpage
    //
    this.render();
  }
  //
  // end of method

  //****************************************************************************
  //
  // rendering methods
  //
  //****************************************************************************
    
  render() {
    /*
    method: ProcessLog::render

    args: None

    return: None

    description:
     This method renders the component to the webpage by setting the
     innerHTML of the shadow root to what is in the string below.
    */

    // write HTML and CSS structure for the ProcessLog component
    //
    this.shadowRoot.innerHTML = `

      <style>
        /* Styling for the scrollable background area */
        .scroll-bg {
          display: block;
          width: 100%;
          height: 100%;
          margin-bottom: 2%;
          margin-left: 0%;
          box-sizing: border-box;
        }

        /* Main scrollable div container styling */
        .scroll-div {
          width: 100%;
          height: 19vh;
          width: 60vw;
          background: white;
          overflow-y: auto;
        }

        /* Content inside the scrollable div styling */
        .scroll-object {
          font-family: 'Inter', sans-serif;
          font-size: 1em;
          padding-right: 0.7em;
        }

        .scroll-object b {
          font-size: 1em;
        }

	.scroll-object h1 {
          margin-top: 0.7em;
          margin-bottom: 0.7em;
        }
	.scroll-object h2 {
          margin-top: 0.5em;
          margin-bottom: 0.5em;
        }
	.scroll-object h3 {
          margin-top: 0.3em;
          margin-bottom: 0.3em
        }

        /* WebKit Browsers (Chrome, Safari) Custom Scrollbar */
        .scroll-div::-webkit-scrollbar {
          width: 0.7em;
        }
        .scroll-div::-webkit-scrollbar {
          background: #c9c9c9;
          border-radius: 100vw;
        }
        .scroll-div::-webkit-scrollbar-thumb {
          background: #7441BA;
          border-radius: 100vw;
        }
        .scroll-div::-webkit-scrollbar-thumb:hover {
          background: #512e82;
          border-radius: 100vw;
        } 

      </style>

      <!-- Structure for log content within a scrollable div -->
      <div>
        <div class="scroll-bg">
          <div class="scroll-div">
            <div class="scroll-object"></div>
          </div>
        </div>
      </div>
    `;
  }
  //
  // end of method

  //****************************************************************************
  //
  // basic log methods
  //
  //****************************************************************************
    
  clear() {
    /*
    method: Toolbar_Button::clear

    args: None

    return: None

    description:
     This method clears the contents of the log container by setting its
     innerHTML to an empty string.
    */

    // get the log container element
    //
    let logObject = this.shadowRoot.querySelector(".scroll-object");

    // clear the content of the log container by setting innerHTML to an
    // empty string
    //
    logObject.innerHTML = "";
  }
  //
  // end of method

  writePlain(log) {
    /*
    method: ProcessLog::writePlain

    args:
     log (String): the log message to be written to the log container

    return: None

    description:
     This method writes a log message to the log container in plain text.
    */

    // get the log object
    //
    let logObject = this.shadowRoot.querySelector(".scroll-object");

    // append the log message to the log container
    //
    logObject.innerHTML += log + "<br>";

    // scroll to bottom of log container; this is the scroll container
    //
    let logDiv = this.shadowRoot.querySelector(".scroll-div"); 
    logDiv.scrollTop = logDiv.scrollHeight;
  }
  //
  // end of method

  writeError(msg) {
    /*
    method: ProcessLog::writeError

    args:
     msg (String): the error message to be written to the log container

    return: None

    description:
     Write an error message to the log container in red text and different
     spacing to make it stand out from other log messages.
    */

    // get the log object
    //
    let logObject = this.shadowRoot.querySelector(".scroll-object");

    // append the log message to the log container
    //
    logObject.innerHTML += `<br><span style="color:red">${msg}</span><br>`;

    // scroll to the bottom of the log container
    //
    let logDiv = this.shadowRoot.querySelector(".scroll-div");
    logDiv.scrollTop = logDiv.scrollHeight;
  }
  //
  // end of method

  writeHeader(txt, type = "h1") {
    /*
    method: ProcessLog::writeHeader

    args:
     txt (String): the text to be written to the log container as a header
     type (String): the type of header to write (h1, h2, h3, ...)
     [default = 'h1']

    return: None

    description:
     This method writes a header message to the log container.
    */

    // get the log object
    //
    let logObject = this.shadowRoot.querySelector(".scroll-object");

    // append the header message to the log container
    //
    logObject.innerHTML += `<${type}>${txt}</${type}>`;

    // scroll to bottom of log container; this is the scroll container
    //
    let logDiv = this.shadowRoot.querySelector(".scroll-div"); 
    logDiv.scrollTop = logDiv.scrollHeight;
  }
  //
  // end of method

  //****************************************************************************
  //
  // parameter logging methods
  //
  //****************************************************************************
    
  parseMatrix(inputString) {
    /*
    method: ProcessLog::parseMatrix

    args:
     inputString (String): The matrix string to be parsed into a 2D array.

    return:
     (Array): A 2D array representation of the input matrix string.

    description:
     This function extracts and converts a string-formatted matrix into
     an actual JavaScript 2D array.
    */

    // return 2D array representing input matrix string
    //  
    return inputString
      .slice(2, -2) // remove the outer brackets
      .split("],[") // split by the inner array separator
      .map((row) => row.split(",").map(Number)); // convert each row into
      // an array of numbers
  }
  //
  // end of method

  writeDataParams(paramValues, param_names) {
    /*
    method: ProcessLog::writeDataParams

    args:
     paramValues (Array): The values of the parameters to be logged.
     param_names (Array): The names of the parameters corresponding to
                          the values.
     type (String): The type of parameters being processed.

    return: None

    description:
     This method iterates over parameter names and values, formatting and
     writing them to the process log. If a parameter represents a matrix,
     it is parsed and logged row by row.
    */

    // counter to keep track of class numbering
    //
    var class_index = 0;

    param_names.forEach((name, index) => {
      let class_padding; // formatting class label
      let name_padding; // formatting parameter name

      // treat parameter name as single-value entry if it has "Number"
      //
      if (name.includes("Number")) {
        this.writeSingleValue(
          `Class ${class_index}`, // label for class
          `${name}: ${paramValues[index]}` // parameter name and value
        );
        class_index = class_index + 1; // increment class counter
      }
      else {

	// else, prepare padded class label for display in white text
	//
        class_padding = `<span style="color: white; user-select: none;">
                        Class ${class_index}:</span>`;

	// parses matrix string into array if parameter value is matrix
	//
        if (paramValues[index].includes("[")) {
          var matrix = this.parseMatrix(paramValues[index]); 
          this.writePlain(`${class_padding}${name}: [${matrix[0]}]`);

	  // writes remaining rows with name padding
	  //
          for (let i = 1; i < matrix.length; i++) {
            name_padding = `<span style="color: white; user-select:
                           none;">${name}:</span>`;
            this.writePlain(`${class_padding}${name_padding} [${matrix[i]}]`);
          }
        }
	else {

	  // for non-matrix values, prints with class label
	  //
          this.writePlain(`${class_padding}${name}: ${paramValues[index]}`);
        }
      }
    });
  }
  //
  // end of method

  writeAlgorithmParams(paramValues, param_names) {
    /*
    method: ProcessLog::writeAlgorithmParams

    args:
     paramValues (Array): The values of the algorithm parameters to be
                          logged.
     param_names (Array): The names of the corresponding algorithm
			  parameters.

    return: None

    description:
     This method iterates over the provided parameter names and values,
     formatting them for logging. If a parameter contains a matrix
     (identified by the presence of square brackets), the matrix is parsed
     and logged row by row. Otherwise, the parameter is logged as a
     single value. Each parameter name is prefixed with three
     non-breaking spaces for indentation.
    */

    // iterate over parameter names and their corresponding values
    //  
    param_names.forEach((name, index) => {

      // if value is 2D array, format it as a matrix
      //	
      if (
        Array.isArray(paramValues[index]) &&
        paramValues[index].every((row) => Array.isArray(row))
      ) {

	// parse 2D array into matrix format
	//  
        var matrix = this.parseMatrix(JSON.stringify(paramValues[index]));

	// write first row of matrix
	//  
        this.writeSingleValue(`&nbsp;&nbsp;&nbsp;${name}`, `[${matrix[0]}]`);

	// write remaining rows with padded name for alignment
	//  
        for (let i = 1; i < matrix.length; i++) {
          name_padding = `<span style="color: white; user-select:
                         none;">${name}:<span>`;
          this.writeSingleValue(
            `&nbsp;&nbsp;&nbsp;${name_padding}`,
            `[${matrix[i]}]`
          );
        }
      }

      // else, write value directly
      //	
      else {
        this.writeSingleValue(`&nbsp;&nbsp;&nbsp;${name}`, paramValues[index]);
      }
    });
  }
  //
  // end of method

  writeEstimatedParams(logOutput) {
    /*
    method: ProcessLog::writeEstimatedParams

    args:
     logOutput (String): The formatted log output returned from the
                         Python function.

    return: None

    description:
     This method processes the formatted string from Python, converts new
     lines to HTML line breaks, bolds all words before ":", and writes it
     to the log container.
    */

    // convert new lines to <br> for HTML display
    //
    logOutput = logOutput.replace(/\n/g, "<br>");

    // bold all words before ":"
    //
    logOutput = logOutput.replace(/(^|\b)([A-Za-z0-9\s]+?):/g, "<b>$2:</b>");

    // add indentation for covariance rows that start with a bracket
    //
    logOutput = logOutput.replace(
      /<br>\s*\[/g,
      '<br><span style="color: white; user-select: none;">Covariancess:</span>['
    );

    // get the log object
    //
    let logObject = this.shadowRoot.querySelector(".scroll-object");

    // append the log message to the log container
    //
    logObject.innerHTML += logOutput;

    // scroll to bottom of log container; this is the scroll container
    //
    let logDiv = this.shadowRoot.querySelector(".scroll-div"); 
    logDiv.scrollTop = logDiv.scrollHeight;
  }
  //
  // end of method

  //****************************************************************************
  //
  // structured writing methods
  //
  //****************************************************************************
    
  writeSingleValue(label, value) {
    /*
    method: ProcessLog::writeSingleValue

    args:
     label (String): The label for the value being logged.
     value (String): The value to be logged.

    return: None

    description:
     This method writes a labeled value to the process log, formatting it 
     for readability.
    */

    // get the log object
    //
    let logObject = this.shadowRoot.querySelector(".scroll-object");

    // append new log entry as HTML, showing label in bold and its value
    //  
    logObject.innerHTML += `
      <div>
        <b>${label}:</b> ${value}
      </div>
    `;

    // scroll to bottom of log container; this is the scroll container
    //
    let logDiv = this.shadowRoot.querySelector(".scroll-div"); 
    logDiv.scrollTop = logDiv.scrollHeight;
  }
  //
  // end of method

  writeMetrics(label, metrics) {
    /*
    method: ProcessLog::writeMetrics

    args:
     metrics (Object): the metrics message to be written to the log
                       container, with the key as the metric name and the
                       value as the metric value

    return: None

    description:
     This method writes a metrics message to the log container.
    */

    // get the log object
    //
    let logObject = this.shadowRoot.querySelector(".scroll-object");

    // write a metrics header
    //
    this.writeHeader(`<br>Performance: ${label}`, "h3");

    // iterate over each metric in the log and write it to the process log
    //
    Object.keys(metrics).forEach((key) => {
      if (key != "Confusion Matrix") {
        logObject.innerHTML += `
          <div>
            <b>&nbsp;&nbsp;&nbsp;${key}:</b> ${metrics[key].toFixed(2)}%
          </div>
        `;
      }
    });

    // scroll to bottom of log container; this is the scroll container
    //
    let logDiv = this.shadowRoot.querySelector(".scroll-div"); 
    logDiv.scrollTop = logDiv.scrollHeight;
  }
  //
  // end of method

  //****************************************************************************
  //
  // utility methods methods
  //
  //****************************************************************************
    
  addFullWidthSeparator() {
    /*
    method: ProcessLog::addFullWidthSeparator

    args: None

    return: None

    description:
     This method generates a full-width separator made up of dashes and
     appends it to the log container. The separator's width is calculated
     based on the current width of the log container, ensuring that it
     spans the entire width of the container. The separator is created by
     determining how many dashes fit within the container's width, and
     the appropriate number of dashes is appended to the log.
    */

    // log container
    //  
    let logDiv = this.shadowRoot.querySelector(".scroll-object"); 

    // exit if logDiv does not exist
    //  
    if (!logDiv) return;

    // get width of log container in pixels
    //  
    let logWidth = logDiv.clientWidth; 

    // approximate width of dash (-) in pixels (depends on font)
    //  
    let charWidth = 5.5; 

    // calculate how many dashes fit
    //      
    let dashCount = Math.floor(logWidth / charWidth); 

    // generate separator line
    //  
    let separator = "-".repeat(dashCount); 

    // add the separator to the log
    //
    logDiv.innerHTML += `<div>${separator}</div>`;
  }
  //
  // end of method
}
//
// end of class

//******************************************************************************
//
// Section 2: Custom Element Registration
//
//******************************************************************************

// register the custom element
//
customElements.define("process-log", ProcessLog);

//
// end of file
