#!/usr/bin/env python
#
# file: $NEDC_NFC/util/python/nedc_imld/v5.0.1/app/extensions/base.py
#
# revision history:
#
# 20250925 (SA): refactored code to fit ISIP standards
#
# This class encapsulates functions that call on ML Tools for the IMLD app.
#-------------------------------------------------------------------------------

# import system modules
#
from flask import Flask, request
import os
from werkzeug.middleware.proxy_fix import ProxyFix

# import nedc modules
#
import nedc_debug_tools as ndt

#-------------------------------------------------------------------------------
#
# global variables are listed here
#
#-------------------------------------------------------------------------------

# set the filename using basename
#
__FILE__ = os.path.basename(__file__)

# directory names
#
DIR_NAME_BACKEND = 'backend'
DIR_NAME_TEMPLATES = 'templates'
DIR_NAME_STATIC = 'static'

# scheduler API config setup
#
SCHEDULER_API_ENABLED = True

# profyfix config settings
#
PROXY_FIX_X_PROTO = 1
PROXY_FIX_X_HOST = 1

# declare a global debug object so we can use it in functions
#
dbgl = ndt.Dbgl()

#-------------------------------------------------------------------------------
#
# classes are listed here
#
#-------------------------------------------------------------------------------

class Config():
    """
    class: PCA

    description:
     This class defines configuration settings for the IMLD Flask app, including
     paths for static files, templates, logging, and backend components.
    """

    def __init__(self, root) -> None:
        """
        method: constructor

        arguments:
         root (str): Path to the root directory of the project.

        return: None

        description:
         Initializes file paths and application settings based on the provided
         root.
        """

        # absolute path to application root
        #
        self.APP = os.path.abspath(os.path.dirname(root))

        # path to backend, templates, and static directory
        #
        self.BACKEND = os.path.join(self.APP, DIR_NAME_BACKEND)
        self.TEMPLATES = os.path.join(self.APP, DIR_NAME_TEMPLATES)
        self.STATIC = os.path.join(self.APP, DIR_NAME_STATIC)
    #
    # end of method
#
# end of class

class App(Flask):
    """
    class: App

    description:
     This class extends the Flask app to include support for reverse proxying
     and dynamic configuration setup through the `set_root` method.
    """

    def __init__(self):
        """
        method: constructor

        arguments: None

        return: None

        description:
         Initializes the Flask app and applies the proxy fix middleware.
        """

        # Initialize the Flask app
        #
        super().__init__(__name__)
        
        # add the proxy fix to the app
        #
        self.wsgi_app = ProxyFix(self.wsgi_app, x_proto=PROXY_FIX_X_PROTO, x_host=PROXY_FIX_X_HOST)
    #
    # end of method

    def set_root(self, root):
        """
        method: set_root

        arguments:
         root (str): The root directory for setting up app paths.

        return: None

        description:
         Creates and applies a configuration object based on the given root 
         path.
        """

        # create the configuration
        #
        config = Config(root)

        # add the configuration to the app
        #
        self.config.from_object(config)

        # Set the templates folder
        #
        self.template_folder = config.TEMPLATES

        # set static folder for CSS, JS, and images
        #
        self.static_folder = config.STATIC
    #
    # end of method
#
# end of class

#-------------------------------------------------------------------------------
#
# global variables dependent on classes are listed here
#
#-------------------------------------------------------------------------------

# global instance of Flask app class
#
app = App()

#
# end of file
